import json
import logging
import logging.config
import paho.mqtt.client as mqtt
import toml

from data_storage import DataStorage


class SubscriberClass:
    def __init__(self, environment):
        #   start: define logging

        if environment == 'PROD':
            with open('/home/admn/python-scripts/ixSmartHome/config/subscriber_logging_PROD.conf') as logging_config:
                log_cfg = logging_config.read()
        else:
            with open('config/subscriber_logging_DEV.conf') as logging_config:
                log_cfg = logging_config.read()

        logging.config.dictConfig(json.loads(log_cfg))
        self.logger = logging.getLogger('mqtt_logger')
        #   end: define logging

        if environment == 'PROD':
            config = toml.load('/home/admn/python-scripts/ixSmartHome/config/subscriber_config.toml')
        else:
            config = toml.load(f'./config/subscriber_config.toml')

        self.logger.info('configuration read')

        # MQTT setup
        client_id = config['MQTT_SETUP'][environment]['client_id']
        host = config['MQTT_SETUP'][environment]['host']
        port = config['MQTT_SETUP'][environment]['port']
        keepalive = config['MQTT_SETUP'][environment]['keepalive']
        username = config['MQTT_SETUP'][environment]['username']
        password = config['MQTT_SETUP'][environment]['password']
        self.topic = config['MQTT_SETUP'][environment]['topic']
        self.logger.info('variables set')

        try:
            self.client = mqtt.Client(client_id)  # Create instance of client with client id
            self.client.username_pw_set(username=username, password=password)
            self.client.on_connect = self.on_connect  # Define callback function for successful connection
            self.client.on_message = self.on_message  # Define callback function for receipt of a message
            self.client.connect(host, port, keepalive)  # Connect to (broker, port, keepalive-time)
            self.logger.info('MQTT client set')
        except Exception as e:
            self.logger.error(f'Could not connect to MQTT host: {host}:{port} / {e}')

        try:
            self.data_storer = DataStorage(self.logger, environment)
            self.logger.info('system completely set up')
        except Exception as e:
            self.logger.error(f'Could not create storage object! {e}')


    # The callback for when the client connects to the broker
    def on_connect(self, client, userdata, flags, rc):
        self.client.subscribe(self.topic)


    # The callback for when a PUBLISH message is received from the server.
    def on_message(self, client, userdata, msg):
        self.data_storer.insert_data(msg.payload)


    def run(self):
        self.client.loop_forever()
