/*
 * Project: aesdroid - AES implementation for Android
 * File   : AesEncryptionByteBased.java
 * Author : Oliver Mueller <oliver@cogito-ergo-sum.org>
 * Purpose: Implementation of class AesEncryption based upon byte[] arrays.
 *
 * $Id: AesEncryptionByteBased.java 39 2012-06-24 15:32:17Z oliver $
 *
 * Copyright (c) 2011, 2012 Oliver Mueller.
 * All rights reserved.
 * http://www.cogito-ergo-sum.org
 * http://oliver-mueller.com
 *
 * This software is provided 'as-is', without any express or implied warranty.
 * In no event will the author be held liable for any damages arising from
 * the use of this software.
 *
 */

package org.cogito_ergo_sum.aes;

public class AesEncryptionByteBased {
    // Substitution box
    private static final byte[] sbox = {
            (byte) 0x63, (byte) 0x7c, (byte) 0x77, (byte) 0x7b, (byte) 0xf2, (byte) 0x6b, (byte) 0x6f, (byte) 0xc5,
            (byte) 0x30, (byte) 0x01, (byte) 0x67, (byte) 0x2b, (byte) 0xfe, (byte) 0xd7, (byte) 0xab, (byte) 0x76,
            (byte) 0xca, (byte) 0x82, (byte) 0xc9, (byte) 0x7d, (byte) 0xfa, (byte) 0x59, (byte) 0x47, (byte) 0xf0,
            (byte) 0xad, (byte) 0xd4, (byte) 0xa2, (byte) 0xaf, (byte) 0x9c, (byte) 0xa4, (byte) 0x72, (byte) 0xc0,
            (byte) 0xb7, (byte) 0xfd, (byte) 0x93, (byte) 0x26, (byte) 0x36, (byte) 0x3f, (byte) 0xf7, (byte) 0xcc,
            (byte) 0x34, (byte) 0xa5, (byte) 0xe5, (byte) 0xf1, (byte) 0x71, (byte) 0xd8, (byte) 0x31, (byte) 0x15,
            (byte) 0x04, (byte) 0xc7, (byte) 0x23, (byte) 0xc3, (byte) 0x18, (byte) 0x96, (byte) 0x05, (byte) 0x9a,
            (byte) 0x07, (byte) 0x12, (byte) 0x80, (byte) 0xe2, (byte) 0xeb, (byte) 0x27, (byte) 0xb2, (byte) 0x75,
            (byte) 0x09, (byte) 0x83, (byte) 0x2c, (byte) 0x1a, (byte) 0x1b, (byte) 0x6e, (byte) 0x5a, (byte) 0xa0,
            (byte) 0x52, (byte) 0x3b, (byte) 0xd6, (byte) 0xb3, (byte) 0x29, (byte) 0xe3, (byte) 0x2f, (byte) 0x84,
            (byte) 0x53, (byte) 0xd1, (byte) 0x00, (byte) 0xed, (byte) 0x20, (byte) 0xfc, (byte) 0xb1, (byte) 0x5b,
            (byte) 0x6a, (byte) 0xcb, (byte) 0xbe, (byte) 0x39, (byte) 0x4a, (byte) 0x4c, (byte) 0x58, (byte) 0xcf,
            (byte) 0xd0, (byte) 0xef, (byte) 0xaa, (byte) 0xfb, (byte) 0x43, (byte) 0x4d, (byte) 0x33, (byte) 0x85,
            (byte) 0x45, (byte) 0xf9, (byte) 0x02, (byte) 0x7f, (byte) 0x50, (byte) 0x3c, (byte) 0x9f, (byte) 0xa8,
            (byte) 0x51, (byte) 0xa3, (byte) 0x40, (byte) 0x8f, (byte) 0x92, (byte) 0x9d, (byte) 0x38, (byte) 0xf5,
            (byte) 0xbc, (byte) 0xb6, (byte) 0xda, (byte) 0x21, (byte) 0x10, (byte) 0xff, (byte) 0xf3, (byte) 0xd2,
            (byte) 0xcd, (byte) 0x0c, (byte) 0x13, (byte) 0xec, (byte) 0x5f, (byte) 0x97, (byte) 0x44, (byte) 0x17,
            (byte) 0xc4, (byte) 0xa7, (byte) 0x7e, (byte) 0x3d, (byte) 0x64, (byte) 0x5d, (byte) 0x19, (byte) 0x73,
            (byte) 0x60, (byte) 0x81, (byte) 0x4f, (byte) 0xdc, (byte) 0x22, (byte) 0x2a, (byte) 0x90, (byte) 0x88,
            (byte) 0x46, (byte) 0xee, (byte) 0xb8, (byte) 0x14, (byte) 0xde, (byte) 0x5e, (byte) 0x0b, (byte) 0xdb,
            (byte) 0xe0, (byte) 0x32, (byte) 0x3a, (byte) 0x0a, (byte) 0x49, (byte) 0x06, (byte) 0x24, (byte) 0x5c,
            (byte) 0xc2, (byte) 0xd3, (byte) 0xac, (byte) 0x62, (byte) 0x91, (byte) 0x95, (byte) 0xe4, (byte) 0x79,
            (byte) 0xe7, (byte) 0xc8, (byte) 0x37, (byte) 0x6d, (byte) 0x8d, (byte) 0xd5, (byte) 0x4e, (byte) 0xa9,
            (byte) 0x6c, (byte) 0x56, (byte) 0xf4, (byte) 0xea, (byte) 0x65, (byte) 0x7a, (byte) 0xae, (byte) 0x08,
            (byte) 0xba, (byte) 0x78, (byte) 0x25, (byte) 0x2e, (byte) 0x1c, (byte) 0xa6, (byte) 0xb4, (byte) 0xc6,
            (byte) 0xe8, (byte) 0xdd, (byte) 0x74, (byte) 0x1f, (byte) 0x4b, (byte) 0xbd, (byte) 0x8b, (byte) 0x8a,
            (byte) 0x70, (byte) 0x3e, (byte) 0xb5, (byte) 0x66, (byte) 0x48, (byte) 0x03, (byte) 0xf6, (byte) 0x0e,
            (byte) 0x61, (byte) 0x35, (byte) 0x57, (byte) 0xb9, (byte) 0x86, (byte) 0xc1, (byte) 0x1d, (byte) 0x9e,
            (byte) 0xe1, (byte) 0xf8, (byte) 0x98, (byte) 0x11, (byte) 0x69, (byte) 0xd9, (byte) 0x8e, (byte) 0x94,
            (byte) 0x9b, (byte) 0x1e, (byte) 0x87, (byte) 0xe9, (byte) 0xce, (byte) 0x55, (byte) 0x28, (byte) 0xdf,
            (byte) 0x8c, (byte) 0xa1, (byte) 0x89, (byte) 0x0d, (byte) 0xbf, (byte) 0xe6, (byte) 0x42, (byte) 0x68,
            (byte) 0x41, (byte) 0x99, (byte) 0x2d, (byte) 0x0f, (byte) 0xb0, (byte) 0x54, (byte) 0xbb, (byte) 0x16
    };

    // Inverse substitution box
    private static final byte[] sboxinv = {
            (byte) 0x52, (byte) 0x09, (byte) 0x6a, (byte) 0xd5, (byte) 0x30, (byte) 0x36, (byte) 0xa5, (byte) 0x38,
            (byte) 0xbf, (byte) 0x40, (byte) 0xa3, (byte) 0x9e, (byte) 0x81, (byte) 0xf3, (byte) 0xd7, (byte) 0xfb,
            (byte) 0x7c, (byte) 0xe3, (byte) 0x39, (byte) 0x82, (byte) 0x9b, (byte) 0x2f, (byte) 0xff, (byte) 0x87,
            (byte) 0x34, (byte) 0x8e, (byte) 0x43, (byte) 0x44, (byte) 0xc4, (byte) 0xde, (byte) 0xe9, (byte) 0xcb,
            (byte) 0x54, (byte) 0x7b, (byte) 0x94, (byte) 0x32, (byte) 0xa6, (byte) 0xc2, (byte) 0x23, (byte) 0x3d,
            (byte) 0xee, (byte) 0x4c, (byte) 0x95, (byte) 0x0b, (byte) 0x42, (byte) 0xfa, (byte) 0xc3, (byte) 0x4e,
            (byte) 0x08, (byte) 0x2e, (byte) 0xa1, (byte) 0x66, (byte) 0x28, (byte) 0xd9, (byte) 0x24, (byte) 0xb2,
            (byte) 0x76, (byte) 0x5b, (byte) 0xa2, (byte) 0x49, (byte) 0x6d, (byte) 0x8b, (byte) 0xd1, (byte) 0x25,
            (byte) 0x72, (byte) 0xf8, (byte) 0xf6, (byte) 0x64, (byte) 0x86, (byte) 0x68, (byte) 0x98, (byte) 0x16,
            (byte) 0xd4, (byte) 0xa4, (byte) 0x5c, (byte) 0xcc, (byte) 0x5d, (byte) 0x65, (byte) 0xb6, (byte) 0x92,
            (byte) 0x6c, (byte) 0x70, (byte) 0x48, (byte) 0x50, (byte) 0xfd, (byte) 0xed, (byte) 0xb9, (byte) 0xda,
            (byte) 0x5e, (byte) 0x15, (byte) 0x46, (byte) 0x57, (byte) 0xa7, (byte) 0x8d, (byte) 0x9d, (byte) 0x84,
            (byte) 0x90, (byte) 0xd8, (byte) 0xab, (byte) 0x00, (byte) 0x8c, (byte) 0xbc, (byte) 0xd3, (byte) 0x0a,
            (byte) 0xf7, (byte) 0xe4, (byte) 0x58, (byte) 0x05, (byte) 0xb8, (byte) 0xb3, (byte) 0x45, (byte) 0x06,
            (byte) 0xd0, (byte) 0x2c, (byte) 0x1e, (byte) 0x8f, (byte) 0xca, (byte) 0x3f, (byte) 0x0f, (byte) 0x02,
            (byte) 0xc1, (byte) 0xaf, (byte) 0xbd, (byte) 0x03, (byte) 0x01, (byte) 0x13, (byte) 0x8a, (byte) 0x6b,
            (byte) 0x3a, (byte) 0x91, (byte) 0x11, (byte) 0x41, (byte) 0x4f, (byte) 0x67, (byte) 0xdc, (byte) 0xea,
            (byte) 0x97, (byte) 0xf2, (byte) 0xcf, (byte) 0xce, (byte) 0xf0, (byte) 0xb4, (byte) 0xe6, (byte) 0x73,
            (byte) 0x96, (byte) 0xac, (byte) 0x74, (byte) 0x22, (byte) 0xe7, (byte) 0xad, (byte) 0x35, (byte) 0x85,
            (byte) 0xe2, (byte) 0xf9, (byte) 0x37, (byte) 0xe8, (byte) 0x1c, (byte) 0x75, (byte) 0xdf, (byte) 0x6e,
            (byte) 0x47, (byte) 0xf1, (byte) 0x1a, (byte) 0x71, (byte) 0x1d, (byte) 0x29, (byte) 0xc5, (byte) 0x89,
            (byte) 0x6f, (byte) 0xb7, (byte) 0x62, (byte) 0x0e, (byte) 0xaa, (byte) 0x18, (byte) 0xbe, (byte) 0x1b,
            (byte) 0xfc, (byte) 0x56, (byte) 0x3e, (byte) 0x4b, (byte) 0xc6, (byte) 0xd2, (byte) 0x79, (byte) 0x20,
            (byte) 0x9a, (byte) 0xdb, (byte) 0xc0, (byte) 0xfe, (byte) 0x78, (byte) 0xcd, (byte) 0x5a, (byte) 0xf4,
            (byte) 0x1f, (byte) 0xdd, (byte) 0xa8, (byte) 0x33, (byte) 0x88, (byte) 0x07, (byte) 0xc7, (byte) 0x31,
            (byte) 0xb1, (byte) 0x12, (byte) 0x10, (byte) 0x59, (byte) 0x27, (byte) 0x80, (byte) 0xec, (byte) 0x5f,
            (byte) 0x60, (byte) 0x51, (byte) 0x7f, (byte) 0xa9, (byte) 0x19, (byte) 0xb5, (byte) 0x4a, (byte) 0x0d,
            (byte) 0x2d, (byte) 0xe5, (byte) 0x7a, (byte) 0x9f, (byte) 0x93, (byte) 0xc9, (byte) 0x9c, (byte) 0xef,
            (byte) 0xa0, (byte) 0xe0, (byte) 0x3b, (byte) 0x4d, (byte) 0xae, (byte) 0x2a, (byte) 0xf5, (byte) 0xb0,
            (byte) 0xc8, (byte) 0xeb, (byte) 0xbb, (byte) 0x3c, (byte) 0x83, (byte) 0x53, (byte) 0x99, (byte) 0x61,
            (byte) 0x17, (byte) 0x2b, (byte) 0x04, (byte) 0x7e, (byte) 0xba, (byte) 0x77, (byte) 0xd6, (byte) 0x26,
            (byte) 0xe1, (byte) 0x69, (byte) 0x14, (byte) 0x63, (byte) 0x55, (byte) 0x21, (byte) 0x0c, (byte) 0x7d
    };

    // rcon values for key expansion
    private static final byte[] rcon = {
            (byte) 0x01, (byte) 0x02, (byte) 0x04, (byte) 0x08,
            (byte) 0x10, (byte) 0x20, (byte) 0x40, (byte) 0x80,
            (byte) 0x1b, (byte) 0x36, (byte) 0x6c, (byte) 0xd8,
            (byte) 0xab, (byte) 0x4d, (byte) 0x9a
    };

    // AES data
    private int knum; // Nk
    private int rnum; // Nr
    protected byte[] key; // expanded AES key

    // Constructor
    public AesEncryptionByteBased(byte[] key) throws AesBadKeyException {
        switch (key.length) {
        case 16:
        case 24:
        case 32:
            knum = key.length / 4;
            rnum = knum + 6;
            break;
        default:
            throw new AesBadKeyException("array has wrong number of elements");
        }
        this.key = new byte[(rnum + 1) * 16];
        System.arraycopy(key, 0, this.key, 0, key.length);
        expandKey();
    }

    // Public encryption and decryption methods
    public byte[] encipher(String plain) {
        byte[] work = plain.getBytes();
        int fill = 16 - work.length % 16;
        byte[] states = new byte[work.length + fill];
        System.arraycopy(work, 0, states, 0, work.length);
        for (int n = work.length; n < states.length; n++)
            states[n] = (byte) 0;
        byte[] state = new byte[16];
        for (int n = 0; n < states.length; n += state.length) {
            System.arraycopy(states, n, state, 0, state.length);
            encipherState(state);
            System.arraycopy(state, 0, states, n, state.length);
        }
        return states;
    }

    public String decipher(byte[] cipher) {
        byte[] states = new byte[cipher.length - cipher.length % 16];
        for (int n = 0; n < states.length; n++)
            states[n] = cipher[n];
        byte[] state = new byte[16];
        for (int n = 0; n < states.length; n += state.length) {
            System.arraycopy(states, n, state, 0, state.length);
            decipherState(state);
            System.arraycopy(state, 0, states, n, state.length);
        }
        return new String(states);
    }

    // Methods implementing encryption
    private void subBytes(byte[] state) {
        for (int n = 0; n < state.length; n++)
            state[n] = sbox[state[n] & 0x000000FF];
    }

    protected void shiftRows(byte[] state) {
        byte tmp;
        // row 1
        // no shifting necessary
        // row 2
        tmp = state[1];
        state[1] = state[5];
        state[5] = state[9];
        state[9] = state[13];
        state[13] = tmp;
        // row 3
        tmp = state[2];
        state[2] = state[10];
        state[10] = tmp;
        tmp = state[6];
        state[6] = state[14];
        state[14] = tmp;
        // row 4
        tmp = state[3];
        state[3] = state[15];
        state[15] = state[11];
        state[11] = state[7];
        state[7] = tmp;
    }

    protected void mixColumns(byte[] state) {
        byte t0, t1, t2, t3;
        for (int n = 0; n < 16; n += 4) {
            t0 = state[0 + n];
            t1 = state[1 + n];
            t2 = state[2 + n];
            t3 = state[3 + n];
            state[0 + n] = (byte) (gmul(t0, (byte) 2) ^ gmul(t1, (byte) 3) ^ t2 ^ t3);
            state[1 + n] = (byte) (t0 ^ gmul(t1, (byte) 2) ^ gmul(t2, (byte) 3) ^ t3);
            state[2 + n] = (byte) (t0 ^ t1 ^ gmul(t2, (byte) 2) ^ gmul(t3, (byte) 3));
            state[3 + n] = (byte) (gmul(t0, (byte) 3) ^ t1 ^ t2 ^ gmul(t3, (byte) 2));
        }
    }

    protected void encipherState(byte[] state) {
        addRoundKey(state, 0);
        for (int r = 1; r < rnum; r++) {
            subBytes(state);
            shiftRows(state);
            mixColumns(state);
            addRoundKey(state, r);

        }
        subBytes(state);
        shiftRows(state);
        addRoundKey(state, rnum);
    }

    // Methods implementing decryption
    private void invSubBytes(byte[] state) {
        for (int n = 0; n < state.length; n++)
            state[n] = sboxinv[state[n] & 0x000000FF];
    }

    protected void invShiftRows(byte[] state) {
        byte tmp;
        // row 1
        // no shifting necessary
        // row 2
        tmp = state[1];
        state[1] = state[13];
        state[13] = state[9];
        state[9] = state[5];
        state[5] = tmp;
        // row 3
        tmp = state[2];
        state[2] = state[10];
        state[10] = tmp;
        tmp = state[6];
        state[6] = state[14];
        state[14] = tmp;
        // row 4
        tmp = state[3];
        state[3] = state[7];
        state[7] = state[11];
        state[11] = state[15];
        state[15] = tmp;
    }

    protected void invMixColumns(byte[] state) {
        byte t0, t1, t2, t3;
        for (int n = 0; n < 16; n += 4) {
            t0 = state[0 + n];
            t1 = state[1 + n];
            t2 = state[2 + n];
            t3 = state[3 + n];
            state[0 + n] = (byte) (gmul(t0, (byte) 0x0e) ^ gmul(t1, (byte) 0x0b) ^ gmul(t2, (byte) 0x0d) ^ gmul(t3,
                    (byte) 0x09));
            state[1 + n] = (byte) (gmul(t0, (byte) 0x09) ^ gmul(t1, (byte) 0x0e) ^ gmul(t2, (byte) 0x0b) ^ gmul(t3,
                    (byte) 0x0d));
            state[2 + n] = (byte) (gmul(t0, (byte) 0x0d) ^ gmul(t1, (byte) 0x09) ^ gmul(t2, (byte) 0x0e) ^ gmul(t3,
                    (byte) 0x0b));
            state[3 + n] = (byte) (gmul(t0, (byte) 0x0b) ^ gmul(t1, (byte) 0x0d) ^ gmul(t2, (byte) 0x09) ^ gmul(t3,
                    (byte) 0x0e));
        }
    }

    protected void decipherState(byte[] state) {
        addRoundKey(state, rnum);
        for (int r = rnum - 1; r > 0; r--) {
            invShiftRows(state);
            invSubBytes(state);
            addRoundKey(state, r);
            invMixColumns(state);
        }
        invShiftRows(state);
        invSubBytes(state);
        addRoundKey(state, 0);
    }

    // Methods uses for encryption and decryption
    protected void addRoundKey(byte[] state, int round) {
        for (int n = 0, m = round * 16; n < 16; n++, m++)
            state[n] ^= key[m];
    }

    // Multiplication in GF(256)
    private byte gmul(byte p, byte q) {
        byte r = 0;
        for (int i = 0; i < 8; i++) {
            if ((q & (byte) 0x01) != 0)
                r ^= p;
            if ((p & (byte) 0x80) != 0) {
                p <<= 1;
                p ^= 0x1b;
            } else
                p <<= 1;
            q >>>= 1;
        }
        return r;
    }

    // Key expansion
    private void expandKey() {
        byte[] tmp = new byte[4];
        for (int i = knum * 4; i < 16 * (rnum + 1); i += 4) {
            tmp[0] = key[i - 4];
            tmp[1] = key[i - 3];
            tmp[2] = key[i - 2];
            tmp[3] = key[i - 1];

            if ((i / 4) % knum == 0) {
                byte trans = tmp[0];
                tmp[0] = tmp[1];
                tmp[1] = tmp[2];
                tmp[2] = tmp[3];
                tmp[3] = trans;
                subBytes(tmp);
                tmp[0] = (byte) (tmp[0] ^ rcon[i / (knum * 4) - 1]);
            } else if (knum > 6 && (i / 4) % knum == 4) {
                subBytes(tmp);
            }

            key[i] = (byte) (key[i - knum * 4] ^ tmp[0]);
            key[i + 1] = (byte) (key[i - knum * 4 + 1] ^ tmp[1]);
            key[i + 2] = (byte) (key[i - knum * 4 + 2] ^ tmp[2]);
            key[i + 3] = (byte) (key[i - knum * 4 + 3] ^ tmp[3]);
        }
    }

    // Getter
    public byte[] getExpandedKey() {
        return key.clone();
    }
}
