/*
 * Project: aesdroid - AES implementation for Android
 * File   : AesAtomicMethodTest.java
 * Author : Oliver Mueller <oliver@cogito-ergo-sum.org>
 * Purpose: Android JUnit test of AES atomic methods of AesEncryption class.
 *
 * $Id: AesAtomicMethodTest.java 45 2012-07-10 06:46:05Z oliver $
 *
 * Copyright (c) 2011, 2012 Oliver Mueller.
 * All rights reserved.
 * http://www.cogito-ergo-sum.org
 * http://oliver-mueller.com
 *
 * This software is provided 'as-is', without any express or implied warranty.
 * In no event will the author be held liable for any damages arising from
 * the use of this software.
 *
 */

package org.cogito_ergo_sum.aesdroid.test;

import org.cogito_ergo_sum.aes.AesBadKeyException;
import org.cogito_ergo_sum.aes.AesEncryption;
import org.cogito_ergo_sum.aes.AesTestHelper;
import org.cogito_ergo_sum.aesdroid.AesdroidActivity;

import android.test.ActivityInstrumentationTestCase2;

public class AesAtomicMethodTest extends
        ActivityInstrumentationTestCase2<AesdroidActivity> {

    // Note: static doesn't work for test data here in all cases on Dalvik.
    // Receiving NullPointerException at runtime while accessing statics!
    // Workaround => object attributes instead of class attributes

    // Test keys
    private String[] key = {
            "2b7e151628aed2a6abf7158809cf4f3c",
            "8e73b0f7da0e6452c810f32b809079e562f8ead2522c6b7b",
            "603deb1015ca71be2b73aef0857d77811f352c073b6108d72d9810a30914dff4"
    };

    // Expanded test keys
    private String[] expkey = {
            "2b7e151628aed2a6abf7158809cf4f3ca0fafe1788542cb123a339392a6c7605f2c295f27a96b9435935807a7359f67f3d80477d4716fe3e1e237e446d7a883bef44a541a8525b7fb671253bdb0bad00d4d1c6f87c839d87caf2b8bc11f915bc6d88a37a110b3efddbf98641ca0093fd4e54f70e5f5fc9f384a64fb24ea6dc4fead27321b58dbad2312bf5607f8d292fac7766f319fadc2128d12941575c006ed014f9a8c9ee2589e13f0cc8b6630ca6",
            "8e73b0f7da0e6452c810f32b809079e562f8ead2522c6b7bfe0c91f72402f5a5ec12068e6c827f6b0e7a95b95c56fec24db7b4bd69b5411885a74796e92538fde75fad44bb095386485af05721efb14fa448f6d94d6dce24aa326360113b30e6a25e7ed583b1cf9a27f939436a94f767c0a69407d19da4e1ec1786eb6fa64971485f703222cb8755e26d135233f0b7b340beeb282f18a2596747d26b458c553ea7e1466c9411f1df821f750aad07d753ca4005388fcc5006282d166abc3ce7b5e98ba06f448c773c8ecc720401002202",
            "603deb1015ca71be2b73aef0857d77811f352c073b6108d72d9810a30914dff49ba354118e6925afa51a8b5f2067fcdea8b09c1a93d194cdbe49846eb75d5b9ad59aecb85bf3c917fee94248de8ebe96b5a9328a2678a647983122292f6c79b3812c81addadf48ba24360af2fab8b46498c5bfc9bebd198e268c3ba709e0421468007bacb2df331696e939e46c518d80c814e20476a9fb8a5025c02d59c58239de1369676ccc5a71fa2563959674ee155886ca5d2e2f31d77e0af1fa27cf73c3749c47ab18501ddae2757e4f7401905acafaaae3e4d59b349adf6acebd10190dfe4890d1e6188d0b046df344706c631e"
    };

    // Reference values for some tests
    //
    // Taken from "Appendix B - Cipher Example" of
    //
    // Federal Information Processing Standards Publication 197:
    // "Announcing the ADVANCED ENCRYPTION STANDARD (AES)"
    // November 26, 2001
    // http://csrc.nist.gov/publications/fips/fips197/fips-197.pdf
    //
    private byte[][] reference = { {
            (byte) 0x19, (byte) 0x3d, (byte) 0xe3, (byte) 0xbe, (byte) 0xa0, (byte) 0xf4, (byte) 0xe2, (byte) 0x2b,
            (byte) 0x9a, (byte) 0xc6, (byte) 0x8d, (byte) 0x2a, (byte) 0xe9, (byte) 0xf8, (byte) 0x48, (byte) 0x08 }, {
            (byte) 0xd4, (byte) 0x27, (byte) 0x11, (byte) 0xae, (byte) 0xe0, (byte) 0xbf, (byte) 0x98, (byte) 0xf1,
            (byte) 0xb8, (byte) 0xb4, (byte) 0x5d, (byte) 0xe5, (byte) 0x1e, (byte) 0x41, (byte) 0x52, (byte) 0x30 }, {
            (byte) 0xd4, (byte) 0xbf, (byte) 0x5d, (byte) 0x30, (byte) 0xe0, (byte) 0xb4, (byte) 0x52, (byte) 0xae,
            (byte) 0xb8, (byte) 0x41, (byte) 0x11, (byte) 0xf1, (byte) 0x1e, (byte) 0x27, (byte) 0x98, (byte) 0xe5 }, {
            (byte) 0x04, (byte) 0x66, (byte) 0x81, (byte) 0xe5, (byte) 0xe0, (byte) 0xcb, (byte) 0x19, (byte) 0x9a,
            (byte) 0x48, (byte) 0xf8, (byte) 0xd3, (byte) 0x7a, (byte) 0x28, (byte) 0x06, (byte) 0x26, (byte) 0x4c }, {
            (byte) 0xa0, (byte) 0xfa, (byte) 0xfe, (byte) 0x17, (byte) 0x88, (byte) 0x54, (byte) 0x2c, (byte) 0xb1,
            (byte) 0x23, (byte) 0xa3, (byte) 0x39, (byte) 0x39, (byte) 0x2a, (byte) 0x6c, (byte) 0x76, (byte) 0x05 }, {

            (byte) 0xa4, (byte) 0x9c, (byte) 0x7f, (byte) 0xf2, (byte) 0x68, (byte) 0x9f, (byte) 0x35, (byte) 0x2b,
            (byte) 0x6b, (byte) 0x5b, (byte) 0xea, (byte) 0x43, (byte) 0x02, (byte) 0x6a, (byte) 0x50, (byte) 0x49 }, {
            (byte) 0x49, (byte) 0xde, (byte) 0xd2, (byte) 0x89, (byte) 0x45, (byte) 0xdb, (byte) 0x96, (byte) 0xf1,
            (byte) 0x7f, (byte) 0x39, (byte) 0x87, (byte) 0x1a, (byte) 0x77, (byte) 0x02, (byte) 0x53, (byte) 0x3b }, {
            (byte) 0x49, (byte) 0xdb, (byte) 0x87, (byte) 0x3b, (byte) 0x45, (byte) 0x39, (byte) 0x53, (byte) 0x89,
            (byte) 0x7f, (byte) 0x02, (byte) 0xd2, (byte) 0xf1, (byte) 0x77, (byte) 0xde, (byte) 0x96, (byte) 0x1a }, {
            (byte) 0x58, (byte) 0x4d, (byte) 0xca, (byte) 0xf1, (byte) 0x1b, (byte) 0x4b, (byte) 0x5a, (byte) 0xac,
            (byte) 0xdb, (byte) 0xe7, (byte) 0xca, (byte) 0xa8, (byte) 0x1b, (byte) 0x6b, (byte) 0xb0, (byte) 0xe5 }, {
            (byte) 0xf2, (byte) 0xc2, (byte) 0x95, (byte) 0xf2, (byte) 0x7a, (byte) 0x96, (byte) 0xb9, (byte) 0x43,
            (byte) 0x59, (byte) 0x35, (byte) 0x80, (byte) 0x7a, (byte) 0x73, (byte) 0x59, (byte) 0xf6, (byte) 0x7f }, {

            (byte) 0xaa, (byte) 0x8f, (byte) 0x5f, (byte) 0x03, (byte) 0x61, (byte) 0xdd, (byte) 0xe3, (byte) 0xef,
            (byte) 0x82, (byte) 0xd2, (byte) 0x4a, (byte) 0xd2, (byte) 0x68, (byte) 0x32, (byte) 0x46, (byte) 0x9a }, {
            (byte) 0xac, (byte) 0x73, (byte) 0xcf, (byte) 0x7b, (byte) 0xef, (byte) 0xc1, (byte) 0x11, (byte) 0xdf,
            (byte) 0x13, (byte) 0xb5, (byte) 0xd6, (byte) 0xb5, (byte) 0x45, (byte) 0x23, (byte) 0x5a, (byte) 0xb8 }, {
            (byte) 0xac, (byte) 0xc1, (byte) 0xd6, (byte) 0xb8, (byte) 0xef, (byte) 0xb5, (byte) 0x5a, (byte) 0x7b,
            (byte) 0x13, (byte) 0x23, (byte) 0xcf, (byte) 0xdf, (byte) 0x45, (byte) 0x73, (byte) 0x11, (byte) 0xb5 }, {
            (byte) 0x75, (byte) 0xec, (byte) 0x09, (byte) 0x93, (byte) 0x20, (byte) 0x0b, (byte) 0x63, (byte) 0x33,
            (byte) 0x53, (byte) 0xc0, (byte) 0xcf, (byte) 0x7c, (byte) 0xbb, (byte) 0x25, (byte) 0xd0, (byte) 0xdc }, {
            (byte) 0x3d, (byte) 0x80, (byte) 0x47, (byte) 0x7d, (byte) 0x47, (byte) 0x16, (byte) 0xfe, (byte) 0x3e,
            (byte) 0x1e, (byte) 0x23, (byte) 0x7e, (byte) 0x44, (byte) 0x6d, (byte) 0x7a, (byte) 0x88, (byte) 0x3b }, {

            (byte) 0x48, (byte) 0x6c, (byte) 0x4e, (byte) 0xee, (byte) 0x67, (byte) 0x1d, (byte) 0x9d, (byte) 0x0d,
            (byte) 0x4d, (byte) 0xe3, (byte) 0xb1, (byte) 0x38, (byte) 0xd6, (byte) 0x5f, (byte) 0x58, (byte) 0xe7 }, {
            (byte) 0x52, (byte) 0x50, (byte) 0x2f, (byte) 0x28, (byte) 0x85, (byte) 0xa4, (byte) 0x5e, (byte) 0xd7,
            (byte) 0xe3, (byte) 0x11, (byte) 0xc8, (byte) 0x07, (byte) 0xf6, (byte) 0xcf, (byte) 0x6a, (byte) 0x94 }, {
            (byte) 0x52, (byte) 0xa4, (byte) 0xc8, (byte) 0x94, (byte) 0x85, (byte) 0x11, (byte) 0x6a, (byte) 0x28,
            (byte) 0xe3, (byte) 0xcf, (byte) 0x2f, (byte) 0xd7, (byte) 0xf6, (byte) 0x50, (byte) 0x5e, (byte) 0x07 }, {
            (byte) 0x0f, (byte) 0xd6, (byte) 0xda, (byte) 0xa9, (byte) 0x60, (byte) 0x31, (byte) 0x38, (byte) 0xbf,
            (byte) 0x6f, (byte) 0xc0, (byte) 0x10, (byte) 0x6b, (byte) 0x5e, (byte) 0xb3, (byte) 0x13, (byte) 0x01 }, {
            (byte) 0xef, (byte) 0x44, (byte) 0xa5, (byte) 0x41, (byte) 0xa8, (byte) 0x52, (byte) 0x5b, (byte) 0x7f,
            (byte) 0xb6, (byte) 0x71, (byte) 0x25, (byte) 0x3b, (byte) 0xdb, (byte) 0x0b, (byte) 0xad, (byte) 0x00 }, {

            (byte) 0xe0, (byte) 0x92, (byte) 0x7f, (byte) 0xe8, (byte) 0xc8, (byte) 0x63, (byte) 0x63, (byte) 0xc0,
            (byte) 0xd9, (byte) 0xb1, (byte) 0x35, (byte) 0x50, (byte) 0x85, (byte) 0xb8, (byte) 0xbe, (byte) 0x01 }, {
            (byte) 0xe1, (byte) 0x4f, (byte) 0xd2, (byte) 0x9b, (byte) 0xe8, (byte) 0xfb, (byte) 0xfb, (byte) 0xba,
            (byte) 0x35, (byte) 0xc8, (byte) 0x96, (byte) 0x53, (byte) 0x97, (byte) 0x6c, (byte) 0xae, (byte) 0x7c }, {
            (byte) 0xe1, (byte) 0xfb, (byte) 0x96, (byte) 0x7c, (byte) 0xe8, (byte) 0xc8, (byte) 0xae, (byte) 0x9b,
            (byte) 0x35, (byte) 0x6c, (byte) 0xd2, (byte) 0xba, (byte) 0x97, (byte) 0x4f, (byte) 0xfb, (byte) 0x53 }, {
            (byte) 0x25, (byte) 0xd1, (byte) 0xa9, (byte) 0xad, (byte) 0xbd, (byte) 0x11, (byte) 0xd1, (byte) 0x68,
            (byte) 0xb6, (byte) 0x3a, (byte) 0x33, (byte) 0x8e, (byte) 0x4c, (byte) 0x4c, (byte) 0xc0, (byte) 0xb0 }, {
            (byte) 0xd4, (byte) 0xd1, (byte) 0xc6, (byte) 0xf8, (byte) 0x7c, (byte) 0x83, (byte) 0x9d, (byte) 0x87,
            (byte) 0xca, (byte) 0xf2, (byte) 0xb8, (byte) 0xbc, (byte) 0x11, (byte) 0xf9, (byte) 0x15, (byte) 0xbc }, {

            (byte) 0xf1, (byte) 0x00, (byte) 0x6f, (byte) 0x55, (byte) 0xc1, (byte) 0x92, (byte) 0x4c, (byte) 0xef,
            (byte) 0x7c, (byte) 0xc8, (byte) 0x8b, (byte) 0x32, (byte) 0x5d, (byte) 0xb5, (byte) 0xd5, (byte) 0x0c }, {
            (byte) 0xa1, (byte) 0x63, (byte) 0xa8, (byte) 0xfc, (byte) 0x78, (byte) 0x4f, (byte) 0x29, (byte) 0xdf,
            (byte) 0x10, (byte) 0xe8, (byte) 0x3d, (byte) 0x23, (byte) 0x4c, (byte) 0xd5, (byte) 0x03, (byte) 0xfe }, {
            (byte) 0xa1, (byte) 0x4f, (byte) 0x3d, (byte) 0xfe, (byte) 0x78, (byte) 0xe8, (byte) 0x03, (byte) 0xfc,
            (byte) 0x10, (byte) 0xd5, (byte) 0xa8, (byte) 0xdf, (byte) 0x4c, (byte) 0x63, (byte) 0x29, (byte) 0x23 }, {
            (byte) 0x4b, (byte) 0x86, (byte) 0x8d, (byte) 0x6d, (byte) 0x2c, (byte) 0x4a, (byte) 0x89, (byte) 0x80,
            (byte) 0x33, (byte) 0x9d, (byte) 0xf4, (byte) 0xe8, (byte) 0x37, (byte) 0xd2, (byte) 0x18, (byte) 0xd8 }, {
            (byte) 0x6d, (byte) 0x88, (byte) 0xa3, (byte) 0x7a, (byte) 0x11, (byte) 0x0b, (byte) 0x3e, (byte) 0xfd,
            (byte) 0xdb, (byte) 0xf9, (byte) 0x86, (byte) 0x41, (byte) 0xca, (byte) 0x00, (byte) 0x93, (byte) 0xfd }, {

            (byte) 0x26, (byte) 0x0e, (byte) 0x2e, (byte) 0x17, (byte) 0x3d, (byte) 0x41, (byte) 0xb7, (byte) 0x7d,
            (byte) 0xe8, (byte) 0x64, (byte) 0x72, (byte) 0xa9, (byte) 0xfd, (byte) 0xd2, (byte) 0x8b, (byte) 0x25 }, {
            (byte) 0xf7, (byte) 0xab, (byte) 0x31, (byte) 0xf0, (byte) 0x27, (byte) 0x83, (byte) 0xa9, (byte) 0xff,
            (byte) 0x9b, (byte) 0x43, (byte) 0x40, (byte) 0xd3, (byte) 0x54, (byte) 0xb5, (byte) 0x3d, (byte) 0x3f }, {
            (byte) 0xf7, (byte) 0x83, (byte) 0x40, (byte) 0x3f, (byte) 0x27, (byte) 0x43, (byte) 0x3d, (byte) 0xf0,
            (byte) 0x9b, (byte) 0xb5, (byte) 0x31, (byte) 0xff, (byte) 0x54, (byte) 0xab, (byte) 0xa9, (byte) 0xd3 }, {
            (byte) 0x14, (byte) 0x15, (byte) 0xb5, (byte) 0xbf, (byte) 0x46, (byte) 0x16, (byte) 0x15, (byte) 0xec,
            (byte) 0x27, (byte) 0x46, (byte) 0x56, (byte) 0xd7, (byte) 0x34, (byte) 0x2a, (byte) 0xd8, (byte) 0x43 }, {
            (byte) 0x4e, (byte) 0x54, (byte) 0xf7, (byte) 0x0e, (byte) 0x5f, (byte) 0x5f, (byte) 0xc9, (byte) 0xf3,
            (byte) 0x84, (byte) 0xa6, (byte) 0x4f, (byte) 0xb2, (byte) 0x4e, (byte) 0xa6, (byte) 0xdc, (byte) 0x4f }, {

            (byte) 0x5a, (byte) 0x41, (byte) 0x42, (byte) 0xb1, (byte) 0x19, (byte) 0x49, (byte) 0xdc, (byte) 0x1f,
            (byte) 0xa3, (byte) 0xe0, (byte) 0x19, (byte) 0x65, (byte) 0x7a, (byte) 0x8c, (byte) 0x04, (byte) 0x0c }, {
            (byte) 0xbe, (byte) 0x83, (byte) 0x2c, (byte) 0xc8, (byte) 0xd4, (byte) 0x3b, (byte) 0x86, (byte) 0xc0,
            (byte) 0x0a, (byte) 0xe1, (byte) 0xd4, (byte) 0x4d, (byte) 0xda, (byte) 0x64, (byte) 0xf2, (byte) 0xfe }, {
            (byte) 0xbe, (byte) 0x3b, (byte) 0xd4, (byte) 0xfe, (byte) 0xd4, (byte) 0xe1, (byte) 0xf2, (byte) 0xc8,
            (byte) 0x0a, (byte) 0x64, (byte) 0x2c, (byte) 0xc0, (byte) 0xda, (byte) 0x83, (byte) 0x86, (byte) 0x4d }, {
            (byte) 0x00, (byte) 0x51, (byte) 0x2f, (byte) 0xd1, (byte) 0xb1, (byte) 0xc8, (byte) 0x89, (byte) 0xff,
            (byte) 0x54, (byte) 0x76, (byte) 0x6d, (byte) 0xcd, (byte) 0xfa, (byte) 0x1b, (byte) 0x99, (byte) 0xea }, {
            (byte) 0xea, (byte) 0xd2, (byte) 0x73, (byte) 0x21, (byte) 0xb5, (byte) 0x8d, (byte) 0xba, (byte) 0xd2,
            (byte) 0x31, (byte) 0x2b, (byte) 0xf5, (byte) 0x60, (byte) 0x7f, (byte) 0x8d, (byte) 0x29, (byte) 0x2f }, {

            (byte) 0xea, (byte) 0x83, (byte) 0x5c, (byte) 0xf0, (byte) 0x04, (byte) 0x45, (byte) 0x33, (byte) 0x2d,
            (byte) 0x65, (byte) 0x5d, (byte) 0x98, (byte) 0xad, (byte) 0x85, (byte) 0x96, (byte) 0xb0, (byte) 0xc5 }, {
            (byte) 0x87, (byte) 0xec, (byte) 0x4a, (byte) 0x8c, (byte) 0xf2, (byte) 0x6e, (byte) 0xc3, (byte) 0xd8,
            (byte) 0x4d, (byte) 0x4c, (byte) 0x46, (byte) 0x95, (byte) 0x97, (byte) 0x90, (byte) 0xe7, (byte) 0xa6 }, {
            (byte) 0x87, (byte) 0x6e, (byte) 0x46, (byte) 0xa6, (byte) 0xf2, (byte) 0x4c, (byte) 0xe7, (byte) 0x8c,
            (byte) 0x4d, (byte) 0x90, (byte) 0x4a, (byte) 0xd8, (byte) 0x97, (byte) 0xec, (byte) 0xc3, (byte) 0x95 }, {
            (byte) 0x47, (byte) 0x37, (byte) 0x94, (byte) 0xed, (byte) 0x40, (byte) 0xd4, (byte) 0xe4, (byte) 0xa5,
            (byte) 0xa3, (byte) 0x70, (byte) 0x3a, (byte) 0xa6, (byte) 0x4c, (byte) 0x9f, (byte) 0x42, (byte) 0xbc }, {
            (byte) 0xac, (byte) 0x77, (byte) 0x66, (byte) 0xf3, (byte) 0x19, (byte) 0xfa, (byte) 0xdc, (byte) 0x21,
            (byte) 0x28, (byte) 0xd1, (byte) 0x29, (byte) 0x41, (byte) 0x57, (byte) 0x5c, (byte) 0x00, (byte) 0x6e }, {

            (byte) 0xeb, (byte) 0x40, (byte) 0xf2, (byte) 0x1e, (byte) 0x59, (byte) 0x2e, (byte) 0x38, (byte) 0x84,
            (byte) 0x8b, (byte) 0xa1, (byte) 0x13, (byte) 0xe7, (byte) 0x1b, (byte) 0xc3, (byte) 0x42, (byte) 0xd2 }
    };
    private String referenceKey = "2b7e151628aed2a6abf7158809cf4f3c";
    private String referencePlain = "3243f6a8885a308d313198a2e0370734";
    private String referenceCipher = "3925841d02dc09fbdc118597196a0b32";

    public AesAtomicMethodTest() {
        super("org.cogito_ergo_sum.aesdroid", AesdroidActivity.class);
    }

    @Override
    protected void setUp() throws Exception {
        super.setUp();
    }

    public void testPreconditions() {
        // assertNotNull(mActivity);
    }

    public void testKeyExpansion() {
        AesEncryption aes;
        byte[] testkey;
        for (int k = 0; k < key.length; k++) {
            testkey =
                    AesdroidActivity.hexStringToByteArray(key[k]);
            try {
                aes = new
                        AesEncryption(testkey);
                String expkey =
                        AesdroidActivity.byteArrayToHexString(aes.getExpandedKey());
                assertEquals(this.expkey[k], expkey);
            } catch (AesBadKeyException e) {
                assertTrue(false);
            }
        }
    }

    public void testShiftRows() {
        try {
            AesTestHelper aes = new
                    AesTestHelper(AesdroidActivity.hexStringToByteArray(
                            "0123456789abcdef2b73aef0857d77811f352c073b6108d72d9810a30914dff4"));
            assertNotNull(reference);
            byte[] state = new byte[16];
            for (int n = 1; n < 45; n += 5) {
                for (int m = 0; m < 16; m++)
                    state[m] = reference[n][m];
                aes.shiftRows(state, 0);
                for (int m = 0; m < 16; m++)
                    assertEquals(state[m], reference[n + 1][m]);
            }
        } catch (AesBadKeyException e) {
            e.printStackTrace();
        }
    }

    public void testInvShiftRows() {
        try {
            AesTestHelper aes = new
                    AesTestHelper(AesdroidActivity.hexStringToByteArray(
                            "0123456789abcdef2b73aef0857d77811f352c073b6108d72d9810a30914dff4"));
            assertNotNull(reference);
            byte[] state = new byte[16];
            for (int n = 2; n < 45; n += 5) {
                for (int m = 0; m < 16; m++)
                    state[m] = reference[n][m];
                aes.invShiftRows(state, 0);
                for (int m = 0; m < 16; m++)
                    assertEquals(state[m], reference[n - 1][m]);
            }
        } catch (AesBadKeyException e) {
            e.printStackTrace();
        }
    }

    public void testMixColumns() {
        try {
            AesTestHelper aes = new
                    AesTestHelper(AesdroidActivity.hexStringToByteArray(
                            "0123456789abcdef2b73aef0857d77811f352c073b6108d72d9810a30914dff4"));
            assertNotNull(reference);
            byte[] state = new byte[16];
            for (int n = 2; n < 45; n += 5) {
                for (int m = 0; m < 16; m++)
                    state[m] = reference[n][m];
                aes.mixColumns(state, 0);
                for (int m = 0; m < 16; m++)
                    assertEquals(state[m], reference[n + 1][m]);
            }
        } catch (AesBadKeyException e) {
            e.printStackTrace();
        }
    }

    public void testInvMixColumns() {
        try {
            AesTestHelper aes = new
                    AesTestHelper(AesdroidActivity.hexStringToByteArray(
                            "0123456789abcdef2b73aef0857d77811f352c073b6108d72d9810a30914dff4"));
            assertNotNull(reference);
            byte[] state = new byte[16];
            for (int n = 3; n < 45; n += 5) {
                for (int m = 0; m < 16; m++)
                    state[m] = reference[n][m];
                aes.invMixColumns(state, 0);
                for (int m = 0; m < 16; m++)
                    assertEquals(state[m], reference[n - 1][m]);
            }
        } catch (AesBadKeyException e) {
            e.printStackTrace();
        }
    }

    public void testAddRoundKey() {
        try {
            AesTestHelper aes = new AesTestHelper(
                    AesdroidActivity
                            .hexStringToByteArray("0123456789abcdef2b73aef0857d77811f352c073b6108d72d9810a30914dff4"));
            assertNotNull(reference);
            byte[] state = new byte[16];
            byte[] key = new byte[16];
            for (int n = 3; n < 45; n += 5) {
                for (int m = 0; m < 16; m++)
                    state[m] = reference[n][m];
                System.arraycopy(reference[n + 1], 0, key, 0, key.length);
                aes.addRoundKey(state, key);
                for (int m = 0; m < 16; m++)
                    assertEquals(state[m], reference[n + 2][m]);
            }
        } catch (AesBadKeyException e) {
            e.printStackTrace();
        }
    }

    public void testEncipherState() {
        try {
            byte[] key = AesdroidActivity.hexStringToByteArray(referenceKey);
            byte[] state = AesdroidActivity.hexStringToByteArray(referencePlain);
            AesTestHelper aes = new AesTestHelper(key);
            aes.encipherState(state, 0);
            assertEquals(referenceCipher, AesdroidActivity.byteArrayToHexString(state));
        } catch (AesBadKeyException e) {
            e.printStackTrace();
        }
    }

    public void testDecipherState() {
        try {
            byte[] key = AesdroidActivity.hexStringToByteArray(referenceKey);
            byte[] state = AesdroidActivity.hexStringToByteArray(referenceCipher);
            AesTestHelper aes = new AesTestHelper(key);
            aes.decipherState(state, 0);
            assertEquals(referencePlain, AesdroidActivity.byteArrayToHexString(state));
        } catch (AesBadKeyException e) {
            e.printStackTrace();
        }
    }

    public void testCiphering() {
        try {
            byte[] key = AesdroidActivity.hexStringToByteArray(referenceKey);
            String testMsg = "This is a little test!";
            AesTestHelper aes = new AesTestHelper(key);
            byte[] cipher = aes.encipher(testMsg);
            String testReMsg = aes.decipher(cipher);
            // Remove trailing null bytes
            while (testReMsg.charAt(testReMsg.length() - 1) == '\0')
                testReMsg = testReMsg.substring(0, testReMsg.length() - 2);
            assertEquals(testMsg, testReMsg);
        } catch (AesBadKeyException e) {
            e.printStackTrace();
        }
    }

    // AES Known Answer Test (KAT)
    // ===========================
    // The following test methods implement the AES Known Answer Test (KAT) for
    // ECB mode.
    // Note: It is not possible to implement the KAT as a single method
    // operating on a
    // table here because it would exceed Java's 64k limit for methods.
    public void testECBGFSbox128d() {
        byte[] key;
        byte[] state;
        String plain;
        String cipher;
        AesTestHelper aes;
        try {
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "f34481ec3cc627bacd5dc3fb08f273e6";
            cipher = "0336763e966d92595a567cc9ce537f5e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "9798c4640bad75c7c3227db910174e72";
            cipher = "a9a1631bf4996954ebc093957b234589";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "96ab5c2ff612d9dfaae8c31f30c42168";
            cipher = "ff4f8391a6a40ca5b25d23bedd44a597";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "6a118a874519e64e9963798a503f1d35";
            cipher = "dc43be40be0e53712f7e2bf5ca707209";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "cb9fceec81286ca3e989bd979b0cb284";
            cipher = "92beedab1895a94faa69b632e5cc47ce";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "b26aeb1874e47ca8358ff22378f09144";
            cipher = "459264f4798f6a78bacb89c15ed3d601";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
        } catch (AesBadKeyException e) {
            e.printStackTrace();
        }
    }

    public void testECBGFSbox128e() {
        byte[] key;
        byte[] state;
        String plain;
        String cipher;
        AesTestHelper aes;
        try {
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "f34481ec3cc627bacd5dc3fb08f273e6";
            cipher = "0336763e966d92595a567cc9ce537f5e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "9798c4640bad75c7c3227db910174e72";
            cipher = "a9a1631bf4996954ebc093957b234589";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "96ab5c2ff612d9dfaae8c31f30c42168";
            cipher = "ff4f8391a6a40ca5b25d23bedd44a597";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "6a118a874519e64e9963798a503f1d35";
            cipher = "dc43be40be0e53712f7e2bf5ca707209";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "cb9fceec81286ca3e989bd979b0cb284";
            cipher = "92beedab1895a94faa69b632e5cc47ce";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "b26aeb1874e47ca8358ff22378f09144";
            cipher = "459264f4798f6a78bacb89c15ed3d601";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
        } catch (AesBadKeyException e) {
            e.printStackTrace();
        }
    }

    public void testECBGFSbox192d() {
        byte[] key;
        byte[] state;
        String plain;
        String cipher;
        AesTestHelper aes;
        try {
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "1b077a6af4b7f98229de786d7516b639";
            cipher = "275cfc0413d8ccb70513c3859b1d0f72";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "9c2d8842e5f48f57648205d39a239af1";
            cipher = "c9b8135ff1b5adc413dfd053b21bd96d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "bff52510095f518ecca60af4205444bb";
            cipher = "4a3650c3371ce2eb35e389a171427440";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "51719783d3185a535bd75adc65071ce1";
            cipher = "4f354592ff7c8847d2d0870ca9481b7c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "26aa49dcfe7629a8901a69a9914e6dfd";
            cipher = "d5e08bf9a182e857cf40b3a36ee248cc";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
        } catch (AesBadKeyException e) {
            e.printStackTrace();
        }
    }

    public void testECBGFSbox192e() {
        byte[] key;
        byte[] state;
        String plain;
        String cipher;
        AesTestHelper aes;
        try {
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "1b077a6af4b7f98229de786d7516b639";
            cipher = "275cfc0413d8ccb70513c3859b1d0f72";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "9c2d8842e5f48f57648205d39a239af1";
            cipher = "c9b8135ff1b5adc413dfd053b21bd96d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "bff52510095f518ecca60af4205444bb";
            cipher = "4a3650c3371ce2eb35e389a171427440";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "51719783d3185a535bd75adc65071ce1";
            cipher = "4f354592ff7c8847d2d0870ca9481b7c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "26aa49dcfe7629a8901a69a9914e6dfd";
            cipher = "d5e08bf9a182e857cf40b3a36ee248cc";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
        } catch (AesBadKeyException e) {
            e.printStackTrace();
        }
    }

    public void testECBGFSbox256d() {
        byte[] key;
        byte[] state;
        String plain;
        String cipher;
        AesTestHelper aes;
        try {
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "014730f80ac625fe84f026c60bfd547d";
            cipher = "5c9d844ed46f9885085e5d6a4f94c7d7";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "0b24af36193ce4665f2825d7b4749c98";
            cipher = "a9ff75bd7cf6613d3731c77c3b6d0c04";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "761c1fe41a18acf20d241650611d90f1";
            cipher = "623a52fcea5d443e48d9181ab32c7421";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "8a560769d605868ad80d819bdba03771";
            cipher = "38f2c7ae10612415d27ca190d27da8b4";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
        } catch (AesBadKeyException e) {
            e.printStackTrace();
        }
    }

    public void testECBGFSbox256e() {
        byte[] key;
        byte[] state;
        String plain;
        String cipher;
        AesTestHelper aes;
        try {
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "014730f80ac625fe84f026c60bfd547d";
            cipher = "5c9d844ed46f9885085e5d6a4f94c7d7";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "0b24af36193ce4665f2825d7b4749c98";
            cipher = "a9ff75bd7cf6613d3731c77c3b6d0c04";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "761c1fe41a18acf20d241650611d90f1";
            cipher = "623a52fcea5d443e48d9181ab32c7421";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "8a560769d605868ad80d819bdba03771";
            cipher = "38f2c7ae10612415d27ca190d27da8b4";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
        } catch (AesBadKeyException e) {
            e.printStackTrace();
        }
    }

    public void testECBKeySbox128d() {
        byte[] key;
        byte[] state;
        String plain;
        String cipher;
        AesTestHelper aes;
        try {
            key = AesdroidActivity.hexStringToByteArray("10a58869d74be5a374cf867cfb473859");
            plain = "00000000000000000000000000000000";
            cipher = "6d251e6944b051e04eaa6fb4dbf78465";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("caea65cdbb75e9169ecd22ebe6e54675");
            plain = "00000000000000000000000000000000";
            cipher = "6e29201190152df4ee058139def610bb";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("a2e2fa9baf7d20822ca9f0542f764a41");
            plain = "00000000000000000000000000000000";
            cipher = "c3b44b95d9d2f25670eee9a0de099fa3";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("b6364ac4e1de1e285eaf144a2415f7a0");
            plain = "00000000000000000000000000000000";
            cipher = "5d9b05578fc944b3cf1ccf0e746cd581";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("64cf9c7abc50b888af65f49d521944b2");
            plain = "00000000000000000000000000000000";
            cipher = "f7efc89d5dba578104016ce5ad659c05";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("47d6742eefcc0465dc96355e851b64d9");
            plain = "00000000000000000000000000000000";
            cipher = "0306194f666d183624aa230a8b264ae7";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("3eb39790678c56bee34bbcdeccf6cdb5");
            plain = "00000000000000000000000000000000";
            cipher = "858075d536d79ccee571f7d7204b1f67";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("64110a924f0743d500ccadae72c13427");
            plain = "00000000000000000000000000000000";
            cipher = "35870c6a57e9e92314bcb8087cde72ce";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("18d8126516f8a12ab1a36d9f04d68e51");
            plain = "00000000000000000000000000000000";
            cipher = "6c68e9be5ec41e22c825b7c7affb4363";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("f530357968578480b398a3c251cd1093");
            plain = "00000000000000000000000000000000";
            cipher = "f5df39990fc688f1b07224cc03e86cea";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("da84367f325d42d601b4326964802e8e");
            plain = "00000000000000000000000000000000";
            cipher = "bba071bcb470f8f6586e5d3add18bc66";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("e37b1c6aa2846f6fdb413f238b089f23");
            plain = "00000000000000000000000000000000";
            cipher = "43c9f7e62f5d288bb27aa40ef8fe1ea8";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("6c002b682483e0cabcc731c253be5674");
            plain = "00000000000000000000000000000000";
            cipher = "3580d19cff44f1014a7c966a69059de5";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("143ae8ed6555aba96110ab58893a8ae1");
            plain = "00000000000000000000000000000000";
            cipher = "806da864dd29d48deafbe764f8202aef";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("b69418a85332240dc82492353956ae0c");
            plain = "00000000000000000000000000000000";
            cipher = "a303d940ded8f0baff6f75414cac5243";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("71b5c08a1993e1362e4d0ce9b22b78d5");
            plain = "00000000000000000000000000000000";
            cipher = "c2dabd117f8a3ecabfbb11d12194d9d0";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("e234cdca2606b81f29408d5f6da21206");
            plain = "00000000000000000000000000000000";
            cipher = "fff60a4740086b3b9c56195b98d91a7b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("13237c49074a3da078dc1d828bb78c6f");
            plain = "00000000000000000000000000000000";
            cipher = "8146a08e2357f0caa30ca8c94d1a0544";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("3071a2a48fe6cbd04f1a129098e308f8");
            plain = "00000000000000000000000000000000";
            cipher = "4b98e06d356deb07ebb824e5713f7be3";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("90f42ec0f68385f2ffc5dfc03a654dce");
            plain = "00000000000000000000000000000000";
            cipher = "7a20a53d460fc9ce0423a7a0764c6cf2";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
        } catch (AesBadKeyException e) {
            e.printStackTrace();
        }
    }

    public void testECBKeySbox128e() {
        byte[] key;
        byte[] state;
        String plain;
        String cipher;
        AesTestHelper aes;
        try {
            key = AesdroidActivity.hexStringToByteArray("10a58869d74be5a374cf867cfb473859");
            plain = "00000000000000000000000000000000";
            cipher = "6d251e6944b051e04eaa6fb4dbf78465";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("caea65cdbb75e9169ecd22ebe6e54675");
            plain = "00000000000000000000000000000000";
            cipher = "6e29201190152df4ee058139def610bb";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("a2e2fa9baf7d20822ca9f0542f764a41");
            plain = "00000000000000000000000000000000";
            cipher = "c3b44b95d9d2f25670eee9a0de099fa3";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("b6364ac4e1de1e285eaf144a2415f7a0");
            plain = "00000000000000000000000000000000";
            cipher = "5d9b05578fc944b3cf1ccf0e746cd581";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("64cf9c7abc50b888af65f49d521944b2");
            plain = "00000000000000000000000000000000";
            cipher = "f7efc89d5dba578104016ce5ad659c05";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("47d6742eefcc0465dc96355e851b64d9");
            plain = "00000000000000000000000000000000";
            cipher = "0306194f666d183624aa230a8b264ae7";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("3eb39790678c56bee34bbcdeccf6cdb5");
            plain = "00000000000000000000000000000000";
            cipher = "858075d536d79ccee571f7d7204b1f67";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("64110a924f0743d500ccadae72c13427");
            plain = "00000000000000000000000000000000";
            cipher = "35870c6a57e9e92314bcb8087cde72ce";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("18d8126516f8a12ab1a36d9f04d68e51");
            plain = "00000000000000000000000000000000";
            cipher = "6c68e9be5ec41e22c825b7c7affb4363";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("f530357968578480b398a3c251cd1093");
            plain = "00000000000000000000000000000000";
            cipher = "f5df39990fc688f1b07224cc03e86cea";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("da84367f325d42d601b4326964802e8e");
            plain = "00000000000000000000000000000000";
            cipher = "bba071bcb470f8f6586e5d3add18bc66";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("e37b1c6aa2846f6fdb413f238b089f23");
            plain = "00000000000000000000000000000000";
            cipher = "43c9f7e62f5d288bb27aa40ef8fe1ea8";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("6c002b682483e0cabcc731c253be5674");
            plain = "00000000000000000000000000000000";
            cipher = "3580d19cff44f1014a7c966a69059de5";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("143ae8ed6555aba96110ab58893a8ae1");
            plain = "00000000000000000000000000000000";
            cipher = "806da864dd29d48deafbe764f8202aef";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("b69418a85332240dc82492353956ae0c");
            plain = "00000000000000000000000000000000";
            cipher = "a303d940ded8f0baff6f75414cac5243";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("71b5c08a1993e1362e4d0ce9b22b78d5");
            plain = "00000000000000000000000000000000";
            cipher = "c2dabd117f8a3ecabfbb11d12194d9d0";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("e234cdca2606b81f29408d5f6da21206");
            plain = "00000000000000000000000000000000";
            cipher = "fff60a4740086b3b9c56195b98d91a7b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("13237c49074a3da078dc1d828bb78c6f");
            plain = "00000000000000000000000000000000";
            cipher = "8146a08e2357f0caa30ca8c94d1a0544";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("3071a2a48fe6cbd04f1a129098e308f8");
            plain = "00000000000000000000000000000000";
            cipher = "4b98e06d356deb07ebb824e5713f7be3";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("90f42ec0f68385f2ffc5dfc03a654dce");
            plain = "00000000000000000000000000000000";
            cipher = "7a20a53d460fc9ce0423a7a0764c6cf2";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
        } catch (AesBadKeyException e) {
            e.printStackTrace();
        }
    }

    public void testECBKeySbox192d() {
        byte[] key;
        byte[] state;
        String plain;
        String cipher;
        AesTestHelper aes;
        try {
            key = AesdroidActivity.hexStringToByteArray("e9f065d7c13573587f7875357dfbb16c53489f6a4bd0f7cd");
            plain = "00000000000000000000000000000000";
            cipher = "0956259c9cd5cfd0181cca53380cde06";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("15d20f6ebc7e649fd95b76b107e6daba967c8a9484797f29");
            plain = "00000000000000000000000000000000";
            cipher = "8e4e18424e591a3d5b6f0876f16f8594";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("a8a282ee31c03fae4f8e9b8930d5473c2ed695a347e88b7c");
            plain = "00000000000000000000000000000000";
            cipher = "93f3270cfc877ef17e106ce938979cb0";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("cd62376d5ebb414917f0c78f05266433dc9192a1ec943300");
            plain = "00000000000000000000000000000000";
            cipher = "7f6c25ff41858561bb62f36492e93c29";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("502a6ab36984af268bf423c7f509205207fc1552af4a91e5");
            plain = "00000000000000000000000000000000";
            cipher = "8e06556dcbb00b809a025047cff2a940";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("25a39dbfd8034f71a81f9ceb55026e4037f8f6aa30ab44ce");
            plain = "00000000000000000000000000000000";
            cipher = "3608c344868e94555d23a120f8a5502d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("e08c15411774ec4a908b64eadc6ac4199c7cd453f3aaef53");
            plain = "00000000000000000000000000000000";
            cipher = "77da2021935b840b7f5dcc39132da9e5";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("3b375a1ff7e8d44409696e6326ec9dec86138e2ae010b980");
            plain = "00000000000000000000000000000000";
            cipher = "3b7c24f825e3bf9873c9f14d39a0e6f4";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("950bb9f22cc35be6fe79f52c320af93dec5bc9c0c2f9cd53");
            plain = "00000000000000000000000000000000";
            cipher = "64ebf95686b353508c90ecd8b6134316";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("7001c487cc3e572cfc92f4d0e697d982e8856fdcc957da40");
            plain = "00000000000000000000000000000000";
            cipher = "ff558c5d27210b7929b73fc708eb4cf1";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("f029ce61d4e5a405b41ead0a883cc6a737da2cf50a6c92ae");
            plain = "00000000000000000000000000000000";
            cipher = "a2c3b2a818075490a7b4c14380f02702";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("61257134a518a0d57d9d244d45f6498cbc32f2bafc522d79");
            plain = "00000000000000000000000000000000";
            cipher = "cfe4d74002696ccf7d87b14a2f9cafc9";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("b0ab0a6a818baef2d11fa33eac947284fb7d748cfb75e570");
            plain = "00000000000000000000000000000000";
            cipher = "d2eafd86f63b109b91f5dbb3a3fb7e13";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ee053aa011c8b428cdcc3636313c54d6a03cac01c71579d6");
            plain = "00000000000000000000000000000000";
            cipher = "9b9fdd1c5975655f539998b306a324af";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("d2926527e0aa9f37b45e2ec2ade5853ef807576104c7ace3");
            plain = "00000000000000000000000000000000";
            cipher = "dd619e1cf204446112e0af2b9afa8f8c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("982215f4e173dfa0fcffe5d3da41c4812c7bcc8ed3540f93");
            plain = "00000000000000000000000000000000";
            cipher = "d4f0aae13c8fe9339fbf9e69ed0ad74d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("98c6b8e01e379fbd14e61af6af891596583565f2a27d59e9");
            plain = "00000000000000000000000000000000";
            cipher = "19c80ec4a6deb7e5ed1033dda933498f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("b3ad5cea1dddc214ca969ac35f37dae1a9a9d1528f89bb35");
            plain = "00000000000000000000000000000000";
            cipher = "3cf5e1d21a17956d1dffad6a7c41c659";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("45899367c3132849763073c435a9288a766c8b9ec2308516");
            plain = "00000000000000000000000000000000";
            cipher = "69fd12e8505f8ded2fdcb197a121b362";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ec250e04c3903f602647b85a401a1ae7ca2f02f67fa4253e");
            plain = "00000000000000000000000000000000";
            cipher = "8aa584e2cc4d17417a97cb9a28ba29c8";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("d077a03bd8a38973928ccafe4a9d2f455130bd0af5ae46a9");
            plain = "00000000000000000000000000000000";
            cipher = "abc786fb1edb504580c4d882ef29a0c7";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("d184c36cf0dddfec39e654195006022237871a47c33d3198");
            plain = "00000000000000000000000000000000";
            cipher = "2e19fb60a3e1de0166f483c97824a978";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("4c6994ffa9dcdc805b60c2c0095334c42d95a8fc0ca5b080");
            plain = "00000000000000000000000000000000";
            cipher = "7656709538dd5fec41e0ce6a0f8e207d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
        } catch (AesBadKeyException e) {
            e.printStackTrace();
        }
    }

    public void testECBKeySbox192e() {
        byte[] key;
        byte[] state;
        String plain;
        String cipher;
        AesTestHelper aes;
        try {
            key = AesdroidActivity.hexStringToByteArray("e9f065d7c13573587f7875357dfbb16c53489f6a4bd0f7cd");
            plain = "00000000000000000000000000000000";
            cipher = "0956259c9cd5cfd0181cca53380cde06";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("15d20f6ebc7e649fd95b76b107e6daba967c8a9484797f29");
            plain = "00000000000000000000000000000000";
            cipher = "8e4e18424e591a3d5b6f0876f16f8594";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("a8a282ee31c03fae4f8e9b8930d5473c2ed695a347e88b7c");
            plain = "00000000000000000000000000000000";
            cipher = "93f3270cfc877ef17e106ce938979cb0";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("cd62376d5ebb414917f0c78f05266433dc9192a1ec943300");
            plain = "00000000000000000000000000000000";
            cipher = "7f6c25ff41858561bb62f36492e93c29";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("502a6ab36984af268bf423c7f509205207fc1552af4a91e5");
            plain = "00000000000000000000000000000000";
            cipher = "8e06556dcbb00b809a025047cff2a940";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("25a39dbfd8034f71a81f9ceb55026e4037f8f6aa30ab44ce");
            plain = "00000000000000000000000000000000";
            cipher = "3608c344868e94555d23a120f8a5502d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("e08c15411774ec4a908b64eadc6ac4199c7cd453f3aaef53");
            plain = "00000000000000000000000000000000";
            cipher = "77da2021935b840b7f5dcc39132da9e5";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("3b375a1ff7e8d44409696e6326ec9dec86138e2ae010b980");
            plain = "00000000000000000000000000000000";
            cipher = "3b7c24f825e3bf9873c9f14d39a0e6f4";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("950bb9f22cc35be6fe79f52c320af93dec5bc9c0c2f9cd53");
            plain = "00000000000000000000000000000000";
            cipher = "64ebf95686b353508c90ecd8b6134316";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("7001c487cc3e572cfc92f4d0e697d982e8856fdcc957da40");
            plain = "00000000000000000000000000000000";
            cipher = "ff558c5d27210b7929b73fc708eb4cf1";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("f029ce61d4e5a405b41ead0a883cc6a737da2cf50a6c92ae");
            plain = "00000000000000000000000000000000";
            cipher = "a2c3b2a818075490a7b4c14380f02702";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("61257134a518a0d57d9d244d45f6498cbc32f2bafc522d79");
            plain = "00000000000000000000000000000000";
            cipher = "cfe4d74002696ccf7d87b14a2f9cafc9";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("b0ab0a6a818baef2d11fa33eac947284fb7d748cfb75e570");
            plain = "00000000000000000000000000000000";
            cipher = "d2eafd86f63b109b91f5dbb3a3fb7e13";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ee053aa011c8b428cdcc3636313c54d6a03cac01c71579d6");
            plain = "00000000000000000000000000000000";
            cipher = "9b9fdd1c5975655f539998b306a324af";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("d2926527e0aa9f37b45e2ec2ade5853ef807576104c7ace3");
            plain = "00000000000000000000000000000000";
            cipher = "dd619e1cf204446112e0af2b9afa8f8c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("982215f4e173dfa0fcffe5d3da41c4812c7bcc8ed3540f93");
            plain = "00000000000000000000000000000000";
            cipher = "d4f0aae13c8fe9339fbf9e69ed0ad74d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("98c6b8e01e379fbd14e61af6af891596583565f2a27d59e9");
            plain = "00000000000000000000000000000000";
            cipher = "19c80ec4a6deb7e5ed1033dda933498f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("b3ad5cea1dddc214ca969ac35f37dae1a9a9d1528f89bb35");
            plain = "00000000000000000000000000000000";
            cipher = "3cf5e1d21a17956d1dffad6a7c41c659";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("45899367c3132849763073c435a9288a766c8b9ec2308516");
            plain = "00000000000000000000000000000000";
            cipher = "69fd12e8505f8ded2fdcb197a121b362";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ec250e04c3903f602647b85a401a1ae7ca2f02f67fa4253e");
            plain = "00000000000000000000000000000000";
            cipher = "8aa584e2cc4d17417a97cb9a28ba29c8";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("d077a03bd8a38973928ccafe4a9d2f455130bd0af5ae46a9");
            plain = "00000000000000000000000000000000";
            cipher = "abc786fb1edb504580c4d882ef29a0c7";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("d184c36cf0dddfec39e654195006022237871a47c33d3198");
            plain = "00000000000000000000000000000000";
            cipher = "2e19fb60a3e1de0166f483c97824a978";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("4c6994ffa9dcdc805b60c2c0095334c42d95a8fc0ca5b080");
            plain = "00000000000000000000000000000000";
            cipher = "7656709538dd5fec41e0ce6a0f8e207d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
        } catch (AesBadKeyException e) {
            e.printStackTrace();
        }
    }

    public void testECBKeySbox256d() {
        byte[] key;
        byte[] state;
        String plain;
        String cipher;
        AesTestHelper aes;
        try {
            key = AesdroidActivity
                    .hexStringToByteArray("c47b0294dbbbee0fec4757f22ffeee3587ca4730c3d33b691df38bab076bc558");
            plain = "00000000000000000000000000000000";
            cipher = "46f2fb342d6f0ab477476fc501242c5f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("28d46cffa158533194214a91e712fc2b45b518076675affd910edeca5f41ac64");
            plain = "00000000000000000000000000000000";
            cipher = "4bf3b0a69aeb6657794f2901b1440ad4";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("c1cc358b449909a19436cfbb3f852ef8bcb5ed12ac7058325f56e6099aab1a1c");
            plain = "00000000000000000000000000000000";
            cipher = "352065272169abf9856843927d0674fd";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("984ca75f4ee8d706f46c2d98c0bf4a45f5b00d791c2dfeb191b5ed8e420fd627");
            plain = "00000000000000000000000000000000";
            cipher = "4307456a9e67813b452e15fa8fffe398";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("b43d08a447ac8609baadae4ff12918b9f68fc1653f1269222f123981ded7a92f");
            plain = "00000000000000000000000000000000";
            cipher = "4663446607354989477a5c6f0f007ef4";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("1d85a181b54cde51f0e098095b2962fdc93b51fe9b88602b3f54130bf76a5bd9");
            plain = "00000000000000000000000000000000";
            cipher = "531c2c38344578b84d50b3c917bbb6e1";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("dc0eba1f2232a7879ded34ed8428eeb8769b056bbaf8ad77cb65c3541430b4cf");
            plain = "00000000000000000000000000000000";
            cipher = "fc6aec906323480005c58e7e1ab004ad";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("f8be9ba615c5a952cabbca24f68f8593039624d524c816acda2c9183bd917cb9");
            plain = "00000000000000000000000000000000";
            cipher = "a3944b95ca0b52043584ef02151926a8";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("797f8b3d176dac5b7e34a2d539c4ef367a16f8635f6264737591c5c07bf57a3e");
            plain = "00000000000000000000000000000000";
            cipher = "a74289fe73a4c123ca189ea1e1b49ad5";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("6838d40caf927749c13f0329d331f448e202c73ef52c5f73a37ca635d4c47707");
            plain = "00000000000000000000000000000000";
            cipher = "b91d4ea4488644b56cf0812fa7fcf5fc";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ccd1bc3c659cd3c59bc437484e3c5c724441da8d6e90ce556cd57d0752663bbc");
            plain = "00000000000000000000000000000000";
            cipher = "304f81ab61a80c2e743b94d5002a126b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("13428b5e4c005e0636dd338405d173ab135dec2a25c22c5df0722d69dcc43887");
            plain = "00000000000000000000000000000000";
            cipher = "649a71545378c783e368c9ade7114f6c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("07eb03a08d291d1b07408bf3512ab40c91097ac77461aad4bb859647f74f00ee");
            plain = "00000000000000000000000000000000";
            cipher = "47cb030da2ab051dfc6c4bf6910d12bb";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("90143ae20cd78c5d8ebdd6cb9dc1762427a96c78c639bccc41a61424564eafe1");
            plain = "00000000000000000000000000000000";
            cipher = "798c7c005dee432b2c8ea5dfa381ecc3";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("b7a5794d52737475d53d5a377200849be0260a67a2b22ced8bbef12882270d07");
            plain = "00000000000000000000000000000000";
            cipher = "637c31dc2591a07636f646b72daabbe7";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
        } catch (AesBadKeyException e) {
            e.printStackTrace();
        }
    }

    public void testECBKeySbox256e() {
        byte[] key;
        byte[] state;
        String plain;
        String cipher;
        AesTestHelper aes;
        try {
            key = AesdroidActivity
                    .hexStringToByteArray("c47b0294dbbbee0fec4757f22ffeee3587ca4730c3d33b691df38bab076bc558");
            plain = "00000000000000000000000000000000";
            cipher = "46f2fb342d6f0ab477476fc501242c5f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("28d46cffa158533194214a91e712fc2b45b518076675affd910edeca5f41ac64");
            plain = "00000000000000000000000000000000";
            cipher = "4bf3b0a69aeb6657794f2901b1440ad4";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("c1cc358b449909a19436cfbb3f852ef8bcb5ed12ac7058325f56e6099aab1a1c");
            plain = "00000000000000000000000000000000";
            cipher = "352065272169abf9856843927d0674fd";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("984ca75f4ee8d706f46c2d98c0bf4a45f5b00d791c2dfeb191b5ed8e420fd627");
            plain = "00000000000000000000000000000000";
            cipher = "4307456a9e67813b452e15fa8fffe398";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("b43d08a447ac8609baadae4ff12918b9f68fc1653f1269222f123981ded7a92f");
            plain = "00000000000000000000000000000000";
            cipher = "4663446607354989477a5c6f0f007ef4";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("1d85a181b54cde51f0e098095b2962fdc93b51fe9b88602b3f54130bf76a5bd9");
            plain = "00000000000000000000000000000000";
            cipher = "531c2c38344578b84d50b3c917bbb6e1";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("dc0eba1f2232a7879ded34ed8428eeb8769b056bbaf8ad77cb65c3541430b4cf");
            plain = "00000000000000000000000000000000";
            cipher = "fc6aec906323480005c58e7e1ab004ad";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("f8be9ba615c5a952cabbca24f68f8593039624d524c816acda2c9183bd917cb9");
            plain = "00000000000000000000000000000000";
            cipher = "a3944b95ca0b52043584ef02151926a8";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("797f8b3d176dac5b7e34a2d539c4ef367a16f8635f6264737591c5c07bf57a3e");
            plain = "00000000000000000000000000000000";
            cipher = "a74289fe73a4c123ca189ea1e1b49ad5";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("6838d40caf927749c13f0329d331f448e202c73ef52c5f73a37ca635d4c47707");
            plain = "00000000000000000000000000000000";
            cipher = "b91d4ea4488644b56cf0812fa7fcf5fc";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ccd1bc3c659cd3c59bc437484e3c5c724441da8d6e90ce556cd57d0752663bbc");
            plain = "00000000000000000000000000000000";
            cipher = "304f81ab61a80c2e743b94d5002a126b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("13428b5e4c005e0636dd338405d173ab135dec2a25c22c5df0722d69dcc43887");
            plain = "00000000000000000000000000000000";
            cipher = "649a71545378c783e368c9ade7114f6c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("07eb03a08d291d1b07408bf3512ab40c91097ac77461aad4bb859647f74f00ee");
            plain = "00000000000000000000000000000000";
            cipher = "47cb030da2ab051dfc6c4bf6910d12bb";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("90143ae20cd78c5d8ebdd6cb9dc1762427a96c78c639bccc41a61424564eafe1");
            plain = "00000000000000000000000000000000";
            cipher = "798c7c005dee432b2c8ea5dfa381ecc3";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("b7a5794d52737475d53d5a377200849be0260a67a2b22ced8bbef12882270d07");
            plain = "00000000000000000000000000000000";
            cipher = "637c31dc2591a07636f646b72daabbe7";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
        } catch (AesBadKeyException e) {
            e.printStackTrace();
        }
    }

    public void testECBVarKey128d() {
        byte[] key;
        byte[] state;
        String plain;
        String cipher;
        AesTestHelper aes;
        try {
            key = AesdroidActivity.hexStringToByteArray("80000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "0edd33d3c621e546455bd8ba1418bec8";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("c0000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "4bc3f883450c113c64ca42e1112a9e87";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("e0000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "72a1da770f5d7ac4c9ef94d822affd97";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("f0000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "970014d634e2b7650777e8e84d03ccd8";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("f8000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "f17e79aed0db7e279e955b5f493875a7";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fc000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "9ed5a75136a940d0963da379db4af26a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fe000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "c4295f83465c7755e8fa364bac6a7ea5";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ff000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "b1d758256b28fd850ad4944208cf1155";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ff800000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "42ffb34c743de4d88ca38011c990890b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffc00000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "9958f0ecea8b2172c0c1995f9182c0f3";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffe00000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "956d7798fac20f82a8823f984d06f7f5";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fff00000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "a01bf44f2d16be928ca44aaf7b9b106b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fff80000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "b5f1a33e50d40d103764c76bd4c6b6f8";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffc0000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "2637050c9fc0d4817e2d69de878aee8d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffe0000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "113ecbe4a453269a0dd26069467fb5b5";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffff0000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "97d0754fe68f11b9e375d070a608c884";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffff8000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "c6a0b3e998d05068a5399778405200b4";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffc000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "df556a33438db87bc41b1752c55e5e49";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffe000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "90fb128d3a1af6e548521bb962bf1f05";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffff000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "26298e9c1db517c215fadfb7d2a8d691";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffff800000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "a6cb761d61f8292d0df393a279ad0380";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffc00000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "12acd89b13cd5f8726e34d44fd486108";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffe00000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "95b1703fc57ba09fe0c3580febdd7ed4";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffff00000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "de11722d893e9f9121c381becc1da59a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffff80000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "6d114ccb27bf391012e8974c546d9bf2";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffc0000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "5ce37e17eb4646ecfac29b9cc38d9340";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffe0000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "18c1b6e2157122056d0243d8a165cddb";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffff0000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "99693e6a59d1366c74d823562d7e1431";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffff8000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "6c7c64dc84a8bba758ed17eb025a57e3";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffc000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "e17bc79f30eaab2fac2cbbe3458d687a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffe000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "1114bc2028009b923f0b01915ce5e7c4";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffff000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "9c28524a16a1e1c1452971caa8d13476";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffff800000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "ed62e16363638360fdd6ad62112794f0";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffc00000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "5a8688f0b2a2c16224c161658ffd4044";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffe00000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "23f710842b9bb9c32f26648c786807ca";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffff00000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "44a98bf11e163f632c47ec6a49683a89";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffff80000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "0f18aff94274696d9b61848bd50ac5e5";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffc0000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "82408571c3e2424540207f833b6dda69";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffe0000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "303ff996947f0c7d1f43c8f3027b9b75";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffff0000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "7df4daf4ad29a3615a9b6ece5c99518a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffff8000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "c72954a48d0774db0b4971c526260415";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffc000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "1df9b76112dc6531e07d2cfda04411f0";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffe000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "8e4d8e699119e1fc87545a647fb1d34f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffff000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "e6c4807ae11f36f091c57d9fb68548d1";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffff800000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "8ebf73aad49c82007f77a5c1ccec6ab4";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffc00000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "4fb288cc2040049001d2c7585ad123fc";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffe00000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "04497110efb9dceb13e2b13fb4465564";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffff00000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "75550e6cb5a88e49634c9ab69eda0430";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffff80000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "b6768473ce9843ea66a81405dd50b345";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffc0000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "cb2f430383f9084e03a653571e065de6";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffe0000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "ff4e66c07bae3e79fb7d210847a3b0ba";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffff0000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "7b90785125505fad59b13c186dd66ce3";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffff8000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "8b527a6aebdaec9eaef8eda2cb7783e5";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffc000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "43fdaf53ebbc9880c228617d6a9b548b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffe000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "53786104b9744b98f052c46f1c850d0b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffff000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "b5ab3013dd1e61df06cbaf34ca2aee78";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffff800000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "7470469be9723030fdcc73a8cd4fbb10";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffc00000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "a35a63f5343ebe9ef8167bcb48ad122e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffe00000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "fd8687f0757a210e9fdf181204c30863";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffff00000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "7a181e84bd5457d26a88fbae96018fb0";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffff80000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "653317b9362b6f9b9e1a580e68d494b5";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffc0000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "995c9dc0b689f03c45867b5faa5c18d1";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffe0000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "77a4d96d56dda398b9aabecfc75729fd";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffff0000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "84be19e053635f09f2665e7bae85b42d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffff8000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "32cd652842926aea4aa6137bb2be2b5e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffc000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "493d4a4f38ebb337d10aa84e9171a554";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffe000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "d9bff7ff454b0ec5a4a2a69566e2cb84";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffff000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "3535d565ace3f31eb249ba2cc6765d7a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffff800000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "f60e91fc3269eecf3231c6e9945697c6";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffc00000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "ab69cfadf51f8e604d9cc37182f6635a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffe00000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "7866373f24a0b6ed56e0d96fcdafb877";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffff00000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "1ea448c2aac954f5d812e9d78494446a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffff80000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "acc5599dd8ac02239a0fef4a36dd1668";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffc0000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "d8764468bb103828cf7e1473ce895073";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffe0000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "1b0d02893683b9f180458e4aa6b73982";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffff0000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "96d9b017d302df410a937dcdb8bb6e43";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffff8000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "ef1623cc44313cff440b1594a7e21cc6";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffc000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "284ca2fa35807b8b0ae4d19e11d7dbd7";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffe000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "f2e976875755f9401d54f36e2a23a594";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffff000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "ec198a18e10e532403b7e20887c8dd80";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffff800000000000");
            plain = "00000000000000000000000000000000";
            cipher = "545d50ebd919e4a6949d96ad47e46a80";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffc00000000000");
            plain = "00000000000000000000000000000000";
            cipher = "dbdfb527060e0a71009c7bb0c68f1d44";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffe00000000000");
            plain = "00000000000000000000000000000000";
            cipher = "9cfa1322ea33da2173a024f2ff0d896d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffff00000000000");
            plain = "00000000000000000000000000000000";
            cipher = "8785b1a75b0f3bd958dcd0e29318c521";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffff80000000000");
            plain = "00000000000000000000000000000000";
            cipher = "38f67b9e98e4a97b6df030a9fcdd0104";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffc0000000000");
            plain = "00000000000000000000000000000000";
            cipher = "192afffb2c880e82b05926d0fc6c448b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffe0000000000");
            plain = "00000000000000000000000000000000";
            cipher = "6a7980ce7b105cf530952d74daaf798c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffff0000000000");
            plain = "00000000000000000000000000000000";
            cipher = "ea3695e1351b9d6858bd958cf513ef6c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffff8000000000");
            plain = "00000000000000000000000000000000";
            cipher = "6da0490ba0ba0343b935681d2cce5ba1";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffc000000000");
            plain = "00000000000000000000000000000000";
            cipher = "f0ea23af08534011c60009ab29ada2f1";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffe000000000");
            plain = "00000000000000000000000000000000";
            cipher = "ff13806cf19cc38721554d7c0fcdcd4b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffff000000000");
            plain = "00000000000000000000000000000000";
            cipher = "6838af1f4f69bae9d85dd188dcdf0688";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffff800000000");
            plain = "00000000000000000000000000000000";
            cipher = "36cf44c92d550bfb1ed28ef583ddf5d7";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffc00000000");
            plain = "00000000000000000000000000000000";
            cipher = "d06e3195b5376f109d5c4ec6c5d62ced";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffe00000000");
            plain = "00000000000000000000000000000000";
            cipher = "c440de014d3d610707279b13242a5c36";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffff00000000");
            plain = "00000000000000000000000000000000";
            cipher = "f0c5c6ffa5e0bd3a94c88f6b6f7c16b9";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffff80000000");
            plain = "00000000000000000000000000000000";
            cipher = "3e40c3901cd7effc22bffc35dee0b4d9";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffc0000000");
            plain = "00000000000000000000000000000000";
            cipher = "b63305c72bedfab97382c406d0c49bc6";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffe0000000");
            plain = "00000000000000000000000000000000";
            cipher = "36bbaab22a6bd4925a99a2b408d2dbae";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffff0000000");
            plain = "00000000000000000000000000000000";
            cipher = "307c5b8fcd0533ab98bc51e27a6ce461";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffff8000000");
            plain = "00000000000000000000000000000000";
            cipher = "829c04ff4c07513c0b3ef05c03e337b5";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffc000000");
            plain = "00000000000000000000000000000000";
            cipher = "f17af0e895dda5eb98efc68066e84c54";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffe000000");
            plain = "00000000000000000000000000000000";
            cipher = "277167f3812afff1ffacb4a934379fc3";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffff000000");
            plain = "00000000000000000000000000000000";
            cipher = "2cb1dc3a9c72972e425ae2ef3eb597cd";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffff800000");
            plain = "00000000000000000000000000000000";
            cipher = "36aeaa3a213e968d4b5b679d3a2c97fe";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffc00000");
            plain = "00000000000000000000000000000000";
            cipher = "9241daca4fdd034a82372db50e1a0f3f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffe00000");
            plain = "00000000000000000000000000000000";
            cipher = "c14574d9cd00cf2b5a7f77e53cd57885";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffff00000");
            plain = "00000000000000000000000000000000";
            cipher = "793de39236570aba83ab9b737cb521c9";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffff80000");
            plain = "00000000000000000000000000000000";
            cipher = "16591c0f27d60e29b85a96c33861a7ef";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffc0000");
            plain = "00000000000000000000000000000000";
            cipher = "44fb5c4d4f5cb79be5c174a3b1c97348";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffe0000");
            plain = "00000000000000000000000000000000";
            cipher = "674d2b61633d162be59dde04222f4740";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffff0000");
            plain = "00000000000000000000000000000000";
            cipher = "b4750ff263a65e1f9e924ccfd98f3e37";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffff8000");
            plain = "00000000000000000000000000000000";
            cipher = "62d0662d6eaeddedebae7f7ea3a4f6b6";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffc000");
            plain = "00000000000000000000000000000000";
            cipher = "70c46bb30692be657f7eaa93ebad9897";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffe000");
            plain = "00000000000000000000000000000000";
            cipher = "323994cfb9da285a5d9642e1759b224a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffff000");
            plain = "00000000000000000000000000000000";
            cipher = "1dbf57877b7b17385c85d0b54851e371";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffff800");
            plain = "00000000000000000000000000000000";
            cipher = "dfa5c097cdc1532ac071d57b1d28d1bd";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffc00");
            plain = "00000000000000000000000000000000";
            cipher = "3a0c53fa37311fc10bd2a9981f513174";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffe00");
            plain = "00000000000000000000000000000000";
            cipher = "ba4f970c0a25c41814bdae2e506be3b4";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffff00");
            plain = "00000000000000000000000000000000";
            cipher = "2dce3acb727cd13ccd76d425ea56e4f6";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffff80");
            plain = "00000000000000000000000000000000";
            cipher = "5160474d504b9b3eefb68d35f245f4b3";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffc0");
            plain = "00000000000000000000000000000000";
            cipher = "41a8a947766635dec37553d9a6c0cbb7";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffe0");
            plain = "00000000000000000000000000000000";
            cipher = "25d6cfe6881f2bf497dd14cd4ddf445b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffff0");
            plain = "00000000000000000000000000000000";
            cipher = "41c78c135ed9e98c096640647265da1e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffff8");
            plain = "00000000000000000000000000000000";
            cipher = "5a4d404d8917e353e92a21072c3b2305";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffc");
            plain = "00000000000000000000000000000000";
            cipher = "02bc96846b3fdc71643f384cd3cc3eaf";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffe");
            plain = "00000000000000000000000000000000";
            cipher = "9ba4a9143f4e5d4048521c4f8877d88e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
        } catch (AesBadKeyException e) {
            e.printStackTrace();
        }
    }

    public void testECBVarKey128e() {
        byte[] key;
        byte[] state;
        String plain;
        String cipher;
        AesTestHelper aes;
        try {
            key = AesdroidActivity.hexStringToByteArray("80000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "0edd33d3c621e546455bd8ba1418bec8";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("c0000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "4bc3f883450c113c64ca42e1112a9e87";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("e0000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "72a1da770f5d7ac4c9ef94d822affd97";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("f0000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "970014d634e2b7650777e8e84d03ccd8";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("f8000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "f17e79aed0db7e279e955b5f493875a7";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fc000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "9ed5a75136a940d0963da379db4af26a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fe000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "c4295f83465c7755e8fa364bac6a7ea5";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ff000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "b1d758256b28fd850ad4944208cf1155";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ff800000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "42ffb34c743de4d88ca38011c990890b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffc00000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "9958f0ecea8b2172c0c1995f9182c0f3";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffe00000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "956d7798fac20f82a8823f984d06f7f5";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fff00000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "a01bf44f2d16be928ca44aaf7b9b106b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fff80000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "b5f1a33e50d40d103764c76bd4c6b6f8";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffc0000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "2637050c9fc0d4817e2d69de878aee8d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffe0000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "113ecbe4a453269a0dd26069467fb5b5";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffff0000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "97d0754fe68f11b9e375d070a608c884";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffff8000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "c6a0b3e998d05068a5399778405200b4";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffc000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "df556a33438db87bc41b1752c55e5e49";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffe000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "90fb128d3a1af6e548521bb962bf1f05";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffff000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "26298e9c1db517c215fadfb7d2a8d691";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffff800000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "a6cb761d61f8292d0df393a279ad0380";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffc00000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "12acd89b13cd5f8726e34d44fd486108";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffe00000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "95b1703fc57ba09fe0c3580febdd7ed4";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffff00000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "de11722d893e9f9121c381becc1da59a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffff80000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "6d114ccb27bf391012e8974c546d9bf2";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffc0000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "5ce37e17eb4646ecfac29b9cc38d9340";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffe0000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "18c1b6e2157122056d0243d8a165cddb";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffff0000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "99693e6a59d1366c74d823562d7e1431";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffff8000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "6c7c64dc84a8bba758ed17eb025a57e3";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffc000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "e17bc79f30eaab2fac2cbbe3458d687a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffe000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "1114bc2028009b923f0b01915ce5e7c4";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffff000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "9c28524a16a1e1c1452971caa8d13476";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffff800000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "ed62e16363638360fdd6ad62112794f0";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffc00000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "5a8688f0b2a2c16224c161658ffd4044";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffe00000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "23f710842b9bb9c32f26648c786807ca";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffff00000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "44a98bf11e163f632c47ec6a49683a89";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffff80000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "0f18aff94274696d9b61848bd50ac5e5";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffc0000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "82408571c3e2424540207f833b6dda69";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffe0000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "303ff996947f0c7d1f43c8f3027b9b75";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffff0000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "7df4daf4ad29a3615a9b6ece5c99518a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffff8000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "c72954a48d0774db0b4971c526260415";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffc000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "1df9b76112dc6531e07d2cfda04411f0";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffe000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "8e4d8e699119e1fc87545a647fb1d34f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffff000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "e6c4807ae11f36f091c57d9fb68548d1";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffff800000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "8ebf73aad49c82007f77a5c1ccec6ab4";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffc00000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "4fb288cc2040049001d2c7585ad123fc";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffe00000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "04497110efb9dceb13e2b13fb4465564";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffff00000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "75550e6cb5a88e49634c9ab69eda0430";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffff80000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "b6768473ce9843ea66a81405dd50b345";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffc0000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "cb2f430383f9084e03a653571e065de6";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffe0000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "ff4e66c07bae3e79fb7d210847a3b0ba";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffff0000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "7b90785125505fad59b13c186dd66ce3";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffff8000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "8b527a6aebdaec9eaef8eda2cb7783e5";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffc000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "43fdaf53ebbc9880c228617d6a9b548b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffe000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "53786104b9744b98f052c46f1c850d0b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffff000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "b5ab3013dd1e61df06cbaf34ca2aee78";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffff800000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "7470469be9723030fdcc73a8cd4fbb10";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffc00000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "a35a63f5343ebe9ef8167bcb48ad122e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffe00000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "fd8687f0757a210e9fdf181204c30863";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffff00000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "7a181e84bd5457d26a88fbae96018fb0";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffff80000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "653317b9362b6f9b9e1a580e68d494b5";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffc0000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "995c9dc0b689f03c45867b5faa5c18d1";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffe0000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "77a4d96d56dda398b9aabecfc75729fd";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffff0000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "84be19e053635f09f2665e7bae85b42d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffff8000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "32cd652842926aea4aa6137bb2be2b5e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffc000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "493d4a4f38ebb337d10aa84e9171a554";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffe000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "d9bff7ff454b0ec5a4a2a69566e2cb84";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffff000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "3535d565ace3f31eb249ba2cc6765d7a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffff800000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "f60e91fc3269eecf3231c6e9945697c6";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffc00000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "ab69cfadf51f8e604d9cc37182f6635a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffe00000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "7866373f24a0b6ed56e0d96fcdafb877";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffff00000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "1ea448c2aac954f5d812e9d78494446a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffff80000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "acc5599dd8ac02239a0fef4a36dd1668";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffc0000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "d8764468bb103828cf7e1473ce895073";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffe0000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "1b0d02893683b9f180458e4aa6b73982";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffff0000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "96d9b017d302df410a937dcdb8bb6e43";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffff8000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "ef1623cc44313cff440b1594a7e21cc6";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffc000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "284ca2fa35807b8b0ae4d19e11d7dbd7";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffe000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "f2e976875755f9401d54f36e2a23a594";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffff000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "ec198a18e10e532403b7e20887c8dd80";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffff800000000000");
            plain = "00000000000000000000000000000000";
            cipher = "545d50ebd919e4a6949d96ad47e46a80";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffc00000000000");
            plain = "00000000000000000000000000000000";
            cipher = "dbdfb527060e0a71009c7bb0c68f1d44";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffe00000000000");
            plain = "00000000000000000000000000000000";
            cipher = "9cfa1322ea33da2173a024f2ff0d896d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffff00000000000");
            plain = "00000000000000000000000000000000";
            cipher = "8785b1a75b0f3bd958dcd0e29318c521";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffff80000000000");
            plain = "00000000000000000000000000000000";
            cipher = "38f67b9e98e4a97b6df030a9fcdd0104";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffc0000000000");
            plain = "00000000000000000000000000000000";
            cipher = "192afffb2c880e82b05926d0fc6c448b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffe0000000000");
            plain = "00000000000000000000000000000000";
            cipher = "6a7980ce7b105cf530952d74daaf798c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffff0000000000");
            plain = "00000000000000000000000000000000";
            cipher = "ea3695e1351b9d6858bd958cf513ef6c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffff8000000000");
            plain = "00000000000000000000000000000000";
            cipher = "6da0490ba0ba0343b935681d2cce5ba1";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffc000000000");
            plain = "00000000000000000000000000000000";
            cipher = "f0ea23af08534011c60009ab29ada2f1";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffe000000000");
            plain = "00000000000000000000000000000000";
            cipher = "ff13806cf19cc38721554d7c0fcdcd4b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffff000000000");
            plain = "00000000000000000000000000000000";
            cipher = "6838af1f4f69bae9d85dd188dcdf0688";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffff800000000");
            plain = "00000000000000000000000000000000";
            cipher = "36cf44c92d550bfb1ed28ef583ddf5d7";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffc00000000");
            plain = "00000000000000000000000000000000";
            cipher = "d06e3195b5376f109d5c4ec6c5d62ced";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffe00000000");
            plain = "00000000000000000000000000000000";
            cipher = "c440de014d3d610707279b13242a5c36";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffff00000000");
            plain = "00000000000000000000000000000000";
            cipher = "f0c5c6ffa5e0bd3a94c88f6b6f7c16b9";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffff80000000");
            plain = "00000000000000000000000000000000";
            cipher = "3e40c3901cd7effc22bffc35dee0b4d9";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffc0000000");
            plain = "00000000000000000000000000000000";
            cipher = "b63305c72bedfab97382c406d0c49bc6";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffe0000000");
            plain = "00000000000000000000000000000000";
            cipher = "36bbaab22a6bd4925a99a2b408d2dbae";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffff0000000");
            plain = "00000000000000000000000000000000";
            cipher = "307c5b8fcd0533ab98bc51e27a6ce461";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffff8000000");
            plain = "00000000000000000000000000000000";
            cipher = "829c04ff4c07513c0b3ef05c03e337b5";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffc000000");
            plain = "00000000000000000000000000000000";
            cipher = "f17af0e895dda5eb98efc68066e84c54";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffe000000");
            plain = "00000000000000000000000000000000";
            cipher = "277167f3812afff1ffacb4a934379fc3";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffff000000");
            plain = "00000000000000000000000000000000";
            cipher = "2cb1dc3a9c72972e425ae2ef3eb597cd";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffff800000");
            plain = "00000000000000000000000000000000";
            cipher = "36aeaa3a213e968d4b5b679d3a2c97fe";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffc00000");
            plain = "00000000000000000000000000000000";
            cipher = "9241daca4fdd034a82372db50e1a0f3f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffe00000");
            plain = "00000000000000000000000000000000";
            cipher = "c14574d9cd00cf2b5a7f77e53cd57885";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffff00000");
            plain = "00000000000000000000000000000000";
            cipher = "793de39236570aba83ab9b737cb521c9";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffff80000");
            plain = "00000000000000000000000000000000";
            cipher = "16591c0f27d60e29b85a96c33861a7ef";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffc0000");
            plain = "00000000000000000000000000000000";
            cipher = "44fb5c4d4f5cb79be5c174a3b1c97348";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffe0000");
            plain = "00000000000000000000000000000000";
            cipher = "674d2b61633d162be59dde04222f4740";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffff0000");
            plain = "00000000000000000000000000000000";
            cipher = "b4750ff263a65e1f9e924ccfd98f3e37";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffff8000");
            plain = "00000000000000000000000000000000";
            cipher = "62d0662d6eaeddedebae7f7ea3a4f6b6";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffc000");
            plain = "00000000000000000000000000000000";
            cipher = "70c46bb30692be657f7eaa93ebad9897";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffe000");
            plain = "00000000000000000000000000000000";
            cipher = "323994cfb9da285a5d9642e1759b224a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffff000");
            plain = "00000000000000000000000000000000";
            cipher = "1dbf57877b7b17385c85d0b54851e371";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffff800");
            plain = "00000000000000000000000000000000";
            cipher = "dfa5c097cdc1532ac071d57b1d28d1bd";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffc00");
            plain = "00000000000000000000000000000000";
            cipher = "3a0c53fa37311fc10bd2a9981f513174";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffe00");
            plain = "00000000000000000000000000000000";
            cipher = "ba4f970c0a25c41814bdae2e506be3b4";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffff00");
            plain = "00000000000000000000000000000000";
            cipher = "2dce3acb727cd13ccd76d425ea56e4f6";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffff80");
            plain = "00000000000000000000000000000000";
            cipher = "5160474d504b9b3eefb68d35f245f4b3";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffc0");
            plain = "00000000000000000000000000000000";
            cipher = "41a8a947766635dec37553d9a6c0cbb7";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffe0");
            plain = "00000000000000000000000000000000";
            cipher = "25d6cfe6881f2bf497dd14cd4ddf445b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffff0");
            plain = "00000000000000000000000000000000";
            cipher = "41c78c135ed9e98c096640647265da1e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffff8");
            plain = "00000000000000000000000000000000";
            cipher = "5a4d404d8917e353e92a21072c3b2305";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffc");
            plain = "00000000000000000000000000000000";
            cipher = "02bc96846b3fdc71643f384cd3cc3eaf";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffe");
            plain = "00000000000000000000000000000000";
            cipher = "9ba4a9143f4e5d4048521c4f8877d88e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
        } catch (AesBadKeyException e) {
            e.printStackTrace();
        }
    }

    public void testECBVarKey192d() {
        byte[] key;
        byte[] state;
        String plain;
        String cipher;
        AesTestHelper aes;
        try {
            key = AesdroidActivity.hexStringToByteArray("800000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "de885dc87f5a92594082d02cc1e1b42c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("c00000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "132b074e80f2a597bf5febd8ea5da55e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("e00000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "6eccedf8de592c22fb81347b79f2db1f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("f00000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "180b09f267c45145db2f826c2582d35c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("f80000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "edd807ef7652d7eb0e13c8b5e15b3bc0";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fc0000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "9978bcf8dd8fd72241223ad24b31b8a4";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fe0000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "5310f654343e8f27e12c83a48d24ff81";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ff0000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "833f71258d53036b02952c76c744f5a1";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ff8000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "eba83ff200cff9318a92f8691a06b09f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffc000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "ff620ccbe9f3292abdf2176b09f04eba";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffe000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "7ababc4b3f516c9aafb35f4140b548f9";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fff000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "aa187824d9c4582b0916493ecbde8c57";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fff800000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "1c0ad553177fd5ea1092c9d626a29dc4";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffc00000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "a5dc46c37261194124ecaebd680408ec";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffe00000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "e4f2f2ae23e9b10bacfa58601531ba54";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffff00000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "b7d67cf1a1e91e8ff3a57a172c7bf412";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffff80000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "26706be06967884e847d137128ce47b3";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffc0000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "b2f8b409b0585909aad3a7b5a219072a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffe0000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "5e4b7bff0290c78344c54a23b722cd20";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffff0000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "07093657552d4414227ce161e9ebf7dd";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffff8000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "e1af1e7d8bc225ed4dffb771ecbb9e67";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffc000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "ef6555253635d8432156cfd9c11b145a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffe000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "fb4035074a5d4260c90cbd6da6c3fceb";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffff000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "446ee416f9ad1c103eb0cc96751c88e1";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffff800000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "198ae2a4637ac0a7890a8fd1485445c9";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffc00000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "562012ec8faded0825fb2fa70ab30cbd";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffe00000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "cc8a64b46b5d88bf7f247d4dbaf38f05";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffff00000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "a168253762e2cc81b42d1e5001762699";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffff80000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "1b41f83b38ce5032c6cd7af98cf62061";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffc0000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "61a89990cd1411750d5fb0dc988447d4";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffe0000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "b5accc8ed629edf8c68a539183b1ea82";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffff0000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "b16fa71f846b81a13f361c43a851f290";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffff8000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "4fad6efdff5975aee7692234bcd54488";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffc000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "ebfdb05a783d03082dfe5fdd80a00b17";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffe000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "eb81b584766997af6ba5529d3bdd8609";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffff000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "0cf4ff4f49c8a0ca060c443499e29313";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffff800000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "cc4ba8a8e029f8b26d8afff9df133bb6";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffc00000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "fefebf64360f38e4e63558f0ffc550c3";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffe00000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "12ad98cbf725137d6a8108c2bed99322";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffff00000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "6afaa996226198b3e2610413ce1b3f78";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffff80000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "2a8ce6747a7e39367828e290848502d9";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffc0000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "223736e8b8f89ca1e37b6deab40facf1";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffe0000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "c0f797e50418b95fa6013333917a9480";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffff0000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "a758de37c2ece2a02c73c01fedc9a132";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffff8000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "3a9b87ae77bae706803966c66c73adbd";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffc000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "d365ab8df8ffd782e358121a4a4fc541";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffe000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "c8dcd9e6f75e6c36c8daee0466f0ed74";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffff000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "c79a637beb1c0304f14014c037e736dd";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffff800000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "105f0a25e84ac930d996281a5f954dd9";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffc00000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "42e4074b2927973e8d17ffa92f7fe615";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffe00000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "4fe2a9d2c1824449c69e3e0398f12963";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffff00000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "b7f29c1e1f62847a15253b28a1e9d712";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffff80000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "36ed5d29b903f31e8983ef8b0a2bf990";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffc0000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "27b8070270810f9d023f9dd7ff3b4aa2";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffe0000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "94d46e155c1228f61d1a0db4815ecc4b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffff0000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "ca6108d1d98071428eeceef1714b96dd";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffff8000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "dc5b25b71b6296cf73dd2cdcac2f70b1";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffc000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "44aba95e8a06a2d9d3530d2677878c80";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffe000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "a570d20e89b467e8f5176061b81dd396";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffff000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "758f4467a5d8f1e7307dc30b34e404f4";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffff800000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "bcea28e9071b5a2302970ff352451bc5";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffc00000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "7523c00bc177d331ad312e09c9015c1c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffe00000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "ccac61e3183747b3f5836da21a1bc4f4";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffff00000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "707b075791878880b44189d3522b8c30";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffff80000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "7132d0c0e4a07593cf12ebb12be7688c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffc0000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "effbac1644deb0c784275fe56e19ead3";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffe0000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "a005063f30f4228b374e2459738f26bb";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffff0000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "29975b5f48bb68fcbbc7cea93b452ed7";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffff8000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "cf3f2576e2afedc74bb1ca7eeec1c0e7";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffc000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "07c403f5f966e0e3d9f296d6226dca28";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffe000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "c8c20908249ab4a34d6dd0a31327ff1a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffff000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "c0541329ecb6159ab23b7fc5e6a21bca";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffff800000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "7aa1acf1a2ed9ba72bc6deb31d88b863";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffc00000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "808bd8eddabb6f3bf0d5a8a27be1fe8a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffe00000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "273c7d7685e14ec66bbb96b8f05b6ddd";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffff00000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "32752eefc8c2a93f91b6e73eb07cca6e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffff80000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "d893e7d62f6ce502c64f75e281f9c000";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffc0000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "8dfd999be5d0cfa35732c0ddc88ff5a5";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffe0000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "02647c76a300c3173b841487eb2bae9f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffff0000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "172df8b02f04b53adab028b4e01acd87";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffff8000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "054b3bf4998aeb05afd87ec536533a36";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffc000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "3783f7bf44c97f065258a666cae03020";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffe000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "aad4c8a63f80954104de7b92cede1be1";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffff000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "cbfe61810fd5467ccdacb75800f3ac07";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffff800000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "830d8a2590f7d8e1b55a737f4af45f34";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffc00000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "fffcd4683f858058e74314671d43fa2c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffe00000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "523d0babbb82f46ebc9e70b1cd41ddd0";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffff00000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "344aab37080d7486f7d542a309e53eed";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffff80000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "56c5609d0906b23ab9caca816f5dbebd";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffc0000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "7026026eedd91adc6d831cdf9894bdc6";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffe0000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "88330baa4f2b618fc9d9b021bf503d5a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffff0000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "fc9e0ea22480b0bac935c8a8ebefcdcf";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffff8000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "29ca779f398fb04f867da7e8a44756cb";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffc000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "51f89c42985786bfc43c6df8ada36832";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffe000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "6ac1de5fb8f21d874e91c53b560c50e3";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffff000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "03aa9058490eda306001a8a9f48d0ca7";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffff800000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "e34ec71d6128d4871865d617c30b37e3";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffc00000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "14be1c535b17cabd0c4d93529d69bf47";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffe00000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "c9ef67756507beec9dd3862883478044";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffff00000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "40e231fa5a5948ce2134e92fc0664d4b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffff80000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "03194b8e5dda5530d0c678c0b48f5d92";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffc0000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "90bd086f237cc4fd99f4d76bde6b4826";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffe0000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "19259761ca17130d6ed86d57cd7951ee";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffff0000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "d7cbb3f34b9b450f24b0e8518e54da6d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffff8000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "725b9caebe9f7f417f4068d0d2ee20b3";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffc000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "9d924b934a90ce1fd39b8a9794f82672";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffe000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "c50562bf094526a91c5bc63c0c224995";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffff000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "d2f11805046743bd74f57188d9188df7";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffff800000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "8dd274bd0f1b58ae345d9e7233f9b8f3";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffc00000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "9d6bdc8f4ce5feb0f3bed2e4b9a9bb0b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffe00000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "fd5548bcf3f42565f7efa94562528d46";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffff00000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "d2ccaebd3a4c3e80b063748131ba4a71";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffff80000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "e03cb23d9e11c9d93f117e9c0a91b576";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffc0000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "78f933a2081ac1db84f69d10f4523fe0";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffe0000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "4061f7412ed320de0edc8851c2e2436f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffff0000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "9064ba1cd04ce6bab98474330814b4d4";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffff8000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "48391bffb9cfff80ac238c886ef0a461";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffc000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "b8d2a67df5a999fdbf93edd0343296c9";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffe000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "aaca7367396b69a221bd632bea386eec";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffff000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "a80fd5020dfe65f5f16293ec92c6fd89";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffff800000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "2162995b8217a67f1abc342e146406f8";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffc00000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "c6a6164b7a60bae4e986ffac28dfadd9";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffe00000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "64e0d7f900e3d9c83e4b8f96717b2146";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffff00000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "1ad2561de8c1232f5d8dbab4739b6cbb";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffff80000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "279689e9a557f58b1c3bf40c97a90964";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffc0000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "c4637e4a5e6377f9cc5a8638045de029";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffe0000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "492e607e5aea4688594b45f3aee3df90";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffff0000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "e8c4e4381feec74054954c05b777a00a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffff8000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "91549514605f38246c9b724ad839f01d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffc000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "74b24e3b6fefe40a4f9ef7ac6e44d76a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffe000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "2437a683dc5d4b52abb4a123a8df86c6";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffff000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "bb2852c891c5947d2ed44032c421b85f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffff800000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "1b9f5fbd5e8a4264c0a85b80409afa5e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffc00000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "30dab809f85a917fe924733f424ac589";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffe00000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "eaef5c1f8d605192646695ceadc65f32";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffff00000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "b8aa90040b4c15a12316b78e0f9586fc";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffff80000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "97fac8297ceaabc87d454350601e0673";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffffc0000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "9b47ef567ac28dfe488492f157e2b2e0";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffffe0000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "1b8426027ddb962b5c5ba7eb8bc9ab63";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffff0000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "e917fc77e71992a12dbe4c18068bec82";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffff8000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "dceebbc98840f8ae6daf76573b7e56f4";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffffc000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "4e11a9f74205125b61e0aee047eca20d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffffe000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "f60467f55a1f17eab88e800120cbc284";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffffff000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "d436649f600b449ee276530f0cd83c11";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffffff800000000000");
            plain = "00000000000000000000000000000000";
            cipher = "3bc0e3656a9e3ac7cd378a737f53b637";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffffffc00000000000");
            plain = "00000000000000000000000000000000";
            cipher = "6bacae63d33b928aa8380f8d54d88c17";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffffffe00000000000");
            plain = "00000000000000000000000000000000";
            cipher = "8935ffbc75ae6251bf8e859f085adcb9";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffffff00000000000");
            plain = "00000000000000000000000000000000";
            cipher = "93dc4970fe35f67747cb0562c06d875a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffffff80000000000");
            plain = "00000000000000000000000000000000";
            cipher = "14f9df858975851797ba604fb0d16cc7";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffffffc0000000000");
            plain = "00000000000000000000000000000000";
            cipher = "02ea0c98dca10b38c21b3b14e8d1b71f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffffffe0000000000");
            plain = "00000000000000000000000000000000";
            cipher = "8f091b1b5b0749b2adc803e63dda9b72";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffffffff0000000000");
            plain = "00000000000000000000000000000000";
            cipher = "05b389e3322c6da08384345a4137fd08";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffffffff8000000000");
            plain = "00000000000000000000000000000000";
            cipher = "381308c438f35b399f10ad71b05027d8";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffc000000000");
            plain = "00000000000000000000000000000000";
            cipher = "68c230fcfa9279c3409fc423e2acbe04";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffe000000000");
            plain = "00000000000000000000000000000000";
            cipher = "1c84a475acb011f3f59f4f46b76274c0";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffffffff000000000");
            plain = "00000000000000000000000000000000";
            cipher = "45119b68cb3f8399ee60066b5611a4d7";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffffffff800000000");
            plain = "00000000000000000000000000000000";
            cipher = "9423762f527a4060ffca312dcca22a16";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffc00000000");
            plain = "00000000000000000000000000000000";
            cipher = "f361a2745a33f056a5ac6ace2f08e344";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffe00000000");
            plain = "00000000000000000000000000000000";
            cipher = "5ef145766eca849f5d011536a6557fdb";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffff00000000");
            plain = "00000000000000000000000000000000";
            cipher = "c9af27b2c89c9b4cf4a0c4106ac80318";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffff80000000");
            plain = "00000000000000000000000000000000";
            cipher = "fb9c4f16c621f4eab7e9ac1d7551dd57";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffc0000000");
            plain = "00000000000000000000000000000000";
            cipher = "138e06fba466fa70854d8c2e524cffb2";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffe0000000");
            plain = "00000000000000000000000000000000";
            cipher = "fb4bc78b225070773f04c40466d4e90c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffff0000000");
            plain = "00000000000000000000000000000000";
            cipher = "8b2cbff1ed0150feda8a4799be94551f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffff8000000");
            plain = "00000000000000000000000000000000";
            cipher = "08b30d7b3f27962709a36bcadfb974bd";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffc000000");
            plain = "00000000000000000000000000000000";
            cipher = "fdf6d32e044d77adcf37fb97ac213326";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffe000000");
            plain = "00000000000000000000000000000000";
            cipher = "93cb284ecdcfd781a8afe32077949e88";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffff000000");
            plain = "00000000000000000000000000000000";
            cipher = "7b017bb02ec87b2b94c96e40a26fc71a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffff800000");
            plain = "00000000000000000000000000000000";
            cipher = "c5c038b6990664ab08a3aaa5df9f3266";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffc00000");
            plain = "00000000000000000000000000000000";
            cipher = "4b7020be37fab6259b2a27f4ec551576";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffe00000");
            plain = "00000000000000000000000000000000";
            cipher = "60136703374f64e860b48ce31f930716";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffff00000");
            plain = "00000000000000000000000000000000";
            cipher = "8d63a269b14d506ccc401ab8a9f1b591";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffff80000");
            plain = "00000000000000000000000000000000";
            cipher = "d317f81dc6aa454aee4bd4a5a5cff4bd";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffc0000");
            plain = "00000000000000000000000000000000";
            cipher = "dddececd5354f04d530d76ed884246eb";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffe0000");
            plain = "00000000000000000000000000000000";
            cipher = "41c5205cc8fd8eda9a3cffd2518f365a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffff0000");
            plain = "00000000000000000000000000000000";
            cipher = "cf42fb474293d96eca9db1b37b1ba676";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffff8000");
            plain = "00000000000000000000000000000000";
            cipher = "a231692607169b4ecdead5cd3b10db3e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffc000");
            plain = "00000000000000000000000000000000";
            cipher = "ace4b91c9c669e77e7acacd19859ed49";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffe000");
            plain = "00000000000000000000000000000000";
            cipher = "75db7cfd4a7b2b62ab78a48f3ddaf4af";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffff000");
            plain = "00000000000000000000000000000000";
            cipher = "c1faba2d46e259cf480d7c38e4572a58";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffff800");
            plain = "00000000000000000000000000000000";
            cipher = "241c45bc6ae16dee6eb7bea128701582";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffc00");
            plain = "00000000000000000000000000000000";
            cipher = "8fd03057cf1364420c2b78069a3e2502";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffe00");
            plain = "00000000000000000000000000000000";
            cipher = "ddb505e6cc1384cbaec1df90b80beb20";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffff00");
            plain = "00000000000000000000000000000000";
            cipher = "5674a3bed27bf4bd3622f9f5fe208306";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffff80");
            plain = "00000000000000000000000000000000";
            cipher = "b687f26a89cfbfbb8e5eeac54055315e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffc0");
            plain = "00000000000000000000000000000000";
            cipher = "0547dd32d3b29ab6a4caeb606c5b6f78";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffe0");
            plain = "00000000000000000000000000000000";
            cipher = "186861f8bc5386d31fb77f720c3226e6";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffff0");
            plain = "00000000000000000000000000000000";
            cipher = "eacf1e6c4224efb38900b185ab1dfd42";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffff8");
            plain = "00000000000000000000000000000000";
            cipher = "d241aab05a42d319de81d874f5c7b90d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffc");
            plain = "00000000000000000000000000000000";
            cipher = "5eb9bc759e2ad8d2140a6c762ae9e1ab";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffe");
            plain = "00000000000000000000000000000000";
            cipher = "018596e15e78e2c064159defce5f3085";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
        } catch (AesBadKeyException e) {
            e.printStackTrace();
        }
    }

    public void testECBVarKey192e() {
        byte[] key;
        byte[] state;
        String plain;
        String cipher;
        AesTestHelper aes;
        try {
            key = AesdroidActivity.hexStringToByteArray("800000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "de885dc87f5a92594082d02cc1e1b42c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("c00000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "132b074e80f2a597bf5febd8ea5da55e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("e00000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "6eccedf8de592c22fb81347b79f2db1f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("f00000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "180b09f267c45145db2f826c2582d35c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("f80000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "edd807ef7652d7eb0e13c8b5e15b3bc0";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fc0000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "9978bcf8dd8fd72241223ad24b31b8a4";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fe0000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "5310f654343e8f27e12c83a48d24ff81";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ff0000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "833f71258d53036b02952c76c744f5a1";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ff8000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "eba83ff200cff9318a92f8691a06b09f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffc000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "ff620ccbe9f3292abdf2176b09f04eba";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffe000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "7ababc4b3f516c9aafb35f4140b548f9";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fff000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "aa187824d9c4582b0916493ecbde8c57";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fff800000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "1c0ad553177fd5ea1092c9d626a29dc4";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffc00000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "a5dc46c37261194124ecaebd680408ec";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffe00000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "e4f2f2ae23e9b10bacfa58601531ba54";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffff00000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "b7d67cf1a1e91e8ff3a57a172c7bf412";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffff80000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "26706be06967884e847d137128ce47b3";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffc0000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "b2f8b409b0585909aad3a7b5a219072a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffe0000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "5e4b7bff0290c78344c54a23b722cd20";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffff0000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "07093657552d4414227ce161e9ebf7dd";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffff8000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "e1af1e7d8bc225ed4dffb771ecbb9e67";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffc000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "ef6555253635d8432156cfd9c11b145a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffe000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "fb4035074a5d4260c90cbd6da6c3fceb";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffff000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "446ee416f9ad1c103eb0cc96751c88e1";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffff800000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "198ae2a4637ac0a7890a8fd1485445c9";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffc00000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "562012ec8faded0825fb2fa70ab30cbd";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffe00000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "cc8a64b46b5d88bf7f247d4dbaf38f05";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffff00000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "a168253762e2cc81b42d1e5001762699";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffff80000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "1b41f83b38ce5032c6cd7af98cf62061";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffc0000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "61a89990cd1411750d5fb0dc988447d4";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffe0000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "b5accc8ed629edf8c68a539183b1ea82";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffff0000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "b16fa71f846b81a13f361c43a851f290";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffff8000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "4fad6efdff5975aee7692234bcd54488";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffc000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "ebfdb05a783d03082dfe5fdd80a00b17";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffe000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "eb81b584766997af6ba5529d3bdd8609";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffff000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "0cf4ff4f49c8a0ca060c443499e29313";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffff800000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "cc4ba8a8e029f8b26d8afff9df133bb6";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffc00000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "fefebf64360f38e4e63558f0ffc550c3";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffe00000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "12ad98cbf725137d6a8108c2bed99322";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffff00000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "6afaa996226198b3e2610413ce1b3f78";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffff80000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "2a8ce6747a7e39367828e290848502d9";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffc0000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "223736e8b8f89ca1e37b6deab40facf1";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffe0000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "c0f797e50418b95fa6013333917a9480";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffff0000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "a758de37c2ece2a02c73c01fedc9a132";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffff8000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "3a9b87ae77bae706803966c66c73adbd";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffc000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "d365ab8df8ffd782e358121a4a4fc541";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffe000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "c8dcd9e6f75e6c36c8daee0466f0ed74";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffff000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "c79a637beb1c0304f14014c037e736dd";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffff800000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "105f0a25e84ac930d996281a5f954dd9";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffc00000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "42e4074b2927973e8d17ffa92f7fe615";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffe00000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "4fe2a9d2c1824449c69e3e0398f12963";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffff00000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "b7f29c1e1f62847a15253b28a1e9d712";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffff80000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "36ed5d29b903f31e8983ef8b0a2bf990";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffc0000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "27b8070270810f9d023f9dd7ff3b4aa2";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffe0000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "94d46e155c1228f61d1a0db4815ecc4b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffff0000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "ca6108d1d98071428eeceef1714b96dd";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffff8000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "dc5b25b71b6296cf73dd2cdcac2f70b1";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffc000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "44aba95e8a06a2d9d3530d2677878c80";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffe000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "a570d20e89b467e8f5176061b81dd396";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffff000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "758f4467a5d8f1e7307dc30b34e404f4";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffff800000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "bcea28e9071b5a2302970ff352451bc5";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffc00000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "7523c00bc177d331ad312e09c9015c1c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffe00000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "ccac61e3183747b3f5836da21a1bc4f4";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffff00000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "707b075791878880b44189d3522b8c30";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffff80000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "7132d0c0e4a07593cf12ebb12be7688c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffc0000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "effbac1644deb0c784275fe56e19ead3";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffe0000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "a005063f30f4228b374e2459738f26bb";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffff0000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "29975b5f48bb68fcbbc7cea93b452ed7";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffff8000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "cf3f2576e2afedc74bb1ca7eeec1c0e7";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffc000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "07c403f5f966e0e3d9f296d6226dca28";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffe000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "c8c20908249ab4a34d6dd0a31327ff1a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffff000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "c0541329ecb6159ab23b7fc5e6a21bca";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffff800000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "7aa1acf1a2ed9ba72bc6deb31d88b863";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffc00000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "808bd8eddabb6f3bf0d5a8a27be1fe8a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffe00000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "273c7d7685e14ec66bbb96b8f05b6ddd";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffff00000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "32752eefc8c2a93f91b6e73eb07cca6e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffff80000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "d893e7d62f6ce502c64f75e281f9c000";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffc0000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "8dfd999be5d0cfa35732c0ddc88ff5a5";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffe0000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "02647c76a300c3173b841487eb2bae9f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffff0000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "172df8b02f04b53adab028b4e01acd87";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffff8000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "054b3bf4998aeb05afd87ec536533a36";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffc000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "3783f7bf44c97f065258a666cae03020";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffe000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "aad4c8a63f80954104de7b92cede1be1";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffff000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "cbfe61810fd5467ccdacb75800f3ac07";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffff800000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "830d8a2590f7d8e1b55a737f4af45f34";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffc00000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "fffcd4683f858058e74314671d43fa2c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffe00000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "523d0babbb82f46ebc9e70b1cd41ddd0";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffff00000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "344aab37080d7486f7d542a309e53eed";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffff80000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "56c5609d0906b23ab9caca816f5dbebd";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffc0000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "7026026eedd91adc6d831cdf9894bdc6";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffe0000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "88330baa4f2b618fc9d9b021bf503d5a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffff0000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "fc9e0ea22480b0bac935c8a8ebefcdcf";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffff8000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "29ca779f398fb04f867da7e8a44756cb";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffc000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "51f89c42985786bfc43c6df8ada36832";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffe000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "6ac1de5fb8f21d874e91c53b560c50e3";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffff000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "03aa9058490eda306001a8a9f48d0ca7";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffff800000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "e34ec71d6128d4871865d617c30b37e3";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffc00000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "14be1c535b17cabd0c4d93529d69bf47";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffe00000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "c9ef67756507beec9dd3862883478044";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffff00000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "40e231fa5a5948ce2134e92fc0664d4b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffff80000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "03194b8e5dda5530d0c678c0b48f5d92";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffc0000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "90bd086f237cc4fd99f4d76bde6b4826";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffe0000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "19259761ca17130d6ed86d57cd7951ee";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffff0000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "d7cbb3f34b9b450f24b0e8518e54da6d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffff8000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "725b9caebe9f7f417f4068d0d2ee20b3";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffc000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "9d924b934a90ce1fd39b8a9794f82672";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffe000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "c50562bf094526a91c5bc63c0c224995";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffff000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "d2f11805046743bd74f57188d9188df7";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffff800000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "8dd274bd0f1b58ae345d9e7233f9b8f3";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffc00000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "9d6bdc8f4ce5feb0f3bed2e4b9a9bb0b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffe00000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "fd5548bcf3f42565f7efa94562528d46";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffff00000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "d2ccaebd3a4c3e80b063748131ba4a71";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffff80000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "e03cb23d9e11c9d93f117e9c0a91b576";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffc0000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "78f933a2081ac1db84f69d10f4523fe0";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffe0000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "4061f7412ed320de0edc8851c2e2436f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffff0000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "9064ba1cd04ce6bab98474330814b4d4";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffff8000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "48391bffb9cfff80ac238c886ef0a461";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffc000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "b8d2a67df5a999fdbf93edd0343296c9";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffe000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "aaca7367396b69a221bd632bea386eec";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffff000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "a80fd5020dfe65f5f16293ec92c6fd89";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffff800000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "2162995b8217a67f1abc342e146406f8";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffc00000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "c6a6164b7a60bae4e986ffac28dfadd9";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffe00000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "64e0d7f900e3d9c83e4b8f96717b2146";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffff00000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "1ad2561de8c1232f5d8dbab4739b6cbb";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffff80000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "279689e9a557f58b1c3bf40c97a90964";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffc0000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "c4637e4a5e6377f9cc5a8638045de029";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffe0000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "492e607e5aea4688594b45f3aee3df90";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffff0000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "e8c4e4381feec74054954c05b777a00a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffff8000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "91549514605f38246c9b724ad839f01d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffc000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "74b24e3b6fefe40a4f9ef7ac6e44d76a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffe000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "2437a683dc5d4b52abb4a123a8df86c6";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffff000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "bb2852c891c5947d2ed44032c421b85f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffff800000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "1b9f5fbd5e8a4264c0a85b80409afa5e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffc00000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "30dab809f85a917fe924733f424ac589";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffe00000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "eaef5c1f8d605192646695ceadc65f32";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffff00000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "b8aa90040b4c15a12316b78e0f9586fc";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffff80000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "97fac8297ceaabc87d454350601e0673";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffffc0000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "9b47ef567ac28dfe488492f157e2b2e0";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffffe0000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "1b8426027ddb962b5c5ba7eb8bc9ab63";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffff0000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "e917fc77e71992a12dbe4c18068bec82";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffff8000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "dceebbc98840f8ae6daf76573b7e56f4";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffffc000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "4e11a9f74205125b61e0aee047eca20d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffffe000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "f60467f55a1f17eab88e800120cbc284";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffffff000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "d436649f600b449ee276530f0cd83c11";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffffff800000000000");
            plain = "00000000000000000000000000000000";
            cipher = "3bc0e3656a9e3ac7cd378a737f53b637";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffffffc00000000000");
            plain = "00000000000000000000000000000000";
            cipher = "6bacae63d33b928aa8380f8d54d88c17";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffffffe00000000000");
            plain = "00000000000000000000000000000000";
            cipher = "8935ffbc75ae6251bf8e859f085adcb9";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffffff00000000000");
            plain = "00000000000000000000000000000000";
            cipher = "93dc4970fe35f67747cb0562c06d875a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffffff80000000000");
            plain = "00000000000000000000000000000000";
            cipher = "14f9df858975851797ba604fb0d16cc7";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffffffc0000000000");
            plain = "00000000000000000000000000000000";
            cipher = "02ea0c98dca10b38c21b3b14e8d1b71f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffffffe0000000000");
            plain = "00000000000000000000000000000000";
            cipher = "8f091b1b5b0749b2adc803e63dda9b72";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffffffff0000000000");
            plain = "00000000000000000000000000000000";
            cipher = "05b389e3322c6da08384345a4137fd08";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffffffff8000000000");
            plain = "00000000000000000000000000000000";
            cipher = "381308c438f35b399f10ad71b05027d8";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffc000000000");
            plain = "00000000000000000000000000000000";
            cipher = "68c230fcfa9279c3409fc423e2acbe04";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffe000000000");
            plain = "00000000000000000000000000000000";
            cipher = "1c84a475acb011f3f59f4f46b76274c0";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffffffff000000000");
            plain = "00000000000000000000000000000000";
            cipher = "45119b68cb3f8399ee60066b5611a4d7";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffffffff800000000");
            plain = "00000000000000000000000000000000";
            cipher = "9423762f527a4060ffca312dcca22a16";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffc00000000");
            plain = "00000000000000000000000000000000";
            cipher = "f361a2745a33f056a5ac6ace2f08e344";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffe00000000");
            plain = "00000000000000000000000000000000";
            cipher = "5ef145766eca849f5d011536a6557fdb";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffff00000000");
            plain = "00000000000000000000000000000000";
            cipher = "c9af27b2c89c9b4cf4a0c4106ac80318";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffff80000000");
            plain = "00000000000000000000000000000000";
            cipher = "fb9c4f16c621f4eab7e9ac1d7551dd57";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffc0000000");
            plain = "00000000000000000000000000000000";
            cipher = "138e06fba466fa70854d8c2e524cffb2";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffe0000000");
            plain = "00000000000000000000000000000000";
            cipher = "fb4bc78b225070773f04c40466d4e90c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffff0000000");
            plain = "00000000000000000000000000000000";
            cipher = "8b2cbff1ed0150feda8a4799be94551f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffff8000000");
            plain = "00000000000000000000000000000000";
            cipher = "08b30d7b3f27962709a36bcadfb974bd";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffc000000");
            plain = "00000000000000000000000000000000";
            cipher = "fdf6d32e044d77adcf37fb97ac213326";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffe000000");
            plain = "00000000000000000000000000000000";
            cipher = "93cb284ecdcfd781a8afe32077949e88";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffff000000");
            plain = "00000000000000000000000000000000";
            cipher = "7b017bb02ec87b2b94c96e40a26fc71a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffff800000");
            plain = "00000000000000000000000000000000";
            cipher = "c5c038b6990664ab08a3aaa5df9f3266";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffc00000");
            plain = "00000000000000000000000000000000";
            cipher = "4b7020be37fab6259b2a27f4ec551576";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffe00000");
            plain = "00000000000000000000000000000000";
            cipher = "60136703374f64e860b48ce31f930716";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffff00000");
            plain = "00000000000000000000000000000000";
            cipher = "8d63a269b14d506ccc401ab8a9f1b591";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffff80000");
            plain = "00000000000000000000000000000000";
            cipher = "d317f81dc6aa454aee4bd4a5a5cff4bd";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffc0000");
            plain = "00000000000000000000000000000000";
            cipher = "dddececd5354f04d530d76ed884246eb";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffe0000");
            plain = "00000000000000000000000000000000";
            cipher = "41c5205cc8fd8eda9a3cffd2518f365a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffff0000");
            plain = "00000000000000000000000000000000";
            cipher = "cf42fb474293d96eca9db1b37b1ba676";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffff8000");
            plain = "00000000000000000000000000000000";
            cipher = "a231692607169b4ecdead5cd3b10db3e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffc000");
            plain = "00000000000000000000000000000000";
            cipher = "ace4b91c9c669e77e7acacd19859ed49";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffe000");
            plain = "00000000000000000000000000000000";
            cipher = "75db7cfd4a7b2b62ab78a48f3ddaf4af";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffff000");
            plain = "00000000000000000000000000000000";
            cipher = "c1faba2d46e259cf480d7c38e4572a58";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffff800");
            plain = "00000000000000000000000000000000";
            cipher = "241c45bc6ae16dee6eb7bea128701582";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffc00");
            plain = "00000000000000000000000000000000";
            cipher = "8fd03057cf1364420c2b78069a3e2502";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffe00");
            plain = "00000000000000000000000000000000";
            cipher = "ddb505e6cc1384cbaec1df90b80beb20";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffff00");
            plain = "00000000000000000000000000000000";
            cipher = "5674a3bed27bf4bd3622f9f5fe208306";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffff80");
            plain = "00000000000000000000000000000000";
            cipher = "b687f26a89cfbfbb8e5eeac54055315e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffc0");
            plain = "00000000000000000000000000000000";
            cipher = "0547dd32d3b29ab6a4caeb606c5b6f78";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffe0");
            plain = "00000000000000000000000000000000";
            cipher = "186861f8bc5386d31fb77f720c3226e6";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffff0");
            plain = "00000000000000000000000000000000";
            cipher = "eacf1e6c4224efb38900b185ab1dfd42";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffff8");
            plain = "00000000000000000000000000000000";
            cipher = "d241aab05a42d319de81d874f5c7b90d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffc");
            plain = "00000000000000000000000000000000";
            cipher = "5eb9bc759e2ad8d2140a6c762ae9e1ab";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffe");
            plain = "00000000000000000000000000000000";
            cipher = "018596e15e78e2c064159defce5f3085";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
        } catch (AesBadKeyException e) {
            e.printStackTrace();
        }
    }

    public void testECBVarKey256d() {
        byte[] key;
        byte[] state;
        String plain;
        String cipher;
        AesTestHelper aes;
        try {
            key = AesdroidActivity
                    .hexStringToByteArray("8000000000000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "e35a6dcb19b201a01ebcfa8aa22b5759";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("c000000000000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "b29169cdcf2d83e838125a12ee6aa400";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("e000000000000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "d8f3a72fc3cdf74dfaf6c3e6b97b2fa6";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("f000000000000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "1c777679d50037c79491a94da76a9a35";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("f800000000000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "9cf4893ecafa0a0247a898e040691559";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fc00000000000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "8fbb413703735326310a269bd3aa94b2";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fe00000000000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "60e32246bed2b0e859e55c1cc6b26502";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ff00000000000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "ec52a212f80a09df6317021bc2a9819e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ff80000000000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "f23e5b600eb70dbccf6c0b1d9a68182c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffc0000000000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "a3f599d63a82a968c33fe26590745970";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffe0000000000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "d1ccb9b1337002cbac42c520b5d67722";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fff0000000000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "cc111f6c37cf40a1159d00fb59fb0488";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fff8000000000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "dc43b51ab609052372989a26e9cdd714";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffc000000000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "4dcede8da9e2578f39703d4433dc6459";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffe000000000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "1a4c1c263bbccfafc11782894685e3a8";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffff000000000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "937ad84880db50613423d6d527a2823d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffff800000000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "610b71dfc688e150d8152c5b35ebc14d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffc00000000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "27ef2495dabf323885aab39c80f18d8b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffe00000000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "633cafea395bc03adae3a1e2068e4b4e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffff00000000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "6e1b482b53761cf631819b749a6f3724";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffff80000000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "976e6f851ab52c771998dbb2d71c75a9";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffc0000000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "85f2ba84f8c307cf525e124c3e22e6cc";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffe0000000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "6bcca98bf6a835fa64955f72de4115fe";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffff0000000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "2c75e2d36eebd65411f14fd0eb1d2a06";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffff8000000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "bd49295006250ffca5100b6007a0eade";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffc000000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "a190527d0ef7c70f459cd3940df316ec";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffe000000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "bbd1097a62433f79449fa97d4ee80dbf";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffff000000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "07058e408f5b99b0e0f061a1761b5b3b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffff800000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "5fd1f13fa0f31e37fabde328f894eac2";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffc00000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "fc4af7c948df26e2ef3e01c1ee5b8f6f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffe00000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "829fd7208fb92d44a074a677ee9861ac";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffff00000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "ad9fc613a703251b54c64a0e76431711";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffff80000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "33ac9eccc4cc75e2711618f80b1548e8";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffc0000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "2025c74b8ad8f4cda17ee2049c4c902d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffe0000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "f85ca05fe528f1ce9b790166e8d551e7";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffff0000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "6f6238d8966048d4967154e0dad5a6c9";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffff8000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "f2b21b4e7640a9b3346de8b82fb41e49";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffc000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "f836f251ad1d11d49dc344628b1884e1";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffe000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "077e9470ae7abea5a9769d49182628c3";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffff000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "e0dcc2d27fc9865633f85223cf0d611f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffff800000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "be66cfea2fecd6bf0ec7b4352c99bcaa";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffc00000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "df31144f87a2ef523facdcf21a427804";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffe00000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "b5bb0f5629fb6aae5e1839a3c3625d63";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffff00000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "3c9db3335306fe1ec612bdbfae6b6028";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffff80000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "3dd5c34634a79d3cfcc8339760e6f5f4";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffc0000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "82bda118a3ed7af314fa2ccc5c07b761";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffe0000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "2937a64f7d4f46fe6fea3b349ec78e38";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffff0000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "225f068c28476605735ad671bb8f39f3";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffff8000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "ae682c5ecd71898e08942ac9aa89875c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffc000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "5e031cb9d676c3022d7f26227e85c38f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffe000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "a78463fb064db5d52bb64bfef64f2dda";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffff000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "8aa9b75e784593876c53a00eae5af52b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffff800000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "3f84566df23da48af692722fe980573a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffc00000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "31690b5ed41c7eb42a1e83270a7ff0e6";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffe00000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "77dd7702646d55f08365e477d3590eda";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffff00000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "4c022ac62b3cb78d739cc67b3e20bb7e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffff80000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "092fa137ce18b5dfe7906f550bb13370";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffc0000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "3e0cdadf2e68353c0027672c97144dd3";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffe0000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "d8c4b200b383fc1f2b2ea677618a1d27";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffff0000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "11825f99b0e9bb3477c1c0713b015aac";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffff8000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "f8b9fffb5c187f7ddc7ab10f4fb77576";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffc000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "ffb4e87a32b37d6f2c8328d3b5377802";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffe000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "d276c13a5d220f4da9224e74896391ce";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffff000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "94efe7a0e2e031e2536da01df799c927";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffff800000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "8f8fd822680a85974e53a5a8eb9d38de";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffc00000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "e0f0a91b2e45f8cc37b7805a3042588d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffe00000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "597a6252255e46d6364dbeeda31e279c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffff00000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "f51a0f694442b8f05571797fec7ee8bf";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffff80000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "9ff071b165b5198a93dddeebc54d09b5";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffc0000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "c20a19fd5758b0c4bc1a5df89cf73877";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffe0000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "97120166307119ca2280e9315668e96f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffff0000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "4b3b9f1e099c2a09dc091e90e4f18f0a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffff8000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "eb040b891d4b37f6851f7ec219cd3f6d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffc000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "9f0fdec08b7fd79aa39535bea42db92a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffe000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "2e70f168fc74bf911df240bcd2cef236";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffff000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "462ccd7f5fd1108dbc152f3cacad328b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffff800000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "a4af534a7d0b643a01868785d86dfb95";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffc00000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "ab980296197e1a5022326c31da4bf6f3";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffe00000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "f97d57b3333b6281b07d486db2d4e20c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffff00000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "f33fa36720231afe4c759ade6bd62eb6";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffff80000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "fdcfac0c02ca538343c68117e0a15938";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffc0000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "ad4916f5ee5772be764fc027b8a6e539";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffe0000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "2e16873e1678610d7e14c02d002ea845";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffff0000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "4e6e627c1acc51340053a8236d579576";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffff8000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "ab0c8410aeeead92feec1eb430d652cb";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffc000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "e86f7e23e835e114977f60e1a592202e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffe000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "e68ad5055a367041fade09d9a70a794b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffff000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "0791823a3c666bb6162825e78606a7fe";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffff800000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "dcca366a9bf47b7b868b77e25c18a364";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffc00000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "684c9efc237e4a442965f84bce20247a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffe00000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "a858411ffbe63fdb9c8aa1bfaed67b52";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffff00000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "04bc3da2179c3015498b0e03910db5b8";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffff80000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "40071eeab3f935dbc25d00841460260f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffc0000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "0ebd7c30ed2016e08ba806ddb008bcc8";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffe0000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "15c6becf0f4cec7129cbd22d1a79b1b8";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffff0000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "0aeede5b91f721700e9e62edbf60b781";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffff8000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "266581af0dcfbed1585e0a242c64b8df";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffc000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "6693dc911662ae473216ba22189a511a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffe000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "7606fa36d86473e6fb3a1bb0e2c0adf5";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffff000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "112078e9e11fbb78e26ffb8899e96b9a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffff800000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "40b264e921e9e4a82694589ef3798262";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffc00000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "8d4595cb4fa7026715f55bd68e2882f9";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffe00000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "b588a302bdbc09197df1edae68926ed9";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffff00000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "33f7502390b8a4a221cfecd0666624ba";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffff80000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "3d20253adbce3be2373767c4d822c566";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffc0000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "a42734a3929bf84cf0116c9856a3c18c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffe0000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "e3abc4939457422bb957da3c56938c6d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffff0000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "972bdd2e7c525130fadc8f76fc6f4b3f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffff8000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "84a83d7b94c699cbcb8a7d9b61f64093";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffc000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "ce61d63514aded03d43e6ebfc3a9001f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffe000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "6c839dd58eeae6b8a36af48ed63d2dc9";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffff000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "cd5ece55b8da3bf622c4100df5de46f9";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffff800000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "3b6f46f40e0ac5fc0a9c1105f800f48d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffc00000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "ba26d47da3aeb028de4fb5b3a854a24b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffe00000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "87f53bf620d3677268445212904389d5";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffff00000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "10617d28b5e0f4605492b182a5d7f9f6";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffff80000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "9aaec4fabbf6fae2a71feff02e372b39";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffc0000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "3a90c62d88b5c42809abf782488ed130";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffe0000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "f1f1c5a40899e15772857ccb65c7a09a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffff0000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "190843d29b25a3897c692ce1dd81ee52";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffff8000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "a866bc65b6941d86e8420a7ffb0964db";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffc000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "8193c6ff85225ced4255e92f6e078a14";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffe000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "9661cb2424d7d4a380d547f9e7ec1cb9";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffff000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "86f93d9ec08453a071e2e2877877a9c8";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffff800000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "27eefa80ce6a4a9d598e3fec365434d2";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffc00000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "d62068444578e3ab39ce7ec95dd045dc";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffe00000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "b5f71d4dd9a71fe5d8bc8ba7e6ea3048";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffff00000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "6825a347ac479d4f9d95c5cb8d3fd7e9";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffff80000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "e3714e94a5778955cc0346358e94783a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffc0000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "d836b44bb29e0c7d89fa4b2d4b677d2a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffe0000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "5d454b75021d76d4b84f873a8f877b92";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffff0000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "c3498f7eced2095314fc28115885b33f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffff8000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "6e668856539ad8e405bd123fe6c88530";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffc000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "8680db7f3a87b8605543cfdbe6754076";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffe000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "6c5d03b13069c3658b3179be91b0800c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffff000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "ef1b384ac4d93eda00c92add0995ea5f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffff800000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "bf8115805471741bd5ad20a03944790f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffc00000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "c64c24b6894b038b3c0d09b1df068b0b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffe00000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "3967a10cffe27d0178545fbf6a40544b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffff00000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "7c85e9c95de1a9ec5a5363a8a053472d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffff80000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "a9eec03c8abec7ba68315c2c8c2316e0";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffc0000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "cac8e414c2f388227ae14986fc983524";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffe0000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "5d942b7f4622ce056c3ce3ce5f1dd9d6";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffff0000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "d240d648ce21a3020282c3f1b528a0b6";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffff8000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "45d089c36d5c5a4efc689e3b0de10dd5";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffc000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "b4da5df4becb5462e03a0ed00d295629";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffe000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "dcf4e129136c1a4b7a0f38935cc34b2b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffff000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "d9a4c7618b0ce48a3d5aee1a1c0114c4";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffff800000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "ca352df025c65c7b0bf306fbee0f36ba";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffc00000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "238aca23fd3409f38af63378ed2f5473";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffe00000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "59836a0e06a79691b36667d5380d8188";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffff00000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "33905080f7acf1cdae0a91fc3e85aee4";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffff80000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "72c9e4646dbc3d6320fc6689d93e8833";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffc0000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "ba77413dea5925b7f5417ea47ff19f59";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffe0000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "6cae8129f843d86dc786a0fb1a184970";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffff0000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "fcfefb534100796eebbd990206754e19";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffff8000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "8c791d5fdddf470da04f3e6dc4a5b5b5";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffc000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "c93bbdc07a4611ae4bb266ea5034a387";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffe000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "c102e38e489aa74762f3efc5bb23205a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffff000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "93201481665cbafc1fcc220bc545fb3d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffff800000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "4960757ec6ce68cf195e454cfd0f32ca";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffc00000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "feec7ce6a6cbd07c043416737f1bbb33";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffe00000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "11c5413904487a805d70a8edd9c35527";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffff00000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "347846b2b2e36f1f0324c86f7f1b98e2";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffff80000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "332eee1a0cbd19ca2d69b426894044f0";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffc0000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "866b5b3977ba6efa5128efbda9ff03cd";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffe0000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "cc1445ee94c0f08cdee5c344ecd1e233";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffff0000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "be288319029363c2622feba4b05dfdfe";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffff8000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "cfd1875523f3cd21c395651e6ee15e56";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffc000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "cb5a408657837c53bf16f9d8465dce19";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffe000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "ca0bf42cb107f55ccff2fc09ee08ca15";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffff000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "fdd9bbb4a7dc2e4a23536a5880a2db67";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffff800000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "ede447b362c484993dec9442a3b46aef";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffc00000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "10dffb05904bff7c4781df780ad26837";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffe00000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "c33bc13e8de88ac25232aa7496398783";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffff00000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "ca359c70803a3b2a3d542e8781dea975";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffff80000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "bcc65b526f88d05b89ce8a52021fdb06";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffc0000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "db91a38855c8c4643851fbfb358b0109";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffe0000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "ca6e8893a114ae8e27d5ab03a5499610";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffff0000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "6629d2b8df97da728cdd8b1e7f945077";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffff8000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "4570a5a18cfc0dd582f1d88d5c9a1720";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffc000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "72bc65aa8e89562e3f274d45af1cd10b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffe000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "98551da1a6503276ae1c77625f9ea615";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffff000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "0ddfe51ced7e3f4ae927daa3fe452cee";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffff800000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "db826251e4ce384b80218b0e1da1dd4c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffc00000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "2cacf728b88abbad7011ed0e64a1680c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffe00000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "330d8ee7c5677e099ac74c9994ee4cfb";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffff00000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "edf61ae362e882ddc0167474a7a77f3a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffff80000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "6168b00ba7859e0970ecfd757efecf7c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffc0000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "d1415447866230d28bb1ea18a4cdfd02";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffe0000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "516183392f7a8763afec68a060264141";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffff0000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "77565c8d73cfd4130b4aa14d8911710f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffff8000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "37232a4ed21ccc27c19c9610078cabac";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffc000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "804f32ea71828c7d329077e712231666";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffe000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "d64424f23cb97215e9c2c6f28d29eab7";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffff000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "023e82b533f68c75c238cebdb2ee89a2";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffff800000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "193a3d24157a51f1ee0893f6777417e7";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffc00000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "84ecacfcd400084d078612b1945f2ef5";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffe00000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "1dcd8bb173259eb33a5242b0de31a455";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffff00000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "35e9eddbc375e792c19992c19165012b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffff80000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "8a772231c01dfdd7c98e4cfddcc0807a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffffc0000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "6eda7ff6b8319180ff0d6e65629d01c3";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffffe0000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "c267ef0e2d01a993944dd397101413cb";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffff0000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "e9f80e9d845bcc0f62926af72eabca39";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffff8000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "6702990727aa0878637b45dcd3a3b074";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffffc000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "2e2e647d5360e09230a5d738ca33471e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffffe000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "1f56413c7add6f43d1d56e4f02190330";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffffff000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "69cd0606e15af729d6bca143016d9842";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffffff800000000000");
            plain = "00000000000000000000000000000000";
            cipher = "a085d7c1a500873a20099c4caa3c3f5b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffffffc00000000000");
            plain = "00000000000000000000000000000000";
            cipher = "4fc0d230f8891415b87b83f95f2e09d1";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffffffe00000000000");
            plain = "00000000000000000000000000000000";
            cipher = "4327d08c523d8eba697a4336507d1f42";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffffff00000000000");
            plain = "00000000000000000000000000000000";
            cipher = "7a15aab82701efa5ae36ab1d6b76290f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffffff80000000000");
            plain = "00000000000000000000000000000000";
            cipher = "5bf0051893a18bb30e139a58fed0fa54";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffffffc0000000000");
            plain = "00000000000000000000000000000000";
            cipher = "97e8adf65638fd9cdf3bc22c17fe4dbd";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffffffe0000000000");
            plain = "00000000000000000000000000000000";
            cipher = "1ee6ee326583a0586491c96418d1a35d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffffffff0000000000");
            plain = "00000000000000000000000000000000";
            cipher = "26b549c2ec756f82ecc48008e529956b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffffffff8000000000");
            plain = "00000000000000000000000000000000";
            cipher = "70377b6da669b072129e057cc28e9ca5";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffffffffc000000000");
            plain = "00000000000000000000000000000000";
            cipher = "9c94b8b0cb8bcc919072262b3fa05ad9";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffffffffe000000000");
            plain = "00000000000000000000000000000000";
            cipher = "2fbb83dfd0d7abcb05cd28cad2dfb523";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffffffff000000000");
            plain = "00000000000000000000000000000000";
            cipher = "96877803de77744bb970d0a91f4debae";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffffffff800000000");
            plain = "00000000000000000000000000000000";
            cipher = "7379f3370cf6e5ce12ae5969c8eea312";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffffffffc00000000");
            plain = "00000000000000000000000000000000";
            cipher = "02dc99fa3d4f98ce80985e7233889313";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffffffffe00000000");
            plain = "00000000000000000000000000000000";
            cipher = "1e38e759075ba5cab6457da51844295a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffffffffff00000000");
            plain = "00000000000000000000000000000000";
            cipher = "70bed8dbf615868a1f9d9b05d3e7a267";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffffffffff80000000");
            plain = "00000000000000000000000000000000";
            cipher = "234b148b8cb1d8c32b287e896903d150";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffffffffffc0000000");
            plain = "00000000000000000000000000000000";
            cipher = "294b033df4da853f4be3e243f7e513f4";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffffffffffe0000000");
            plain = "00000000000000000000000000000000";
            cipher = "3f58c950f0367160adec45f2441e7411";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffffffffff0000000");
            plain = "00000000000000000000000000000000";
            cipher = "37f655536a704e5ace182d742a820cf4";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffffffffff8000000");
            plain = "00000000000000000000000000000000";
            cipher = "ea7bd6bb63418731aeac790fe42d61e8";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffffffffffc000000");
            plain = "00000000000000000000000000000000";
            cipher = "e74a4c999b4c064e48bb1e413f51e5ea";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffffffffffe000000");
            plain = "00000000000000000000000000000000";
            cipher = "ba9ebefdb4ccf30f296cecb3bc1943e8";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffffffffffff000000");
            plain = "00000000000000000000000000000000";
            cipher = "3194367a4898c502c13bb7478640a72d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffffffffffff800000");
            plain = "00000000000000000000000000000000";
            cipher = "da797713263d6f33a5478a65ef60d412";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffc00000");
            plain = "00000000000000000000000000000000";
            cipher = "d1ac39bb1ef86b9c1344f214679aa376";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffe00000");
            plain = "00000000000000000000000000000000";
            cipher = "2fdea9e650532be5bc0e7325337fd363";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffffffffffff00000");
            plain = "00000000000000000000000000000000";
            cipher = "d3a204dbd9c2af158b6ca67a5156ce4a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffffffffffff80000");
            plain = "00000000000000000000000000000000";
            cipher = "3a0a0e75a8da36735aee6684d965a778";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffffffffffffc0000");
            plain = "00000000000000000000000000000000";
            cipher = "52fc3e620492ea99641ea168da5b6d52";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffffffffffffe0000");
            plain = "00000000000000000000000000000000";
            cipher = "d2e0c7f15b4772467d2cfc873000b2ca";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff0000");
            plain = "00000000000000000000000000000000";
            cipher = "563531135e0c4d70a38f8bdb190ba04e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff8000");
            plain = "00000000000000000000000000000000";
            cipher = "a8a39a0f5663f4c0fe5f2d3cafff421a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffc000");
            plain = "00000000000000000000000000000000";
            cipher = "d94b5e90db354c1e42f61fabe167b2c0";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffe000");
            plain = "00000000000000000000000000000000";
            cipher = "50e6d3c9b6698a7cd276f96b1473f35a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff000");
            plain = "00000000000000000000000000000000";
            cipher = "9338f08e0ebee96905d8f2e825208f43";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff800");
            plain = "00000000000000000000000000000000";
            cipher = "8b378c86672aa54a3a266ba19d2580ca";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffc00");
            plain = "00000000000000000000000000000000";
            cipher = "cca7c3086f5f9511b31233da7cab9160";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffe00");
            plain = "00000000000000000000000000000000";
            cipher = "5b40ff4ec9be536ba23035fa4f06064c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff00");
            plain = "00000000000000000000000000000000";
            cipher = "60eb5af8416b257149372194e8b88749";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff80");
            plain = "00000000000000000000000000000000";
            cipher = "2f005a8aed8a361c92e440c15520cbd1";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffc0");
            plain = "00000000000000000000000000000000";
            cipher = "7b03627611678a997717578807a800e2";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffe0");
            plain = "00000000000000000000000000000000";
            cipher = "cf78618f74f6f3696e0a4779b90b5a77";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff0");
            plain = "00000000000000000000000000000000";
            cipher = "03720371a04962eaea0a852e69972858";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff8");
            plain = "00000000000000000000000000000000";
            cipher = "1f8a8133aa8ccf70e2bd3285831ca6b7";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffc");
            plain = "00000000000000000000000000000000";
            cipher = "27936bd27fb1468fc8b48bc483321725";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffe");
            plain = "00000000000000000000000000000000";
            cipher = "b07d4f3e2cd2ef2eb545980754dfea0f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
        } catch (AesBadKeyException e) {
            e.printStackTrace();
        }
    }

    public void testECBVarKey256e() {
        byte[] key;
        byte[] state;
        String plain;
        String cipher;
        AesTestHelper aes;
        try {
            key = AesdroidActivity
                    .hexStringToByteArray("8000000000000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "e35a6dcb19b201a01ebcfa8aa22b5759";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("c000000000000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "b29169cdcf2d83e838125a12ee6aa400";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("e000000000000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "d8f3a72fc3cdf74dfaf6c3e6b97b2fa6";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("f000000000000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "1c777679d50037c79491a94da76a9a35";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("f800000000000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "9cf4893ecafa0a0247a898e040691559";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fc00000000000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "8fbb413703735326310a269bd3aa94b2";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fe00000000000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "60e32246bed2b0e859e55c1cc6b26502";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ff00000000000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "ec52a212f80a09df6317021bc2a9819e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ff80000000000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "f23e5b600eb70dbccf6c0b1d9a68182c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffc0000000000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "a3f599d63a82a968c33fe26590745970";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffe0000000000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "d1ccb9b1337002cbac42c520b5d67722";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fff0000000000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "cc111f6c37cf40a1159d00fb59fb0488";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fff8000000000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "dc43b51ab609052372989a26e9cdd714";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffc000000000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "4dcede8da9e2578f39703d4433dc6459";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffe000000000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "1a4c1c263bbccfafc11782894685e3a8";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffff000000000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "937ad84880db50613423d6d527a2823d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffff800000000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "610b71dfc688e150d8152c5b35ebc14d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffc00000000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "27ef2495dabf323885aab39c80f18d8b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffe00000000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "633cafea395bc03adae3a1e2068e4b4e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffff00000000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "6e1b482b53761cf631819b749a6f3724";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffff80000000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "976e6f851ab52c771998dbb2d71c75a9";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffc0000000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "85f2ba84f8c307cf525e124c3e22e6cc";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffe0000000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "6bcca98bf6a835fa64955f72de4115fe";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffff0000000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "2c75e2d36eebd65411f14fd0eb1d2a06";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffff8000000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "bd49295006250ffca5100b6007a0eade";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffc000000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "a190527d0ef7c70f459cd3940df316ec";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffe000000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "bbd1097a62433f79449fa97d4ee80dbf";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffff000000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "07058e408f5b99b0e0f061a1761b5b3b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffff800000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "5fd1f13fa0f31e37fabde328f894eac2";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffc00000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "fc4af7c948df26e2ef3e01c1ee5b8f6f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffe00000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "829fd7208fb92d44a074a677ee9861ac";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffff00000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "ad9fc613a703251b54c64a0e76431711";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffff80000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "33ac9eccc4cc75e2711618f80b1548e8";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffc0000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "2025c74b8ad8f4cda17ee2049c4c902d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffe0000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "f85ca05fe528f1ce9b790166e8d551e7";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffff0000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "6f6238d8966048d4967154e0dad5a6c9";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffff8000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "f2b21b4e7640a9b3346de8b82fb41e49";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffc000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "f836f251ad1d11d49dc344628b1884e1";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffe000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "077e9470ae7abea5a9769d49182628c3";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffff000000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "e0dcc2d27fc9865633f85223cf0d611f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffff800000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "be66cfea2fecd6bf0ec7b4352c99bcaa";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffc00000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "df31144f87a2ef523facdcf21a427804";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffe00000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "b5bb0f5629fb6aae5e1839a3c3625d63";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffff00000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "3c9db3335306fe1ec612bdbfae6b6028";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffff80000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "3dd5c34634a79d3cfcc8339760e6f5f4";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffc0000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "82bda118a3ed7af314fa2ccc5c07b761";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffe0000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "2937a64f7d4f46fe6fea3b349ec78e38";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffff0000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "225f068c28476605735ad671bb8f39f3";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffff8000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "ae682c5ecd71898e08942ac9aa89875c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffc000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "5e031cb9d676c3022d7f26227e85c38f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffe000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "a78463fb064db5d52bb64bfef64f2dda";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffff000000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "8aa9b75e784593876c53a00eae5af52b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffff800000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "3f84566df23da48af692722fe980573a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffc00000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "31690b5ed41c7eb42a1e83270a7ff0e6";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffe00000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "77dd7702646d55f08365e477d3590eda";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffff00000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "4c022ac62b3cb78d739cc67b3e20bb7e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffff80000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "092fa137ce18b5dfe7906f550bb13370";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffc0000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "3e0cdadf2e68353c0027672c97144dd3";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffe0000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "d8c4b200b383fc1f2b2ea677618a1d27";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffff0000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "11825f99b0e9bb3477c1c0713b015aac";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffff8000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "f8b9fffb5c187f7ddc7ab10f4fb77576";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffc000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "ffb4e87a32b37d6f2c8328d3b5377802";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffe000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "d276c13a5d220f4da9224e74896391ce";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffff000000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "94efe7a0e2e031e2536da01df799c927";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffff800000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "8f8fd822680a85974e53a5a8eb9d38de";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffc00000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "e0f0a91b2e45f8cc37b7805a3042588d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffe00000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "597a6252255e46d6364dbeeda31e279c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffff00000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "f51a0f694442b8f05571797fec7ee8bf";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffff80000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "9ff071b165b5198a93dddeebc54d09b5";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffc0000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "c20a19fd5758b0c4bc1a5df89cf73877";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffe0000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "97120166307119ca2280e9315668e96f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffff0000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "4b3b9f1e099c2a09dc091e90e4f18f0a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffff8000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "eb040b891d4b37f6851f7ec219cd3f6d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffc000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "9f0fdec08b7fd79aa39535bea42db92a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffe000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "2e70f168fc74bf911df240bcd2cef236";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffff000000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "462ccd7f5fd1108dbc152f3cacad328b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffff800000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "a4af534a7d0b643a01868785d86dfb95";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffc00000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "ab980296197e1a5022326c31da4bf6f3";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffe00000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "f97d57b3333b6281b07d486db2d4e20c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffff00000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "f33fa36720231afe4c759ade6bd62eb6";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffff80000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "fdcfac0c02ca538343c68117e0a15938";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffc0000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "ad4916f5ee5772be764fc027b8a6e539";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffe0000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "2e16873e1678610d7e14c02d002ea845";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffff0000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "4e6e627c1acc51340053a8236d579576";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffff8000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "ab0c8410aeeead92feec1eb430d652cb";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffc000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "e86f7e23e835e114977f60e1a592202e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffe000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "e68ad5055a367041fade09d9a70a794b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffff000000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "0791823a3c666bb6162825e78606a7fe";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffff800000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "dcca366a9bf47b7b868b77e25c18a364";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffc00000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "684c9efc237e4a442965f84bce20247a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffe00000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "a858411ffbe63fdb9c8aa1bfaed67b52";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffff00000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "04bc3da2179c3015498b0e03910db5b8";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffff80000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "40071eeab3f935dbc25d00841460260f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffc0000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "0ebd7c30ed2016e08ba806ddb008bcc8";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffe0000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "15c6becf0f4cec7129cbd22d1a79b1b8";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffff0000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "0aeede5b91f721700e9e62edbf60b781";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffff8000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "266581af0dcfbed1585e0a242c64b8df";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffc000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "6693dc911662ae473216ba22189a511a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffe000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "7606fa36d86473e6fb3a1bb0e2c0adf5";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffff000000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "112078e9e11fbb78e26ffb8899e96b9a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffff800000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "40b264e921e9e4a82694589ef3798262";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffc00000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "8d4595cb4fa7026715f55bd68e2882f9";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffe00000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "b588a302bdbc09197df1edae68926ed9";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffff00000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "33f7502390b8a4a221cfecd0666624ba";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffff80000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "3d20253adbce3be2373767c4d822c566";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffc0000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "a42734a3929bf84cf0116c9856a3c18c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffe0000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "e3abc4939457422bb957da3c56938c6d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffff0000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "972bdd2e7c525130fadc8f76fc6f4b3f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffff8000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "84a83d7b94c699cbcb8a7d9b61f64093";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffc000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "ce61d63514aded03d43e6ebfc3a9001f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffe000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "6c839dd58eeae6b8a36af48ed63d2dc9";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffff000000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "cd5ece55b8da3bf622c4100df5de46f9";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffff800000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "3b6f46f40e0ac5fc0a9c1105f800f48d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffc00000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "ba26d47da3aeb028de4fb5b3a854a24b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffe00000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "87f53bf620d3677268445212904389d5";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffff00000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "10617d28b5e0f4605492b182a5d7f9f6";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffff80000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "9aaec4fabbf6fae2a71feff02e372b39";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffc0000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "3a90c62d88b5c42809abf782488ed130";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffe0000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "f1f1c5a40899e15772857ccb65c7a09a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffff0000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "190843d29b25a3897c692ce1dd81ee52";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffff8000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "a866bc65b6941d86e8420a7ffb0964db";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffc000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "8193c6ff85225ced4255e92f6e078a14";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffe000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "9661cb2424d7d4a380d547f9e7ec1cb9";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffff000000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "86f93d9ec08453a071e2e2877877a9c8";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffff800000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "27eefa80ce6a4a9d598e3fec365434d2";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffc00000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "d62068444578e3ab39ce7ec95dd045dc";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffe00000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "b5f71d4dd9a71fe5d8bc8ba7e6ea3048";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffff00000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "6825a347ac479d4f9d95c5cb8d3fd7e9";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffff80000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "e3714e94a5778955cc0346358e94783a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffc0000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "d836b44bb29e0c7d89fa4b2d4b677d2a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffe0000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "5d454b75021d76d4b84f873a8f877b92";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffff0000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "c3498f7eced2095314fc28115885b33f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffff8000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "6e668856539ad8e405bd123fe6c88530";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffc000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "8680db7f3a87b8605543cfdbe6754076";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffe000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "6c5d03b13069c3658b3179be91b0800c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffff000000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "ef1b384ac4d93eda00c92add0995ea5f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffff800000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "bf8115805471741bd5ad20a03944790f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffc00000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "c64c24b6894b038b3c0d09b1df068b0b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffe00000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "3967a10cffe27d0178545fbf6a40544b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffff00000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "7c85e9c95de1a9ec5a5363a8a053472d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffff80000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "a9eec03c8abec7ba68315c2c8c2316e0";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffc0000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "cac8e414c2f388227ae14986fc983524";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffe0000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "5d942b7f4622ce056c3ce3ce5f1dd9d6";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffff0000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "d240d648ce21a3020282c3f1b528a0b6";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffff8000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "45d089c36d5c5a4efc689e3b0de10dd5";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffc000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "b4da5df4becb5462e03a0ed00d295629";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffe000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "dcf4e129136c1a4b7a0f38935cc34b2b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffff000000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "d9a4c7618b0ce48a3d5aee1a1c0114c4";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffff800000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "ca352df025c65c7b0bf306fbee0f36ba";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffc00000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "238aca23fd3409f38af63378ed2f5473";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffe00000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "59836a0e06a79691b36667d5380d8188";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffff00000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "33905080f7acf1cdae0a91fc3e85aee4";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffff80000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "72c9e4646dbc3d6320fc6689d93e8833";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffc0000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "ba77413dea5925b7f5417ea47ff19f59";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffe0000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "6cae8129f843d86dc786a0fb1a184970";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffff0000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "fcfefb534100796eebbd990206754e19";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffff8000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "8c791d5fdddf470da04f3e6dc4a5b5b5";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffc000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "c93bbdc07a4611ae4bb266ea5034a387";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffe000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "c102e38e489aa74762f3efc5bb23205a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffff000000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "93201481665cbafc1fcc220bc545fb3d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffff800000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "4960757ec6ce68cf195e454cfd0f32ca";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffc00000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "feec7ce6a6cbd07c043416737f1bbb33";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffe00000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "11c5413904487a805d70a8edd9c35527";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffff00000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "347846b2b2e36f1f0324c86f7f1b98e2";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffff80000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "332eee1a0cbd19ca2d69b426894044f0";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffc0000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "866b5b3977ba6efa5128efbda9ff03cd";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffe0000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "cc1445ee94c0f08cdee5c344ecd1e233";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffff0000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "be288319029363c2622feba4b05dfdfe";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffff8000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "cfd1875523f3cd21c395651e6ee15e56";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffc000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "cb5a408657837c53bf16f9d8465dce19";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffe000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "ca0bf42cb107f55ccff2fc09ee08ca15";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffff000000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "fdd9bbb4a7dc2e4a23536a5880a2db67";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffff800000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "ede447b362c484993dec9442a3b46aef";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffc00000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "10dffb05904bff7c4781df780ad26837";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffe00000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "c33bc13e8de88ac25232aa7496398783";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffff00000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "ca359c70803a3b2a3d542e8781dea975";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffff80000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "bcc65b526f88d05b89ce8a52021fdb06";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffc0000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "db91a38855c8c4643851fbfb358b0109";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffe0000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "ca6e8893a114ae8e27d5ab03a5499610";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffff0000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "6629d2b8df97da728cdd8b1e7f945077";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffff8000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "4570a5a18cfc0dd582f1d88d5c9a1720";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffc000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "72bc65aa8e89562e3f274d45af1cd10b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffe000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "98551da1a6503276ae1c77625f9ea615";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffff000000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "0ddfe51ced7e3f4ae927daa3fe452cee";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffff800000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "db826251e4ce384b80218b0e1da1dd4c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffc00000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "2cacf728b88abbad7011ed0e64a1680c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffe00000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "330d8ee7c5677e099ac74c9994ee4cfb";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffff00000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "edf61ae362e882ddc0167474a7a77f3a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffff80000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "6168b00ba7859e0970ecfd757efecf7c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffc0000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "d1415447866230d28bb1ea18a4cdfd02";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffe0000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "516183392f7a8763afec68a060264141";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffff0000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "77565c8d73cfd4130b4aa14d8911710f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffff8000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "37232a4ed21ccc27c19c9610078cabac";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffc000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "804f32ea71828c7d329077e712231666";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffe000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "d64424f23cb97215e9c2c6f28d29eab7";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffff000000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "023e82b533f68c75c238cebdb2ee89a2";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffff800000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "193a3d24157a51f1ee0893f6777417e7";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffc00000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "84ecacfcd400084d078612b1945f2ef5";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffe00000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "1dcd8bb173259eb33a5242b0de31a455";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffff00000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "35e9eddbc375e792c19992c19165012b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffff80000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "8a772231c01dfdd7c98e4cfddcc0807a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffffc0000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "6eda7ff6b8319180ff0d6e65629d01c3";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffffe0000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "c267ef0e2d01a993944dd397101413cb";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffff0000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "e9f80e9d845bcc0f62926af72eabca39";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffff8000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "6702990727aa0878637b45dcd3a3b074";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffffc000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "2e2e647d5360e09230a5d738ca33471e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffffe000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "1f56413c7add6f43d1d56e4f02190330";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffffff000000000000");
            plain = "00000000000000000000000000000000";
            cipher = "69cd0606e15af729d6bca143016d9842";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffffff800000000000");
            plain = "00000000000000000000000000000000";
            cipher = "a085d7c1a500873a20099c4caa3c3f5b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffffffc00000000000");
            plain = "00000000000000000000000000000000";
            cipher = "4fc0d230f8891415b87b83f95f2e09d1";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffffffe00000000000");
            plain = "00000000000000000000000000000000";
            cipher = "4327d08c523d8eba697a4336507d1f42";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffffff00000000000");
            plain = "00000000000000000000000000000000";
            cipher = "7a15aab82701efa5ae36ab1d6b76290f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffffff80000000000");
            plain = "00000000000000000000000000000000";
            cipher = "5bf0051893a18bb30e139a58fed0fa54";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffffffc0000000000");
            plain = "00000000000000000000000000000000";
            cipher = "97e8adf65638fd9cdf3bc22c17fe4dbd";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffffffe0000000000");
            plain = "00000000000000000000000000000000";
            cipher = "1ee6ee326583a0586491c96418d1a35d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffffffff0000000000");
            plain = "00000000000000000000000000000000";
            cipher = "26b549c2ec756f82ecc48008e529956b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffffffff8000000000");
            plain = "00000000000000000000000000000000";
            cipher = "70377b6da669b072129e057cc28e9ca5";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffffffffc000000000");
            plain = "00000000000000000000000000000000";
            cipher = "9c94b8b0cb8bcc919072262b3fa05ad9";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffffffffe000000000");
            plain = "00000000000000000000000000000000";
            cipher = "2fbb83dfd0d7abcb05cd28cad2dfb523";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffffffff000000000");
            plain = "00000000000000000000000000000000";
            cipher = "96877803de77744bb970d0a91f4debae";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffffffff800000000");
            plain = "00000000000000000000000000000000";
            cipher = "7379f3370cf6e5ce12ae5969c8eea312";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffffffffc00000000");
            plain = "00000000000000000000000000000000";
            cipher = "02dc99fa3d4f98ce80985e7233889313";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffffffffe00000000");
            plain = "00000000000000000000000000000000";
            cipher = "1e38e759075ba5cab6457da51844295a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffffffffff00000000");
            plain = "00000000000000000000000000000000";
            cipher = "70bed8dbf615868a1f9d9b05d3e7a267";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffffffffff80000000");
            plain = "00000000000000000000000000000000";
            cipher = "234b148b8cb1d8c32b287e896903d150";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffffffffffc0000000");
            plain = "00000000000000000000000000000000";
            cipher = "294b033df4da853f4be3e243f7e513f4";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffffffffffe0000000");
            plain = "00000000000000000000000000000000";
            cipher = "3f58c950f0367160adec45f2441e7411";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffffffffff0000000");
            plain = "00000000000000000000000000000000";
            cipher = "37f655536a704e5ace182d742a820cf4";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffffffffff8000000");
            plain = "00000000000000000000000000000000";
            cipher = "ea7bd6bb63418731aeac790fe42d61e8";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffffffffffc000000");
            plain = "00000000000000000000000000000000";
            cipher = "e74a4c999b4c064e48bb1e413f51e5ea";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffffffffffe000000");
            plain = "00000000000000000000000000000000";
            cipher = "ba9ebefdb4ccf30f296cecb3bc1943e8";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffffffffffff000000");
            plain = "00000000000000000000000000000000";
            cipher = "3194367a4898c502c13bb7478640a72d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffffffffffff800000");
            plain = "00000000000000000000000000000000";
            cipher = "da797713263d6f33a5478a65ef60d412";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffc00000");
            plain = "00000000000000000000000000000000";
            cipher = "d1ac39bb1ef86b9c1344f214679aa376";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffe00000");
            plain = "00000000000000000000000000000000";
            cipher = "2fdea9e650532be5bc0e7325337fd363";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffffffffffff00000");
            plain = "00000000000000000000000000000000";
            cipher = "d3a204dbd9c2af158b6ca67a5156ce4a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffffffffffff80000");
            plain = "00000000000000000000000000000000";
            cipher = "3a0a0e75a8da36735aee6684d965a778";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffffffffffffc0000");
            plain = "00000000000000000000000000000000";
            cipher = "52fc3e620492ea99641ea168da5b6d52";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffffffffffffe0000");
            plain = "00000000000000000000000000000000";
            cipher = "d2e0c7f15b4772467d2cfc873000b2ca";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff0000");
            plain = "00000000000000000000000000000000";
            cipher = "563531135e0c4d70a38f8bdb190ba04e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff8000");
            plain = "00000000000000000000000000000000";
            cipher = "a8a39a0f5663f4c0fe5f2d3cafff421a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffc000");
            plain = "00000000000000000000000000000000";
            cipher = "d94b5e90db354c1e42f61fabe167b2c0";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffe000");
            plain = "00000000000000000000000000000000";
            cipher = "50e6d3c9b6698a7cd276f96b1473f35a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff000");
            plain = "00000000000000000000000000000000";
            cipher = "9338f08e0ebee96905d8f2e825208f43";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff800");
            plain = "00000000000000000000000000000000";
            cipher = "8b378c86672aa54a3a266ba19d2580ca";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffc00");
            plain = "00000000000000000000000000000000";
            cipher = "cca7c3086f5f9511b31233da7cab9160";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffe00");
            plain = "00000000000000000000000000000000";
            cipher = "5b40ff4ec9be536ba23035fa4f06064c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff00");
            plain = "00000000000000000000000000000000";
            cipher = "60eb5af8416b257149372194e8b88749";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff80");
            plain = "00000000000000000000000000000000";
            cipher = "2f005a8aed8a361c92e440c15520cbd1";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffc0");
            plain = "00000000000000000000000000000000";
            cipher = "7b03627611678a997717578807a800e2";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffe0");
            plain = "00000000000000000000000000000000";
            cipher = "cf78618f74f6f3696e0a4779b90b5a77";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff0");
            plain = "00000000000000000000000000000000";
            cipher = "03720371a04962eaea0a852e69972858";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff8");
            plain = "00000000000000000000000000000000";
            cipher = "1f8a8133aa8ccf70e2bd3285831ca6b7";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffc");
            plain = "00000000000000000000000000000000";
            cipher = "27936bd27fb1468fc8b48bc483321725";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("fffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffe");
            plain = "00000000000000000000000000000000";
            cipher = "b07d4f3e2cd2ef2eb545980754dfea0f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
        } catch (AesBadKeyException e) {
            e.printStackTrace();
        }
    }

    public void testECBVarTxt128d() {
        byte[] key;
        byte[] state;
        String plain;
        String cipher;
        AesTestHelper aes;
        try {
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "80000000000000000000000000000000";
            cipher = "3ad78e726c1ec02b7ebfe92b23d9ec34";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "c0000000000000000000000000000000";
            cipher = "aae5939c8efdf2f04e60b9fe7117b2c2";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "e0000000000000000000000000000000";
            cipher = "f031d4d74f5dcbf39daaf8ca3af6e527";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "f0000000000000000000000000000000";
            cipher = "96d9fd5cc4f07441727df0f33e401a36";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "f8000000000000000000000000000000";
            cipher = "30ccdb044646d7e1f3ccea3dca08b8c0";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fc000000000000000000000000000000";
            cipher = "16ae4ce5042a67ee8e177b7c587ecc82";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fe000000000000000000000000000000";
            cipher = "b6da0bb11a23855d9c5cb1b4c6412e0a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ff000000000000000000000000000000";
            cipher = "db4f1aa530967d6732ce4715eb0ee24b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ff800000000000000000000000000000";
            cipher = "a81738252621dd180a34f3455b4baa2f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffc00000000000000000000000000000";
            cipher = "77e2b508db7fd89234caf7939ee5621a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffe00000000000000000000000000000";
            cipher = "b8499c251f8442ee13f0933b688fcd19";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fff00000000000000000000000000000";
            cipher = "965135f8a81f25c9d630b17502f68e53";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fff80000000000000000000000000000";
            cipher = "8b87145a01ad1c6cede995ea3670454f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffc0000000000000000000000000000";
            cipher = "8eae3b10a0c8ca6d1d3b0fa61e56b0b2";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffe0000000000000000000000000000";
            cipher = "64b4d629810fda6bafdf08f3b0d8d2c5";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffff0000000000000000000000000000";
            cipher = "d7e5dbd3324595f8fdc7d7c571da6c2a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffff8000000000000000000000000000";
            cipher = "f3f72375264e167fca9de2c1527d9606";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffc000000000000000000000000000";
            cipher = "8ee79dd4f401ff9b7ea945d86666c13b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffe000000000000000000000000000";
            cipher = "dd35cea2799940b40db3f819cb94c08b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffff000000000000000000000000000";
            cipher = "6941cb6b3e08c2b7afa581ebdd607b87";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffff800000000000000000000000000";
            cipher = "2c20f439f6bb097b29b8bd6d99aad799";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffc00000000000000000000000000";
            cipher = "625d01f058e565f77ae86378bd2c49b3";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffe00000000000000000000000000";
            cipher = "c0b5fd98190ef45fbb4301438d095950";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffff00000000000000000000000000";
            cipher = "13001ff5d99806efd25da34f56be854b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffff80000000000000000000000000";
            cipher = "3b594c60f5c8277a5113677f94208d82";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffc0000000000000000000000000";
            cipher = "e9c0fc1818e4aa46bd2e39d638f89e05";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffe0000000000000000000000000";
            cipher = "f8023ee9c3fdc45a019b4e985c7e1a54";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffff0000000000000000000000000";
            cipher = "35f40182ab4662f3023baec1ee796b57";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffff8000000000000000000000000";
            cipher = "3aebbad7303649b4194a6945c6cc3694";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffc000000000000000000000000";
            cipher = "a2124bea53ec2834279bed7f7eb0f938";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffe000000000000000000000000";
            cipher = "b9fb4399fa4facc7309e14ec98360b0a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffff000000000000000000000000";
            cipher = "c26277437420c5d634f715aea81a9132";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffff800000000000000000000000";
            cipher = "171a0e1b2dd424f0e089af2c4c10f32f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffc00000000000000000000000";
            cipher = "7cadbe402d1b208fe735edce00aee7ce";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffe00000000000000000000000";
            cipher = "43b02ff929a1485af6f5c6d6558baa0f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffff00000000000000000000000";
            cipher = "092faacc9bf43508bf8fa8613ca75dea";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffff80000000000000000000000";
            cipher = "cb2bf8280f3f9742c7ed513fe802629c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffc0000000000000000000000";
            cipher = "215a41ee442fa992a6e323986ded3f68";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffe0000000000000000000000";
            cipher = "f21e99cf4f0f77cea836e11a2fe75fb1";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffff0000000000000000000000";
            cipher = "95e3a0ca9079e646331df8b4e70d2cd6";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffff8000000000000000000000";
            cipher = "4afe7f120ce7613f74fc12a01a828073";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffc000000000000000000000";
            cipher = "827f000e75e2c8b9d479beed913fe678";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffe000000000000000000000";
            cipher = "35830c8e7aaefe2d30310ef381cbf691";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffff000000000000000000000";
            cipher = "191aa0f2c8570144f38657ea4085ebe5";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffff800000000000000000000";
            cipher = "85062c2c909f15d9269b6c18ce99c4f0";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffc00000000000000000000";
            cipher = "678034dc9e41b5a560ed239eeab1bc78";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffe00000000000000000000";
            cipher = "c2f93a4ce5ab6d5d56f1b93cf19911c1";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffff00000000000000000000";
            cipher = "1c3112bcb0c1dcc749d799743691bf82";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffff80000000000000000000";
            cipher = "00c55bd75c7f9c881989d3ec1911c0d4";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffc0000000000000000000";
            cipher = "ea2e6b5ef182b7dff3629abd6a12045f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffe0000000000000000000";
            cipher = "22322327e01780b17397f24087f8cc6f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffff0000000000000000000";
            cipher = "c9cacb5cd11692c373b2411768149ee7";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffff8000000000000000000";
            cipher = "a18e3dbbca577860dab6b80da3139256";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffc000000000000000000";
            cipher = "79b61c37bf328ecca8d743265a3d425c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffe000000000000000000";
            cipher = "d2d99c6bcc1f06fda8e27e8ae3f1ccc7";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffff000000000000000000";
            cipher = "1bfd4b91c701fd6b61b7f997829d663b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffff800000000000000000";
            cipher = "11005d52f25f16bdc9545a876a63490a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffc00000000000000000";
            cipher = "3a4d354f02bb5a5e47d39666867f246a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffe00000000000000000";
            cipher = "d451b8d6e1e1a0ebb155fbbf6e7b7dc3";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffff00000000000000000";
            cipher = "6898d4f42fa7ba6a10ac05e87b9f2080";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffff80000000000000000";
            cipher = "b611295e739ca7d9b50f8e4c0e754a3f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffc0000000000000000";
            cipher = "7d33fc7d8abe3ca1936759f8f5deaf20";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffe0000000000000000";
            cipher = "3b5e0f566dc96c298f0c12637539b25c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffff0000000000000000";
            cipher = "f807c3e7985fe0f5a50e2cdb25c5109e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffff8000000000000000";
            cipher = "41f992a856fb278b389a62f5d274d7e9";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffc000000000000000";
            cipher = "10d3ed7a6fe15ab4d91acbc7d0767ab1";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffe000000000000000";
            cipher = "21feecd45b2e675973ac33bf0c5424fc";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffff000000000000000";
            cipher = "1480cb3955ba62d09eea668f7c708817";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffff800000000000000";
            cipher = "66404033d6b72b609354d5496e7eb511";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffc00000000000000";
            cipher = "1c317a220a7d700da2b1e075b00266e1";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffe00000000000000";
            cipher = "ab3b89542233f1271bf8fd0c0f403545";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffff00000000000000";
            cipher = "d93eae966fac46dca927d6b114fa3f9e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffff80000000000000";
            cipher = "1bdec521316503d9d5ee65df3ea94ddf";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffffc0000000000000";
            cipher = "eef456431dea8b4acf83bdae3717f75f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffffe0000000000000";
            cipher = "06f2519a2fafaa596bfef5cfa15c21b9";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffff0000000000000";
            cipher = "251a7eac7e2fe809e4aa8d0d7012531a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffff8000000000000";
            cipher = "3bffc16e4c49b268a20f8d96a60b4058";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffffc000000000000";
            cipher = "e886f9281999c5bb3b3e8862e2f7c988";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffffe000000000000";
            cipher = "563bf90d61beef39f48dd625fcef1361";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffffff000000000000";
            cipher = "4d37c850644563c69fd0acd9a049325b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffffff800000000000";
            cipher = "b87c921b91829ef3b13ca541ee1130a6";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffffffc00000000000";
            cipher = "2e65eb6b6ea383e109accce8326b0393";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffffffe00000000000";
            cipher = "9ca547f7439edc3e255c0f4d49aa8990";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffffff00000000000";
            cipher = "a5e652614c9300f37816b1f9fd0c87f9";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffffff80000000000";
            cipher = "14954f0b4697776f44494fe458d814ed";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffffffc0000000000";
            cipher = "7c8d9ab6c2761723fe42f8bb506cbcf7";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffffffe0000000000";
            cipher = "db7e1932679fdd99742aab04aa0d5a80";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffffffff0000000000";
            cipher = "4c6a1c83e568cd10f27c2d73ded19c28";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffffffff8000000000";
            cipher = "90ecbe6177e674c98de412413f7ac915";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffffffffc000000000";
            cipher = "90684a2ac55fe1ec2b8ebd5622520b73";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffffffffe000000000";
            cipher = "7472f9a7988607ca79707795991035e6";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffffffff000000000";
            cipher = "56aff089878bf3352f8df172a3ae47d8";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffffffff800000000";
            cipher = "65c0526cbe40161b8019a2a3171abd23";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffffffffc00000000";
            cipher = "377be0be33b4e3e310b4aabda173f84f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffffffffe00000000";
            cipher = "9402e9aa6f69de6504da8d20c4fcaa2f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffffffffff00000000";
            cipher = "123c1f4af313ad8c2ce648b2e71fb6e1";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffffffffff80000000";
            cipher = "1ffc626d30203dcdb0019fb80f726cf4";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffc0000000";
            cipher = "76da1fbe3a50728c50fd2e621b5ad885";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffe0000000";
            cipher = "082eb8be35f442fb52668e16a591d1d6";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffffffffff0000000";
            cipher = "e656f9ecf5fe27ec3e4a73d00c282fb3";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffffffffff8000000";
            cipher = "2ca8209d63274cd9a29bb74bcd77683a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffc000000";
            cipher = "79bf5dce14bb7dd73a8e3611de7ce026";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffe000000";
            cipher = "3c849939a5d29399f344c4a0eca8a576";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffff000000";
            cipher = "ed3c0a94d59bece98835da7aa4f07ca2";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffff800000";
            cipher = "63919ed4ce10196438b6ad09d99cd795";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffc00000";
            cipher = "7678f3a833f19fea95f3c6029e2bc610";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffe00000";
            cipher = "3aa426831067d36b92be7c5f81c13c56";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffff00000";
            cipher = "9272e2d2cdd11050998c845077a30ea0";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffff80000";
            cipher = "088c4b53f5ec0ff814c19adae7f6246c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffc0000";
            cipher = "4010a5e401fdf0a0354ddbcc0d012b17";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffe0000";
            cipher = "a87a385736c0a6189bd6589bd8445a93";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffff0000";
            cipher = "545f2b83d9616dccf60fa9830e9cd287";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffff8000";
            cipher = "4b706f7f92406352394037a6d4f4688d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffffc000";
            cipher = "b7972b3941c44b90afa7b264bfba7387";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffffe000";
            cipher = "6f45732cf10881546f0fd23896d2bb60";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffff000";
            cipher = "2e3579ca15af27f64b3c955a5bfc30ba";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffff800";
            cipher = "34a2c5a91ae2aec99b7d1b5fa6780447";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffffc00";
            cipher = "a4d6616bd04f87335b0e53351227a9ee";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffffe00";
            cipher = "7f692b03945867d16179a8cefc83ea3f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffffff00";
            cipher = "3bd141ee84a0e6414a26e7a4f281f8a2";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffffff80";
            cipher = "d1788f572d98b2b16ec5d5f3922b99bc";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffffffc0";
            cipher = "0833ff6f61d98a57b288e8c3586b85a6";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffffffe0";
            cipher = "8568261797de176bf0b43becc6285afb";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffffff0";
            cipher = "f9b0fda0c4a898f5b9e6f661c4ce4d07";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffffff8";
            cipher = "8ade895913685c67c5269f8aae42983e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffffffc";
            cipher = "39bde67d5c8ed8a8b1c37eb8fa9f5ac0";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffffffe";
            cipher = "5c005e72c1418c44f569f2ea33ba54f3";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
        } catch (AesBadKeyException e) {
            e.printStackTrace();
        }
    }

    public void testECBVarTxt128e() {
        byte[] key;
        byte[] state;
        String plain;
        String cipher;
        AesTestHelper aes;
        try {
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "80000000000000000000000000000000";
            cipher = "3ad78e726c1ec02b7ebfe92b23d9ec34";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "c0000000000000000000000000000000";
            cipher = "aae5939c8efdf2f04e60b9fe7117b2c2";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "e0000000000000000000000000000000";
            cipher = "f031d4d74f5dcbf39daaf8ca3af6e527";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "f0000000000000000000000000000000";
            cipher = "96d9fd5cc4f07441727df0f33e401a36";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "f8000000000000000000000000000000";
            cipher = "30ccdb044646d7e1f3ccea3dca08b8c0";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fc000000000000000000000000000000";
            cipher = "16ae4ce5042a67ee8e177b7c587ecc82";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fe000000000000000000000000000000";
            cipher = "b6da0bb11a23855d9c5cb1b4c6412e0a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ff000000000000000000000000000000";
            cipher = "db4f1aa530967d6732ce4715eb0ee24b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ff800000000000000000000000000000";
            cipher = "a81738252621dd180a34f3455b4baa2f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffc00000000000000000000000000000";
            cipher = "77e2b508db7fd89234caf7939ee5621a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffe00000000000000000000000000000";
            cipher = "b8499c251f8442ee13f0933b688fcd19";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fff00000000000000000000000000000";
            cipher = "965135f8a81f25c9d630b17502f68e53";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fff80000000000000000000000000000";
            cipher = "8b87145a01ad1c6cede995ea3670454f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffc0000000000000000000000000000";
            cipher = "8eae3b10a0c8ca6d1d3b0fa61e56b0b2";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffe0000000000000000000000000000";
            cipher = "64b4d629810fda6bafdf08f3b0d8d2c5";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffff0000000000000000000000000000";
            cipher = "d7e5dbd3324595f8fdc7d7c571da6c2a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffff8000000000000000000000000000";
            cipher = "f3f72375264e167fca9de2c1527d9606";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffc000000000000000000000000000";
            cipher = "8ee79dd4f401ff9b7ea945d86666c13b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffe000000000000000000000000000";
            cipher = "dd35cea2799940b40db3f819cb94c08b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffff000000000000000000000000000";
            cipher = "6941cb6b3e08c2b7afa581ebdd607b87";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffff800000000000000000000000000";
            cipher = "2c20f439f6bb097b29b8bd6d99aad799";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffc00000000000000000000000000";
            cipher = "625d01f058e565f77ae86378bd2c49b3";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffe00000000000000000000000000";
            cipher = "c0b5fd98190ef45fbb4301438d095950";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffff00000000000000000000000000";
            cipher = "13001ff5d99806efd25da34f56be854b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffff80000000000000000000000000";
            cipher = "3b594c60f5c8277a5113677f94208d82";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffc0000000000000000000000000";
            cipher = "e9c0fc1818e4aa46bd2e39d638f89e05";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffe0000000000000000000000000";
            cipher = "f8023ee9c3fdc45a019b4e985c7e1a54";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffff0000000000000000000000000";
            cipher = "35f40182ab4662f3023baec1ee796b57";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffff8000000000000000000000000";
            cipher = "3aebbad7303649b4194a6945c6cc3694";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffc000000000000000000000000";
            cipher = "a2124bea53ec2834279bed7f7eb0f938";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffe000000000000000000000000";
            cipher = "b9fb4399fa4facc7309e14ec98360b0a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffff000000000000000000000000";
            cipher = "c26277437420c5d634f715aea81a9132";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffff800000000000000000000000";
            cipher = "171a0e1b2dd424f0e089af2c4c10f32f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffc00000000000000000000000";
            cipher = "7cadbe402d1b208fe735edce00aee7ce";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffe00000000000000000000000";
            cipher = "43b02ff929a1485af6f5c6d6558baa0f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffff00000000000000000000000";
            cipher = "092faacc9bf43508bf8fa8613ca75dea";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffff80000000000000000000000";
            cipher = "cb2bf8280f3f9742c7ed513fe802629c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffc0000000000000000000000";
            cipher = "215a41ee442fa992a6e323986ded3f68";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffe0000000000000000000000";
            cipher = "f21e99cf4f0f77cea836e11a2fe75fb1";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffff0000000000000000000000";
            cipher = "95e3a0ca9079e646331df8b4e70d2cd6";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffff8000000000000000000000";
            cipher = "4afe7f120ce7613f74fc12a01a828073";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffc000000000000000000000";
            cipher = "827f000e75e2c8b9d479beed913fe678";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffe000000000000000000000";
            cipher = "35830c8e7aaefe2d30310ef381cbf691";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffff000000000000000000000";
            cipher = "191aa0f2c8570144f38657ea4085ebe5";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffff800000000000000000000";
            cipher = "85062c2c909f15d9269b6c18ce99c4f0";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffc00000000000000000000";
            cipher = "678034dc9e41b5a560ed239eeab1bc78";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffe00000000000000000000";
            cipher = "c2f93a4ce5ab6d5d56f1b93cf19911c1";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffff00000000000000000000";
            cipher = "1c3112bcb0c1dcc749d799743691bf82";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffff80000000000000000000";
            cipher = "00c55bd75c7f9c881989d3ec1911c0d4";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffc0000000000000000000";
            cipher = "ea2e6b5ef182b7dff3629abd6a12045f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffe0000000000000000000";
            cipher = "22322327e01780b17397f24087f8cc6f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffff0000000000000000000";
            cipher = "c9cacb5cd11692c373b2411768149ee7";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffff8000000000000000000";
            cipher = "a18e3dbbca577860dab6b80da3139256";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffc000000000000000000";
            cipher = "79b61c37bf328ecca8d743265a3d425c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffe000000000000000000";
            cipher = "d2d99c6bcc1f06fda8e27e8ae3f1ccc7";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffff000000000000000000";
            cipher = "1bfd4b91c701fd6b61b7f997829d663b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffff800000000000000000";
            cipher = "11005d52f25f16bdc9545a876a63490a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffc00000000000000000";
            cipher = "3a4d354f02bb5a5e47d39666867f246a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffe00000000000000000";
            cipher = "d451b8d6e1e1a0ebb155fbbf6e7b7dc3";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffff00000000000000000";
            cipher = "6898d4f42fa7ba6a10ac05e87b9f2080";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffff80000000000000000";
            cipher = "b611295e739ca7d9b50f8e4c0e754a3f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffc0000000000000000";
            cipher = "7d33fc7d8abe3ca1936759f8f5deaf20";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffe0000000000000000";
            cipher = "3b5e0f566dc96c298f0c12637539b25c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffff0000000000000000";
            cipher = "f807c3e7985fe0f5a50e2cdb25c5109e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffff8000000000000000";
            cipher = "41f992a856fb278b389a62f5d274d7e9";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffc000000000000000";
            cipher = "10d3ed7a6fe15ab4d91acbc7d0767ab1";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffe000000000000000";
            cipher = "21feecd45b2e675973ac33bf0c5424fc";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffff000000000000000";
            cipher = "1480cb3955ba62d09eea668f7c708817";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffff800000000000000";
            cipher = "66404033d6b72b609354d5496e7eb511";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffc00000000000000";
            cipher = "1c317a220a7d700da2b1e075b00266e1";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffe00000000000000";
            cipher = "ab3b89542233f1271bf8fd0c0f403545";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffff00000000000000";
            cipher = "d93eae966fac46dca927d6b114fa3f9e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffff80000000000000";
            cipher = "1bdec521316503d9d5ee65df3ea94ddf";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffffc0000000000000";
            cipher = "eef456431dea8b4acf83bdae3717f75f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffffe0000000000000";
            cipher = "06f2519a2fafaa596bfef5cfa15c21b9";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffff0000000000000";
            cipher = "251a7eac7e2fe809e4aa8d0d7012531a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffff8000000000000";
            cipher = "3bffc16e4c49b268a20f8d96a60b4058";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffffc000000000000";
            cipher = "e886f9281999c5bb3b3e8862e2f7c988";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffffe000000000000";
            cipher = "563bf90d61beef39f48dd625fcef1361";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffffff000000000000";
            cipher = "4d37c850644563c69fd0acd9a049325b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffffff800000000000";
            cipher = "b87c921b91829ef3b13ca541ee1130a6";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffffffc00000000000";
            cipher = "2e65eb6b6ea383e109accce8326b0393";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffffffe00000000000";
            cipher = "9ca547f7439edc3e255c0f4d49aa8990";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffffff00000000000";
            cipher = "a5e652614c9300f37816b1f9fd0c87f9";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffffff80000000000";
            cipher = "14954f0b4697776f44494fe458d814ed";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffffffc0000000000";
            cipher = "7c8d9ab6c2761723fe42f8bb506cbcf7";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffffffe0000000000";
            cipher = "db7e1932679fdd99742aab04aa0d5a80";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffffffff0000000000";
            cipher = "4c6a1c83e568cd10f27c2d73ded19c28";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffffffff8000000000";
            cipher = "90ecbe6177e674c98de412413f7ac915";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffffffffc000000000";
            cipher = "90684a2ac55fe1ec2b8ebd5622520b73";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffffffffe000000000";
            cipher = "7472f9a7988607ca79707795991035e6";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffffffff000000000";
            cipher = "56aff089878bf3352f8df172a3ae47d8";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffffffff800000000";
            cipher = "65c0526cbe40161b8019a2a3171abd23";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffffffffc00000000";
            cipher = "377be0be33b4e3e310b4aabda173f84f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffffffffe00000000";
            cipher = "9402e9aa6f69de6504da8d20c4fcaa2f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffffffffff00000000";
            cipher = "123c1f4af313ad8c2ce648b2e71fb6e1";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffffffffff80000000";
            cipher = "1ffc626d30203dcdb0019fb80f726cf4";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffc0000000";
            cipher = "76da1fbe3a50728c50fd2e621b5ad885";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffe0000000";
            cipher = "082eb8be35f442fb52668e16a591d1d6";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffffffffff0000000";
            cipher = "e656f9ecf5fe27ec3e4a73d00c282fb3";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffffffffff8000000";
            cipher = "2ca8209d63274cd9a29bb74bcd77683a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffc000000";
            cipher = "79bf5dce14bb7dd73a8e3611de7ce026";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffe000000";
            cipher = "3c849939a5d29399f344c4a0eca8a576";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffff000000";
            cipher = "ed3c0a94d59bece98835da7aa4f07ca2";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffff800000";
            cipher = "63919ed4ce10196438b6ad09d99cd795";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffc00000";
            cipher = "7678f3a833f19fea95f3c6029e2bc610";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffe00000";
            cipher = "3aa426831067d36b92be7c5f81c13c56";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffff00000";
            cipher = "9272e2d2cdd11050998c845077a30ea0";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffff80000";
            cipher = "088c4b53f5ec0ff814c19adae7f6246c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffc0000";
            cipher = "4010a5e401fdf0a0354ddbcc0d012b17";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffe0000";
            cipher = "a87a385736c0a6189bd6589bd8445a93";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffff0000";
            cipher = "545f2b83d9616dccf60fa9830e9cd287";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffff8000";
            cipher = "4b706f7f92406352394037a6d4f4688d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffffc000";
            cipher = "b7972b3941c44b90afa7b264bfba7387";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffffe000";
            cipher = "6f45732cf10881546f0fd23896d2bb60";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffff000";
            cipher = "2e3579ca15af27f64b3c955a5bfc30ba";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffff800";
            cipher = "34a2c5a91ae2aec99b7d1b5fa6780447";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffffc00";
            cipher = "a4d6616bd04f87335b0e53351227a9ee";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffffe00";
            cipher = "7f692b03945867d16179a8cefc83ea3f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffffff00";
            cipher = "3bd141ee84a0e6414a26e7a4f281f8a2";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffffff80";
            cipher = "d1788f572d98b2b16ec5d5f3922b99bc";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffffffc0";
            cipher = "0833ff6f61d98a57b288e8c3586b85a6";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffffffe0";
            cipher = "8568261797de176bf0b43becc6285afb";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffffff0";
            cipher = "f9b0fda0c4a898f5b9e6f661c4ce4d07";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffffff8";
            cipher = "8ade895913685c67c5269f8aae42983e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffffffc";
            cipher = "39bde67d5c8ed8a8b1c37eb8fa9f5ac0";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("00000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffffffe";
            cipher = "5c005e72c1418c44f569f2ea33ba54f3";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
        } catch (AesBadKeyException e) {
            e.printStackTrace();
        }
    }

    public void testECBVarTxt192d() {
        byte[] key;
        byte[] state;
        String plain;
        String cipher;
        AesTestHelper aes;
        try {
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "80000000000000000000000000000000";
            cipher = "6cd02513e8d4dc986b4afe087a60bd0c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "c0000000000000000000000000000000";
            cipher = "2ce1f8b7e30627c1c4519eada44bc436";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "e0000000000000000000000000000000";
            cipher = "9946b5f87af446f5796c1fee63a2da24";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "f0000000000000000000000000000000";
            cipher = "2a560364ce529efc21788779568d5555";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "f8000000000000000000000000000000";
            cipher = "35c1471837af446153bce55d5ba72a0a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fc000000000000000000000000000000";
            cipher = "ce60bc52386234f158f84341e534cd9e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fe000000000000000000000000000000";
            cipher = "8c7c27ff32bcf8dc2dc57c90c2903961";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ff000000000000000000000000000000";
            cipher = "32bb6a7ec84499e166f936003d55a5bb";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ff800000000000000000000000000000";
            cipher = "a5c772e5c62631ef660ee1d5877f6d1b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffc00000000000000000000000000000";
            cipher = "030d7e5b64f380a7e4ea5387b5cd7f49";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffe00000000000000000000000000000";
            cipher = "0dc9a2610037009b698f11bb7e86c83e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fff00000000000000000000000000000";
            cipher = "0046612c766d1840c226364f1fa7ed72";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fff80000000000000000000000000000";
            cipher = "4880c7e08f27befe78590743c05e698b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffc0000000000000000000000000000";
            cipher = "2520ce829a26577f0f4822c4ecc87401";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffe0000000000000000000000000000";
            cipher = "8765e8acc169758319cb46dc7bcf3dca";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffff0000000000000000000000000000";
            cipher = "e98f4ba4f073df4baa116d011dc24a28";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffff8000000000000000000000000000";
            cipher = "f378f68c5dbf59e211b3a659a7317d94";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffc000000000000000000000000000";
            cipher = "283d3b069d8eb9fb432d74b96ca762b4";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffe000000000000000000000000000";
            cipher = "a7e1842e8a87861c221a500883245c51";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffff000000000000000000000000000";
            cipher = "77aa270471881be070fb52c7067ce732";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffff800000000000000000000000000";
            cipher = "01b0f476d484f43f1aeb6efa9361a8ac";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffc00000000000000000000000000";
            cipher = "1c3a94f1c052c55c2d8359aff2163b4f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffe00000000000000000000000000";
            cipher = "e8a067b604d5373d8b0f2e05a03b341b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffff00000000000000000000000000";
            cipher = "a7876ec87f5a09bfea42c77da30fd50e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffff80000000000000000000000000";
            cipher = "0cf3e9d3a42be5b854ca65b13f35f48d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffc0000000000000000000000000";
            cipher = "6c62f6bbcab7c3e821c9290f08892dda";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffe0000000000000000000000000";
            cipher = "7f5e05bd2068738196fee79ace7e3aec";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffff0000000000000000000000000";
            cipher = "440e0d733255cda92fb46e842fe58054";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffff8000000000000000000000000";
            cipher = "aa5d5b1c4ea1b7a22e5583ac2e9ed8a7";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffc000000000000000000000000";
            cipher = "77e537e89e8491e8662aae3bc809421d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffe000000000000000000000000";
            cipher = "997dd3e9f1598bfa73f75973f7e93b76";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffff000000000000000000000000";
            cipher = "1b38d4f7452afefcb7fc721244e4b72e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffff800000000000000000000000";
            cipher = "0be2b18252e774dda30cdda02c6906e3";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffc00000000000000000000000";
            cipher = "d2695e59c20361d82652d7d58b6f11b2";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffe00000000000000000000000";
            cipher = "902d88d13eae52089abd6143cfe394e9";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffff00000000000000000000000";
            cipher = "d49bceb3b823fedd602c305345734bd2";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffff80000000000000000000000";
            cipher = "707b1dbb0ffa40ef7d95def421233fae";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffc0000000000000000000000";
            cipher = "7ca0c1d93356d9eb8aa952084d75f913";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffe0000000000000000000000";
            cipher = "f2cbf9cb186e270dd7bdb0c28febc57d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffff0000000000000000000000";
            cipher = "c94337c37c4e790ab45780bd9c3674a0";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffff8000000000000000000000";
            cipher = "8e3558c135252fb9c9f367ed609467a1";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffc000000000000000000000";
            cipher = "1b72eeaee4899b443914e5b3a57fba92";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffe000000000000000000000";
            cipher = "011865f91bc56868d051e52c9efd59b7";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffff000000000000000000000";
            cipher = "e4771318ad7a63dd680f6e583b7747ea";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffff800000000000000000000";
            cipher = "61e3d194088dc8d97e9e6db37457eac5";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffc00000000000000000000";
            cipher = "36ff1ec9ccfbc349e5d356d063693ad6";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffe00000000000000000000";
            cipher = "3cc9e9a9be8cc3f6fb2ea24088e9bb19";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffff00000000000000000000";
            cipher = "1ee5ab003dc8722e74905d9a8fe3d350";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffff80000000000000000000";
            cipher = "245339319584b0a412412869d6c2eada";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffc0000000000000000000";
            cipher = "7bd496918115d14ed5380852716c8814";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffe0000000000000000000";
            cipher = "273ab2f2b4a366a57d582a339313c8b1";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffff0000000000000000000";
            cipher = "113365a9ffbe3b0ca61e98507554168b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffff8000000000000000000";
            cipher = "afa99c997ac478a0dea4119c9e45f8b1";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffc000000000000000000";
            cipher = "9216309a7842430b83ffb98638011512";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffe000000000000000000";
            cipher = "62abc792288258492a7cb45145f4b759";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffff000000000000000000";
            cipher = "534923c169d504d7519c15d30e756c50";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffff800000000000000000";
            cipher = "fa75e05bcdc7e00c273fa33f6ee441d2";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffc00000000000000000";
            cipher = "7d350fa6057080f1086a56b17ec240db";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffe00000000000000000";
            cipher = "f34e4a6324ea4a5c39a661c8fe5ada8f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffff00000000000000000";
            cipher = "0882a16f44088d42447a29ac090ec17e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffff80000000000000000";
            cipher = "3a3c15bfc11a9537c130687004e136ee";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffc0000000000000000";
            cipher = "22c0a7678dc6d8cf5c8a6d5a9960767c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffe0000000000000000";
            cipher = "b46b09809d68b9a456432a79bdc2e38c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffff0000000000000000";
            cipher = "93baaffb35fbe739c17c6ac22eecf18f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffff8000000000000000";
            cipher = "c8aa80a7850675bc007c46df06b49868";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffc000000000000000";
            cipher = "12c6f3877af421a918a84b775858021d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffe000000000000000";
            cipher = "33f123282c5d633924f7d5ba3f3cab11";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffff000000000000000";
            cipher = "a8f161002733e93ca4527d22c1a0c5bb";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffff800000000000000";
            cipher = "b72f70ebf3e3fda23f508eec76b42c02";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffc00000000000000";
            cipher = "6a9d965e6274143f25afdcfc88ffd77c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffe00000000000000";
            cipher = "a0c74fd0b9361764ce91c5200b095357";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffff00000000000000";
            cipher = "091d1fdc2bd2c346cd5046a8c6209146";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffff80000000000000";
            cipher = "e2a37580116cfb71856254496ab0aca8";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffc0000000000000";
            cipher = "e0b3a00785917c7efc9adba322813571";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffe0000000000000";
            cipher = "733d41f4727b5ef0df4af4cf3cffa0cb";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffff0000000000000";
            cipher = "a99ebb030260826f981ad3e64490aa4f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffff8000000000000";
            cipher = "73f34c7d3eae5e80082c1647524308ee";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffc000000000000";
            cipher = "40ebd5ad082345b7a2097ccd3464da02";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffe000000000000";
            cipher = "7cc4ae9a424b2cec90c97153c2457ec5";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffff000000000000";
            cipher = "54d632d03aba0bd0f91877ebdd4d09cb";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffff800000000000";
            cipher = "d3427be7e4d27cd54f5fe37b03cf0897";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffc00000000000";
            cipher = "b2099795e88cc158fd75ea133d7e7fbe";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffe00000000000";
            cipher = "a6cae46fb6fadfe7a2c302a34242817b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffff00000000000";
            cipher = "026a7024d6a902e0b3ffccbaa910cc3f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffff80000000000";
            cipher = "156f07767a85a4312321f63968338a01";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffc0000000000";
            cipher = "15eec9ebf42b9ca76897d2cd6c5a12e2";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffe0000000000";
            cipher = "db0d3a6fdcc13f915e2b302ceeb70fd8";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffff0000000000";
            cipher = "71dbf37e87a2e34d15b20e8f10e48924";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffff8000000000";
            cipher = "c745c451e96ff3c045e4367c833e3b54";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffc000000000";
            cipher = "340da09c2dd11c3b679d08ccd27dd595";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffe000000000";
            cipher = "8279f7c0c2a03ee660c6d392db025d18";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffff000000000";
            cipher = "a4b2c7d8eba531ff47c5041a55fbd1ec";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffff800000000";
            cipher = "74569a2ca5a7bd5131ce8dc7cbfbf72f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffc00000000";
            cipher = "3713da0c0219b63454035613b5a403dd";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffe00000000";
            cipher = "8827551ddcc9df23fa72a3de4e9f0b07";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffff00000000";
            cipher = "2e3febfd625bfcd0a2c06eb460da1732";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffff80000000";
            cipher = "ee82e6ba488156f76496311da6941deb";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffc0000000";
            cipher = "4770446f01d1f391256e85a1b30d89d3";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffe0000000";
            cipher = "af04b68f104f21ef2afb4767cf74143c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffff0000000";
            cipher = "cf3579a9ba38c8e43653173e14f3a4c6";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffff8000000";
            cipher = "b3bba904f4953e09b54800af2f62e7d4";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffc000000";
            cipher = "fc4249656e14b29eb9c44829b4c59a46";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffe000000";
            cipher = "9b31568febe81cfc2e65af1c86d1a308";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffff000000";
            cipher = "9ca09c25f273a766db98a480ce8dfedc";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffff800000";
            cipher = "b909925786f34c3c92d971883c9fbedf";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffc00000";
            cipher = "82647f1332fe570a9d4d92b2ee771d3b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffe00000";
            cipher = "3604a7e80832b3a99954bca6f5b9f501";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffff00000";
            cipher = "884607b128c5de3ab39a529a1ef51bef";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffff80000";
            cipher = "670cfa093d1dbdb2317041404102435e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffc0000";
            cipher = "7a867195f3ce8769cbd336502fbb5130";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffe0000";
            cipher = "52efcf64c72b2f7ca5b3c836b1078c15";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffff0000";
            cipher = "4019250f6eefb2ac5ccbcae044e75c7e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffff8000";
            cipher = "022c4f6f5a017d292785627667ddef24";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffffc000";
            cipher = "e9c21078a2eb7e03250f71000fa9e3ed";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffffe000";
            cipher = "a13eaeeb9cd391da4e2b09490b3e7fad";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffff000";
            cipher = "c958a171dca1d4ed53e1af1d380803a9";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffff800";
            cipher = "21442e07a110667f2583eaeeee44dc8c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffffc00";
            cipher = "59bbb353cf1dd867a6e33737af655e99";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffffe00";
            cipher = "43cd3b25375d0ce41087ff9fe2829639";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffffff00";
            cipher = "6b98b17e80d1118e3516bd768b285a84";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffffff80";
            cipher = "ae47ed3676ca0c08deea02d95b81db58";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffffffc0";
            cipher = "34ec40dc20413795ed53628ea748720b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffffffe0";
            cipher = "4dc68163f8e9835473253542c8a65d46";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffffff0";
            cipher = "2aabb999f43693175af65c6c612c46fb";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffffff8";
            cipher = "e01f94499dac3547515c5b1d756f0f58";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffffffc";
            cipher = "9d12435a46480ce00ea349f71799df9a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffffffe";
            cipher = "cef41d16d266bdfe46938ad7884cc0cf";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
        } catch (AesBadKeyException e) {
            e.printStackTrace();
        }
    }

    public void testECBVarTxt192e() {
        byte[] key;
        byte[] state;
        String plain;
        String cipher;
        AesTestHelper aes;
        try {
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "80000000000000000000000000000000";
            cipher = "6cd02513e8d4dc986b4afe087a60bd0c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "c0000000000000000000000000000000";
            cipher = "2ce1f8b7e30627c1c4519eada44bc436";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "e0000000000000000000000000000000";
            cipher = "9946b5f87af446f5796c1fee63a2da24";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "f0000000000000000000000000000000";
            cipher = "2a560364ce529efc21788779568d5555";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "f8000000000000000000000000000000";
            cipher = "35c1471837af446153bce55d5ba72a0a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fc000000000000000000000000000000";
            cipher = "ce60bc52386234f158f84341e534cd9e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fe000000000000000000000000000000";
            cipher = "8c7c27ff32bcf8dc2dc57c90c2903961";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ff000000000000000000000000000000";
            cipher = "32bb6a7ec84499e166f936003d55a5bb";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ff800000000000000000000000000000";
            cipher = "a5c772e5c62631ef660ee1d5877f6d1b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffc00000000000000000000000000000";
            cipher = "030d7e5b64f380a7e4ea5387b5cd7f49";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffe00000000000000000000000000000";
            cipher = "0dc9a2610037009b698f11bb7e86c83e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fff00000000000000000000000000000";
            cipher = "0046612c766d1840c226364f1fa7ed72";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fff80000000000000000000000000000";
            cipher = "4880c7e08f27befe78590743c05e698b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffc0000000000000000000000000000";
            cipher = "2520ce829a26577f0f4822c4ecc87401";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffe0000000000000000000000000000";
            cipher = "8765e8acc169758319cb46dc7bcf3dca";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffff0000000000000000000000000000";
            cipher = "e98f4ba4f073df4baa116d011dc24a28";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffff8000000000000000000000000000";
            cipher = "f378f68c5dbf59e211b3a659a7317d94";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffc000000000000000000000000000";
            cipher = "283d3b069d8eb9fb432d74b96ca762b4";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffe000000000000000000000000000";
            cipher = "a7e1842e8a87861c221a500883245c51";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffff000000000000000000000000000";
            cipher = "77aa270471881be070fb52c7067ce732";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffff800000000000000000000000000";
            cipher = "01b0f476d484f43f1aeb6efa9361a8ac";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffc00000000000000000000000000";
            cipher = "1c3a94f1c052c55c2d8359aff2163b4f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffe00000000000000000000000000";
            cipher = "e8a067b604d5373d8b0f2e05a03b341b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffff00000000000000000000000000";
            cipher = "a7876ec87f5a09bfea42c77da30fd50e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffff80000000000000000000000000";
            cipher = "0cf3e9d3a42be5b854ca65b13f35f48d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffc0000000000000000000000000";
            cipher = "6c62f6bbcab7c3e821c9290f08892dda";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffe0000000000000000000000000";
            cipher = "7f5e05bd2068738196fee79ace7e3aec";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffff0000000000000000000000000";
            cipher = "440e0d733255cda92fb46e842fe58054";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffff8000000000000000000000000";
            cipher = "aa5d5b1c4ea1b7a22e5583ac2e9ed8a7";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffc000000000000000000000000";
            cipher = "77e537e89e8491e8662aae3bc809421d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffe000000000000000000000000";
            cipher = "997dd3e9f1598bfa73f75973f7e93b76";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffff000000000000000000000000";
            cipher = "1b38d4f7452afefcb7fc721244e4b72e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffff800000000000000000000000";
            cipher = "0be2b18252e774dda30cdda02c6906e3";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffc00000000000000000000000";
            cipher = "d2695e59c20361d82652d7d58b6f11b2";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffe00000000000000000000000";
            cipher = "902d88d13eae52089abd6143cfe394e9";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffff00000000000000000000000";
            cipher = "d49bceb3b823fedd602c305345734bd2";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffff80000000000000000000000";
            cipher = "707b1dbb0ffa40ef7d95def421233fae";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffc0000000000000000000000";
            cipher = "7ca0c1d93356d9eb8aa952084d75f913";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffe0000000000000000000000";
            cipher = "f2cbf9cb186e270dd7bdb0c28febc57d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffff0000000000000000000000";
            cipher = "c94337c37c4e790ab45780bd9c3674a0";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffff8000000000000000000000";
            cipher = "8e3558c135252fb9c9f367ed609467a1";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffc000000000000000000000";
            cipher = "1b72eeaee4899b443914e5b3a57fba92";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffe000000000000000000000";
            cipher = "011865f91bc56868d051e52c9efd59b7";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffff000000000000000000000";
            cipher = "e4771318ad7a63dd680f6e583b7747ea";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffff800000000000000000000";
            cipher = "61e3d194088dc8d97e9e6db37457eac5";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffc00000000000000000000";
            cipher = "36ff1ec9ccfbc349e5d356d063693ad6";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffe00000000000000000000";
            cipher = "3cc9e9a9be8cc3f6fb2ea24088e9bb19";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffff00000000000000000000";
            cipher = "1ee5ab003dc8722e74905d9a8fe3d350";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffff80000000000000000000";
            cipher = "245339319584b0a412412869d6c2eada";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffc0000000000000000000";
            cipher = "7bd496918115d14ed5380852716c8814";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffe0000000000000000000";
            cipher = "273ab2f2b4a366a57d582a339313c8b1";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffff0000000000000000000";
            cipher = "113365a9ffbe3b0ca61e98507554168b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffff8000000000000000000";
            cipher = "afa99c997ac478a0dea4119c9e45f8b1";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffc000000000000000000";
            cipher = "9216309a7842430b83ffb98638011512";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffe000000000000000000";
            cipher = "62abc792288258492a7cb45145f4b759";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffff000000000000000000";
            cipher = "534923c169d504d7519c15d30e756c50";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffff800000000000000000";
            cipher = "fa75e05bcdc7e00c273fa33f6ee441d2";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffc00000000000000000";
            cipher = "7d350fa6057080f1086a56b17ec240db";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffe00000000000000000";
            cipher = "f34e4a6324ea4a5c39a661c8fe5ada8f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffff00000000000000000";
            cipher = "0882a16f44088d42447a29ac090ec17e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffff80000000000000000";
            cipher = "3a3c15bfc11a9537c130687004e136ee";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffc0000000000000000";
            cipher = "22c0a7678dc6d8cf5c8a6d5a9960767c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffe0000000000000000";
            cipher = "b46b09809d68b9a456432a79bdc2e38c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffff0000000000000000";
            cipher = "93baaffb35fbe739c17c6ac22eecf18f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffff8000000000000000";
            cipher = "c8aa80a7850675bc007c46df06b49868";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffc000000000000000";
            cipher = "12c6f3877af421a918a84b775858021d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffe000000000000000";
            cipher = "33f123282c5d633924f7d5ba3f3cab11";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffff000000000000000";
            cipher = "a8f161002733e93ca4527d22c1a0c5bb";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffff800000000000000";
            cipher = "b72f70ebf3e3fda23f508eec76b42c02";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffc00000000000000";
            cipher = "6a9d965e6274143f25afdcfc88ffd77c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffe00000000000000";
            cipher = "a0c74fd0b9361764ce91c5200b095357";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffff00000000000000";
            cipher = "091d1fdc2bd2c346cd5046a8c6209146";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffff80000000000000";
            cipher = "e2a37580116cfb71856254496ab0aca8";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffc0000000000000";
            cipher = "e0b3a00785917c7efc9adba322813571";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffe0000000000000";
            cipher = "733d41f4727b5ef0df4af4cf3cffa0cb";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffff0000000000000";
            cipher = "a99ebb030260826f981ad3e64490aa4f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffff8000000000000";
            cipher = "73f34c7d3eae5e80082c1647524308ee";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffc000000000000";
            cipher = "40ebd5ad082345b7a2097ccd3464da02";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffe000000000000";
            cipher = "7cc4ae9a424b2cec90c97153c2457ec5";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffff000000000000";
            cipher = "54d632d03aba0bd0f91877ebdd4d09cb";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffff800000000000";
            cipher = "d3427be7e4d27cd54f5fe37b03cf0897";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffc00000000000";
            cipher = "b2099795e88cc158fd75ea133d7e7fbe";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffe00000000000";
            cipher = "a6cae46fb6fadfe7a2c302a34242817b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffff00000000000";
            cipher = "026a7024d6a902e0b3ffccbaa910cc3f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffff80000000000";
            cipher = "156f07767a85a4312321f63968338a01";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffc0000000000";
            cipher = "15eec9ebf42b9ca76897d2cd6c5a12e2";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffe0000000000";
            cipher = "db0d3a6fdcc13f915e2b302ceeb70fd8";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffff0000000000";
            cipher = "71dbf37e87a2e34d15b20e8f10e48924";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffff8000000000";
            cipher = "c745c451e96ff3c045e4367c833e3b54";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffc000000000";
            cipher = "340da09c2dd11c3b679d08ccd27dd595";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffe000000000";
            cipher = "8279f7c0c2a03ee660c6d392db025d18";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffff000000000";
            cipher = "a4b2c7d8eba531ff47c5041a55fbd1ec";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffff800000000";
            cipher = "74569a2ca5a7bd5131ce8dc7cbfbf72f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffc00000000";
            cipher = "3713da0c0219b63454035613b5a403dd";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffe00000000";
            cipher = "8827551ddcc9df23fa72a3de4e9f0b07";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffff00000000";
            cipher = "2e3febfd625bfcd0a2c06eb460da1732";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffff80000000";
            cipher = "ee82e6ba488156f76496311da6941deb";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffc0000000";
            cipher = "4770446f01d1f391256e85a1b30d89d3";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffe0000000";
            cipher = "af04b68f104f21ef2afb4767cf74143c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffff0000000";
            cipher = "cf3579a9ba38c8e43653173e14f3a4c6";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffff8000000";
            cipher = "b3bba904f4953e09b54800af2f62e7d4";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffc000000";
            cipher = "fc4249656e14b29eb9c44829b4c59a46";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffe000000";
            cipher = "9b31568febe81cfc2e65af1c86d1a308";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffff000000";
            cipher = "9ca09c25f273a766db98a480ce8dfedc";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffff800000";
            cipher = "b909925786f34c3c92d971883c9fbedf";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffc00000";
            cipher = "82647f1332fe570a9d4d92b2ee771d3b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffe00000";
            cipher = "3604a7e80832b3a99954bca6f5b9f501";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffff00000";
            cipher = "884607b128c5de3ab39a529a1ef51bef";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffff80000";
            cipher = "670cfa093d1dbdb2317041404102435e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffc0000";
            cipher = "7a867195f3ce8769cbd336502fbb5130";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffe0000";
            cipher = "52efcf64c72b2f7ca5b3c836b1078c15";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffff0000";
            cipher = "4019250f6eefb2ac5ccbcae044e75c7e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffff8000";
            cipher = "022c4f6f5a017d292785627667ddef24";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffffc000";
            cipher = "e9c21078a2eb7e03250f71000fa9e3ed";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffffe000";
            cipher = "a13eaeeb9cd391da4e2b09490b3e7fad";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffff000";
            cipher = "c958a171dca1d4ed53e1af1d380803a9";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffff800";
            cipher = "21442e07a110667f2583eaeeee44dc8c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffffc00";
            cipher = "59bbb353cf1dd867a6e33737af655e99";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffffe00";
            cipher = "43cd3b25375d0ce41087ff9fe2829639";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffffff00";
            cipher = "6b98b17e80d1118e3516bd768b285a84";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffffff80";
            cipher = "ae47ed3676ca0c08deea02d95b81db58";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffffffc0";
            cipher = "34ec40dc20413795ed53628ea748720b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffffffe0";
            cipher = "4dc68163f8e9835473253542c8a65d46";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffffff0";
            cipher = "2aabb999f43693175af65c6c612c46fb";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffffff8";
            cipher = "e01f94499dac3547515c5b1d756f0f58";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffffffc";
            cipher = "9d12435a46480ce00ea349f71799df9a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity.hexStringToByteArray("000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffffffe";
            cipher = "cef41d16d266bdfe46938ad7884cc0cf";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
        } catch (AesBadKeyException e) {
            e.printStackTrace();
        }
    }

    public void testECBVarTxt256d() {
        byte[] key;
        byte[] state;
        String plain;
        String cipher;
        AesTestHelper aes;
        try {
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "80000000000000000000000000000000";
            cipher = "ddc6bf790c15760d8d9aeb6f9a75fd4e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "c0000000000000000000000000000000";
            cipher = "0a6bdc6d4c1e6280301fd8e97ddbe601";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "e0000000000000000000000000000000";
            cipher = "9b80eefb7ebe2d2b16247aa0efc72f5d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "f0000000000000000000000000000000";
            cipher = "7f2c5ece07a98d8bee13c51177395ff7";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "f8000000000000000000000000000000";
            cipher = "7818d800dcf6f4be1e0e94f403d1e4c2";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fc000000000000000000000000000000";
            cipher = "e74cd1c92f0919c35a0324123d6177d3";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fe000000000000000000000000000000";
            cipher = "8092a4dcf2da7e77e93bdd371dfed82e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ff000000000000000000000000000000";
            cipher = "49af6b372135acef10132e548f217b17";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ff800000000000000000000000000000";
            cipher = "8bcd40f94ebb63b9f7909676e667f1e7";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffc00000000000000000000000000000";
            cipher = "fe1cffb83f45dcfb38b29be438dbd3ab";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffe00000000000000000000000000000";
            cipher = "0dc58a8d886623705aec15cb1e70dc0e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fff00000000000000000000000000000";
            cipher = "c218faa16056bd0774c3e8d79c35a5e4";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fff80000000000000000000000000000";
            cipher = "047bba83f7aa841731504e012208fc9e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffc0000000000000000000000000000";
            cipher = "dc8f0e4915fd81ba70a331310882f6da";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffe0000000000000000000000000000";
            cipher = "1569859ea6b7206c30bf4fd0cbfac33c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffff0000000000000000000000000000";
            cipher = "300ade92f88f48fa2df730ec16ef44cd";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffff8000000000000000000000000000";
            cipher = "1fe6cc3c05965dc08eb0590c95ac71d0";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffc000000000000000000000000000";
            cipher = "59e858eaaa97fec38111275b6cf5abc0";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffe000000000000000000000000000";
            cipher = "2239455e7afe3b0616100288cc5a723b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffff000000000000000000000000000";
            cipher = "3ee500c5c8d63479717163e55c5c4522";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffff800000000000000000000000000";
            cipher = "d5e38bf15f16d90e3e214041d774daa8";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffc00000000000000000000000000";
            cipher = "b1f4066e6f4f187dfe5f2ad1b17819d0";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffe00000000000000000000000000";
            cipher = "6ef4cc4de49b11065d7af2909854794a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffff00000000000000000000000000";
            cipher = "ac86bc606b6640c309e782f232bf367f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffff80000000000000000000000000";
            cipher = "36aff0ef7bf3280772cf4cac80a0d2b2";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffc0000000000000000000000000";
            cipher = "1f8eedea0f62a1406d58cfc3ecea72cf";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffe0000000000000000000000000";
            cipher = "abf4154a3375a1d3e6b1d454438f95a6";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffff0000000000000000000000000";
            cipher = "96f96e9d607f6615fc192061ee648b07";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffff8000000000000000000000000";
            cipher = "cf37cdaaa0d2d536c71857634c792064";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffc000000000000000000000000";
            cipher = "fbd6640c80245c2b805373f130703127";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffe000000000000000000000000";
            cipher = "8d6a8afe55a6e481badae0d146f436db";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffff000000000000000000000000";
            cipher = "6a4981f2915e3e68af6c22385dd06756";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffff800000000000000000000000";
            cipher = "42a1136e5f8d8d21d3101998642d573b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffc00000000000000000000000";
            cipher = "9b471596dc69ae1586cee6158b0b0181";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffe00000000000000000000000";
            cipher = "753665c4af1eff33aa8b628bf8741cfd";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffff00000000000000000000000";
            cipher = "9a682acf40be01f5b2a4193c9a82404d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffff80000000000000000000000";
            cipher = "54fafe26e4287f17d1935f87eb9ade01";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffc0000000000000000000000";
            cipher = "49d541b2e74cfe73e6a8e8225f7bd449";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffe0000000000000000000000";
            cipher = "11a45530f624ff6f76a1b3826626ff7b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffff0000000000000000000000";
            cipher = "f96b0c4a8bc6c86130289f60b43b8fba";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffff8000000000000000000000";
            cipher = "48c7d0e80834ebdc35b6735f76b46c8b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffc000000000000000000000";
            cipher = "2463531ab54d66955e73edc4cb8eaa45";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffe000000000000000000000";
            cipher = "ac9bd8e2530469134b9d5b065d4f565b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffff000000000000000000000";
            cipher = "3f5f9106d0e52f973d4890e6f37e8a00";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffff800000000000000000000";
            cipher = "20ebc86f1304d272e2e207e59db639f0";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffc00000000000000000000";
            cipher = "e67ae6426bf9526c972cff072b52252c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffe00000000000000000000";
            cipher = "1a518dddaf9efa0d002cc58d107edfc8";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffff00000000000000000000";
            cipher = "ead731af4d3a2fe3b34bed047942a49f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffff80000000000000000000";
            cipher = "b1d4efe40242f83e93b6c8d7efb5eae9";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffc0000000000000000000";
            cipher = "cd2b1fec11fd906c5c7630099443610a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffe0000000000000000000";
            cipher = "a1853fe47fe29289d153161d06387d21";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffff0000000000000000000";
            cipher = "4632154179a555c17ea604d0889fab14";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffff8000000000000000000";
            cipher = "dd27cac6401a022e8f38f9f93e774417";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffc000000000000000000";
            cipher = "c090313eb98674f35f3123385fb95d4d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffe000000000000000000";
            cipher = "cc3526262b92f02edce548f716b9f45c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffff000000000000000000";
            cipher = "c0838d1a2b16a7c7f0dfcc433c399c33";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffff800000000000000000";
            cipher = "0d9ac756eb297695eed4d382eb126d26";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffc00000000000000000";
            cipher = "56ede9dda3f6f141bff1757fa689c3e1";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffe00000000000000000";
            cipher = "768f520efe0f23e61d3ec8ad9ce91774";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffff00000000000000000";
            cipher = "b1144ddfa75755213390e7c596660490";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffff80000000000000000";
            cipher = "1d7c0c4040b355b9d107a99325e3b050";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffc0000000000000000";
            cipher = "d8e2bb1ae8ee3dcf5bf7d6c38da82a1a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffe0000000000000000";
            cipher = "faf82d178af25a9886a47e7f789b98d7";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffff0000000000000000";
            cipher = "9b58dbfd77fe5aca9cfc190cd1b82d19";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffff8000000000000000";
            cipher = "77f392089042e478ac16c0c86a0b5db5";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffc000000000000000";
            cipher = "19f08e3420ee69b477ca1420281c4782";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffe000000000000000";
            cipher = "a1b19beee4e117139f74b3c53fdcb875";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffff000000000000000";
            cipher = "a37a5869b218a9f3a0868d19aea0ad6a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffff800000000000000";
            cipher = "bc3594e865bcd0261b13202731f33580";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffc00000000000000";
            cipher = "811441ce1d309eee7185e8c752c07557";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffe00000000000000";
            cipher = "959971ce4134190563518e700b9874d1";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffff00000000000000";
            cipher = "76b5614a042707c98e2132e2e805fe63";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffff80000000000000";
            cipher = "7d9fa6a57530d0f036fec31c230b0cc6";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffc0000000000000";
            cipher = "964153a83bf6989a4ba80daa91c3e081";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffe0000000000000";
            cipher = "a013014d4ce8054cf2591d06f6f2f176";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffff0000000000000";
            cipher = "d1c5f6399bf382502e385eee1474a869";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffff8000000000000";
            cipher = "0007e20b8298ec354f0f5fe7470f36bd";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffc000000000000";
            cipher = "b95ba05b332da61ef63a2b31fcad9879";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffe000000000000";
            cipher = "4620a49bd967491561669ab25dce45f4";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffff000000000000";
            cipher = "12e71214ae8e04f0bb63d7425c6f14d5";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffff800000000000";
            cipher = "4cc42fc1407b008fe350907c092e80ac";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffc00000000000";
            cipher = "08b244ce7cbc8ee97fbba808cb146fda";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffe00000000000";
            cipher = "39b333e8694f21546ad1edd9d87ed95b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffff00000000000";
            cipher = "3b271f8ab2e6e4a20ba8090f43ba78f3";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffff80000000000";
            cipher = "9ad983f3bf651cd0393f0a73cccdea50";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffc0000000000";
            cipher = "8f476cbff75c1f725ce18e4bbcd19b32";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffe0000000000";
            cipher = "905b6267f1d6ab5320835a133f096f2a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffff0000000000";
            cipher = "145b60d6d0193c23f4221848a892d61a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffff8000000000";
            cipher = "55cfb3fb6d75cad0445bbc8dafa25b0f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffc000000000";
            cipher = "7b8e7098e357ef71237d46d8b075b0f5";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffe000000000";
            cipher = "2bf27229901eb40f2df9d8398d1505ae";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffff000000000";
            cipher = "83a63402a77f9ad5c1e931a931ecd706";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffff800000000";
            cipher = "6f8ba6521152d31f2bada1843e26b973";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffc00000000";
            cipher = "e5c3b8e30fd2d8e6239b17b44bd23bbd";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffe00000000";
            cipher = "1ac1f7102c59933e8b2ddc3f14e94baa";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffff00000000";
            cipher = "21d9ba49f276b45f11af8fc71a088e3d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffff80000000";
            cipher = "649f1cddc3792b4638635a392bc9bade";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffc0000000";
            cipher = "e2775e4b59c1bc2e31a2078c11b5a08c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffe0000000";
            cipher = "2be1fae5048a25582a679ca10905eb80";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffff0000000";
            cipher = "da86f292c6f41ea34fb2068df75ecc29";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffff8000000";
            cipher = "220df19f85d69b1b562fa69a3c5beca5";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffc000000";
            cipher = "1f11d5d0355e0b556ccdb6c7f5083b4d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffe000000";
            cipher = "62526b78be79cb384633c91f83b4151b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffff000000";
            cipher = "90ddbcb950843592dd47bbef00fdc876";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffff800000";
            cipher = "2fd0e41c5b8402277354a7391d2618e2";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffc00000";
            cipher = "3cdf13e72dee4c581bafec70b85f9660";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffe00000";
            cipher = "afa2ffc137577092e2b654fa199d2c43";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffff00000";
            cipher = "8d683ee63e60d208e343ce48dbc44cac";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffff80000";
            cipher = "705a4ef8ba2133729c20185c3d3a4763";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffc0000";
            cipher = "0861a861c3db4e94194211b77ed761b9";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffe0000";
            cipher = "4b00c27e8b26da7eab9d3a88dec8b031";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffff0000";
            cipher = "5f397bf03084820cc8810d52e5b666e9";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffff8000";
            cipher = "63fafabb72c07bfbd3ddc9b1203104b8";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffffc000";
            cipher = "683e2140585b18452dd4ffbb93c95df9";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffffe000";
            cipher = "286894e48e537f8763b56707d7d155c8";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffff000";
            cipher = "a423deabc173dcf7e2c4c53e77d37cd1";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffff800";
            cipher = "eb8168313e1cfdfdb5e986d5429cf172";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffffc00";
            cipher = "27127daafc9accd2fb334ec3eba52323";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffffe00";
            cipher = "ee0715b96f72e3f7a22a5064fc592f4c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffffff00";
            cipher = "29ee526770f2a11dcfa989d1ce88830f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffffff80";
            cipher = "0493370e054b09871130fe49af730a5a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffffffc0";
            cipher = "9b7b940f6c509f9e44a4ee140448ee46";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffffffe0";
            cipher = "2915be4a1ecfdcbe3e023811a12bb6c7";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffffff0";
            cipher = "7240e524bc51d8c4d440b1be55d1062c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffffff8";
            cipher = "da63039d38cb4612b2dc36ba26684b93";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffffffc";
            cipher = "0f59cb5a4b522e2ac56c1a64f558ad9a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffffffe";
            cipher = "7bfe9d876c6d63c1d035da8fe21c409d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(cipher);
            aes.decipherState(state, 0);
            assertEquals(plain, AesdroidActivity.byteArrayToHexString(state));
        } catch (AesBadKeyException e) {
            e.printStackTrace();
        }
    }

    public void testECBVarTxt256e() {
        byte[] key;
        byte[] state;
        String plain;
        String cipher;
        AesTestHelper aes;
        try {
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "80000000000000000000000000000000";
            cipher = "ddc6bf790c15760d8d9aeb6f9a75fd4e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "c0000000000000000000000000000000";
            cipher = "0a6bdc6d4c1e6280301fd8e97ddbe601";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "e0000000000000000000000000000000";
            cipher = "9b80eefb7ebe2d2b16247aa0efc72f5d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "f0000000000000000000000000000000";
            cipher = "7f2c5ece07a98d8bee13c51177395ff7";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "f8000000000000000000000000000000";
            cipher = "7818d800dcf6f4be1e0e94f403d1e4c2";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fc000000000000000000000000000000";
            cipher = "e74cd1c92f0919c35a0324123d6177d3";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fe000000000000000000000000000000";
            cipher = "8092a4dcf2da7e77e93bdd371dfed82e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ff000000000000000000000000000000";
            cipher = "49af6b372135acef10132e548f217b17";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ff800000000000000000000000000000";
            cipher = "8bcd40f94ebb63b9f7909676e667f1e7";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffc00000000000000000000000000000";
            cipher = "fe1cffb83f45dcfb38b29be438dbd3ab";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffe00000000000000000000000000000";
            cipher = "0dc58a8d886623705aec15cb1e70dc0e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fff00000000000000000000000000000";
            cipher = "c218faa16056bd0774c3e8d79c35a5e4";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fff80000000000000000000000000000";
            cipher = "047bba83f7aa841731504e012208fc9e";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffc0000000000000000000000000000";
            cipher = "dc8f0e4915fd81ba70a331310882f6da";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffe0000000000000000000000000000";
            cipher = "1569859ea6b7206c30bf4fd0cbfac33c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffff0000000000000000000000000000";
            cipher = "300ade92f88f48fa2df730ec16ef44cd";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffff8000000000000000000000000000";
            cipher = "1fe6cc3c05965dc08eb0590c95ac71d0";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffc000000000000000000000000000";
            cipher = "59e858eaaa97fec38111275b6cf5abc0";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffe000000000000000000000000000";
            cipher = "2239455e7afe3b0616100288cc5a723b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffff000000000000000000000000000";
            cipher = "3ee500c5c8d63479717163e55c5c4522";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffff800000000000000000000000000";
            cipher = "d5e38bf15f16d90e3e214041d774daa8";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffc00000000000000000000000000";
            cipher = "b1f4066e6f4f187dfe5f2ad1b17819d0";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffe00000000000000000000000000";
            cipher = "6ef4cc4de49b11065d7af2909854794a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffff00000000000000000000000000";
            cipher = "ac86bc606b6640c309e782f232bf367f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffff80000000000000000000000000";
            cipher = "36aff0ef7bf3280772cf4cac80a0d2b2";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffc0000000000000000000000000";
            cipher = "1f8eedea0f62a1406d58cfc3ecea72cf";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffe0000000000000000000000000";
            cipher = "abf4154a3375a1d3e6b1d454438f95a6";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffff0000000000000000000000000";
            cipher = "96f96e9d607f6615fc192061ee648b07";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffff8000000000000000000000000";
            cipher = "cf37cdaaa0d2d536c71857634c792064";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffc000000000000000000000000";
            cipher = "fbd6640c80245c2b805373f130703127";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffe000000000000000000000000";
            cipher = "8d6a8afe55a6e481badae0d146f436db";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffff000000000000000000000000";
            cipher = "6a4981f2915e3e68af6c22385dd06756";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffff800000000000000000000000";
            cipher = "42a1136e5f8d8d21d3101998642d573b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffc00000000000000000000000";
            cipher = "9b471596dc69ae1586cee6158b0b0181";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffe00000000000000000000000";
            cipher = "753665c4af1eff33aa8b628bf8741cfd";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffff00000000000000000000000";
            cipher = "9a682acf40be01f5b2a4193c9a82404d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffff80000000000000000000000";
            cipher = "54fafe26e4287f17d1935f87eb9ade01";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffc0000000000000000000000";
            cipher = "49d541b2e74cfe73e6a8e8225f7bd449";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffe0000000000000000000000";
            cipher = "11a45530f624ff6f76a1b3826626ff7b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffff0000000000000000000000";
            cipher = "f96b0c4a8bc6c86130289f60b43b8fba";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffff8000000000000000000000";
            cipher = "48c7d0e80834ebdc35b6735f76b46c8b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffc000000000000000000000";
            cipher = "2463531ab54d66955e73edc4cb8eaa45";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffe000000000000000000000";
            cipher = "ac9bd8e2530469134b9d5b065d4f565b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffff000000000000000000000";
            cipher = "3f5f9106d0e52f973d4890e6f37e8a00";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffff800000000000000000000";
            cipher = "20ebc86f1304d272e2e207e59db639f0";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffc00000000000000000000";
            cipher = "e67ae6426bf9526c972cff072b52252c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffe00000000000000000000";
            cipher = "1a518dddaf9efa0d002cc58d107edfc8";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffff00000000000000000000";
            cipher = "ead731af4d3a2fe3b34bed047942a49f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffff80000000000000000000";
            cipher = "b1d4efe40242f83e93b6c8d7efb5eae9";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffc0000000000000000000";
            cipher = "cd2b1fec11fd906c5c7630099443610a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffe0000000000000000000";
            cipher = "a1853fe47fe29289d153161d06387d21";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffff0000000000000000000";
            cipher = "4632154179a555c17ea604d0889fab14";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffff8000000000000000000";
            cipher = "dd27cac6401a022e8f38f9f93e774417";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffc000000000000000000";
            cipher = "c090313eb98674f35f3123385fb95d4d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffe000000000000000000";
            cipher = "cc3526262b92f02edce548f716b9f45c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffff000000000000000000";
            cipher = "c0838d1a2b16a7c7f0dfcc433c399c33";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffff800000000000000000";
            cipher = "0d9ac756eb297695eed4d382eb126d26";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffc00000000000000000";
            cipher = "56ede9dda3f6f141bff1757fa689c3e1";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffe00000000000000000";
            cipher = "768f520efe0f23e61d3ec8ad9ce91774";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffff00000000000000000";
            cipher = "b1144ddfa75755213390e7c596660490";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffff80000000000000000";
            cipher = "1d7c0c4040b355b9d107a99325e3b050";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffc0000000000000000";
            cipher = "d8e2bb1ae8ee3dcf5bf7d6c38da82a1a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffe0000000000000000";
            cipher = "faf82d178af25a9886a47e7f789b98d7";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffff0000000000000000";
            cipher = "9b58dbfd77fe5aca9cfc190cd1b82d19";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffff8000000000000000";
            cipher = "77f392089042e478ac16c0c86a0b5db5";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffc000000000000000";
            cipher = "19f08e3420ee69b477ca1420281c4782";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffe000000000000000";
            cipher = "a1b19beee4e117139f74b3c53fdcb875";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffff000000000000000";
            cipher = "a37a5869b218a9f3a0868d19aea0ad6a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffff800000000000000";
            cipher = "bc3594e865bcd0261b13202731f33580";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffc00000000000000";
            cipher = "811441ce1d309eee7185e8c752c07557";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffe00000000000000";
            cipher = "959971ce4134190563518e700b9874d1";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffff00000000000000";
            cipher = "76b5614a042707c98e2132e2e805fe63";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffff80000000000000";
            cipher = "7d9fa6a57530d0f036fec31c230b0cc6";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffc0000000000000";
            cipher = "964153a83bf6989a4ba80daa91c3e081";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffe0000000000000";
            cipher = "a013014d4ce8054cf2591d06f6f2f176";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffff0000000000000";
            cipher = "d1c5f6399bf382502e385eee1474a869";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffff8000000000000";
            cipher = "0007e20b8298ec354f0f5fe7470f36bd";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffc000000000000";
            cipher = "b95ba05b332da61ef63a2b31fcad9879";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffe000000000000";
            cipher = "4620a49bd967491561669ab25dce45f4";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffff000000000000";
            cipher = "12e71214ae8e04f0bb63d7425c6f14d5";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffff800000000000";
            cipher = "4cc42fc1407b008fe350907c092e80ac";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffc00000000000";
            cipher = "08b244ce7cbc8ee97fbba808cb146fda";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffe00000000000";
            cipher = "39b333e8694f21546ad1edd9d87ed95b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffff00000000000";
            cipher = "3b271f8ab2e6e4a20ba8090f43ba78f3";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffff80000000000";
            cipher = "9ad983f3bf651cd0393f0a73cccdea50";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffc0000000000";
            cipher = "8f476cbff75c1f725ce18e4bbcd19b32";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffe0000000000";
            cipher = "905b6267f1d6ab5320835a133f096f2a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffff0000000000";
            cipher = "145b60d6d0193c23f4221848a892d61a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffff8000000000";
            cipher = "55cfb3fb6d75cad0445bbc8dafa25b0f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffc000000000";
            cipher = "7b8e7098e357ef71237d46d8b075b0f5";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffe000000000";
            cipher = "2bf27229901eb40f2df9d8398d1505ae";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffff000000000";
            cipher = "83a63402a77f9ad5c1e931a931ecd706";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffff800000000";
            cipher = "6f8ba6521152d31f2bada1843e26b973";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffc00000000";
            cipher = "e5c3b8e30fd2d8e6239b17b44bd23bbd";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffe00000000";
            cipher = "1ac1f7102c59933e8b2ddc3f14e94baa";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffff00000000";
            cipher = "21d9ba49f276b45f11af8fc71a088e3d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffff80000000";
            cipher = "649f1cddc3792b4638635a392bc9bade";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffc0000000";
            cipher = "e2775e4b59c1bc2e31a2078c11b5a08c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffe0000000";
            cipher = "2be1fae5048a25582a679ca10905eb80";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffff0000000";
            cipher = "da86f292c6f41ea34fb2068df75ecc29";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffff8000000";
            cipher = "220df19f85d69b1b562fa69a3c5beca5";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffc000000";
            cipher = "1f11d5d0355e0b556ccdb6c7f5083b4d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffe000000";
            cipher = "62526b78be79cb384633c91f83b4151b";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffff000000";
            cipher = "90ddbcb950843592dd47bbef00fdc876";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffff800000";
            cipher = "2fd0e41c5b8402277354a7391d2618e2";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffc00000";
            cipher = "3cdf13e72dee4c581bafec70b85f9660";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffe00000";
            cipher = "afa2ffc137577092e2b654fa199d2c43";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffff00000";
            cipher = "8d683ee63e60d208e343ce48dbc44cac";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffff80000";
            cipher = "705a4ef8ba2133729c20185c3d3a4763";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffc0000";
            cipher = "0861a861c3db4e94194211b77ed761b9";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffe0000";
            cipher = "4b00c27e8b26da7eab9d3a88dec8b031";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffff0000";
            cipher = "5f397bf03084820cc8810d52e5b666e9";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffff8000";
            cipher = "63fafabb72c07bfbd3ddc9b1203104b8";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffffc000";
            cipher = "683e2140585b18452dd4ffbb93c95df9";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffffe000";
            cipher = "286894e48e537f8763b56707d7d155c8";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffff000";
            cipher = "a423deabc173dcf7e2c4c53e77d37cd1";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffff800";
            cipher = "eb8168313e1cfdfdb5e986d5429cf172";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffffc00";
            cipher = "27127daafc9accd2fb334ec3eba52323";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffffe00";
            cipher = "ee0715b96f72e3f7a22a5064fc592f4c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffffff00";
            cipher = "29ee526770f2a11dcfa989d1ce88830f";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffffff80";
            cipher = "0493370e054b09871130fe49af730a5a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffffffc0";
            cipher = "9b7b940f6c509f9e44a4ee140448ee46";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "ffffffffffffffffffffffffffffffe0";
            cipher = "2915be4a1ecfdcbe3e023811a12bb6c7";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffffff0";
            cipher = "7240e524bc51d8c4d440b1be55d1062c";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffffff8";
            cipher = "da63039d38cb4612b2dc36ba26684b93";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffffffc";
            cipher = "0f59cb5a4b522e2ac56c1a64f558ad9a";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
            key = AesdroidActivity
                    .hexStringToByteArray("0000000000000000000000000000000000000000000000000000000000000000");
            plain = "fffffffffffffffffffffffffffffffe";
            cipher = "7bfe9d876c6d63c1d035da8fe21c409d";
            aes = new AesTestHelper(key);
            state = AesdroidActivity.hexStringToByteArray(plain);
            aes.encipherState(state, 0);
            assertEquals(cipher, AesdroidActivity.byteArrayToHexString(state));
        } catch (AesBadKeyException e) {
            e.printStackTrace();
        }
    }
}
