/*
 * Project: aesdroid - AES implementation for Android
 * File   : AesdroidActivity.java
 * Author : Oliver Mueller <oliver@cogito-ergo-sum.org>
 * Purpose: Main Activity
 *
 * $Id: AesdroidActivity.java 47 2012-07-10 07:05:36Z oliver $
 *
 * Copyright (c) 2011, 2012 Oliver Mueller.
 * All rights reserved.
 * http://www.cogito-ergo-sum.org
 * http://oliver-mueller.com
 *
 * This software is provided 'as-is', without any express or implied warranty.
 * In no event will the author be held liable for any damages arising from
 * the use of this software.
 *
 */

package org.cogito_ergo_sum.aesdroid;

import java.io.UnsupportedEncodingException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;

import org.cogito_ergo_sum.aes.AesBadKeyException;
import org.cogito_ergo_sum.aes.AesEncryption;
import org.cogito_ergo_sum.aes.AesEncryptionReference;
import org.cogito_ergo_sum.aes.AesInterface;

import android.app.Activity;
import android.app.AlertDialog;
import android.content.DialogInterface;
import android.content.Intent;
import android.net.Uri;
import android.os.Bundle;
import android.text.InputType;
import android.view.View;
import android.widget.Button;
import android.widget.CheckBox;
import android.widget.EditText;

public class AesdroidActivity extends Activity {

    private EditText messageText;
    private Button encryptButton;
    private Button decryptButton;
    private CheckBox useRefCheckBox;
    private AesInterface aes;
    // private AesEncryption aes;
    // private AesEncryptionReference aesRef;
    private static final String version = "V1.00";

    private final DialogInterface.OnClickListener omcListener =
            new DialogInterface.OnClickListener() {
                public void onClick(DialogInterface dialogInterface, int i) {
                    Intent intent = new Intent(Intent.ACTION_VIEW, Uri.parse("http://oliver-mueller.com/"));
                    startActivity(intent);
                }
            };

    private final DialogInterface.OnClickListener cesListener =
            new DialogInterface.OnClickListener() {
                public void onClick(DialogInterface dialogInterface, int i) {
                    Intent intent = new Intent(Intent.ACTION_VIEW, Uri.parse("http://www.cogito-ergo-sum.org/"));
                    startActivity(intent);
                }
            };

    /** Called when the activity is first created. */
    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.main);
        messageText = (EditText) findViewById(R.id.messageText);
        encryptButton = (Button) findViewById(R.id.encryptButton);
        encryptButton.setEnabled(false);
        decryptButton = (Button) findViewById(R.id.decryptButton);
        decryptButton.setEnabled(false);
        useRefCheckBox = (CheckBox) findViewById(R.id.useRefCheckBox);
    }

    public void onClickEncryptButton(View view) {
        if (aes != null) {
            try {
                String plain = messageText.getText().toString();
                String cipher = byteArrayToHexString(aes.encipher(plain));
                messageText.setText(cipher);
            } catch (IndexOutOfBoundsException e) {
                AlertDialog.Builder builder = new AlertDialog.Builder(this);
                builder.setTitle(getString(R.string.error));
                builder.setMessage(e.getMessage());
                builder.setNegativeButton(R.string.ok, null);
                builder.show();
            }
        }
    }

    public void onClickDecryptButton(View view) {
        if (aes != null) {
            byte[] cipher = hexStringToByteArray(messageText.getText().toString());
            String plain = aes.decipher(cipher);
            int m = 0;
            for (int n = plain.length() - 1; n >= 0; n--, m++)
                if (plain.charAt(n) != '\0')
                    break;
            plain = plain.substring(0, plain.length() - m);
            messageText.setText(plain);
        }
    }

    public void onClickGenerateKeyButton(View view) {
        AlertDialog.Builder alert = new AlertDialog.Builder(this);

        alert.setTitle(getString(R.string.gen_key));
        alert.setMessage(getString(R.string.passphrase_prompt));

        final EditText input1 = new EditText(this);
        input1.setInputType(InputType.TYPE_CLASS_TEXT |
                InputType.TYPE_TEXT_VARIATION_PASSWORD);
        alert.setView(input1);

        alert.setPositiveButton("Ok", new DialogInterface.OnClickListener() {
            public void onClick(DialogInterface dialog, int whichButton) {
                byte[] hash;
                try {
                    hash = getSha256(input1.getText().toString());
                    byte[] key = new byte[32];
                    for (int n = 0; n < key.length; n++)
                        key[n] = hash[n];
                    if(useRefCheckBox.isChecked())
                        aes = new AesEncryptionReference(key);
                    else
                        aes = new AesEncryption(key);
                    encryptButton.setEnabled(true);
                    decryptButton.setEnabled(true);
                } catch (NoSuchAlgorithmException e) {
                    aes = null;
                    AlertDialog.Builder builder = new AlertDialog.Builder(getParent());
                    builder.setTitle(getString(R.string.error));
                    builder.setMessage(e.getMessage());
                    builder.setNegativeButton(R.string.ok, null);
                    builder.show();
                } catch (UnsupportedEncodingException e) {
                    aes = null;
                    AlertDialog.Builder builder = new AlertDialog.Builder(getParent());
                    builder.setTitle(getString(R.string.error));
                    builder.setMessage(e.getMessage());
                    builder.setNegativeButton(R.string.ok, null);
                    builder.show();
                } catch (AesBadKeyException e) {
                    aes = null;
                    AlertDialog.Builder builder = new AlertDialog.Builder(getParent());
                    builder.setTitle(getString(R.string.error));
                    builder.setMessage(e.getMessage());
                    builder.setNegativeButton(R.string.ok, null);
                    builder.show();
                }
            }
        });

        alert.setNegativeButton("Cancel", new
                DialogInterface.OnClickListener() {
                    public void onClick(DialogInterface dialog, int whichButton) {
                        // Canceled.
                    }
                });

        alert.show();
    }

    public void onClickUseRefCheckBox(View view) {
        encryptButton.setEnabled(false);
        decryptButton.setEnabled(false);
    }
    
    public void onClickAboutButton(View view) {
        AlertDialog.Builder builder = new AlertDialog.Builder(this);
        builder.setTitle(getString(R.string.about));
        builder.setMessage(getString(R.string.app_name) + " " + version + "\n\n" + getString(R.string.copyright) + "\nOliver Mueller\n" + getString(R.string.rights)
                + "\n\na) http://oliver-mueller.com/\nb) http://www.cogito-ergo-sum.org/");
        builder.setIcon(R.drawable.ic_launcher);
        builder.setPositiveButton(getString(R.string.open) + " a)", omcListener);
        builder.setNeutralButton(getString(R.string.open) + " b)", cesListener);
        builder.setNegativeButton(R.string.ok, null);
        builder.show();
    }

    // Conversion methods
    public static String byteArrayToHexString(byte[] a) {
        String result = "";
        for (int i = 0; i < a.length; i++) {
            result += Integer.toString((a[i] & 0xff) + 0x100, 16).substring(1);
        }
        return result;
    }

    public static byte[] hexStringToByteArray(String s) {
        int len = s.length();
        if (len % 2 != 0)
            len--;
        byte[] data = new byte[len / 2];
        for (int i = 0; i < len; i += 2) {
            data[i / 2] = (byte) ((Character.digit(s.charAt(i), 16) << 4) + Character.digit(s.charAt(i + 1), 16));
        }
        return data;
    }

    // Cryptographic hash methods
    public static byte[] getSha256(String text)
            throws NoSuchAlgorithmException, UnsupportedEncodingException {
        MessageDigest md = MessageDigest.getInstance("SHA-256");
        md.update(text.getBytes("utf-8"), 0, text.length());
        return md.digest();
    }
}