#include <math.h>
#include <QtGui/QPainter>
#include "data.h"
#include "juliaview.h"

JuliaView::JuliaView(Data *p_data, QWidget *parent, int w, int h) : QWidget(parent) {

  data = p_data;
  mousePos.setX(0);
  mousePos.setY(0);
  mousePressed = false;
  setMinimumSize(w, h);
  setMouseTracking(true);
  juliaDataCache = (int *)malloc(data->getImageWidth() * data->getImageHeight() * sizeof(int));
}

JuliaView::~JuliaView() {

  free(juliaDataCache);  
}

void JuliaView::paintEvent(QPaintEvent *) {

  QPainter p;
  long double tmpScale[2], w, h;
  
  p.begin(this);
  p.drawImage(0, 0, *data->juliaImage);
  if (data->showZoomRect[1]) {
    p.setPen(QColor(255, 255, 255));
    tmpScale[0] = 8.0 * data->tmpZoom[1] / (long double)data->getImageWidth();
    tmpScale[1] = 8.0 * data->tmpZoom[1] / (long double)data->getImageHeight();
    w = tmpScale[0] / data->getXScale(1) * (long double)data->getImageWidth();
    h = tmpScale[1] / data->getYScale(1) * (long double)data->getImageHeight();
    p.drawRect((data->getImageWidth()>>1) - (int)(w/2.0), (data->getImageHeight()>>1) - (int)(h/2.0),(int)w,(int)h);
  }
  p.end();
}

void JuliaView::mousePressEvent (QMouseEvent *ev) {

  if (ev->button() == Qt::LeftButton) {
    mousePos = ev->pos();
    mousePressed = true;
  }  
}

void JuliaView::mouseReleaseEvent (QMouseEvent *ev) {

  if (ev->button() == Qt::LeftButton) {
    mousePressed = false;
  }  
}

void JuliaView::mouseMoveEvent (QMouseEvent *ev) {

  if (mousePressed && !data->getImageBusy(1)) {
    data->moveOfs(1, mousePos - ev->pos());   
    moveImage(ev->pos() - mousePos);
    emit imageMoved(ev->pos() - mousePos);
    mousePos = ev->pos();
  }  
}

void JuliaView::mouseDoubleClickEvent (QMouseEvent *ev) {

  if (!data->getImageBusy(1)) {
    data->moveOfs(1, mousePos - QPoint(data->getImageWidth()>>1, data->getImageHeight()>>1));
    moveImage(QPoint(data->getImageWidth()>>1, data->getImageHeight()>>1) - mousePos);
    emit imageMoved(QPoint(data->getImageWidth()>>1, data->getImageHeight()>>1) - mousePos);
    mousePos = QPoint(data->getImageWidth()>>1, data->getImageHeight()>>1);
  }  
}

void JuliaView::moveImage (QPoint dp) {

  int dx, dy, i1, i2, w, left, right, up, down;
  QRgb colorPixel;
  long double scale;
  
  dx = dp.x();
  dy = dp.y();
  w = data->getImageWidth();
  if (dx >= 0) {
    left = dx;
    right = data->getImageWidth();
  } else {
    left = 0;
    right = data->getImageWidth() + dx;
  }
  if (dy >= 0) {
    up = dy;
    down = data->getImageHeight();
  } else {
    up = 0;
    down = data->getImageHeight() + dy;
  }
  if (left < 0) left = 0;
  if (right < 0) right = 0;
  if (up < 0) up = 0;
  if (down < 0) down = 0;
  if (left > data->getImageWidth()) left = data->getImageWidth();
  if (right > data->getImageWidth()) right = data->getImageWidth();
  if (up > data->getImageHeight()) up = data->getImageHeight();
  if (down > data->getImageHeight()) down = data->getImageHeight();
  data->juliaMutex.lock();
  scale = (data->getNormalize()) ? (long double)data->getMaxIterations(1) / (long double)data->getMax(1) : 1.0;
  for (i1 = up; i1 < down; i1++) {
    for (i2 = left; i2 < right; i2++) {
      juliaDataCache[i2 + i1 * w] = data->juliaData[i2-dx + (i1-dy) * w];
      colorPixel = data->getPalette(1, (int)(scale * (long double)data->juliaData[i2-dx + (i1-dy) * w])).rgb();
      data->juliaImage->setPixel(i2, i1, colorPixel);
    }  
  }
  for (i1 = up; i1 < down; i1++) {
    for (i2 = left; i2 < right; i2++) {
      data->juliaData[i2 + i1 * w] = juliaDataCache[i2 + i1 * w];
    }
  }
  data->juliaMutex.unlock();
  repaint();
}

void JuliaView::changeColor() {

  int i1, i2, w;
  QRgb colorPixel;
  long double scale;
  
  w = data->getImageWidth();
  data->juliaMutex.lock();
  scale = (data->getNormalize()) ? (long double)data->getMaxIterations(1) / (long double)data->getMax(1) : 1.0;
  for (i1 = 0; i1 < data->getImageHeight(); i1++) {
    for (i2 = 0; i2 < data->getImageWidth(); i2++) {
      colorPixel = data->getPalette(1, (int)(scale * (long double)data->juliaData[i2 + i1 * w])).rgb();      
      data->juliaImage->setPixel(i2, i1, colorPixel);
    }
  }
  data->juliaMutex.unlock();
  repaint();
}
