#include <QtGui/QWidget>
#include <QtGui/QHBoxLayout>
#include <QtGui/QVBoxLayout>
#include <QtGui/QGridLayout>
#include <QtGui/QMainWindow>
#include <QtGui/QLabel>
#include <QtGui/QSlider>
#include <QtGui/QSplitter>
#include <QtGui/QFileDialog>
#include <QtGui/QMessageBox>
#include <QtCore/QXmlStreamWriter>
#include <QtXml/QXmlDefaultHandler>
#include "data.h"
#include "xmlhandler.h"
#include "gui.h"

Gui::Gui(QWidget *parent) : QWidget(parent) {

  QWidget *mandelFrame, *juliaFrame;
  QVBoxLayout *mandelLayout, *juliaLayout;
  QWidget *mandelZoomFrame, *juliaZoomFrame;
  QHBoxLayout *mandelZoomLayout, *juliaZoomLayout;
  QVBoxLayout *guiLayout;
  int i;
  long double c[2];
      
  data = new Data(this);
  colorDialog = new ColorDialog(data, this, Qt::Window);
  colorDialog->setWindowTitle("QJulia - Color Settings");
  iterationDialog = new IterationDialog(data, this, Qt::Window);
  iterationDialog->setWindowTitle("QJulia - Options");
  hiresDialog = new HiresDialog(data, this, Qt::Window);
  hiresDialog->setWindowTitle("QJulia - High Resolution Window");
  mandelLayout = new QVBoxLayout;
  mandelFrame = new QWidget;
  mandelFrame->setLayout(mandelLayout);
  mandelView = new MandelView(data, this, WIDTH, HEIGHT);
  mandelLayout->addWidget(mandelView); 
  juliaLayout = new QVBoxLayout;
  juliaFrame = new QWidget;
  juliaFrame->setLayout(juliaLayout);
  juliaView = new JuliaView(data, this, WIDTH, HEIGHT);
  juliaLayout->addWidget(juliaView);
  mandelZoomLayout = new QHBoxLayout;
  mandelZoomFrame = new QWidget;
  mandelZoomFrame->setLayout(mandelZoomLayout);
  QLabel *mandelZoomLabel = new QLabel("Zoom: ");
  mandelZoomLayout->addWidget(mandelZoomLabel);
  mandelZoomSlider = new QSlider(Qt::Horizontal);
  mandelZoomLayout->addWidget(mandelZoomSlider);
  mandelZoomSlider->setMinimum(1);
  mandelZoomSlider->setMaximum(1000);
  mandelZoomSlider->setTickInterval(1);
  mandelZoomSlider->setTracking(false);
  QObject::connect(mandelZoomSlider, SIGNAL(valueChanged(int)), this, SLOT(setMandelZoom(int)));  
  QObject::connect(mandelZoomSlider, SIGNAL(sliderMoved(int)), this, SLOT(mandelZoomMoved(int)));  
  mandelLayout->addWidget(mandelZoomFrame);
  juliaZoomLayout = new QHBoxLayout;
  juliaZoomFrame = new QWidget;
  juliaZoomFrame->setLayout(juliaZoomLayout);
  QLabel *juliaZoomLabel = new QLabel("Zoom: ");
  juliaZoomLayout->addWidget(juliaZoomLabel);
  juliaZoomSlider = new QSlider(Qt::Horizontal);
  juliaZoomLayout->addWidget(juliaZoomSlider);
  juliaZoomSlider->setMinimum(1);
  juliaZoomSlider->setMaximum(1000);
  juliaZoomSlider->setTickInterval(1);  
  juliaZoomSlider->setTracking(false);
  QObject::connect(juliaZoomSlider, SIGNAL(valueChanged(int)), this, SLOT(setJuliaZoom(int)));
  QObject::connect(juliaZoomSlider, SIGNAL(sliderMoved(int)), this, SLOT(juliaZoomMoved(int)));  
  juliaLayout->addWidget(juliaZoomFrame);
  QSplitter *splitter = new QSplitter(this);
  splitter->addWidget(mandelFrame);
  splitter->addWidget(juliaFrame);
  guiLayout = new QVBoxLayout;
  setLayout(guiLayout);
  guiLayout->addWidget(splitter);
  QGridLayout *parameterLayout = new QGridLayout;
  parameterLayout->setContentsMargins(QMargins(70, 10, 30, 10));
  guiLayout->addLayout(parameterLayout);
  mandelLabel[0] = new QLabel("   Re(z):   min = " + QString::number((long double)data->getXOfs(0) * data->getXScale(0), 'g', 20));
  mandelLabel[1] = new QLabel("   max = " + QString::number((long double)(data->getXOfs(0) + data->getImageWidth()) * data->getXScale(0), 'g', 20));
  mandelLabel[2] = new QLabel("   Im(z):   min = " + QString::number(data->getYOfs(0) * data->getYScale(0), 'g', 20));
  mandelLabel[3] = new QLabel("   max = " + QString::number((long double)(data->getYOfs(0) + data->getImageHeight()) * data->getYScale(0), 'g', 20));
  parameterLayout->addWidget(new QLabel("Mandelbrot Set:"), 1, 1);
  parameterLayout->addWidget(mandelLabel[0], 2, 1);
  parameterLayout->addWidget(mandelLabel[1], 2, 2);
  parameterLayout->addWidget(mandelLabel[2], 3, 1);
  parameterLayout->addWidget(mandelLabel[3], 3, 2);
  juliaLabel[0] = new QLabel("   Re(z):   min = " + QString::number((long double)data->getXOfs(1) * data->getXScale(1), 'g', 20));
  juliaLabel[1] = new QLabel("   max = " + QString::number((long double)(data->getXOfs(1) + data->getImageWidth()) * data->getXScale(1), 'g', 20));
  juliaLabel[2] = new QLabel("   Im(z):   min = " + QString::number(data->getYOfs(1) * data->getYScale(1), 'g', 20));
  juliaLabel[3] = new QLabel("   max = " + QString::number((long double)(data->getYOfs(1) + data->getImageHeight()) * data->getYScale(1), 'g', 20));
  data->getC(c[0], c[1]);
  juliaLabel[4] = new QLabel("   Re(c) = " + QString::number(c[0], 'g', 20));
  juliaLabel[5] = new QLabel("   Im(c) = " + QString::number(c[1], 'g', 20));
  parameterLayout->addWidget(new QLabel("Julia Set:"), 1, 3);
  parameterLayout->addWidget(juliaLabel[0], 2, 3);
  parameterLayout->addWidget(juliaLabel[1], 2, 4);
  parameterLayout->addWidget(juliaLabel[2], 3, 3);
  parameterLayout->addWidget(juliaLabel[3], 3, 4);
  parameterLayout->addWidget(juliaLabel[4], 4, 3);
  parameterLayout->addWidget(juliaLabel[5], 4, 4);
  data->calcTiles(0);
  mandelCalc = (MandelCalc**)malloc(data->getThreadCount() * sizeof(MandelCalc*));  
  for (i = 0; i < data->getThreadCount(); i++) {
    mandelCalc[i] = new MandelCalc(data, i);
    QObject::connect(mandelCalc[i], SIGNAL(finished()), this, SLOT(mandelCalcFinished()));
    mandelCalc[i]->start();
  }  
  data->calcTiles(1);
  juliaCalc = (JuliaCalc**)malloc(data->getThreadCount() * sizeof(JuliaCalc*));
  for (i = 0; i < data->getThreadCount(); i++) {
    juliaCalc[i] = new JuliaCalc(data, i);
    QObject::connect(juliaCalc[i], SIGNAL(finished()), this, SLOT(juliaCalcFinished()));
    juliaCalc[i]->start();
  }  
  hiresCalc = (HiresCalc**)malloc(data->getThreadCount() * sizeof(HiresCalc*));
  data->calcChunks(CHUNK_LEN);
  for (i = 0; i < data->getThreadCount(); i++) {
    hiresCalc[i] = new HiresCalc(data, i);
    QObject::connect(hiresCalc[i], SIGNAL(finished()), this, SLOT(hiresCalcFinished()));
    hiresCalc[i]->start();
  }  
  QObject::connect(juliaView, SIGNAL(imageMoved(QPoint)), this, SLOT(moveJulia(QPoint)));
  QObject::connect(mandelView, SIGNAL(imageMoved(QPoint)), this, SLOT(moveMandel(QPoint)));
  QObject::connect(mandelView, SIGNAL(newJuliaC()), this, SLOT(updateJulia()));
  QObject::connect(colorDialog, SIGNAL(mandelColorChanged()), mandelView, SLOT(changeColor()));
  QObject::connect(colorDialog, SIGNAL(juliaColorChanged()), juliaView, SLOT(changeColor()));
  QObject::connect(iterationDialog, SIGNAL(mandelIterationChanged()), this, SLOT(updateMandelPalette()));
  QObject::connect(iterationDialog, SIGNAL(juliaIterationChanged()), this, SLOT(updateJuliaPalette()));
  QObject::connect(iterationDialog, SIGNAL(normalizeChanged()), this, SLOT(updateNormalization()));
  QObject::connect(hiresDialog, SIGNAL(refreshHires()), this, SLOT(refreshHires()));
  QObject::connect(this, SIGNAL(refreshHiresView()), hiresDialog, SLOT(refreshHiresView()));
}

Gui::~Gui() {

  int i;
  
  data->shutdownFlag = true;
  for (i = 0; i < data->getThreadCount(); i++) {
    mandelCalc[i]->wait();
    juliaCalc[i]->wait();
    hiresCalc[i]->wait();
  }  
}

void Gui::loadPalette(int index, QFile *f) {

  QXmlSimpleReader xmlReader;
  bool ok;

  data->clearPalette(index);  
  QXmlInputSource *source = new QXmlInputSource(f);
  XmlHandler *xmlHandler = new XmlHandler(data, index);
  xmlReader.setContentHandler(xmlHandler);
  ok = xmlReader.parse(source);
  while (ok) {
    ok = xmlReader.parseContinue();
  }  
  f->close();
}

void Gui::savePalette(int index, QFile *f) {

  QXmlStreamWriter xmlWriter(f);
  int i, pos;
  QColor color;

  xmlWriter.setAutoFormatting(true);
  xmlWriter.writeStartDocument();
  xmlWriter.writeStartElement("Palette");
  for (i = 0; i < data->getColorCount(index); i++) {
    xmlWriter.writeStartElement("PalettePoint");
    xmlWriter.writeAttribute("index", QString::number(i));
    color = data->getColor(index, i, pos);
    xmlWriter.writeTextElement("Pos", QString::number(pos));
    xmlWriter.writeTextElement("Red", QString::number(color.red()));
    xmlWriter.writeTextElement("Green", QString::number(color.green()));
    xmlWriter.writeTextElement("Blue", QString::number(color.blue()));
    xmlWriter.writeEndElement();
  }  
  xmlWriter.writeEndElement();
  xmlWriter.writeEndDocument();
  f->close();
}

void Gui::mandelZoomMoved(int value) {

  data->showZoomRect[0] = true;
  data->tmpZoom[0] = 1.0 / exp((long double)value/60.0);
  mandelView->repaint();
}

void Gui::setMandelZoom(int value) {

  if (!data->getImageBusy(0)) {
    data->setMandelZoom(1.0 / exp((long double)value/60.0));
    updateMandel();
  }  
  data->showZoomRect[0] = false;
}

void Gui::juliaZoomMoved(int value) {

  data->showZoomRect[1] = true;
  data->tmpZoom[1] = 1.0 / exp((long double)value/60.0);
  juliaView->repaint();
}

void Gui::setJuliaZoom(int value) {

  if (!data->getImageBusy(1)) {
    data->setJuliaZoom(1.0 / exp((long double)value/60.0));
    updateJulia();
  }  
  data->showZoomRect[1] = false;
} 

void Gui::moveMandel(QPoint dp) {

  int i;
  
  if (!data->getImageBusy(0)) {
     data->calcTiles(0, dp.x(), dp.y()); 
     for (i = 0; i < data->getThreadCount(); i++) {
       mandelCalc[i]->start();
     } 
  }
}

void Gui::updateMandel() {

  int i;
  
  if (!data->getImageBusy(0)) {
     data->calcTiles(0); 
     for (i = 0; i < data->getThreadCount(); i++) {
       mandelCalc[i]->start();
     } 
  }
}

void Gui::updateMandelPalette() {

  colorDialog->updateMandelPalette();
  updateMandel();
}

void Gui::mandelCalcFinished() {

  data->mandelMutex.lock();
  data->threadReady(0);
  if (data->getThreadsReady(0) == data->getThreadCount()) {
    data->imageReady(0);
    updateImage(0);
  }    
  data->mandelMutex.unlock();
}

void Gui::moveJulia(QPoint dp) {

  int i;
  
  if (!data->getImageBusy(1)) {
     data->calcTiles(1, dp.x(), dp.y()); 
     for (i = 0; i < data->getThreadCount(); i++) {
       juliaCalc[i]->start();
     } 
  }
}

void Gui::updateJulia() {

  int i;
  
  if (!data->getImageBusy(1)) {
    data->calcTiles(1);
    for (i = 0; i < data->getThreadCount(); i++) {
      juliaCalc[i]->start();
    }
  }
}

void Gui::updateJuliaPalette() {

  colorDialog->updateJuliaPalette();
  updateJulia();
}

void Gui::juliaCalcFinished() {

  data->juliaMutex.lock();
  data->threadReady(1);
  if (data->getThreadsReady(1) == data->getThreadCount()) {
    data->imageReady(1);
    updateImage(1);
  }  
  data->juliaMutex.unlock();    
}

void Gui::updateImage(int index) {

  long double c[2];

  switch (index) {
  case 0:
    data->calcMax(0);
    mandelView->repaint();
    mandelLabel[0]->setText("   Re(z):   min = " + QString::number((long double)data->getXOfs(0) * data->getXScale(0), 'g', 20));
    mandelLabel[1]->setText("   max = " + QString::number((long double)(data->getXOfs(0) + data->getImageWidth()) * data->getXScale(0), 'g', 20));
    mandelLabel[2]->setText("   Im(z):   min = " + QString::number(data->getYOfs(0) * data->getYScale(0), 'g', 20));
    mandelLabel[3]->setText("   max = " + QString::number((long double)(data->getYOfs(0) + data->getImageHeight()) * data->getYScale(0), 'g', 20));
    break;
  case 1:
    data->calcMax(1);
    juliaView->repaint();
    juliaLabel[0]->setText("   Re(z):   min = " + QString::number((long double)data->getXOfs(1) * data->getXScale(1), 'g', 20));
    juliaLabel[1]->setText("   max = " + QString::number((long double)(data->getXOfs(1) + data->getImageWidth()) * data->getXScale(1), 'g', 20));
    juliaLabel[2]->setText("   Im(z):   min = " + QString::number(data->getYOfs(1) * data->getYScale(1), 'g', 20));
    juliaLabel[3]->setText("   max = " + QString::number((long double)(data->getYOfs(1) + data->getImageHeight()) * data->getYScale(1), 'g', 20));
    data->getC(c[0], c[1]); 
    juliaLabel[4]->setText("   Re(c) = " + QString::number(c[0], 'g', 20));
    juliaLabel[5]->setText("   Im(c) = " + QString::number(c[1], 'g', 20));
    break;
  case 2:
    emit refreshHiresView();
    break; 
  }  
}

void Gui::updateNormalization() {

  mandelView->changeColor();
  juliaView->changeColor();  
}

void Gui::showColorDialog() {

  colorDialog->show();
  colorDialog->raise();
}

void Gui::showIterationDialog() {

  iterationDialog->show();
  iterationDialog->raise();
}

void Gui::showHiRes() {
  
  hiresDialog->show();
  hiresDialog->raise();
}

void Gui::refreshHires() {

  int i;
  
  if (data->getChunkCount()<=0) {
    data->calcChunks(CHUNK_LEN);
    for (i = 0; i < data->getThreadCount(); i++) {
      hiresCalc[i]->start();
    }
  }
}

void Gui::hiresCalcFinished() {

  data->hiresMutex.lock();
  data->threadReady(2);
  if (data->getThreadsReady(2) == data->getThreadCount()) {
    data->imageReady(2);
    updateImage(2);
  }    
  data->hiresMutex.unlock();
}

void Gui::loadMandelPalette() {

  QString fn, qs;
  QFile *f;
            
  QFileDialog *dialog = new QFileDialog;
  fn = dialog->getOpenFileName();
  if (!fn.isEmpty()) {
    f = new QFile(fn);
    if (!f->open(QIODevice::ReadOnly)) {
      QMessageBox::warning(this, "QMandelJul", "Could not open file.");
    } else {
      loadPalette(0, f);
      updateMandelPalette();
    }
  }  
}

void Gui::saveMandelPalette() {

  QString fn, qs;
  QFile *f;
            
  QFileDialog *dialog = new QFileDialog;
  fn = dialog->getSaveFileName();
  if (!fn.isEmpty()) {
    f = new QFile(fn);
    if (!f->open(QIODevice::WriteOnly)) {
      QMessageBox::warning(this, "QMandelJul", "Could not save file.");
    } else {
      savePalette(0, f);
    }
  }  
}

void Gui::loadJuliaPalette() {

  QString fn, qs;
  QFile *f;
            
  QFileDialog *dialog = new QFileDialog;
  fn = dialog->getOpenFileName();
  if (!fn.isEmpty()) {
    f = new QFile(fn);
    if (!f->open(QIODevice::ReadOnly)) {
      QMessageBox::warning(this, "QMandelJul", "Could not open file.");
    } else {
      loadPalette(1, f);
      updateJuliaPalette();
    }
  }  
}

void Gui::saveJuliaPalette() {

  QString fn, qs;
  QFile *f;
            
  QFileDialog *dialog = new QFileDialog;
  fn = dialog->getSaveFileName();
  if (!fn.isEmpty()) {
    f = new QFile(fn);
    if (!f->open(QIODevice::WriteOnly)) {
      QMessageBox::warning(this, "QMandelJul", "Could not save file.");
    } else {
      savePalette(1, f);
    }
  }  
}

void Gui::saveMandel() {

  QString fn, qs;
  QFile *f;
            
  QFileDialog *dialog = new QFileDialog;
  fn = dialog->getSaveFileName();
  if (!fn.isEmpty()) {
    f = new QFile(fn);
    if (!f->open(QIODevice::WriteOnly)) {
      QMessageBox::warning(this, "QMandelJul", "Could not save file.");
    } else {
      data->mandelImage->save(fn);
    }
  }  
}

void Gui::saveJulia() {

  QString fn, qs;
  QFile *f;
            
  QFileDialog *dialog = new QFileDialog;
  fn = dialog->getSaveFileName();
  if (!fn.isEmpty()) {
    f = new QFile(fn);
    if (!f->open(QIODevice::WriteOnly)) {
      QMessageBox::warning(this, "QMandelJul", "Could not save file.");
    } else {
      data->juliaImage->save(fn);
    }
  }  
}

void Gui::saveHiRes() {

  QString fn, qs;
  QFile *f;
            
  QFileDialog *dialog = new QFileDialog;
  fn = dialog->getSaveFileName();
  if (!fn.isEmpty()) {
    f = new QFile(fn);
    if (!f->open(QIODevice::WriteOnly)) {
      QMessageBox::warning(this, "QMandelJul", "Could not save file.");
    } else {
      data->hiresImage->save(fn);
    }
  }  
}

void Gui::loadState() {

  QString fn, qs;
  QFile *f;
  QXmlSimpleReader xmlReader;
  bool ok;
  int i;

  QFileDialog *dialog = new QFileDialog;
  fn = dialog->getOpenFileName();
  if (!fn.isEmpty()) {
    f = new QFile(fn);
    if (!f->open(QIODevice::ReadOnly)) {
      QMessageBox::warning(this, "QMandelJul", "Could not open file.");
    } else {
      data->clearPalette(0);  
      data->clearPalette(1); 
      QXmlInputSource *source = new QXmlInputSource(f);
      XmlHandler *xmlHandler = new XmlHandler(data, 0);
      xmlReader.setContentHandler(xmlHandler);
      ok = xmlReader.parse(source);
      while (ok) {
        ok = xmlReader.parseContinue();
      }  
      f->close();
      updateMandelPalette();
      updateJuliaPalette();
    }  
  }  
  mandelZoomSlider->setValue((int)(60.0 * log(4.0 / (data->getXScale(0) * (long double)data->getImageWidth()))));
  juliaZoomSlider->setValue((int)(60.0 * log(8.0 / (data->getXScale(1) * (long double)data->getImageWidth()))));
  for (i = 0; i < 2; i++) {
    iterationDialog->setMaxIterations(i, data->getMaxIterations(i));
  }  
  for (i = 0; i < 2; i++) {
    iterationDialog->setMaxAbsoluteValue(i, data->getMaxAbsoluteValue(i));
  }  
}

void Gui::saveState() {

  int i, pos;
  QColor color;
  QString fn;
  QFile *f;
  long double c_re, c_im;
            
  QFileDialog *dialog = new QFileDialog;
  fn = dialog->getSaveFileName();
  if (!fn.isEmpty()) {
    f = new QFile(fn);
    if (!f->open(QIODevice::WriteOnly)) {
      QMessageBox::warning(this, "QMandelJul", "Could not save file.");
    } else {
      QXmlStreamWriter xmlWriter(f);
      xmlWriter.setAutoFormatting(true);
      xmlWriter.writeStartDocument();
      xmlWriter.writeStartElement("State");
      xmlWriter.writeStartElement("MandelPalette");
      for (i = 0; i < data->getColorCount(0); i++) {
        xmlWriter.writeStartElement("PalettePoint");
        xmlWriter.writeAttribute("index", QString::number(i));
        color = data->getColor(0, i, pos);
        xmlWriter.writeTextElement("Pos", QString::number(pos));
        xmlWriter.writeTextElement("Red", QString::number(color.red()));
        xmlWriter.writeTextElement("Green", QString::number(color.green()));
        xmlWriter.writeTextElement("Blue", QString::number(color.blue()));
        xmlWriter.writeEndElement();
      }  
      xmlWriter.writeEndElement();
      xmlWriter.writeStartElement("JuliaPalette");
      for (i = 0; i < data->getColorCount(1); i++) {
        xmlWriter.writeStartElement("PalettePoint");
        xmlWriter.writeAttribute("index", QString::number(i));
        color = data->getColor(1, i, pos);
        xmlWriter.writeTextElement("Pos", QString::number(pos));
        xmlWriter.writeTextElement("Red", QString::number(color.red()));
        xmlWriter.writeTextElement("Green", QString::number(color.green()));
        xmlWriter.writeTextElement("Blue", QString::number(color.blue()));
        xmlWriter.writeEndElement();
      }  
      xmlWriter.writeEndElement();
      xmlWriter.writeStartElement("Parameters");
      xmlWriter.writeTextElement("MandelZoom", QString::number(data->getMandelZoom(), 'g', 16));
      xmlWriter.writeTextElement("JuliaZoom", QString::number(data->getJuliaZoom(), 'g', 16));
      data->getC(c_re, c_im);
      xmlWriter.writeTextElement("C_Re", QString::number(c_re, 'g', 16));
      xmlWriter.writeTextElement("C_Im", QString::number(c_im, 'g', 16));
      xmlWriter.writeTextElement("MandelXOfs", QString::number(data->getXOfs(0)));
      xmlWriter.writeTextElement("MandelYOfs", QString::number(data->getYOfs(0)));
      xmlWriter.writeTextElement("JuliaXOfs", QString::number(data->getXOfs(1)));
      xmlWriter.writeTextElement("JuliaYOfs", QString::number(data->getYOfs(1)));
      xmlWriter.writeTextElement("MandelIterations", QString::number(data->getMaxIterations(0)));
      xmlWriter.writeTextElement("JuliaIterations", QString::number(data->getMaxIterations(1)));
      xmlWriter.writeTextElement("MandelMaxAbsoluteValue", QString::number(data->getMaxAbsoluteValue(0)));
      xmlWriter.writeTextElement("JuliaMaxAbsoluteValue", QString::number(data->getMaxAbsoluteValue(1)));
      xmlWriter.writeEndElement();
      xmlWriter.writeEndElement();
      xmlWriter.writeEndDocument();
      f->close();
    }
  }    
}
