//
//  PersonManager.swift
//  SwiftTest
//
//  Created by Kai Brüning on 14.7.14.
//  Copyright (c) 2014 Hippocampus Software GmbH. All rights reserved.
//

import Foundation

class PersonManager {
    class var sharedInstance: PersonManager {
        if (sSharedInstance == nil) {
            sSharedInstance = PersonManager()
        }
        return sSharedInstance!
    }

    // Nachschlagen nach Namen
    func personWithName (name:String) -> Person? {
        return self.personByName[name]
    }
    
    // Nachschlagen nach Namen und Geburtstag
    func personWithName (name:String, birthday:NSDate) -> Person? {
        // Die Schlüsselstruktur NameAndBirthday wird temporär auf dem Stack angelegt, entsprechend effizient ist das Nachschlagen.
        return self.personByNameAndBirthday[NameAndBirthday (name: name, birthday: birthday)]
    }


    func addPerson (person:Person) {
        self.personByName[person.name] = person
        // Wenn kein Geburtstag gesetzt ist, wird kein Eintrag in die Tabelle nach Namen und Geburtstag gemacht.
        if let birthday = person.birthday {
            self.personByNameAndBirthday[NameAndBirthday (name: person.name, birthday: birthday)] = person
        }
    }
    func removePerson (person:Person) {
        self.personByName[person.name] = nil
        if let birthday = person.birthday {
            self.personByNameAndBirthday[NameAndBirthday (name: person.name, birthday: birthday)] = nil
        }
    }

    // Die beiden Dictionaries, die als Nachschlagetabellen benutzt werden. Als Implementierungsdetail sind sie privat; jeglicher Zugriff geschieht über das öffentliche API.
    
    private var personByName = [String:Person]()

    private struct NameAndBirthday : Hashable {
        // Als Schlüssel sollten die Properties von NameAndBirthday tunlichst unveränderlich sein, daher mit let deklariert.
        let name:     String
        // 'birthday' ist nicht optional: Einträge in diese Tabelle werden nur gemacht, wenn ein Geburtstag bekannt ist.
        let birthday: NSDate

        // Property aus dem Protokoll "Hashable".
        // Alle berechneten Properties müssen als "var" deklariert werden, auch wenn der Wert tatsächlich konstant ist.
        var hashValue: Int {
            var hash = name.hash
            hash = 37 * hash + birthday.hash
            return hash
        }
    }
    private var personByNameAndBirthday = [NameAndBirthday:Person]()
}

// "Hashable" basiert auf "Equatable", welches diese Funktion verlangt.
private func == (lhs: PersonManager.NameAndBirthday, rhs: PersonManager.NameAndBirthday) -> Bool {
    return lhs.name == rhs.name && lhs.birthday.isEqualToDate(rhs.birthday)
}

// Leider gibt es keine gespeicherte Klassen-Properties, deshalb muss die gemeinsame Instanz in einer globalen Variablen hinterlegt werden.
private var sSharedInstance: PersonManager?

