//
//  ViewController.m
//  Gaston
//
//  Created by Richard Kurz on 18/09/14.
//  Copyright (c) 2014 Richard Kurz (Public Domain). No rights reserved.
//


@import Metal;

#import "ViewController.h"

#import <sys/types.h>
#import <sys/sysctl.h>


#define W               512
#define H               512
#define MAX_ITERATIONS  1024

#define FRACTAL_LEFT    0.5226f
#define FRACTAL_TOP     -0.0614f
#define FRACTAL_WIDTH   0.012f
#define FRACTAL_HEIGHT  0.012f
#define FRACTAL_A       -0.7453f
#define FRACTAL_B       -0.11301f


void fractal_scalar(size_t index, int maxDepth, int pixWidth, int pixHeight,
                    float left, float top, float width, float height, float a, float b,
                    float* output);

void colorit_scalar(size_t index, int maxDepth, float* input, int* output);


typedef NS_ENUM(NSInteger, GastonJobType)
{
  jobTypeSingleThread = 0,
  jobTypeGCD = 1,
  jobTypeGPU = 2
};


@implementation ViewController
{
  BOOL metalInitialized;
  
  id<MTLDevice> device;
  id<MTLCommandQueue> commandQueue;
  id<MTLLibrary> defaultLibrary;
  id<MTLBuffer> parameter;
  id<MTLComputePipelineState> fractalPipeline, coloritPipeline;
  
  id<MTLBuffer> bufferFractal, bufferColor;
}


#pragma mark - Helper -


- (UIImage*) imageFromMemory: (void*) imageMemory width: (int) w height: (int) h
{
  CGColorSpaceRef colorSpace = CGColorSpaceCreateDeviceRGB();
  CGContextRef bitmapContext = CGBitmapContextCreate(imageMemory, w, h, 8, h * 4, colorSpace,
                                                     kCGBitmapByteOrderDefault + kCGImageAlphaNoneSkipLast);
  CGImageRef cgImage = CGBitmapContextCreateImage(bitmapContext);
  
  UIImage* image = [UIImage imageWithCGImage: cgImage];
  
  CGImageRelease(cgImage);
  CGContextRelease(bitmapContext);
  CFRelease(colorSpace);
  
  return image;
}


- (NSString*) hardwareName
{
  size_t size;
  sysctlbyname("hw.machine", NULL, &size, NULL, 0);
  char name[size + 1];
  sysctlbyname("hw.machine", name, &size, NULL, 0);
  return [NSString stringWithUTF8String: name];
}


#pragma mark - Metal Computing -


- (BOOL) setupMetalComputing
{
  device = MTLCreateSystemDefaultDevice();
  if(!device) return NO;
  
  commandQueue = [device newCommandQueue];
  if(!commandQueue) return NO;
  
  defaultLibrary = [device newDefaultLibrary];
  if(!defaultLibrary) return NO;
  
  parameter = [device newBufferWithLength: 16 * sizeof(float) options: MTLResourceOptionCPUCacheModeDefault];
  if(!parameter) return NO;
  
  id<MTLFunction> fractalFunction = [defaultLibrary newFunctionWithName: @"fractal"];
  if(!fractalFunction) return NO;
  
  NSError* pError = nil;
  fractalPipeline = [device newComputePipelineStateWithFunction: fractalFunction error: &pError];
  if(pError) NSLog(@"ERROR: Creating fractal kernel failed: %@", pError);
  if(!fractalPipeline) return NO;
  
  id<MTLFunction> coloritFunction = [defaultLibrary newFunctionWithName: @"colorit"];
  if(!coloritFunction) return NO;
  
  pError = nil;
  coloritPipeline = [device newComputePipelineStateWithFunction: coloritFunction error: &pError];
  if(pError) NSLog(@"ERROR: Creating colorit kernel failed: %@", pError);
  if(!coloritPipeline) return NO;
  
  return YES;
}


- (dispatch_block_t) renderBlockForJob: (GastonJobType) jobType width: (int) w height: (int) h
{
  dispatch_block_t renderBlock = NULL;
  size_t n = w * h;
  
  if (jobType == jobTypeGPU)
  {
    renderBlock = ^{
      
      float* param = parameter.contents;
      param[0] = MAX_ITERATIONS;
      param[1] = w;
      param[2] = h;
      param[3] = FRACTAL_LEFT;
      param[4] = FRACTAL_TOP;
      param[5] = FRACTAL_WIDTH;
      param[6] = FRACTAL_HEIGHT;
      param[7] = FRACTAL_A;
      param[8] = FRACTAL_B;
      
      id<MTLCommandBuffer> commandBuffer = [commandQueue commandBuffer];
      
      //-----
      
      NSUInteger gs = fractalPipeline.threadExecutionWidth;
      MTLSize threadGroupSize = MTLSizeMake(gs, 1, 1);
      MTLSize threadGroupCount = MTLSizeMake(n / gs, 1, 1);
      
      id<MTLComputeCommandEncoder> fractalEncoder = [commandBuffer computeCommandEncoder];
      [fractalEncoder setComputePipelineState: fractalPipeline];
      
      [fractalEncoder setBuffer: parameter offset: 0 atIndex: 0];
      [fractalEncoder setBuffer: bufferFractal offset: 0 atIndex: 1];
      
      [fractalEncoder dispatchThreadgroups: threadGroupCount threadsPerThreadgroup: threadGroupSize];
      [fractalEncoder endEncoding];
      
      //-----
      
      gs = coloritPipeline.threadExecutionWidth;
      threadGroupSize = MTLSizeMake(gs, 1, 1);
      threadGroupCount = MTLSizeMake(n / gs, 1, 1);
      
      id<MTLComputeCommandEncoder> coloritEncoder = [commandBuffer computeCommandEncoder];
      [coloritEncoder setComputePipelineState: coloritPipeline];
      
      [coloritEncoder setBuffer: parameter offset: 0 atIndex: 0];
      [coloritEncoder setBuffer: bufferFractal offset: 0 atIndex: 1];
      [coloritEncoder setBuffer: bufferColor offset: 0 atIndex: 2];
      
      [coloritEncoder dispatchThreadgroups: threadGroupCount threadsPerThreadgroup: threadGroupSize];
      [coloritEncoder endEncoding];
      
      //-----
      
      [commandBuffer commit];
      [commandBuffer waitUntilCompleted];
    };
  }
  else if (jobType == jobTypeGCD)
  {
    renderBlock = ^{
      
      float* fractalBuffer = bufferFractal.contents;
      int* colorBuffer = bufferColor.contents;
      
      dispatch_apply(n, dispatch_get_global_queue(0, 0), ^(size_t index) {
        fractal_scalar(index, MAX_ITERATIONS, w, h,
                       FRACTAL_LEFT, FRACTAL_TOP, FRACTAL_WIDTH, FRACTAL_HEIGHT, FRACTAL_A, FRACTAL_B,
                       fractalBuffer);
      });

      dispatch_apply(n, dispatch_get_global_queue(0, 0), ^(size_t index) {
        colorit_scalar(index, MAX_ITERATIONS, fractalBuffer, colorBuffer);
      });
    };
  }
  else
  {
    renderBlock = ^{
      
      float* fractalBuffer = bufferFractal.contents;
      int* colorBuffer = bufferColor.contents;
      
      for (int index = 0; index < n; ++index)
        fractal_scalar(index, MAX_ITERATIONS, w, h,
                       FRACTAL_LEFT, FRACTAL_TOP, FRACTAL_WIDTH, FRACTAL_HEIGHT, FRACTAL_A, FRACTAL_B,
                       fractalBuffer);
      
      for (int index = 0; index < n; ++index)
        colorit_scalar(index, MAX_ITERATIONS, fractalBuffer, colorBuffer);
    };
  }
  
  return renderBlock;
}


#pragma mark - Benchmark -


- (void) renderJob: (GastonJobType) jobType withOversampling: (int) aa
{
  NSLog(@"Start...");

  [[UIApplication sharedApplication] beginIgnoringInteractionEvents];
  
  self.imageView.image = nil;
  
  int w = W * aa;
  int h = H * aa;
  
  bufferFractal = [device newBufferWithLength: w * h * sizeof(float) options: MTLResourceOptionCPUCacheModeDefault];
  bufferColor = [device newBufferWithLength: w * h * sizeof(int) options: MTLResourceOptionCPUCacheModeDefault];
  
  dispatch_block_t renderBlock = [self renderBlockForJob: jobType width: w height: h];

  dispatch_after(dispatch_time(DISPATCH_TIME_NOW, NSEC_PER_SEC / 4), dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0), ^{
    
    double fastestRun = HUGE_VAL;
    double timeElapsed = 0.0;
    double begin = CACurrentMediaTime();
    
    do
    {
      for (int iterations = 0; iterations < 5; iterations += 1)
      {
        double start = CACurrentMediaTime();
        
        renderBlock();
        
        double duration = CACurrentMediaTime() - start;
        if (duration < fastestRun) fastestRun = duration;
      }
      
      timeElapsed = CACurrentMediaTime() - begin;
    }
    while (timeElapsed < 4.0); // Hier kann die Mindestdauer in Sekunden eingestellt werden.
    
    dispatch_async(dispatch_get_main_queue(), ^{
      
      [[UIApplication sharedApplication] endIgnoringInteractionEvents];

      self.imageView.image = [self imageFromMemory: bufferColor.contents width: w height: h];
      
      NSString* jobDescription = @"CPU (Single Thread)";
      if (jobType == jobTypeGCD) jobDescription = @"CPU (Grand Central Dispatch)";
      else if (jobType == jobTypeGPU) jobDescription = @"GPU (Metal)";
      
      NSString* deviceDescription = [NSString stringWithFormat: @"iOS %@ on %@",
                                     [UIDevice currentDevice].systemVersion,
                                     self.hardwareName];
      
      double mps = (double)(w * h) / fastestRun / 1000000.0;
      NSString* result = [NSString stringWithFormat: @"%@\n%@\n%d x %d pixel in %.2f ms\n%.2f megapixel per second",
                          deviceDescription, jobDescription, w, h, fastestRun * 1000.0, mps];
      
      NSLog(@"Result:\n%@", result);
      
      UIAlertController* alert = [UIAlertController alertControllerWithTitle: @"Result:" message: result preferredStyle: UIAlertControllerStyleAlert];
    
      [alert addAction: [UIAlertAction actionWithTitle: @"Copy" style: UIAlertActionStyleDefault handler: ^(UIAlertAction* action) {
        [[UIPasteboard generalPasteboard] setString: result];
      }]];
      
      [alert addAction: [UIAlertAction actionWithTitle: @"OK" style: UIAlertActionStyleDefault handler: ^(UIAlertAction* action) {
      }]];
    
      [self presentViewController: alert animated: YES completion: nil];
    });
  });
}


#pragma mark - UI -


- (void) showMenu
{
  UIAlertController* alert = [UIAlertController alertControllerWithTitle: @"Gaston Benchmark" message: @"Run on:" preferredStyle: UIAlertControllerStyleAlert];
  
  if (metalInitialized)
  {
    [alert addAction: [UIAlertAction actionWithTitle: @"GPU (Metal)" style: UIAlertActionStyleDefault handler: ^(UIAlertAction* action) {
      [self renderJob: jobTypeGPU withOversampling: 2];
    }]];
  }
  
  [alert addAction: [UIAlertAction actionWithTitle: @"CPU (Grand Central Dispatch)" style: UIAlertActionStyleDefault handler: ^(UIAlertAction* action) {
    [self renderJob: jobTypeGCD withOversampling: 2];
  }]];
  
  [alert addAction: [UIAlertAction actionWithTitle: @"CPU (Single Thread)" style: UIAlertActionStyleDefault handler: ^(UIAlertAction* action) {
    [self renderJob: jobTypeSingleThread withOversampling: 2];
  }]];
  
  [alert addAction: [UIAlertAction actionWithTitle: @"Cancel" style: UIAlertActionStyleCancel handler: ^(UIAlertAction* action) {
  }]];
  
  [self presentViewController: alert animated: YES completion: nil];
}


- (void) viewDidLoad
{
  [super viewDidLoad];

  metalInitialized = [self setupMetalComputing];
}


- (void) viewDidAppear: (BOOL) animated
{
  [self showMenu];
}


- (IBAction) screenTouched: (id) sender
{
  [self showMenu];
}


@end
