//
//  VideoMessageTableViewCell.swift
//  HeiseMultipeerConnectivity
//
//  Created by Gero Gerber on 24.02.19.
//  Copyright © 2019 Gero Gerber. All rights reserved.
//

import AVFoundation
import UIKit

class VideoMessageTableViewCell: UITableViewCell, MessageTableViewCellProtocol {

    @IBOutlet var messageSender: UILabel!
    @IBOutlet var messageDate: UILabel!
    @IBOutlet var videoPlayerView: UIView!
    @IBOutlet var bgView: UIView!
    
    var delegate: MessageTableViewCellDelegate?
    
    private let dateFormatter = DateFormatter()
    private var videoPlayer: AVPlayer?
    private var videoPlayerLayer: AVPlayerLayer?
    private var videoURL: URL?
    
    override func awakeFromNib() {
        super.awakeFromNib()
        
        bgView.layer.cornerRadius = 10
        dateFormatter.dateFormat = "HH:mm:ss"
        
        let tapGesture = UITapGestureRecognizer(target: self, action: #selector(videoViewTapped))
        videoPlayerView.addGestureRecognizer(tapGesture)
    }

    func setChatMessage(message: ChatMessage) {

        videoURL = message.videoURL
        messageSender.text = message.sentFromLocalDevice ? "You" : message.sender!
        messageDate.text = dateFormatter.string(from: message.date!)
        
        if let layer = videoPlayerLayer {
            layer.removeFromSuperlayer()
            videoPlayerLayer = nil
        }
        
        if let videoURL = videoURL {
            videoPlayer = AVPlayer(url: videoURL)
            videoPlayerLayer = AVPlayerLayer(player: videoPlayer)
            videoPlayerLayer?.videoGravity = .resizeAspect
            videoPlayerLayer?.needsDisplayOnBoundsChange = true
            videoPlayerLayer?.frame = videoPlayerView.bounds
            videoPlayerView.layer.addSublayer(videoPlayerLayer!)
            
            if var repeatCount = message.videoRepeatCount {

                if repeatCount > 0 {
                    startVideo()
                    
                    NotificationCenter.default.addObserver(forName: .AVPlayerItemDidPlayToEndTime, object: videoPlayer?.currentItem, queue: .main) { [weak self] _ in
                        repeatCount -= 1
                        
                        if repeatCount > 0 {
                            self?.startVideo()
                        }
                    }
                }
            }
        }
    }
    
    @objc private func videoViewTapped(_ sender: UITapGestureRecognizer) {
        if let videoURL = videoURL {
            let activityController = UIActivityViewController(activityItems: [videoURL], applicationActivities: nil)
            activityController.popoverPresentationController?.sourceView = bgView
            delegate?.presentViewController(vc: activityController)
        }
    }
    
    @IBAction private func btnPlayTouchUpInside(_ sender: Any) {
        startVideo()
    }
    
    private func startVideo() {
        if let videoPlayer = videoPlayer {
            videoPlayer.seek(to: CMTime.zero)
            videoPlayer.play()
        }
    }
}
