//
//  ChatMessage.swift
//  HeiseMultipeerConnectivity
//
//  Created by Gero Gerber on 19.02.19.
//  Copyright © 2019 Gero Gerber. All rights reserved.
//

import AVFoundation
import Foundation
import UIKit

class ChatMessage: NSObject, NSCoding {
    enum MessageType: Int {
        case text
        case image
        case video
    }
    
    lazy var text: String? = {
        if type == .text {
            let text = String(data: data, encoding: .utf8)
            data = nil
            return text
        }
            
        return nil
    }()
    
    lazy var photo: UIImage? = {
        if type == .image {
            let image = UIImage(data: data)
            data = nil
            return image
        }
        
        return nil
    }()
    
    lazy var videoURL: URL? = {
        if type == .video {
            let directory = NSTemporaryDirectory()
            let fileName = NSUUID().uuidString + "." + videoExtension!
            if let fullURL = NSURL.fileURL(withPathComponents: [directory, fileName]) {
                do {
                    try data.write(to: fullURL)
                    data = nil
                    return fullURL
                } catch {
                    print("Error writing data to disk: '\(fullURL)' - Error: \(error.localizedDescription)")
                }
            }
        }
        
        return nil
    }()
    
    var sender: String!
    var date: Date!
    var type: MessageType!
    var data: Data!
    var videoExtension: String?
    var videoRepeatCount: Int?
    var sentFromLocalDevice = false
    
    required convenience init(coder decoder: NSCoder) {
        self.init()
        sender = decoder.decodeObject(forKey: "sender") as? String
        date = decoder.decodeObject(forKey: "date") as? Date
        type = MessageType(rawValue: decoder.decodeInteger(forKey: "type"))
        data = decoder.decodeObject(forKey: "data") as? Data
        videoExtension = decoder.decodeObject(forKey: "videoExtension") as? String
        videoRepeatCount = decoder.decodeObject(forKey: "videoRepeatCount") as? Int
    }
    
    convenience init(messageSender: String, messageDate: Date, messageType: MessageType, messageData: Data, pathExtension: String? = nil, repeatCount: Int? = nil) {
        self.init()
        sentFromLocalDevice = true
        sender = messageSender
        date = messageDate
        type = messageType
        data = messageData
        videoExtension = pathExtension
        videoRepeatCount = repeatCount
    }
    
    func encode(with aCoder: NSCoder) {
        aCoder.encode(sender, forKey: "sender")
        aCoder.encode(date, forKey: "date")
        aCoder.encode(type.rawValue, forKey: "type")
        aCoder.encode(data, forKey: "data")
        aCoder.encode(videoExtension, forKey: "videoExtension")
        aCoder.encode(videoRepeatCount, forKey: "videoRepeatCount")
    }
}
