//
//  MultipeerConnectivityProvider.swift
//  HeiseMultipeerConnectivity
//
//  Created by Gero Gerber on 19.02.19.
//  Copyright © 2019 Gero Gerber. All rights reserved.
//

import Foundation
import MultipeerConnectivity
import UIKit

class MultipeerConnectivityProvider {
    var localPeer: LocalPeer {
        get {
            return localPeerId
        }
    }
    
    var remotePeers: Set<MCPeerID> {
        get {
            return serviceBrowser.peers
        }
    }
    
    var sessions: Set<MCSession> {
        get {
            return sessionContainer.sessions
        }
    }
    
    var peersDelegate: PeersDelegate? {
        get {
            return serviceBrowser.delegate
        }
        
        set {
            serviceBrowser.delegate = newValue
        }
    }
    
    var sessionsDelegate: SessionsDelegate? {
        get {
            return sessionContainer.delegate
        }
        
        set {
            sessionContainer.delegate = newValue
        }
    }
    
    var chatsDelegate: ChatHistoryDelegate? {
        get {
            return chatHistory.delegate
        }
        
        set {
            chatHistory.delegate = newValue
        }
    }
    
    private let serviceType = "heise-txtchat"
    private let localPeerId: LocalPeer
    private let serviceAdvertiser: ServiceAdvertiser
    private let serviceBrowser: ServiceBrowser
    private let sessionContainer: SessionContainer
    private let chatHistory: ChatHistory
    
    init() {
        localPeerId = LocalPeer()
        chatHistory = ChatHistory()
        sessionContainer = SessionContainer(localPeer: localPeerId, chatHistory: chatHistory)
        serviceAdvertiser = ServiceAdvertiser(localPeer: localPeerId, serviceType: serviceType, sessionContainer: sessionContainer)
        serviceBrowser = ServiceBrowser(localPeer: localPeerId, serviceType: serviceType)
    }
    
    func enterSession(with peerId: MCPeerID) -> MCSession {
        if let session = sessionContainer.findSession(with: peerId) {
            return session
        } else {
            let session = sessionContainer.createSession()
            serviceBrowser.invite(peer: peerId, session: session)
            return session
        }
    }
    
    func invite(peer: MCPeerID, to session: MCSession) {
        serviceBrowser.invite(peer: peer, session: session)
    }
    
    func sendTextMessage(to session: MCSession, with message: String) {
        if let message = message.data(using: .utf8) {
            let chatMessage = ChatMessage(messageSender: localPeer.peerId.displayName, messageDate: Date(), messageType: .text, messageData: message)
            sendMessage(to: session, with: chatMessage)
        }
    }
    
    func sendImageMessage(to session: MCSession, with image: UIImage) {
        if let imageData = image.jpegData(compressionQuality: 0.5) {
            let chatMessage = ChatMessage(messageSender: localPeer.peerId.displayName, messageDate: Date(), messageType: .image, messageData: imageData)
            sendMessage(to: session, with: chatMessage)
        }
    }
    
    func sendVideoMessage(to session: MCSession, with videoURL: URL) {
        do {
            let videoData = try Data(contentsOf: videoURL)
            sendVideoMessage(to: session, with: videoData, type: videoURL.pathExtension)
        } catch {
            print("Failed to read video data form URL '\(videoURL)' - Error: \(error.localizedDescription)")
        }
    }
    
    func sendVideoMessage(to session: MCSession, with videoData: Data, type pathExtension: String, repeat repeatCount: Int? = nil) {
        let chatMessage = ChatMessage(messageSender: localPeer.peerId.displayName, messageDate: Date(), messageType: .video, messageData: videoData, pathExtension: pathExtension, repeatCount: repeatCount)
        sendMessage(to: session, with: chatMessage)
    }
        
    func sendMessage(to session: MCSession, with message: ChatMessage) {
        do {
            let messageData = try NSKeyedArchiver.archivedData(withRootObject: message, requiringSecureCoding: false)
            try session.send(messageData, toPeers: session.connectedPeers, with: .reliable)
            chatHistory.addChatMessage(session: session, message: message)
        } catch {
            print("Failed to send message: \(error.localizedDescription)")
        }
    }
    
    func getChatMessages(for session: MCSession) -> [ChatMessage]? {
        return chatHistory.getMessages(for: session)
    }
}
