//
//  GiphyTableViewController.swift
//  HeiseMultipeerConnectivity
//
//  Created by Gero Gerber on 26.02.19.
//  Copyright © 2019 Gero Gerber. All rights reserved.
//

import UIKit

protocol GiphyTableViewControllerDelegate: class {
    func giphySelected(data: Data, pathExtension: String)
}

class GiphyTableViewController: UITableViewController {
    
    let giphyAPIKey = "<YOUR_API_KEY_GOES_HERE>" // https://developers.giphy.com
    var result: GiphyResponse?
    var pendingTasks = Set<URLSessionDataTask>()
    weak var delegate: GiphyTableViewControllerDelegate?
    
    override func viewDidLoad() {
        super.viewDidLoad()
        
        definesPresentationContext = true
        
        tableView.separatorStyle = .none
        
        let searchController = UISearchController(searchResultsController: nil)
        navigationItem.searchController = searchController
        navigationItem.hidesSearchBarWhenScrolling = false
        searchController.obscuresBackgroundDuringPresentation = false
        searchController.searchBar.delegate = self
        searchController.searchBar.placeholder = "Search GIPHY"
        
        title = "GIPHY"
    }
    
    override func viewDidAppear(_ animated: Bool) {
        super.viewDidAppear(animated)
        
        let urlRequest = getURLRequestByTrending()
        startGiphyDataTask(request: urlRequest)
    }
    
    func refresh(result: GiphyResponse) {
        self.result = result
        tableView.reloadData()
    }
    
    private func getURLRequestByTrending() -> URLRequest {
        var urlComponents = URLComponents(string: "https://api.giphy.com/v1/gifs/trending")!
        urlComponents.queryItems = getDefaultQueryItems()
        return createURLRequest(urlComponents: urlComponents)
    }
    
    private func getURLRequestByKeyword(keyword: String) -> URLRequest {
        var urlComponents = URLComponents(string: "https://api.giphy.com/v1/gifs/search")!
        urlComponents.queryItems = getDefaultQueryItems()
        urlComponents.queryItems?.append(URLQueryItem(name: "q", value: keyword))
        
        return createURLRequest(urlComponents: urlComponents)
    }
    
    private func getDefaultQueryItems() -> [URLQueryItem] {
        let apiKeyQuery = URLQueryItem(name: "api_key", value: giphyAPIKey)
        let limitQuery = URLQueryItem(name: "limit", value: "25")
        let offsetQuery = URLQueryItem(name: "offset", value: "0")
        let ratingQuery = URLQueryItem(name: "rating", value: "G")
        let languageQuery = URLQueryItem(name: "lang", value: "en")
        
        return [ apiKeyQuery, limitQuery, offsetQuery, ratingQuery, languageQuery ]
    }
    
    private func createURLRequest(urlComponents: URLComponents) -> URLRequest {
        var request = URLRequest(url: urlComponents.url!)
        request.httpMethod = "GET"
        return request
    }
    
    private func startGiphyDataTask(request: URLRequest) {
        for t in pendingTasks {
            t.cancel()
        }
        
        pendingTasks.removeAll()
        
        let defaultSession = URLSession(configuration: .default)
        
        let task = defaultSession.dataTask(with: request) { data, response, error in
            if let data = data {
                let result = try! JSONDecoder().decode(GiphyResponse.self, from: data)
                
                DispatchQueue.main.async {
                    self.refresh(result: result)
                }
            }
        }
        
        self.pendingTasks.insert(task)
        task.resume()
    }
}

// MARK: - UITableViewDataSource

extension GiphyTableViewController {
    override func tableView(_ tableView: UITableView, cellForRowAt indexPath: IndexPath) -> UITableViewCell {
        let cell = tableView.dequeueReusableCell(withIdentifier: "GiphyCell", for: indexPath) as! GiphyTableViewCell
        
        if let item = result?.data[indexPath.row] {
            if let original = item.images["preview"], let url = original.mp4 {
                cell.loadVideo(url: URL(string: url)!)
            }
        }
        
        return cell
    }
    
    override func numberOfSections(in tableView: UITableView) -> Int {
        return 1
    }
    
    override func tableView(_ tableView: UITableView, numberOfRowsInSection section: Int) -> Int {
        if let result = result {
            return result.data.count
        }
        
        return 0
    }
}

// MARK: - UITableViewDelegate

extension GiphyTableViewController {
    override func tableView(_ tableView: UITableView, heightForRowAt indexPath: IndexPath) -> CGFloat {
        if let item = result?.data[indexPath.row] {
            if let preview = item.images["preview"], let height = preview.height, let width = preview.width {
                if let imageHeight = Int(height), let imageWidth = Int(width) {
                    let gifAspectRatio = CGFloat(Float(imageHeight) / Float(imageWidth))
                    return CGFloat(tableView.frame.size.width * gifAspectRatio)
                }
            }
        }
        
        return 0.0
    }
    
    override func tableView(_ tableView: UITableView, didEndDisplaying cell: UITableViewCell, forRowAt indexPath: IndexPath) {
        if let giphyCell = cell as? GiphyTableViewCell {
            giphyCell.reset()
        }
    }
    
    override func tableView(_ tableView: UITableView, didSelectRowAt indexPath: IndexPath) {
        if let item = result?.data[indexPath.row] {
            if let preview = item.images["preview"] {
                if let previewUrl = preview.mp4 {
                    if let url = URL(string: previewUrl) {
                        URLSession.shared.dataTask(with: url, completionHandler: { [weak self] data, response, error in
                            DispatchQueue.main.async {
                                if let data = data {
                                    if let strongSelf = self {
                                        strongSelf.delegate?.giphySelected(data: data, pathExtension: url.pathExtension)
                                        strongSelf.navigationController!.popViewController(animated: true)
                                    }
                                }
                            }
                        }).resume()
                    }
                }
            }
        }
    }
}

// MARK: - UISearchBarDelegate

extension GiphyTableViewController: UISearchBarDelegate {
    func searchBarSearchButtonClicked(_ searchBar: UISearchBar) {
        if let searchText = searchBar.text {
            let urlRequest = getURLRequestByKeyword(keyword: searchText)
            startGiphyDataTask(request: urlRequest)
        }
    }
}

