//
//  BeaconManager.swift
//  HeiseMultipeerConnectivity
//
//  Created by Gero Gerber on 26.02.19.
//  Copyright © 2019 Gero Gerber. All rights reserved.
//

import CoreBluetooth
import CoreLocation
import Foundation
import UserNotifications

class BeaconManager: NSObject {
    private static let uuidValue = "EE9F792D-CC1D-4FB4-B3D0-1755F9E06233"
    private static let beaconRegionID = "de.apptects.HeiseMultipeerConnectivityRegion"
    private static let beaconRegionMajorID: CLBeaconMajorValue = 1
    private static let beaconRegionMinorID: CLBeaconMinorValue = 1
    
    private var beaconRegion: CLBeaconRegion!
    private var locationManager: CLLocationManager?
    private var peripheralManager: CBPeripheralManager?
    
    func initialize() {
        initializeNotifications()
        initializeAdvertising()
        initializeBeaconMonitoring()
    }
    
    private func initializeNotifications() {
        UNUserNotificationCenter.current().delegate = self
        UNUserNotificationCenter.current().requestAuthorization(options: [.alert, .sound, .badge], completionHandler: { didAllow, error in
            if let error = error {
                print("Error requesting authorization for nitifications: \(error.localizedDescription)")
            }
        })
    }
    
    private func initializeAdvertising() {
        beaconRegion = BeaconManager.createBeaconRegion()
        peripheralManager = CBPeripheralManager(delegate: self, queue: nil)
    }
    
    private func startAdvertising() {
        let peripheralData = beaconRegion.peripheralData(withMeasuredPower: nil)
        peripheralManager?.startAdvertising(((peripheralData as NSDictionary) as! [String : Any]))
        print("Started advertising with Beacon Region ID \(BeaconManager.beaconRegionID)")
    }
    
    private func initializeBeaconMonitoring() {
        locationManager = CLLocationManager()
        locationManager?.allowsBackgroundLocationUpdates = true
        locationManager?.delegate = self
        
        locationManager?.requestAlwaysAuthorization()        
    }
    
    private func startBeaconMonitoring() {
        if CLLocationManager.isMonitoringAvailable(for: CLBeaconRegion.self) {
            locationManager?.startMonitoring(for: beaconRegion)
            print("Started monitoring Beacons")
        }
    }
    
    private func triggerNearbyPeerNotification() {
        let content = UNMutableNotificationContent()
        content.title = "Nearby Peer Alert"
        content.subtitle = "A peer is near you"
        content.body = "We found a peer nearby. Tap to open app and start chatting!"
        
        let trigger = UNTimeIntervalNotificationTrigger(timeInterval: 1, repeats: false)
        let request = UNNotificationRequest(identifier: "NearbyPeerEnteredNotification", content: content, trigger: trigger)
        
        UNUserNotificationCenter.current().add(request, withCompletionHandler: nil)
    }
    
    private class func createBeaconRegion() -> CLBeaconRegion {
        let uuid = UUID(uuidString: uuidValue)!
        let beaconRegion = CLBeaconRegion(proximityUUID: uuid, major: beaconRegionMajorID, minor: beaconRegionMinorID, identifier: beaconRegionID)
        beaconRegion.notifyOnEntry = true
        beaconRegion.notifyEntryStateOnDisplay = true
        return beaconRegion
    }
}

// MARK: - CBPeripheralManagerDelegate

extension BeaconManager: CBPeripheralManagerDelegate {
    func peripheralManagerDidUpdateState(_ peripheral: CBPeripheralManager) {
        if peripheral.state == .poweredOn {
            startAdvertising()
        }
    }
}

// MARK: - CLLocationManagerDelegate

extension BeaconManager: CLLocationManagerDelegate {
    func locationManager(_ manager: CLLocationManager, didEnterRegion region: CLRegion) {
        // Device crossing from outside the region to inside the region
        triggerNearbyPeerNotification()
    }
    
    func locationManager(_ manager: CLLocationManager, didChangeAuthorization status: CLAuthorizationStatus) {
        if status == .authorizedAlways {
            startBeaconMonitoring()
        }
    }
    
    func locationManager(_ manager: CLLocationManager, didDetermineState state: CLRegionState, for region: CLRegion) {
        if state == .inside {
            // User turns on the display and the device is already inside the region
            triggerNearbyPeerNotification()
        }
    }
    
    func locationManager(_ manager: CLLocationManager, monitoringDidFailFor region: CLRegion?, withError error: Error) {
        print("Region Monitoring failed: \(error.localizedDescription)")
    }
}

// MARK: - UNUserNotificationCenterDelegate

extension BeaconManager: UNUserNotificationCenterDelegate {
    func userNotificationCenter(_ center: UNUserNotificationCenter, willPresent notification: UNNotification, withCompletionHandler completionHandler: @escaping (UNNotificationPresentationOptions) -> Void) {
        completionHandler([.sound, .badge, .alert])
    }
}
