unit BillboardU; // 14-MAR-2000 as (Arne Schpers)
// Auf Delphi umgesetztes Sample des DirectX-SDKs
// Setzt DirectX 7 und die Header-Dateien von Erik Unger voraus

//-----------------------------------------------------------------------------
// File: Billboard.h & .cpp
//
// Delphi translation by Arne Schpers
//
// Example code showing how to do billboarding. The sample uses
// billboarding to draw some trees.
//
// Note: this implementation is for billboards that are fixed to rotate
// about the Y-axis, which is good for things like trees. For
// unconstrained billboards, like explosions in a flight sim, the
// technique is the same, but the the billboards are positioned slightly
// different. Try using the inverse of the view matrix, TL-vertices, or
// some other technique.
//
// Note: This code uses the D3D Framework helper library.
//
// Copyright (c) 1995-1999 Microsoft Corporation. All rights reserved.
//-----------------------------------------------------------------------------


interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls,
  Forms, Dialogs, Math,  // floor
  Direct3D, DirectDraw,  // Header von Erik Unger
  // D3D-Rahmenprogramm aus dem DX-SDK, auf Delphi umgesetzt
  D3DEnum,D3DApp,D3DFrame,D3DFile,D3DTextr,D3DUtil,D3DMath;

//-----------------------------------------------------------------------------
// Defines, constants, and global variables
//-----------------------------------------------------------------------------
const
  NUM_DEBRIS = 512;     // Tree, shadow, and point data
  NUM_TREES  = 128;
  MAX_WIDTH  = 50.0;

type
  TBBForm = class(TD3DApplication)
    procedure FormCreate(Sender: TObject);
  private
    m_bUseAlphaTest: Boolean;
    m_BackgroundMesh: Array[0..3] of TD3DTLVERTEX;
    m_Debris: Array[0..NUM_DEBRIS-1] of TD3DLVERTEX;
    m_TreeMesh: Array[0..3] of TD3DLVERTEX;
    m_TreePositions: Array[0..NUM_TREES-1] of TD3DVECTOR;
    m_fViewAngle: FLOAT; // Angle off Y-axis, used to rotate billboards
  protected
    function DrawBackground: HRESULT;
    function DrawDebris: HRESULT;
    function DrawTreeShadows: HRESULT;
    function DrawTrees: HRESULT;
  protected
    function ConfirmDevice(Caps: PDDCaps;
    Desc: PD3DDeviceDesc7): HResult;
  protected  // Overrides fr TD3DApplication
    function OneTimeSceneInit: HResult; override;
    function InitDeviceObjects: HResult; override;
    function DeleteDeviceObjects: HResult; override;
    function Render: HResult; override;
    function FrameMove(fTimeKey: FLOAT): HResult; override;
  end;

var
  BBForm: TBBForm;

implementation
{$R *.DFM}

function RandomPos: FLOAT;  // inlines in C++
begin
  Result := MAX_WIDTH* (Random - Random);
end;

function RandomColor: FLOAT;
begin
  Result := 0.1 + Random / 2;
end;


{ TBBForm }
procedure TBBForm.FormCreate(Sender: TObject);
begin
  Caption := 'Billboard: D3D Billboarding Example';
  m_bAppUseZBuffer  := TRUE;
  m_bAppUseStereo   := TRUE;
  m_bShowStats      := TRUE;
  // nur D3D-Gerte mit Alphacaps
  OnConfirmDevice := ConfirmDevice;
  // D3D-Rahmen anlegen, ruft OneTimeSceneInit auf
  CreateFramework;
end;


//--------------------------------------------------------------
// Called during device intialization, this code checks the device
// for some minimum set of capabilities
//--------------------------------------------------------------
function TBBForm.ConfirmDevice(Caps: PDDCaps;
  Desc: PD3DDeviceDesc7): HResult;
begin
  // This sample uses alpha textures and/or straight alpha.
  // Make sure the device supports them
  with Desc^.dpcTriCaps do
    if (dwTextureCaps and D3DPTEXTURECAPS_ALPHAPALETTE <> 0)
     or (dwTextureCaps and D3DPTEXTURECAPS_ALPHA <> 0)
  then Result := S_OK
  else Result := E_FAIL;
end;

// Wird einmal beim Programmstart aufgerufen, bernimmt alle
// von Grafiktreiber und -modus unabhngigen Initialisierungen
function TBBForm.OneTimeSceneInit: HResult;
var x: Integer;

  // damit man hier im Falle eines Falles eine Fehlermeldung zu sehen kriegt
  function LoadTex(const FName: String; dwState, dwFlags: DWord): HResult;
  begin
    Result := D3DTextr_CreateTextureFromFile(FName, dwState, dwFlags);
    if FAILED(Result) then ShowMessage('Error Loading '+FName);
  end;

begin
  Randomize;
  // Initialize the tree and debris data
  for x := 0 to NUM_TREES-1 do
      m_TreePositions[x] := D3DVECTOR(RandomPos, 0, RandomPos);

  for x := 0 to NUM_DEBRIS-1 do
      m_Debris[x] := D3DLVERTEX(D3DVECTOR(RandomPos, 0, RandomPos),
        D3DRGBA(RandomColor, RandomColor, 0.0, 1.0), 0, 0.0, 0.0);

  // Initialize the tree and background meshes
  m_TreeMesh[0] := D3DLVERTEX(D3DVECTOR(-1, 0, 0), $ffffffff, 0, 0.0, 1.0);
  m_TreeMesh[1] := D3DLVERTEX(D3DVECTOR(-1, 2, 0), $ffffffff, 0, 0.0, 0.0);
  m_TreeMesh[2] := D3DLVERTEX(D3DVECTOR( 1, 0, 0), $ffffffff, 0, 1.0, 1.0);
  m_TreeMesh[3] := D3DLVERTEX(D3DVECTOR( 1, 2, 0), $ffffffff, 0, 1.0, 0.0);

  m_BackgroundMesh[0] := D3DTLVERTEX(D3DVECTOR(0,0,0.99), 0.5, $ffffffff, 0, 0, 1 );
  m_BackgroundMesh[1] := D3DTLVERTEX(D3DVECTOR(0,0,0.99), 0.5, $ffffffff, 0, 0, 0 );
  m_BackgroundMesh[2] := D3DTLVERTEX(D3DVECTOR(0,0,0.99), 0.5, $ffffffff, 0, 1, 1 );
  m_BackgroundMesh[3] := D3DTLVERTEX(D3DVECTOR(0,0,0.99), 0.5, $ffffffff, 0, 1, 0 );

  // Create some textures
  if SUCCEEDED(LoadTex('Cloud3.bmp', 0, 0)) and
     SUCCEEDED(LoadTex('Shadow1.bmp', 0, D3DTEXTR_TRANSPARENTWHITE )) and 
     SUCCEEDED(LoadTex('Tree0.bmp',   0, D3DTEXTR_TRANSPARENTBLACK ))
   then Result := S_OK
   else Result := E_FAIL;
end;


// Von Grafiktreiber und -modus abhngige Initialisierungen,
// wird bei jedem Wechsel erneut aufgerufen
function TBBForm.InitDeviceObjects: HResult;
var vp: TD3DViewport7; matProj: TD3DMatrix;
begin
  // Check the alpha test caps of the device. (The alpha test offers a
  // performance boost to not render pixels less than some alpha threshold.)
  m_bUseAlphaTest := m_pDeviceInfo^.ddDeviceDesc.dpcTriCaps.dwAlphaCmpCaps
      and D3DPCMPCAPS_GREATEREQUAL <> 0;

  // Set up the dimensions for the background image
  m_pd3dDevice.GetViewport(vp);
  m_BackgroundMesh[0].sy := vp.dwHeight;
  m_BackgroundMesh[2].sy := vp.dwHeight;
  m_BackgroundMesh[2].sx := vp.dwWidth;
  m_BackgroundMesh[3].sx := vp.dwWidth;

  // Set the transform matrices (view and world are updated per frame)
  D3DUtil_SetProjectionMatrix(matProj, Pi/2, 1.0, 1.0, 100.0);
  with m_pd3dDevice do
  begin
    SetTransform(D3DTRANSFORMSTATE_PROJECTION, matProj);

    // Set up the default texture states
    D3DTextr_RestoreAllTextures(m_pd3dDevice);
    SetTextureStageState(0, D3DTSS_COLORARG1, Ord(D3DTA_TEXTURE));
    SetTextureStageState(0, D3DTSS_COLORARG2, Ord(D3DTA_DIFFUSE));
    SetTextureStageState(0, D3DTSS_COLOROP,   Ord(D3DTOP_MODULATE));
    SetTextureStageState(0, D3DTSS_ALPHAOP,   Ord(D3DTOP_SELECTARG1));
    SetTextureStageState(0, D3DTSS_ALPHAARG1, Ord(D3DTA_TEXTURE));
    SetTextureStageState(0, D3DTSS_MINFILTER, Ord(D3DTFN_LINEAR));
    SetTextureStageState(0, D3DTSS_MAGFILTER, Ord(D3DTFG_LINEAR));
    SetRenderState(D3DRENDERSTATE_DITHERENABLE, Ord(TRUE));

    // Note: in DX7, setting D3DRENDERSTATE_LIGHTING to FALSE is needed to
    // turn off vertex lighting (and use the color in the D3DLVERTEX instead.)
    SetRenderState(D3DRENDERSTATE_LIGHTING, Ord(FALSE));
  end;
  Result := S_OK;
end;


// Gibt die bei InitDeviceObjects angelegten Objekte wieder frei
// Bei jedem Wechsel von Grafiktreiber und -modus aufgerufen
function TBBForm.DeleteDeviceObjects: HResult;
begin
  // DirectDraw-Oberflchen der Texturen. Textur-Bitmaps
  // bleiben davon unberhrt
  D3DTextr_InvalidateAllTextures;
  // Lichtquellen etc. werden beim Abbau der Gerteschnittstelle
  // automatisch beseitigt
  Result := S_OK;
end;

// ===============================================
function TBBForm.DrawBackground: HRESULT;
begin // Draw the background
  with m_pd3dDevice do
  begin
    SetTexture(0, D3DTextr_GetSurface('Cloud3.bmp'));
    SetRenderState(D3DRENDERSTATE_ZENABLE, Ord(False));
    SetRenderState(D3DRENDERSTATE_TEXTUREPERSPECTIVE, Ord(False));
    Result := DrawPrimitive(D3DPT_TRIANGLESTRIP, D3DFVF_TLVERTEX, m_BackgroundMesh, 4, 0 );
  end;
end;

function TBBForm.DrawDebris: HRESULT;
var matIdentity: TD3DMatrix;
begin
  D3DUtil_SetIdentityMatrix(matIdentity);

  // Render the debris
  with m_pd3dDevice do
  begin
    SetTexture(0, nil);
    SetTransform(D3DTRANSFORMSTATE_WORLD, matIdentity);
    Result := DrawPrimitive(D3DPT_POINTLIST, D3DFVF_LVERTEX, m_Debris, NUM_DEBRIS, 0);
  end;
end;

function TBBForm.DrawTrees: HRESULT;
var matWorld: TD3DMatrix; x: Integer;
begin
  with m_pd3dDevice do
  begin
    // Set state for drawing trees
    SetTexture(0, D3DTextr_GetSurface('Tree0.bmp'));

    // Billboards are in XY plane, so turn off texture perpective correction
    SetRenderState(D3DRENDERSTATE_TEXTUREPERSPECTIVE, Ord(False));

    // Trees should be using the zbuffer
    SetRenderState(D3DRENDERSTATE_ZENABLE, Ord(True));

    // Set diffuse blending for alpha set in vertices.
    SetRenderState(D3DRENDERSTATE_ALPHABLENDENABLE, Ord(True));
    SetRenderState(D3DRENDERSTATE_SRCBLEND, Ord(D3DBLEND_SRCALPHA));
    SetRenderState(D3DRENDERSTATE_DESTBLEND, Ord(D3DBLEND_INVSRCALPHA));

    // Enable alpha testing (avoids drawing pixels with less
    // than a certain alpha.)
    SetRenderState(D3DRENDERSTATE_ALPHATESTENABLE, Ord(m_bUseAlphaTest));
    SetRenderState(D3DRENDERSTATE_ALPHAREF,$08);
    SetRenderState(D3DRENDERSTATE_ALPHAFUNC, Ord(D3DCMP_GREATEREQUAL));

    // Set up a rotation matrix to orient the billboard towards the camera.
    // This is for billboards that are fixed about the Y-axis. See note at top
    // of file for more info.
    D3DUtil_SetRotateYMatrix(matWorld, -m_fViewAngle);

    // Loop through and render all trees
    for x := 0 to NUM_TREES-1 do
    begin
      // Translate the billboard into place
      matWorld._41 := m_TreePositions[x].x;
      matWorld._42 := m_TreePositions[x].y;
      matWorld._43 := m_TreePositions[x].z;
      SetTransform(D3DTRANSFORMSTATE_WORLD, matWorld);

      // Render the billboards polygons
      DrawPrimitive(D3DPT_TRIANGLESTRIP, D3DFVF_LVERTEX, m_TreeMesh, 4, 0);
    end;

    // Restore state
    SetRenderState(D3DRENDERSTATE_ALPHATESTENABLE,  Ord(False));
    SetRenderState(D3DRENDERSTATE_ALPHABLENDENABLE, Ord(False));
  end;
  Result := S_OK;
end;


function TBBForm.DrawTreeShadows: HRESULT;
var matWorld: TD3DMatrix; x: Integer;
begin
  with m_pd3dDevice do
  begin
    // Set state for rendering shadows
    SetTexture(0, D3DTextr_GetSurface('Shadow1.bmp'));
    SetRenderState(D3DRENDERSTATE_TEXTUREPERSPECTIVE, Ord(True));
    SetRenderState(D3DRENDERSTATE_ALPHABLENDENABLE, Ord(True));
    SetRenderState(D3DRENDERSTATE_SRCBLEND, Ord(D3DBLEND_ZERO));
    SetRenderState(D3DRENDERSTATE_DESTBLEND, Ord(D3DBLEND_SRCCOLOR));

    // Enable alpha testing (avoids drawing pixels with less
    // than a certain alpha.)
    SetRenderState(D3DRENDERSTATE_ALPHATESTENABLE, Ord(m_bUseAlphaTest));
    SetRenderState(D3DRENDERSTATE_ALPHAREF, $08);
    SetRenderState(D3DRENDERSTATE_ALPHAFUNC, Ord(D3DCMP_GREATEREQUAL));

    // Rotate the world matrix, to lay the shadow on the ground
    D3DUtil_SetRotateXMatrix(matWorld, Pi/2.0);

    // Loop through the trees rendering the shadows
    for x := 0 to NUM_TREES-1 do
    begin
      // Translate the world matrix to move the shadow into place
      matWorld._41 := m_TreePositions[x].x;
      matWorld._42 := m_TreePositions[x].y;
      matWorld._43 := m_TreePositions[x].z;
      SetTransform(D3DTRANSFORMSTATE_WORLD, matWorld);

      // Render the shadow's polygons
      DrawPrimitive(D3DPT_TRIANGLESTRIP, D3DFVF_LVERTEX, m_TreeMesh, 4, 0);
    end;
    // Restore state
    SetRenderState(D3DRENDERSTATE_ALPHATESTENABLE,  Ord(False));
    SetRenderState(D3DRENDERSTATE_ALPHABLENDENABLE, Ord(False));
  end;
  Result := S_OK;
end;

function TBBForm.FrameMove(fTimeKey: FLOAT): HResult;
var vUp, vFrom, vTo, vAt: TD3DVector; tu: FLOAT;
begin
  // Move the camera about a large circle through the trees
  m_fViewAngle := ( (fTimeKey/12)-floor(fTimeKey/12) ) * 2 * Pi;

  vUp := D3DVECTOR(0,1,0);  // head straight up
  vFrom := D3DVECTOR(20.0*cos(m_fViewAngle), 3.0, 20.0*sin(m_fViewAngle));
  vTo := D3DVECTOR(20.0*cos(m_fViewAngle+0.1), 3.0, 20.0*sin(m_fViewAngle+0.1));

  vAt := VectorAdd(vFrom, VectorMulS(VectorNormalize(VectorSub(vTo,vFrom)),10));

  SetViewParams( @vFrom, @vAt, @vUp, 0.1);

  // Scroll the background texture
  tu := (fTimeKey/9)-floor(fTimeKey/9);
  m_BackgroundMesh[0].tu := tu; m_BackgroundMesh[1].tu := tu;
  m_BackgroundMesh[2].tu := tu - 1.0; m_BackgroundMesh[3].tu := tu - 1.0;

  Result := S_OK;
end;


//-----------------------------------------------------------------------------
// Called once per frame, the call is the entry point for 3d
// rendering. This function sets up render states, clears the
// viewport, and renders the scene.
//-----------------------------------------------------------------------------
function TBBForm.Render: HResult;
begin
  // Clear the viewport
  m_pd3dDevice.Clear( 0, nil, D3DCLEAR_ZBUFFER, 0, 1.0, 0);

  // Begin the scene
  if SUCCEEDED(m_pd3dDevice.BeginScene) then
  begin
    // Draw all items in the scene
    DrawBackground;
    DrawDebris;
    DrawTreeShadows;
    DrawTrees;

    // End the scene.
    m_pd3dDevice.EndScene;
  end;
  Result := S_OK;
end;

end.

