unit SphereMapU; // 15-MAR-2000 as (Arne Schpers)
// Auf Delphi umgesetztes Sample des DirectX-SDKs
// Setzt DirectX 7 und die Header-Dateien von Erik Unger voraus

//-----------------------------------------------------------------------------
// File: SphereMap.cpp
//
// Desc: Example code showing how to use sphere-mapping in D3DIM. The main part
//       of the code is how the sample must do it's own world-view
//       transformation of the object's normals. Based of the value of the
//       transformed normals, the corresponding vertex's texture coords are set
//       to the appopriate spot in the spheremap.
//
//       Note: This code uses the D3D Framework helper library.
//
//
// Copyright (c) 1997-1999 Microsoft Corporation. All rights reserved.
//-----------------------------------------------------------------------------

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls,
  Forms, Dialogs, Math,  // floor
  Direct3D, DirectDraw,  // Header von Erik Unger
  // D3D-Rahmenprogramm aus dem DX-SDK, auf Delphi umgesetzt
  D3DEnum,D3DApp,D3DFrame,D3DFile,D3DTextr,D3DUtil,D3DMath;

type
  TSMForm = class(TD3DApplication)
    procedure FormCreate(Sender: TObject);
  private
    m_pTeapotFile: TD3DFile;
    m_pTeapotVertices: PD3DVertex;
    m_dwTeapotNumVertices: Cardinal;
    m_pTeapotIndices: PWord;
    m_dwTeapotNumIndices: Cardinal;

    function ApplySphereMapToObject(pvVertices: PD3DVertex;
      dwNumVertices: Cardinal): HResult;

  protected  // Overrides fr TD3DApplication
    function OneTimeSceneInit: HResult; override;
    function InitDeviceObjects: HResult; override;
    function DeleteDeviceObjects: HResult; override;
    function Render: HResult; override;
    function FrameMove(fTimeKey: FLOAT): HResult; override;
    function FinalCleanup: HResult; override;
  end;



var
  SMForm: TSMForm;

implementation

{$R *.DFM}

procedure TSMForm.FormCreate(Sender: TObject);
begin
  Caption := 'SphereMap: Using Spheremaps in Direct3D';
  m_bAppUseZBuffer  := TRUE;
  m_bAppUseStereo   := TRUE;
  m_bShowStats      := TRUE;

  OnConfirmDevice := nil;  // alle D3D-Gerte abzhlen
  // D3D-Rahmen anlegen, ruft OneTimeSceneInit auf
  CreateFramework;
end;

// Wird einmal beim Programmstart aufgerufen, bernimmt alle
// von Grafiktreiber und -modus unabhngigen Initialisierungen
function TSMForm.OneTimeSceneInit: HResult;
begin
  m_pTeapotFile := TD3DFile.Create;

  Result := m_pTeapotFile.Load('teapot.x');
  Result := Result
    or m_pTeapotFile.GetMeshVertices('',
           m_pTeapotVertices, m_dwTeapotNumVertices);
  Result := Result
    or m_pTeapotFile.GetMeshIndices('',
           m_pTeapotIndices, m_dwTeapotNumIndices);

  if FAILED(Result) then
  begin
    ShowMessage('Error loading TEAPOT.X file');
    Result := E_FAIL; Exit;
  end;

  // Load the spheremap texture
  if FAILED(D3DTextr_CreateTextureFromFile('SphereMap.bmp',0,0)) then
  begin
    ShowMessage('Error loading SPHEREMAP.BMP texture');
    Result := E_FAIL; Exit;
  end;
end;

// Gegenstck zu OneTimeSceneInit
function TSMForm.FinalCleanup: HResult;
begin
  m_pTeapotFile.Free;
  Result := S_OK;
end;

// Von Grafiktreiber und -modus abhngige Initialisierungen,
// wird bei jedem Wechsel erneut aufgerufen
function TSMForm.InitDeviceObjects: HResult;
var vp: TD3DViewport7; fAspect: FLOAT;
    matWorld, matProj: TD3DMatrix;
    vEyePt, vLookatPt, vUpVec: TD3DVector;
    mtrl: TD3DMaterial7;
begin
  D3DTextr_RestoreAllTextures(m_pd3dDevice);
  with m_pd3dDevice do
  begin
    SetTexture( 0, D3DTextr_GetSurface('SphereMap.bmp') );
    SetTextureStageState( 0, D3DTSS_COLORARG1, Ord(D3DTA_TEXTURE) );
    SetTextureStageState( 0, D3DTSS_COLORARG2, Ord(D3DTA_DIFFUSE) );
    SetTextureStageState( 0, D3DTSS_COLOROP,   Ord(D3DTOP_MODULATE) );
    SetTextureStageState( 0, D3DTSS_MAGFILTER, Ord(D3DTFG_LINEAR) );
    SetTextureStageState( 0, D3DTSS_MINFILTER, Ord(D3DTFN_LINEAR) );
    SetRenderState( D3DRENDERSTATE_DITHERENABLE,   Ord(TRUE) );
    SetRenderState( D3DRENDERSTATE_SPECULARENABLE, Ord(FALSE) );
    SetRenderState( D3DRENDERSTATE_ZENABLE,        Ord(TRUE) );
    SetRenderState( D3DRENDERSTATE_AMBIENT,        $ffffffff );
  end;

  // Get the aspect ratio
  m_pd3dDevice.GetViewport(vp);
  fAspect := vp.dwHeight / vp.dwWidth;

  // Set the transform matrices
  vEyePt    := D3DVECTOR(0.0, 0.0, -4.5);
  vLookatPt := D3DVECTOR(0.0, 0.0,  0.0);
  vUpVec    := D3DVECTOR(0.0, 1.0,  0.0);

  D3DUtil_SetIdentityMatrix(matWorld );
  D3DUtil_SetProjectionMatrix(matProj, Pi/4, fAspect, 1.0, 20.0);

  m_pd3dDevice.SetTransform(D3DTRANSFORMSTATE_WORLD, matWorld );
  m_pd3dDevice.SetTransform(D3DTRANSFORMSTATE_PROJECTION, matProj);
  SetViewParams(@vEyePt, @vLookatPt, @vUpVec, 0.1);

  D3DUtil_InitMaterial(mtrl, 1.0, 1.0, 1.0, 1.0);
  m_pd3dDevice.SetMaterial(mtrl);

  Result := S_OK;
end;


// Gibt die bei InitDeviceObjects angelegten Objekte wieder frei
// Bei jedem Wechsel von Grafiktreiber und -modus aufgerufen
function TSMForm.DeleteDeviceObjects: HResult;
begin
  D3DTextr_InvalidateAllTextures;
  Result := S_OK;
end;

// Fortschreibung der Bewegung abhngig von fTimeKey
function TSMForm.FrameMove(fTimeKey: FLOAT): HResult;
var matRotate: TD3DMatrix;
begin
  // Setup the world spin matrix
  D3DUtil_SetRotationMatrix(matRotate, D3DVECTOR(1.0,1.0,0.0), fTimeKey/2);
  m_pd3dDevice.SetTransform(D3DTRANSFORMSTATE_WORLD, matRotate);

  ApplySphereMapToObject(m_pTeapotVertices, m_dwTeapotNumVertices);

  Result := S_OK;
end;


function TSMForm.ApplySphereMapToObject(pvVertices: PD3DVertex;
  dwNumVertices: Cardinal): HResult;
var matWorld, matView, matWV: TD3DMatrix;
  i: Integer;
begin
  // Get the current world-view matrix
  m_pd3dDevice.GetTransform(D3DTRANSFORMSTATE_VIEW,  matView);
  m_pd3dDevice.GetTransform(D3DTRANSFORMSTATE_WORLD, matWorld);

  D3DMath_MatrixMultiply(matWV, matWorld, matView );

  // Loop through the vertices, transforming each one and calculating
  // the correct texture coordinates.
  for i := 0 to Integer(dwNumVertices-1) do
  with pvVertices^ do
  begin
    // Check the z-component, to skip any vertices that face backwards
    if nx * matWV._13 + ny* matWV._23 + nz * matWV._33 <= 0 then
    begin
      // Assign the spheremap's texture coordinates
      tu := 0.5 * (1.0 + (nx * matWV._11 + ny * matWV._21 + nz * matWV._31));
      tv := 0.5 * (1.0 - (nx * matWV._12 + ny * matWV._22 + nz * matWV._32));
    end;
    Inc(pvVertices);
  end;
  Result := S_OK;
end;


// Wird einmal pro Frame aufgerufen (Transformationsmatrizen,
// Objekte, Viewports etc. bereits eingerichtet). Lscht den
// Viewport und zeichnet die gesamte Szene
function TSMForm.Render: HResult;
begin
  // Clear the viewport
  m_pd3dDevice.Clear(0, nil, D3DCLEAR_TARGET or D3DCLEAR_ZBUFFER,
                     $000000ff, 1.0, 0);

  // Begin the scene
  if SUCCEEDED(m_pd3dDevice.BeginScene) then
  begin
   // = m_pTeapotFile.Render(m_pd3dDevice);
   m_pd3dDevice.DrawIndexedPrimitive(D3DPT_TRIANGLELIST, D3DFVF_VERTEX,
                            m_pTeapotVertices^, m_dwTeapotNumVertices,
                            m_pTeapotIndices^, m_dwTeapotNumIndices, 0);
    m_pd3dDevice.EndScene;
  end;
  Result := S_OK;
end;

end.

