// to compile: javac -O de/heise/ct/bench/Benchmark.java
// to run: java de.heise.ct.bench.Benchmark

package de.heise.ct.bench;

/**
 * Title: ctbench
 * Description: a small benchmark for C#, Java, and C++
 * Copyright: Copyright (c) 2002
 * Company: c't magazine, www.heise.de/ct
 * @author Jrn Loviscach
 * @author Ulrich Breymann
 * @version 1.0
 */

import java.util.*;

public class Benchmark {

	public static void main(String args[]) {
		System.out.println("Please wait ...");
		// use the bits of sqrt(2) as pseudo-random numbers
		RandomNumberCollection r = new RandomNumberCollection(benchSqrtTwo());
		System.gc();
		benchVector(r);
		System.gc();
		benchList(r);
		System.gc();
		benchBlur(r);
		System.gc();
		benchRungeKutta();
		System.gc();
		benchMemorySimple(r);
	}

	/** call and bench the initializer of the square root class */
	private static LongNumber benchSqrtTwo() {
		Stopwatch s = new Stopwatch();
		s.start();
		int i = SqrtTwo.init();
		s.stop();
		s.print("Square Root", i);
		return SqrtTwo.getValue();
	}

	/** search in a vector-like container */
	private static void benchVector(RandomNumberCollection r) {
		final short count = 20000;
		final short size = 10000;
		Vector c = new Vector();
		for(int i=0; i < size; i++)
			c.add(new Integer(i));
		int seeked;
		int pos;
		Stopwatch s = new Stopwatch();
		s.start();
		for(int j = 0; j < count; j++) {
			seeked = r.next(size);
			pos = c.indexOf(new Integer(seeked));
		}
		s.stop();
		s.print("Vector", (count*size)/2); // (count*size)/2 is an estimate
	}

	/** search in a list-like container */
	private static void benchList(RandomNumberCollection r) {
		final short count = 5000;
		final short size = 10000;
		LinkedList c = new LinkedList();
		for(int i = 0; i < size; i++)
			c.add(new Integer(i));
		int seeked;
		int pos;
		Stopwatch s = new Stopwatch();
		s.start();
		for(int j = 0; j < count; j++) {
			seeked = r.next((short)size);
			pos = c.indexOf(new Integer(seeked));
		}
		s.stop();
		s.print("List", (count*size)/2); // (count*size)/2 is an estimate
	}

	/** Gaussian blur of a random grayscale picture */
	private static void benchBlur(RandomNumberCollection r) {
		final short size = 1000;
		final short num  = 5;
		Stopwatch s = new Stopwatch();
		byte arr1[][] = new byte[size][size];
		byte arr2[][] = new byte[size][size];

		for(int i = 0; i < size; i++)
			for(int j = 0; j < size; j++)
				arr1[i][j] = (byte) r.next((short)128); // Java has no unsigned byte type, so use only 0...127
		s.start();
		for(int n = 0; n < num; n++) { // num rounds of blurring
			for(int i = 3; i < size-3; i++) // vertical blur arr1 -> arr2
				for(int j = 0; j < size; j++)
					arr2[i][j] = (byte)((arr1[i-3][j] + arr1[i+3][j]
						+ 6*(arr1[i-2][j]+arr1[i+2][j])
						+ 15*(arr1[i-1][j]+arr1[i+1][j])
						+ 20*arr1[i][j] + 32)>>6);

			for(int j = 3; j < size-3; j++) // horizontal blur arr1 -> arr2
				for(int i = 0; i < size; i++)
					arr1[i][j] = (byte)((arr2[i][j-3] + arr2[i][j+3]
						+ 6*(arr2[i][j-2]+arr2[i][j+2])
						+ 15*(arr2[i][j-1]+arr2[i][j+1])
						+ 20*arr2[i][j] + 32)>>6);
		}
		s.stop();
		s.print("Blur", num*(size-6)*(size-6));
	}

	/** solve the Lorenz differential equation by Runge-Kutta integration */
	private static void benchRungeKutta() {
		Stopwatch s = new Stopwatch();
		final double dt = 1e-4;
		double t = 0.0;
		Vector3D v = new Vector3D(2.0, 2.0, 1.0);

		s.start();
		while(t < 50.0 + 0.5*dt){ // 0.5*dt to compensate for possible roundoff errors
			Vector3D dv1 = Vector3D.mult(dt, Vector3D.lorenz(v));
			t += 0.5*dt;
			Vector3D dv2 = Vector3D.mult(dt, Vector3D.lorenz(
				Vector3D.add(v, Vector3D.mult(0.5, dv1))));
			Vector3D dv3 = Vector3D.mult(dt,Vector3D.lorenz(
				Vector3D.add(v, Vector3D.mult(0.5, dv2))));
			t += 0.5*dt;
			Vector3D dv4 = Vector3D.mult(dt, Vector3D.lorenz(
				Vector3D.add(v, dv3)));
			// v = v + (1.0/6.0)*dv1 + (1.0/3.0)*dv2 + (1.0/3.0)*dv3 + (1.0/6.0)*dv4;
			v = Vector3D.add(v, Vector3D.mult(1.0/6.0, dv1));
			v = Vector3D.add(v, Vector3D.mult(1.0/3.0, dv2));
			v = Vector3D.add(v, Vector3D.mult(1.0/3.0, dv3));
			v = Vector3D.add(v, Vector3D.mult(1.0/6.0, dv4));
		}
		s.stop();
		s.print("Runge-Kutta", (int)(50.0/dt + 0.5));
	}

	/** allocate and free memory, no crisscross pointers involved */
	private static void benchMemorySimple(RandomNumberCollection r) {
		Stopwatch s = new Stopwatch();
		final short maxSize = 2000;
		final short minSize = 1;
		final short num = 2000;
		int[][] arr = new int[num][]; // 32 bit
		final short count = 20000;

		for(int i = 0; i < num; i++) {
			arr[i] = new int[minSize + r.next((short)(maxSize - minSize + 1))];
		}

		s.start();
		for(int i = 0; i < count; i++) {
			int n = r.next(num);
			int size = minSize + r.next((short)(maxSize - minSize + 1));
			arr[n] = new int[size];
			arr[n][size-1] = 42; // only to make sure that memory is being used
		}
		s.stop();
		s.print("Memory", count);
	}

	/** simple stop timer with formatted output */
	private static class Stopwatch {
		private long startTime, stopTime;
		public void start() { startTime = System.currentTimeMillis(); }
		public void stop() { stopTime = System.currentTimeMillis(); }
		public void print(String name, int num) {
			String tUnit = "ns";
			double tPerIteration = (1e9/1000.0/num)*(stopTime-startTime);
			if (tPerIteration > 1e8) {
				tPerIteration /= 1e6;
				tUnit = "ms";
			} else if (tPerIteration > 1e5) {
				tPerIteration /= 1e3;
				tUnit = "s";
			}
			System.out.println(
				name + ": "
				+ (int)(0.5 + (1000.0/1000.0)*(stopTime-startTime))
				+ " ms for " + num + " iterations; "
				+ (int)(0.5 + tPerIteration)
				+ " " + tUnit + " per iteration"
			);
		}
	}

	/** calculate sqrt(2) up to "arbitrary" precision */
	private static class SqrtTwo {
		final private static int size = 10000;
		private static LongNumber sqrtTwoValue = new LongNumber(size);
		/** calculate sqrt(2) and return number of iterations needed*/
		public static int init() {
			int i = 1;
			LongNumber n = new LongNumber(size,(short)1);
			sqrtTwoValue = iterate(n);
			while(! n.almostEqual(sqrtTwoValue) ) {
				n = sqrtTwoValue;
				sqrtTwoValue = iterate(n);
				i++;
			}
			return i;
		}

		/** return copy of internal LongNumber object */
		public static LongNumber getValue() {
			return new LongNumber(sqrtTwoValue);
		}

		/** iteration step for calculating sqrt(2): n = n*(3-n*n/2)/2 */
		private static LongNumber iterate(LongNumber n) {
			LongNumber x = new LongNumber(size, (short)3);
			LongNumber y = new LongNumber(size);
			y.addProd(n, n);
			y.half();
			x.sub(y);
			x.half();
			LongNumber z = new LongNumber(size);
			z.addProd(n, x);
			return z;
		}
	}

	/** pseudo-random number generator taking values from a LongNumber object */
	private static class RandomNumberCollection {
		private LongNumber val;
		private int num = 0;
		/** initialize by a LongNumber */
		public RandomNumberCollection (LongNumber ln) {
			val = new LongNumber(ln); // use a copy so that later changes to ln don't matter
		}
		/** get the next short as random number */
		public short next(short max) {
			num++;
			if(num >= val.getLength()) {
				num = 0;
			}
			return (short)((max*(int)val.get(num)) >> 15);
		}
	}

	/** unsigned fixed point type of "arbitrary" precision */
	private static class LongNumber {
		private int length;
		private short[] array; // using 15 bits in lack of an unsigned type in Java
		public LongNumber(int len) {
			length = len;
			array = new short[length];
		}
		public LongNumber(int len, short x) {
			length = len;
			array = new short[length];
			array[0] = x;
		}
		public LongNumber(LongNumber ln) {
			length = ln.length;
			array = new short[length];
			for (int i=0; i<length; i++) {
				array[i] = ln.array[i];
			}
		}
		private void addAndCarry(int i, int s) {
			int s1 = array[i] + s; // array may contain 0x7FFF, s may contain 0x3FFFF001
			if(s1 <= 0x7FFF) { // only 15 bits because Java doesn't do unsigned types
				array[i] = (short) s1;
			} else {
				int twoShort = array[i-1]*0x8000 + s1;
				if(twoShort >= 0x8000*0x8000) { // need to carry over more than 2*15 bits
					twoShort -= 0x8000*0x8000; // enough in any case, because s1 is smaller than this number
					int j = i-2;
					while(array[j] == 0x7FFF) { // exception on j = -1 means overflow
						array[j] = 0;
						j--;
					}
					array[j]++;
				}
				array[i] = (short)(twoShort&0x7FFF);
				array[i-1] = (short)(twoShort>>15);
			}
		}
		private void subAndCarry(int i, int s) { // note: s may be filled up to 31 bit
			if(array[i] >= s) {
				array[i] = (short)(array[i]-s);
			} else {
				int twoShort = array[i-1]*0x8000 + array[i] - s;
				if(twoShort <= 0) { // need to carry over more than 2*15 bits
					twoShort += 0x8000*0x8000; // enough in any case, because s is smaller than this number
					int j = i-2;
					while(array[j] == 0) { // exception on j = -1 means overflow
						array[j] = 0x7FFF;
						j--;
					}
					array[j]--;
				}
				array[i] = (short)(twoShort&0x7FFF);
				array[i-1] = (short)(twoShort>>15);
			}
		}
		/** subtraction: a has to be smaller than the objects itself: unsigned op! */
		public void sub(LongNumber a) {
			int len = length;
			if(a.length < len) {
				len = a.length;
			}
			for(int i = len-1; i >= 0; i--) {
				subAndCarry(i, a.array[i]);
			}
		}
		public void half() {
			array[length-1]>>=1;
			for(int i = length-2; i >= 0; i--) {
				if( (array[i] & 0x1) == 1 ) {
				array[i+1] += 0x4000; // this bit has been cleared in the step before
			}
			array[i]>>=1;
		}
		}
		public void addProd(LongNumber a, LongNumber b) {
			int len = length;
			if(a.length < len) {
				len = a.length;
			}
			if(b.length < len) {
				len = b.length;
			}
			for(int i = len-1; i >= 0; i--) {
				for(int j = len-1-i; j >= 0; j--) {
					addAndCarry(i+j, a.array[i] * b.array[j]);
				}
			}
		}
		public boolean almostEqual(LongNumber a) {
			int len = length;
			if(a.length < len) {
				len = a.length;
			}
			int carry = 0;
			for(int i = 0; i < len-2; i++) { // skip the last two
				if(carry == 0) {
					if(array[i] != a.array[i]) {
						carry = array[i]-a.array[i];
						if( (carry != 1) && (carry != -1) ) return false;
					}
				} else if(carry > 0) { // carry == +1
					if(array[i] != 0) return false;
					if(a.array[i] != 0x7FFF) return false;
				} else { // carry == -1
					if(a.array[i] != 0) return false;
					if(array[i] != 0x7FFF) return false;
				}
			}
			return true;
		}
		public short get(int i) {
			return array[i];
		}
		public int getLength() {
			return length;
		}
	}

	/** 3D vector with arithmetics and a pre-defined chaos vector field */
	private static class Vector3D {
		public double x, y, z;

		public Vector3D(double x_, double y_, double z_) {
			x = x_; y = y_; z = z_;
		}

		public static Vector3D mult(double c, Vector3D v) {
			return new Vector3D(c*v.x, c*v.y, c*v.z);
		}

		public static Vector3D add(Vector3D v, Vector3D u) {
			return new Vector3D(v.x+u.x, v.y+u.y, v.z+u.z);
		}

		/** vector field of the Lorenz differential equation */
		public static Vector3D lorenz(Vector3D v) {
			final double a = 10.0;
			final double r = 28.0;
			final double b = 8.0/3.0;
			return new Vector3D(a*(v.y-v.x), r*v.x-v.y-v.x*v.z, v.x*v.y-b*v.z);
		}
	}
}