//----------------------------------------------------------------------------
// Identify.cpp
// IDENTIFY-Daten einer IDE-Festplatte anzeigen
// fr Windows 9x/ME/NT/2000/XP
// 2001 c't/Matthias Withopf


#include <stdio.h>
#include <stdlib.h>
#define WIN32_LEAN_AND_MEAN
#include <windows.h>


//----------------------------------------------------------------------------
// Datentypen und Definitionen der SMART-Schnittstelle

// Alignment fr Sturkturen umstellen, 
// damit die Strukturgre jener des Treibers entspricht
#pragma pack(push,1)

// Funktionscode fr DeviceIOControl

const unsigned int SMART_RECEIVE_DRIVE_DATA = 0x0007C088;
                                              
// Registersatz des ATA-Hostadapter

typedef struct
  {
    BYTE bFeaturesReg;
    BYTE bSectorCountReg;
    BYTE bSectorNumberReg;
    BYTE bCylLowReg;
    BYTE bCylHighReg;
    BYTE bDriveHeadReg;
    BYTE bCommandReg;
    BYTE bReserved;
  } IDERegisters;

// Struktur zur Datenbergabe 

typedef struct
  {
    DWORD        cBufferSize;
    IDERegisters irDriveRegs;
    BYTE         bDriveNumber;
    BYTE         bReserved[3];
    DWORD        dwReserved[4];
    BYTE         bBuffer[1];
  } ATAInParam;

// Status- und Fehlermeldungen des SMART-Treibers

typedef struct
  {
    BYTE  bDriverError;
    BYTE  bIDEStatus;
    BYTE  bReserved[2];
    DWORD dwReserved[2];
  } DriverStat;

// Struktur der zurckgebenen Daten

typedef struct
  {
    DWORD      cBufferSize;
    DriverStat DriverStatus;
    BYTE       bBuffer[1];
  } ATAOutParam;


// Alignment zurckstellen 
#pragma pack(pop)

//----------------------------------------------------------------------------
// GetIdentifyStr kopiert Strings aus dem IDE-Konfigurationssektor

static void GetIdentifyStr(char *s, PWORD InqBuf, unsigned int InqBufSize)
{
  PWORD p  = InqBuf;
  char *p1 = s;
  for (unsigned int i = 0; i < InqBufSize; ++i, ++p)
    {
      // wandelt Big Endian in Little Endian
      *p1++ = (char)(*p >> 8);
      *p1++ = (char)*p;
    }
  *p1 = '\0';
  
  // fhrende Leerzeichen entfernen
  while (s[0] == ' ')
    memmove(s, s + 1, strlen(s));
  unsigned int l = strlen(s);
  
  // abschlieende Leerzeichen entfernen
  while ((l > 0) && (s[l - 1] == ' '))
    s[--l] = '\0';
}

//----------------------------------------------------------------------------
// DoIdentify liest den Konfigurationssektor einer Festplatte 
// und gibt ihn als Text aus

BOOL DoIdentify(HANDLE DiskDevice, BYTE DriveNum)
{
  BOOL Result = FALSE;
  
  // Speicherplatz fr Befehle und Daten anfordern
  unsigned int InDataSize  =   0, ISize = sizeof(ATAInParam)  + InDataSize;
  unsigned int OutDataSize = 512, OSize = sizeof(ATAOutParam) + OutDataSize; 

  // VirtualAlloc sorgt dafr, dass der Gertetreiber den Speicher ansprechen kann
  ATAInParam  *IP = (ATAInParam  *)VirtualAlloc(NULL, ISize, MEM_COMMIT, PAGE_READWRITE);
  ATAOutParam *OP = (ATAOutParam *)VirtualAlloc(NULL, OSize, MEM_COMMIT, PAGE_READWRITE);
  
  if (IP && OP)
    {
      memset(IP, 0, ISize);
      // Gre des Puffers fr die Daten der Festplatte
	  IP->cBufferSize                  = OutDataSize;
      // Register setzen
	  IP->bDriveNumber                 = DriveNum;
      IP->irDriveRegs.bSectorCountReg  = 1;
      IP->irDriveRegs.bSectorNumberReg = 1;
      // Laufwerksangabe
      IP->irDriveRegs.bDriveHeadReg    = 0xA0 | ((DriveNum & 1) << 4);
      // ATA-Befehl IDENTIFY
      IP->irDriveRegs.bCommandReg      = 0xEC;  
      
      memset(OP, 0, OSize);
      OP->cBufferSize = OSize;
      
      // Gertetreiber meldet Gre der zurckgegebenen Daten
      DWORD BytesReturned;
      
      // Gertetreiber aufrufen, bei Erfolg liefert DeviceIoControl einen Wert ungleich 0      
      if (DeviceIoControl(DiskDevice, SMART_RECEIVE_DRIVE_DATA, IP, ISize -1, OP, OSize -1, &BytesReturned, NULL))
        {
          Result = TRUE;

          // Modell, Firmware und Seriennummer kopieren ...
          char ModelStr   [40 + 1];
          char FirmwareStr[ 8 + 1];
          char SerialStr  [20 + 1];
          GetIdentifyStr(ModelStr,   ((PWORD)OP->bBuffer) + 27, 20);
          GetIdentifyStr(FirmwareStr, ((PWORD)OP->bBuffer) + 23, 4);
          GetIdentifyStr(SerialStr,  ((PWORD)OP->bBuffer) + 10, 10);
          // ... und ausgeben
          printf("Modell:       %s\n", ModelStr);
          printf("Firmware:     %s\n", FirmwareStr);
          printf("Seriennummer: %s\n", SerialStr);
        }
	    else
		  printf("GetLastError %u, BytesReturned: %u,\nDriverStatus %u, IDE Status: %u\n", GetLastError(), BytesReturned, OP->DriverStatus.bDriverError, OP->DriverStatus.bIDEStatus);
    }
  // Speicher freigeben
  if (IP) VirtualFree(IP, 0, MEM_RELEASE);
  if (OP) VirtualFree(OP, 0, MEM_RELEASE);
  
  return Result;
}

//----------------------------------------------------------------------------
// Hauptprogramm

int main(void)
{
  int Result;
  printf("ATA Identify 1.0  2001 c't/Matthias Withopf\n");
  
  // erste Festplatte abfragen
  const BYTE DriveNum = 0;
    
  // Windows 9x oder NT-basiertes System?
  BOOL IsWin9X = (GetVersion() & 0x80000000) != 0;
  
  // Name des Gertetreibers hngt vom Betriebssystem ab
  char DeviceName[128];
  if (IsWin9X)
    // Windows 9x oder ME
    strcpy(DeviceName, "\\\\.\\SMARTVSD");
  else
    // Windows NT oder Nachfolger
	sprintf(DeviceName, "\\\\.\\PhysicalDrive%u",DriveNum);
  
  // Gertetreiber als Datei ffnen
  HANDLE DiskDevice = CreateFile(DeviceName, GENERIC_READ | GENERIC_WRITE, FILE_SHARE_READ | FILE_SHARE_WRITE, NULL, OPEN_EXISTING, 0, NULL);
  
  if (DiskDevice == INVALID_HANDLE_VALUE)
    { // Fehler beim ffnen
      printf("Kann Geraet/Treiber '%s' nicht ffnen, Fehler %u\n", DeviceName, GetLastError());
	  if(IsWin9X)
		  printf("Die Datei SMARTVSD.VXD muss im Windows-Verzeichnis System\\Iosubsys liegen!\n");
      Result = 10;
    }
  else
    {
      if (DoIdentify(DiskDevice, DriveNum))
        Result = 0;
      else
        {
          printf("IDENTIFY fehlgeschlagen!\n");
          Result = 11;
        }
      // Gertetreiber schlieen
      CloseHandle(DiskDevice);
    }
  return Result;
}

