/***
 * c't Motorsteuerung
 * usbdemo.c
 *
 * User Interface
 *
 * (c) 2002 Redaktion c't, Hannover
 *          Kai-Uwe Mrkor / Herbert Schmid
 ***/


#include <windows.h>
#include <winuser.h>
#include <stdio.h>
#include <math.h>

#include "resource.h"

#include "motor.h"


/* Drehwinkel am jeweiligen Anschlag des Poti */
unsigned DrawPos0=0, DrawPos255=0;

/* Einstellungen in der Registry sichern */
void RegistryRead();
void RegistryWrite();

/* Fehlerbehandlung */
int ErrorBox(HWND hWnd, int err, BOOL restartMotor);


/***
 * Schnittstelle zur Motorsteuerung
 *
 * Dialogfenster anpassen
 ***/
void MotorSetGui(HWND hWnd, BOOL MotorRuns)
{
/* 	static BOOL OldMotorRuns=TRUE; */

	/*if (OldMotorRuns != MotorRuns)*/
	{   /*OldMotorRuns = MotorRuns;*/
		/* aktuellen Werte eintragen */
	    CheckDlgButton(hWnd, IDC_Poti_Invert, MotorInvertPoti ? BST_CHECKED : BST_UNCHECKED );
	    CheckDlgButton(hWnd, IDC_Motor_Invert, MotorExchangeLR ? BST_CHECKED : BST_UNCHECKED );
  
		EnableWindow(GetDlgItem(hWnd, IDC_Hand_Stop), MotorRuns);

		EnableWindow(GetDlgItem(hWnd, IDC_Hand_Left), !MotorRuns);
		EnableWindow(GetDlgItem(hWnd, IDC_Hand_Right), !MotorRuns);
		
		EnableWindow(GetDlgItem(hWnd, IDC_Start), !MotorRuns);
		EnableWindow(GetDlgItem(hWnd, IDC_Pos), !MotorRuns);
		EnableWindow(GetDlgItem(hWnd, IDC_Middle), !MotorRuns);
		EnableWindow(GetDlgItem(hWnd, IDC_Left), !MotorRuns);
		EnableWindow(GetDlgItem(hWnd, IDC_Right), !MotorRuns);
		
		EnableWindow(GetDlgItem(hWnd, IDC_Edit0), !MotorRuns);
		EnableWindow(GetDlgItem(hWnd, IDC_Edit255), !MotorRuns);
		EnableWindow(GetDlgItem(hWnd, IDC_Pos0), !MotorRuns);
		EnableWindow(GetDlgItem(hWnd, IDC_Pos255), !MotorRuns);
		
		EnableWindow(GetDlgItem(hWnd, IDC_Poti_Invert), !MotorRuns);
		EnableWindow(GetDlgItem(hWnd, IDC_Motor_Invert), !MotorRuns);

		EnableWindow(GetDlgItem(hWnd, IDC_SpeedEnabled), TRUE);	
	}

}


/***
 * Windrose zeichnen
 ***/
LRESULT DrawPosition(LPDRAWITEMSTRUCT lpdis)
{
	if (lpdis == NULL)
		return FALSE;

	if(lpdis->itemAction == ODA_DRAWENTIRE)
	{
		RECT* pRect = &(lpdis->rcItem);
		LONG widthY = labs( lpdis->rcItem.top - lpdis->rcItem.bottom ) ;
		LONG widthX = labs( lpdis->rcItem.left - lpdis->rcItem.right ) ;
		LONG halfMin = __min( widthX, widthY ) / 2;
		
		/* zuerst in Bitmap zeichnen, um Flackern zu verhindern */
		HDC hDC = CreateCompatibleDC(lpdis->hDC);
		HGDIOBJ hPic = CreateCompatibleBitmap(lpdis->hDC, widthX, widthY);

		TEXTMETRIC tm;
		int x,y;
		double grad;
		static int oldPoti=0x100;
		char szPos[15] = "";
				
		SelectObject(hDC, hPic );
		
		SelectObject( hDC, GetStockObject(BLACK_PEN) );
		SelectObject( hDC, GetStockObject(WHITE_BRUSH) );
        SelectObject( hDC, GetStockObject(DEFAULT_GUI_FONT) );
		GetTextMetrics( hDC, &tm);
		
		/* Weier Hintergrund */
		FillRect( hDC, &(lpdis->rcItem), GetStockObject(WHITE_BRUSH) );
		/* Windrose */
		Ellipse(hDC, (int)(widthX/2-halfMin+tm.tmMaxCharWidth*1.2),   (int)(widthY/2 - halfMin + tm.tmMaxCharWidth*1.2), 
					 (int)(widthX/2 + halfMin-tm.tmMaxCharWidth*1.2), (int)(widthY/2 + halfMin - tm.tmMaxCharWidth*1.2) );
         
		/* Beschriftung */
		SetTextAlign( hDC, TA_TOP | TA_LEFT);
		TextOut(hDC, (widthX - tm.tmAveCharWidth)/2 , 0, "N", 1);
		TextOut(hDC, (widthX - tm.tmAveCharWidth)/2 , widthY/2 + halfMin - tm.tmHeight, "S", 1);
        TextOut(hDC, (widthX/2 + halfMin-tm.tmMaxCharWidth), (widthY - tm.tmHeight) / 2 , "O", 1);
		TextOut(hDC, (widthX/2 - halfMin), (widthY - tm.tmHeight) / 2, "W", 1);
		MoveToEx(hDC, widthX/2, widthX/2, NULL);
		
		/* MotorPoti schwankt im letzten Bit,
		   das stellen wir aber nicht dar */
		if( ((oldPoti&0x1FE) != (MotorPoti&0xFE)) )
			oldPoti=MotorPoti;
		
		/* Koordinaten umrechnen */
		grad = ( (DrawPos0>DrawPos255) ? (360+DrawPos255-DrawPos0) : (DrawPos255-DrawPos0));
		if (grad < 1) /* gleiche Werte fr Position 0 und 255? */
			grad = 359; /* dann den vollen Kreis ntzen */

		/* Berechnung ab hier in 1000stel Grad */
		grad *= (double)oldPoti * 1000 / 255;
		grad += DrawPos0 * 1000;
		
		sprintf(szPos," %3i ", (int)(grad / 1000) % 360);

		grad = ((int)grad + 270000) % 360000;
		grad = 2*3.14159265358979323846264/360000*grad;

		halfMin -= 2*tm.tmMaxCharWidth;
		x = (int)((widthX/2) + halfMin * cos( grad ) );
        y = (int)((widthY/2) + halfMin * sin( grad ) );
		
		/* Pfeil */
		LineTo( hDC, x,y);
		/* Pfeilspitzen */
		halfMin -= 3*tm.tmMaxCharWidth;
		LineTo(hDC, (int)( (widthX/2) + halfMin * cos( grad + (double)0.1 ) ),
					(int)( (widthY/2) + halfMin * sin( grad + (double)0.1 ) ) );
		MoveToEx(hDC,x,y,NULL);
		LineTo(hDC, (int)( (widthX/2) + halfMin * cos( grad - (double)0.1 ) ), 
					(int)( (widthY/2) + halfMin * sin( grad - (double)0.1 ) ) );

		
		/* aktuelle Position in Grad ausgeben */
		SetTextAlign(hDC, TA_RIGHT | TA_CENTER);
		TextOut(hDC, (int)(widthX/2+tm.tmAveCharWidth), widthY/2, szPos, strlen(szPos) );
			         
		BitBlt(lpdis->hDC, pRect->left, pRect->top, widthX, widthY, hDC, 0, 0, SRCCOPY); 
		DeleteDC(hDC);
		DeleteObject(hPic);
	}

	return TRUE;
}

/***
 * Parameter zum Zeichnen der Windrose vom Dialog holen
 ***/
void GetDrawParams(HWND hDlg)
{
	int x;

	DrawPos0 = GetDlgItemInt( hDlg, IDC_Edit0, NULL, TRUE);
	DrawPos255 = GetDlgItemInt( hDlg, IDC_Edit255, NULL, TRUE);
	
	if(DrawPos0 > DrawPos255)
	{  x=DrawPos255;
	   DrawPos255=DrawPos0;
	   DrawPos0=x;
	}
	
	while(DrawPos0 <0 || DrawPos255 <0) 
	{ DrawPos0+=360; DrawPos255+=360; 
	}

	if(DrawPos0 == DrawPos255)
			DrawPos255+=359; /* vollen Kreis nutzen */
	
	/* Werte minimieren */
	x=DrawPos255-DrawPos0;
	DrawPos0 = DrawPos0 % 360;
	DrawPos255 = DrawPos0 + x;

	SetDlgItemInt(hDlg, IDC_Edit0, DrawPos0, TRUE);
	SetDlgItemInt(hDlg, IDC_Edit255, DrawPos255, TRUE);
}


/***
 * Fehlercode der Motor-Routinen auswerten
 * 
 * behandelte Codes (err):
 *   0 .. alles ok (nichts tun)
 *   1 .. USB-Baustein nicht erreichbar, Motor stand still
 *  11 .. USB-Baustein nicht erreichbar, Motor lief
 *   2 .. Motor lief nicht an oder blieb hngen
 *   3 .. Ziel trotz Richtungskorrektur nicht erreicht
 *   9 .. Fehler bei Initialisierung
 *
 *  restartMotor: TRUE: Motor bei Fehlercode 1 erneut starten
 *
 *  Rckgabewert:
 *   0 .. leichter Fehler bzw. Fehler behoben
 *   1 .. schwerer Fehler (Programmende eingeleitet)
 ***/
int ErrorBox(HWND hWnd, int err, BOOL restartMotor)
{
  
  static BOOL SentQuit=FALSE;
  char *msg=NULL;

  if (SentQuit)
	  return 1;
	
  

  switch(err)
  {  
	case 0: return 0;
  
	case 11: err-=10; restartMotor=TRUE;
	case 1:
	case 9:
		msg = (err==1) ? "Fehler beim Zugriff auf das USB-Gert!\nErneut versuchen?" :
					   "Fehler beim Initialisieren der Motorsteuerung!\nErneut Wollen Sie es erneut versuchen oder das Programm beenden?";
		if (  MessageBoxEx(hWnd, msg, "c't Motorsteuerung", 
			               MB_ICONERROR | MB_RETRYCANCEL |MB_SETFOREGROUND,
						   MAKELANGID(LANG_GERMAN, SUBLANG_GERMAN) )
		   == IDRETRY )
		{
			err=MotorInit(hWnd);
			if( !err )
			{ /* gelingt es doch noch? */
				if(restartMotor)
					err=MotorStart(hWnd);
			}
			if (!err) 
				return 0;
			else
				
				/* auch der zweite Versuch schlug fehl */
				MessageBoxEx(hWnd, "Auch der zweite Versuch schlug fehl!\nDas Programm wird beendet.", 
							 "c't Motorsteuerung", MB_ICONERROR | MB_OK |MB_SETFOREGROUND,
						   MAKELANGID(LANG_GERMAN, SUBLANG_GERMAN) );
		}
		
		/* Programm beenden */
		SentQuit=TRUE;
		PostMessage(hWnd, WM_SYSCOMMAND, SC_CLOSE, 0);
		return 1;
	
	case 2:
	case 3:
	    MotorStop(hWnd);  /* gelegentlich doppelt gemoppelt */

		msg = (err==2) ? "Der Motor lief nicht an oder stockte\nund wurde deshalb abgeschaltet." :
						 "Der Motor erreichte das Ziel trotz Richtungskorrektur nicht und wurde deshalb angehalten.";
		
		MessageBoxEx(hWnd, msg, "c't Motorsteuerung", 
			               MB_ICONINFORMATION | MB_OK |MB_SETFOREGROUND,
						   MAKELANGID(LANG_GERMAN, SUBLANG_GERMAN) );
		return 0;

	default:
		MessageBoxEx(hWnd, "Unbekannter Fehler!\nDas Programm wird beendet.", 
					 "c't Motorsteuerung", MB_ICONERROR | MB_OK |MB_SETFOREGROUND,
		        	 MAKELANGID(LANG_GERMAN, SUBLANG_GERMAN) );

		SentQuit=TRUE;
		PostMessage(hWnd, WM_SYSCOMMAND, SC_CLOSE, 0);
		return 1;
  }

  return 0;
}


/***
 * Windows-Messages abarbeiten
 ***/ 
LRESULT CALLBACK DialogProc(HWND hDlg, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
  unsigned pos;
  static InTimer=FALSE;

  switch (uMsg)
  {

   	case WM_CREATE:	/* Fenster wird erzeugt -> Motor initialisieren */
					ErrorBox(hDlg, MotorInit(hDlg), FALSE );
					
	case WM_TIMER:	if(InTimer)
					  return TRUE;  /* passiert etwa whrend der Fehlerbehandlung mittels MessageBox() */
					else 
					  InTimer=TRUE;
					
					if( !ErrorBox(hDlg, MotorTimer(hDlg), FALSE) )
					{
					  SetDlgItemInt(hDlg, IDC_Poti, (int)MotorPoti, TRUE);
					  /* Windrose neu zeichnen */
					  InvalidateRect( GetDlgItem(hDlg, IDC_Position), NULL, FALSE);
					}
					
					InTimer=FALSE;
					return TRUE;
  			
					
	case WM_DRAWITEM: /* zeichne Control */
		            if ((UINT)wParam == IDC_Position)
						return DrawPosition((LPDRAWITEMSTRUCT)lParam);
					break;
					
	case WM_COMMAND:		
						switch( LOWORD(wParam))
						{	
							/* Position anfahren */
							case IDC_Left:		GetDrawParams(hDlg);
												SetDlgItemInt(hDlg, IDC_Pos, DrawPos0, TRUE);
												break;

							case IDC_Right:		GetDrawParams(hDlg);
												SetDlgItemInt(hDlg, IDC_Pos, DrawPos255, TRUE);
												break;
														
							case IDC_Middle:	GetDrawParams(hDlg);
												SetDlgItemInt(hDlg, IDC_Pos, (int)((DrawPos0+DrawPos255)/2), TRUE);
												break;

							case IDC_Start:		GetDrawParams(hDlg);
												
												pos=GetDlgItemInt(hDlg, IDC_Pos, NULL, TRUE);
												if(pos<DrawPos0 || pos > DrawPos255)
													MessageBox(hDlg, "Ziel liegt auerhalb der erreichbaren Winkel!","c't Motorsteuerung", MB_ICONINFORMATION); 
												else
												{ /* Steuerung starten */
												  MotorGoal=(int)( (pos -DrawPos0) * ((double)255/(DrawPos255-DrawPos0) ));
												  ErrorBox(hDlg, MotorStart(hDlg), TRUE);
												}
												break;

							/* Handsteuerung und Kalibrierung */				
							case IDC_Pos0:
							case IDC_Hand_Left: GetDrawParams(hDlg);
												
												if (MotorPoti==0) 
													break;
								
												MotorGoal=0;
												ErrorBox(hDlg, MotorStart(hDlg), TRUE);
													  
												break;

							case IDC_Pos255:
							case IDC_Hand_Right:GetDrawParams(hDlg);

												/* Motor schon am Anschlag? */
												if (MotorPoti==255)
													break;

												/* unsichtbare Zielvorgabe fr Motor */
												MotorGoal=255;
												ErrorBox(hDlg, MotorStart(hDlg), TRUE);
												break;

							case IDC_Stop:		
							case IDC_Hand_Stop: ErrorBox(hDlg,MotorStop(hDlg),FALSE);
												break;

							/* Parameter der Checkboxen bernehmen */
							case IDC_Poti_Invert:
												MotorInvertPoti = ( BST_CHECKED == IsDlgButtonChecked(hDlg, IDC_Poti_Invert) );
												break;

							case IDC_Motor_Invert:
												MotorExchangeLR = ( BST_CHECKED == IsDlgButtonChecked(hDlg,IDC_Motor_Invert) );
							/* die Werte von IDC_Edit0, IDC_Edit255 und IDC_Pos bernimmt ein Click auf Start oder die Handsteuerung */
												break;

							case IDC_Edit0:
							case IDC_Edit255:	
												if ( HIWORD(wParam) == EN_KILLFOCUS  )
													GetDrawParams(hDlg);
												break;
							/* Programmende einleiten */
							case IDC_End:		PostMessage(hDlg, WM_SYSCOMMAND, SC_CLOSE, 0L);
												break;			
							default:			break;
						}
						return 0;	
						break;

	case WM_INITMENU:   /* Systemmen korrigieren: Maximieren noch ndern der Gre verhindern */
						{ 
						  HMENU hmenuInit = (HMENU) wParam; 
						  EnableMenuItem (hmenuInit, SC_SIZE, MF_BYCOMMAND | MF_GRAYED | MF_DISABLED);
						  EnableMenuItem (hmenuInit, SC_MAXIMIZE, MF_BYCOMMAND | MF_GRAYED | MF_DISABLED);
						}
						break;
	case WM_SYSCOMMAND:	if( (wParam&0xFFF0) == SC_MAXIMIZE || (wParam&0xFFF0) == SC_SIZE    ||
							(wParam&0xFFF0) == SC_HSCROLL  || (wParam&0xFFF0) == SC_VSCROLL  )
							return 0;
						break;

	/* Programmende */
	case WM_CLOSE:      MotorClose(hDlg); /* jetzt gibts keine Fehlermeldung mehr */
						/* alle Werte noch mal aus der Maske holen */
						GetDrawParams(hDlg);
						/* und Einstellungen in der Registry sichern */
						RegistryWrite();
						break;

	case WM_DESTROY:	PostQuitMessage(0);
						break;

	default:			break;


  }
  return DefWindowProc(hDlg, uMsg, wParam, lParam); 
}

/***************/
/*   WinMain   */
/***************/
int WINAPI WinMain( HINSTANCE hInstance, 
				    HINSTANCE hPrevInstance,
					LPSTR lpCmdLine,int nCmdShow)
{
  MSG			msg;					  /* Nachrichten-Struktur */
  HWND			hwDialog;				  /* Handle auf das Fenster */
  WNDCLASS		wnd;					  /* Fensterklassen-Struktur */  

  static TCHAR	szAppName[]=TEXT("Motor");

  /* Fensterklasse registrieren */
  wnd.style=CS_HREDRAW | CS_VREDRAW;
  wnd.lpfnWndProc=DialogProc;
  wnd.cbClsExtra=0;
  wnd.cbWndExtra=DLGWINDOWEXTRA;
  wnd.hInstance=hInstance;
  wnd.hIcon=LoadIcon( hInstance, "MotorIcon");
  wnd.hCursor=LoadCursor(NULL,IDC_ARROW);
  wnd.hbrBackground=(HBRUSH)(COLOR_BTNFACE+1);
  wnd.lpszMenuName=NULL;
  wnd.lpszClassName=szAppName; /* Dialog und Dialog-Klasse muss genauso heissen */
  RegisterClass( &wnd);

  /* Dialogfenster erzeugen und anzeigen */
  hwDialog=CreateDialog( hInstance, "MOTOR1" /*szAppName*/, 0, 0);
  ShowWindow(hwDialog, nCmdShow);

  /* DefaultWerte, falls noch nichts in der Registry steht */
  DrawPos0=0;
  DrawPos255=359;
  MotorInvertPoti=0;
  MotorExchangeLR=0;
  MotorGoal=128;
  
  /* Registry auslesen */
  RegistryRead();		  
  
  /* Werte im Dialogfenster eintragen */
  SetDlgItemInt(hwDialog, IDC_Edit0, DrawPos0, FALSE);
  SetDlgItemInt(hwDialog, IDC_Edit255, DrawPos255, FALSE);
  GetDrawParams(hwDialog); /* Werte normieren */
  SetDlgItemInt(hwDialog, IDC_Pos, 
	  DrawPos0 + MotorGoal*(DrawPos255-DrawPos0)/255,
	  FALSE);
  MotorSetGui(hwDialog, FALSE);
  EnableWindow(GetDlgItem(hwDialog, IDC_Hand_Stop), FALSE);
    
  
  /* Windows Message Queue abarbeiten */
  while ( GetMessage( &msg, NULL,0,0)) 
  {
	 
	 /* IsDialogMessage() sorgt fr das Dialog-Feeling */
	 if( !IsDialogMessage(hwDialog,&msg) ) 
	 {
		 TranslateMessage(&msg);
	 	 DispatchMessage(&msg);
	 }
  }
  return msg.wParam;			
}




/*** 
 * Einstellungen aus der Registry holen
 ***/
void RegistryRead()
{

  HKEY	hReg=INVALID_HANDLE_VALUE; /* Handle zum Lesen der Registry */
  DWORD lpdwDispo;

  
  if( RegCreateKeyEx(HKEY_CURRENT_USER, "\\Software\\c't\\USB Motorsteuerung", 
							0, "", REG_OPTION_NON_VOLATILE , KEY_ALL_ACCESS | KEY_READ | KEY_EXECUTE, NULL, &hReg, &lpdwDispo)  
							== ERROR_SUCCESS )
  {	
	  DWORD lValType=REG_DWORD;
	  DWORD data;	
	  DWORD dataSize; 

	  dataSize = sizeof(data);
	  if (RegQueryValueEx(hReg, "DrawPos0", NULL, &lValType, (BYTE*)&data, &dataSize) 
		  == ERROR_SUCCESS )
	      if(dataSize == sizeof(data)) DrawPos0 = (unsigned)LOWORD(data);	

	  dataSize = sizeof(data);
	  if (RegQueryValueEx(hReg, "DrawPos255", NULL, &lValType, (BYTE*)&data, &dataSize)
		  == ERROR_SUCCESS )
	      if(dataSize == sizeof(data)) DrawPos255 = (unsigned)LOWORD(data);	
		  
	  dataSize = sizeof(data);
	  if (RegQueryValueEx(hReg, "MotorInvertPoti", NULL, &lValType, (BYTE*)&data, &dataSize)
		  == ERROR_SUCCESS )
	      if(dataSize == sizeof(data)) MotorInvertPoti = (unsigned)LOWORD(data);	
	  
	  dataSize = sizeof(data);
	  if (RegQueryValueEx(hReg, "MotorExchangeLR", NULL, &lValType, (BYTE*)&data, &dataSize)
		  == ERROR_SUCCESS )
	      if(dataSize == sizeof(data)) MotorExchangeLR = (unsigned)LOWORD(data);	

	  dataSize = sizeof(data);
	  if (RegQueryValueEx(hReg, "MotorGoal", NULL, &lValType, (BYTE*)&data, &dataSize)
		  == ERROR_SUCCESS )
	      if(dataSize == sizeof(data)) MotorGoal = (unsigned)LOWORD(data);	

	  RegCloseKey(hReg);
  }

}


/***
 * Einstellungen in der Registry sichern 
 ***/
void RegistryWrite()
{
		
	HKEY hReg=INVALID_HANDLE_VALUE; /* Handle des Registry-Keys */
	DWORD lpdwDispo=0;
	

	if( RegCreateKeyEx(HKEY_CURRENT_USER, "\\Software\\c't\\USB Motorsteuerung", 
		0, "", REG_OPTION_NON_VOLATILE , KEY_ALL_ACCESS | KEY_EXECUTE, NULL, &hReg, &lpdwDispo)  
		== ERROR_SUCCESS )
	{	
		
		DWORD data;

		data=DrawPos0;
		RegSetValueEx(hReg, "DrawPos0", 0, REG_DWORD, (BYTE*)&data, sizeof(data) );
		
		data=DrawPos255;
		RegSetValueEx(hReg, "DrawPos255", 0, REG_DWORD, (BYTE*)&data, sizeof(data) );

		data=MotorInvertPoti;
		RegSetValueEx(hReg, "MotorInvertPoti", 0, REG_DWORD, (BYTE*)&data, sizeof(data) );
		
		data=MotorExchangeLR;
		RegSetValueEx(hReg, "MotorExchangeLR", 0, REG_DWORD, (BYTE*)&data, sizeof(data) );

 	    data=MotorGoal;
		RegSetValueEx(hReg, "MotorGoal", 0, REG_DWORD, (BYTE*)&data, sizeof(data) );

	
		RegCloseKey(hReg);
	}

}