/* smd.c: Schrittmotorsteuerung

   Jens Altenburg / c't
*/


/***************************************************************************
* HEADER-FILES (Only those that are needed in this file)                   *
****************************************************************************/
/* System-headerfiles */
#include "cdef.h"                      /* General definitions and types */

/* Own headerfiles */
#include "smd.h"

/* Foreign headerfiles */
#include <stdlib.h>

/***************************************************************************
*    Konstaten                                                             *
****************************************************************************/
const byte abHalfPattern[] = {0x15, 0x31, 0x17, 0x0f, 0x16, 0x32, 0x14, 0x0c};
const byte abFullPattern[] = {0x00, 0x02, 0x03, 0x01, 0x00, 0x02, 0x03, 0x01};

#define nTableSize       7          /* Maske fr Index in Bitmustertabelle */
#define nSmdMask         0x3f
#define mcSetDriver(x)   (P1OUT = (P1OUT & ~nSmdMask) | x)
#define nRampSize        8
#define nSmdLowSpeed     60000          /* niedrige Geschwindigkeit */
#define nSmdHighSpeed    5000           /* Hchstgeschwindigkeit */

/***************************************************************************
*    Variablen                                                             *
****************************************************************************/
byte bSmdState; /*
* Description : Variable fr Zustandsautomat
*/

fCallBackSmd fSmdFunc; /*
* Description : Callbackfunktion fr Benachrichtigung
*/

const byte *pBitMuster; /*
* Description : Index auf Bitmuster
*/

byte bRamp; /*
* Description : Index auf Rampe
*/

dword dwSteps; /*
* Description : Schrittzahl
*/

int8 i8Dir; /*
* Description : Decrement/Increment
*/

byte bSmdCount; /*
* Description : Schrittzhler
*/

word awRamp[nRampSize]; /*
* Description : Beschleunigung/Bremsung
*/


/* Zugriffsmakros fr Merkflags */



/***************************************************************************
* Defines und Makros Zugriff                                               *
****************************************************************************/
enum {
     enSmdInit = 0,                /* tut nix */
     enSmdPrep1,                   /* Start-Stop vorbereiten */
     enSmdPrep2,                   /* Normalmode vorbereiten */
     enSmdStartStop,               /* Start-Stop-Mode */
     enSmdRampUp,                  /* Beschleunigung */
     enSmdMove,                    /* Drehung */
     enSmdRampDown,                /* Bremsen */
     };

#define mcPowerOn()	      (P1OUT &= ~0x40)
#define mcPowerOff()	  (P1OUT |=  0x40)						   


/***************************************************************************
*    Code                                                                  *
****************************************************************************/

/***************************************************************************
*    Funktion: Dummy fr Benachrichtigungsfunktion                         *
*    Autor:    Jens Altenburg                                              *
****************************************************************************/
void vSmdDummy( void ){
      }

/***************************************************************************
*    Funktion: Not-Aus, Motor wird sofort gestoppt                         *
*    Autor:    Jens Altenburg                                              *
****************************************************************************/
void SMD_vCritical( void ){

     }

/***************************************************************************
*    Funktion: Initialisierung                                             *
*                                                                          *
*    Autor:    Jens Altenburg                                              *
****************************************************************************/
void SMD_vInit( void ){
     fSmdFunc = vSmdDummy;                   /* Callbackfunktion initialisieren */
     bSmdState = enSmdInit;
     }


/***************************************************************************
*    Funktion: Kommunikationsinterface zum Schrittmotortreiber             *
*              - Benachritigungsfunktion initialisieren                    *
*              - Bewegungsmode einstellen                                  *
*              - Hochlaufkurve berechnen                                   *
*              - Schrittzahl einstellen                                    *
*    Autor:    Jens Altenburg                                              *
****************************************************************************/
void SMD_vStart ( fCallBackSmd fSmd,    /* Benachrichtigung */
                  byte bTyp,            /* Bewegungstyp, Halbschritt, Vollschritt */
                  word wMax,            /* Maximalgeschwindigkeit */
                  int32 iW              /* Schrittzahl, Vorzeichen = Drehrichtung */
                ){
     word w;
     byte i;
	 fSmdFunc = fSmd;                   /* Benachrichtigungsfunktion */
     if(iW == 0){
	 	   fSmdFunc();
		   return;                		/* Schrittzahl 0 abfangen */
		   }
     if(iW >= 0){                       /* Drehrichtung festlegen */
          dwSteps = iW;
          i8Dir = 1;
          }
     else {
          dwSteps = -iW;
          i8Dir = -1;
          }
     if(bTyp){                          /* Unterscheidung Halb/Vollschritt */
          pBitMuster = &abHalfPattern[0];
          }
     else {
          pBitMuster = &abFullPattern[0];
          }
     /* Hochlauframpe berechnen */
     if(wMax < nSmdHighSpeed) wMax = nSmdHighSpeed; /* Maximalgeschwindigkeit begrenzen */
     if(wMax > (nSmdLowSpeed - nRampSize)){
          wMax = nSmdLowSpeed - nRampSize;
          }
     if(dwSteps < (2 * nRampSize)){     /* nur Start-Stop Geschwindigkeit */
          awRamp[0] = nSmdLowSpeed;
          bSmdState = enSmdPrep1;       /* Bewegung vorbereiten */
          }
     else {
          w = nSmdLowSpeed - wMax;      /* Differenzgeschwindigkeit */
          w = w / nRampSize;            /* Geschwindigkeitsschritte */
		  for(i = 0; i < nRampSize; i++){
               awRamp[i] = nSmdLowSpeed - w * (i + 1);
               }
          dwSteps -= 2 * nRampSize;     /* Schritte mit Hchstgeschwindigkeit */
          bSmdState = enSmdPrep2;       /* Bewegung vorbereiten */
          }
     }


/***************************************************************************
*    Funktion: Interruptservive-Funktion zur Motorsteuerung                *
*                                                                          *
*    Autor:    Jens Altenburg                                              *
****************************************************************************/
word SMD_wMove  ( void ){                    /* Bewegungserzeugung */
     byte i;
     word wTime;
     switch(bSmdState){
          case enSmdInit:                    /* tut nix */
               mcPowerOff();                 /* Haltestrom einstellen */
               wTime = nSmdLowSpeed;         /* Zeitspanne bis zum nchsten Aufruf */
               break;
          case enSmdPrep1:                   /* Start-Stop vorbereiten */
               mcPowerOn();
               wTime = awRamp[0];            /* Zeit aus Rampentabelle */
               bSmdState = enSmdStartStop;
               break;
          case enSmdPrep2:                   /* Normalmode vorbereiten */
               mcPowerOn();
               //P2OUT &= ~0x01;
			   wTime = awRamp[0];
               bRamp = 0;
			   bSmdState = enSmdRampUp;
               break;
          case enSmdStartStop:               /* Start-Stop-Mode */
               i = bSmdCount & nTableSize;   /* Offset fr Zeiger auf Tabelle begrenzen */
               mcSetDriver(*(pBitMuster+i));
               bSmdCount += i8Dir;           /* nchster Schritt */
               dwSteps--;
               if(dwSteps == 0){
			          bSmdState = enSmdInit;
 					  fSmdFunc();
					  } 
               wTime = awRamp[0];            /* Schrittzeit */
               break;
          case enSmdRampUp:                  /* Beschleunigung */
               i = bSmdCount & nTableSize;   /* Zeiger auf Bitmuster berechnen */
               mcSetDriver(*(pBitMuster+i));
               bSmdCount += i8Dir;           /* nchster Schritt */
               bRamp++;
               wTime = awRamp[bRamp];
               if(bRamp >= nRampSize - 1) bSmdState = enSmdMove;
               break;
          case enSmdMove:                    /* Drehung */
               i = bSmdCount & nTableSize;   /* Zeiger auf Bitmuster berechnen */
               mcSetDriver(*(pBitMuster+i));
               bSmdCount += i8Dir;           /* nchster Schritt */
               dwSteps--;
               if(dwSteps == 0) bSmdState = enSmdRampDown;
               wTime = awRamp[nRampSize - 1];
               break;
          case enSmdRampDown:                /* Bremsen */
               i = bSmdCount & nTableSize;   /* Zeiger auf Bitmuster berechnen */
               mcSetDriver(*(pBitMuster+i));
               bSmdCount += i8Dir;           /* nchster Schritt */
               bRamp--;
               wTime = awRamp[bRamp];
               if(bRamp == 0){
                    bSmdState = enSmdInit;
                    fSmdFunc();              /* Motorbewegung ist beendet */
                    //P2OUT |= 0x01;
					}
               break;
          }
     return wTime;
     }


