#import <Foundation/Foundation.h>
#import "BlueMP3Object.h"
#import <IOBluetooth/objc/IOBluetoothDevice.h>
#import <IOBluetooth/objc/IOBluetoothSDPUUID.h>
#import <IOBluetoothUI/objc/IOBluetoothDeviceSelectorController.h>
#import <IOBluetooth/IOBluetoothUtilities.h>

unsigned char gBlueMP3UUID[] =	//50B2FO9B-9B75-45DC-A54B-15F15E3B4CE7
{
    0x50, 0xb2, 0xf0, 0x9b,
    0x9b, 0x75,
    0x45, 0xdc,
    0xa5, 0x4b,
    0x15, 0xf1, 0x5e, 0x3b, 0x4c, 0xe7
};

/*
 Sample code to generate UUID:
 =============================

#include <CoreFoundation/CoreFoundation.h>

 int main()
 {
     CFUUIDRef       uuid;
     CFStringRef     string;

     uuid = CFUUIDCreate( NULL );
     string = CFUUIDCreateString( NULL, uuid );

     CFShow( string );
 }
 */

@implementation BlueMP3Object

- (id)init
{
    self = [super init];
    if (self == nil)
        return nil;

    blueMP3Volume = 30;
    loopMode = FALSE;

    return self;
}

- initWithVolume:(char)value andLoopMode:(BOOL)loop
{
    self = [self init];
    if ( self != nil )
    {
        blueMP3Volume = value;
        loopMode = loop;
    }

    return self;
}

- (void)dealloc
{
    [asCallbackScript release];
    [super dealloc];
}

- (BOOL)selectBlueMP3
{
    IOBluetoothDeviceSelectorController	*deviceSelector;
    IOBluetoothSDPUUID			*BlueMP3UUID;
    NSArray				*deviceArray;
    IOBluetoothDevice			*selectedDevice;
    NSUserDefaults			*thePrefs;

    // Dialog zur Bluetooth-Gerteauswahl generieren
    deviceSelector = [IOBluetoothDeviceSelectorController deviceSelector];
    if ( deviceSelector == nil )
    {
	NSLog( @"Fehler: kein IOBluetoothDeviceSelectorController\n" );
	goto exit;
    }

    // nur Gerte mit der UUID des BlueMP3 akzeptieren
    BlueMP3UUID = [IOBluetoothSDPUUID uuidWithBytes:gBlueMP3UUID length:16];
    [deviceSelector addAllowedUUID:BlueMP3UUID];

    // Bluetooth-Gerteauswahl anzeigen
    if ( (int)[deviceSelector runModal] != (int)kIOBluetoothUISuccess )
    {
	NSLog( @"Fehler: Anwender hat Gerteauswahl abgebrochen\n" );
	goto exit;
    }

    // Gerteauswahl ermitteln
    deviceArray = [deviceSelector getResults];
    if ( ( deviceArray == nil ) || ( [deviceArray count] == 0 ) )
    {
	NSLog( @"Fehler bei der Bluetooth-Gerteauswahl (eigentlich unmglich)\n" );
	goto exit;
    }

    // Das erste Gert ist "unseres". 
    selectedDevice = [deviceArray objectAtIndex:0];

    // MAC-Adresse in den Preferences ablegen
    thePrefs = [NSUserDefaults standardUserDefaults];
    [thePrefs setObject:[selectedDevice getAddressString] forKey:@"lastBlueMP3"];

    // Einstellungen jetzt auf Festplatte schreiben
    [thePrefs synchronize];

    return TRUE;

exit:
    return FALSE;
}

- (BOOL)connectToBlueMP3
{
    BOOL			returnValue = FALSE;
    IOBluetoothDevice		*selectedDevice;
    IOReturn			status;
    NSUserDefaults		*thePrefs;
    NSString			*lastBlueMP3;
    BluetoothDeviceAddress	btAddress;

    // Ist schon ein BlueMP3 in den Einstellungen gespeichert?
    thePrefs = [NSUserDefaults standardUserDefaults];
    lastBlueMP3 = [thePrefs objectForKey:@"lastBlueMP3"];

    // Nein, dann Auswahldialog aufrufen
    if ( [lastBlueMP3 length] == 0 )
    {
	if ( [self selectBlueMP3] )
	    lastBlueMP3 = [thePrefs objectForKey:@"lastBlueMP3"];
	else
	    goto exit;
    }

    // Gibt es jetzt einen Verweis auf einen BlueMP3?
    if ( [lastBlueMP3 length] > 0 )
    {
        // String der MAC-Adresse in binre Adresse umwandeln ...
	status = IOBluetoothNSStringToDeviceAddress(lastBlueMP3, &btAddress);
        if ( status != kIOReturnSuccess )
        {
            NSLog(@"Fehler (0x%lx) beim Umwandeln der MAC-Adresse\n", status );
            goto exit;
        }

        // ... und daraus einen "echtes" BluetoothDevice machen
	selectedDevice = [IOBluetoothDevice withAddress:&btAddress];
    }

    NSLog( @"Device selected: '%@'\n", [selectedDevice getNameOrAddress]);

    // Verbindung mit dem BlueMP3 aufnehmen
    status = [selectedDevice openConnection];
    if ( status != kIOReturnSuccess )
    {
        NSLog( @"Fehler (0x%lx) beim ffnen der Verbindung\n", status );
        goto exit;
    }

    // L2CAP-Kanal mit MP3_PSM=11 ffnen
    status = (IOReturn)[selectedDevice openL2CAPChannelSync:&mL2CAPChannel withPSM: MP3_PSM delegate:self];
    if ( ( status == kIOReturnSuccess ) && ( mL2CAPChannel != nil ) )
    {
	// MTU-Gre als maximale Sendepuffergre bestimmen
	channelMTU = [mL2CAPChannel getOutgoingMTU];

        // Kanalreferenz vor dem Lschen schtzen
        [mL2CAPChannel retain];

        // alle Meldungen bezglich des Kanals an dieses Objekt schicken
	[mL2CAPChannel setDelegate: self];

        returnValue = TRUE;
    }
    else
    {
        NSLog( @"Fehler (0x%lx) beim ffnen des L2CAP-Kanals\n", status );
        goto exit;
    }

exit:
    return returnValue;
}

- (BOOL)disconnectFromBlueMP3
{
    if ( mL2CAPChannel != nil )
    {
        // Gert unseres Kanals ermitteln
	IOBluetoothDevice *device = [mL2CAPChannel getDevice];

        // Kanal schlieen
	[mL2CAPChannel closeChannel];

        // die zugehrigen Objekte freigeben
        [mL2CAPChannel release];
        mL2CAPChannel = nil;

        // Verbindung zum BlueMP3 abbrechen
        [device closeConnection];

        return TRUE;
    }
    else
	return FALSE;
}

- (BOOL)isConnected
{
    return ( mL2CAPChannel != nil );
}

- (BOOL)startPlaying:(NSArray *)songs
{
    // Grnes Licht fr den Thread geben
    continuePlaying = TRUE;

    // Transferroutine als eigenen Thread abkoppeln
    [NSThread detachNewThreadSelector:@selector(playThread:) toTarget:self withObject:songs];

    // Alles klar bis hierher?
    return TRUE;
}

- (void)playThread:(NSArray *)songs
{
    NSString		*oneSong;
    NSAutoreleasePool	*pool;
    NSData		*data;
    unsigned char	*buffer;
    unsigned int	i, len, count;

    // in einem Thread muss ein AutoreleasePool angelegt werden
    pool = [[NSAutoreleasePool alloc] init];

    // Anzahl der zu bertragenden Songs bestimmen
    count = [songs count];

    // Jetzt gehts (gleich) los!
    isPlaying = TRUE;

    // NSArrays sind 0-basiert.
    i = 0;
    while ( (continuePlaying) && (i < count) )
    {
        // Pfad der MP3-Datei holen und ...
	oneSong = [songs objectAtIndex:i];

	// ... merken
	currentSong = [NSString stringWithString:oneSong];
	currentSongIndex = i;

	// Song einlesen, Lnge bestimmen und ...
	data = [NSData dataWithContentsOfFile:oneSong];
	len = [data length] + 1;	// wegen des 0-Bytes am Anfang eines Datenblocks
	
	// ... in einen Puffer schreiben
	buffer = malloc(len);

        // erstes Byte fr MP3_DATA-Byte frei lassen
        [data getBytes:buffer + 1];

	// Puffer bertragen und ...
	if ( [self sendMP3Data:buffer + 1 length:len] != TRUE )
	{
	    // Irgendetwas lief schief, eine Fehlerbehandlung
	    // an dieser Stelle wre deshalb nicht schlecht.
	}
	
	// ... anschlieend freigeben
	free(buffer);

	// Wurde zwischenzeitlich die NEXT-Taste am BlueMP3, der Next- oder
        // Prev-Button der Software gedrckt?
	if ( internalCmd == CMD_NEXT )
	{
	    // fr den Anfang der while-Schleife
	    continuePlaying = TRUE;
	    internalCmd = NO_CMD;
            if ( i == count - 1 )
                i = -1;
	}
        else if ( internalCmd == CMD_PREV )
        {
            continuePlaying = TRUE;
            internalCmd = NO_CMD;
            i = ( ( i == 0 ) ? count - 2 :  i - 2 );
        }

        // Index erhhen
        i++;

	// Am Ende wieder von vorne?
	if ( loopMode && (i == count) )
	    i = 0;
    }

    // Momentan spielt nichts.
    isPlaying = FALSE;

    // den AutoreleasePool freigeben
    [pool release];
}

- (BOOL)sendMP3Data:(unsigned char *)buffer length:(UInt32)length
{
    UInt32	numBytesRemaining, numBytesToSend;
    IOReturn	result;

    if ( mL2CAPChannel != nil )
    {
	// vor jedem Titel BlueMP3 resetten und Lautstrke auf gespeicherten Wert setzen
	[self resetBlueMP3:FALSE];
	[self setVolume:blueMP3Volume];

	numBytesRemaining = length;
	result = kIOReturnSuccess;

	while ( ( result == kIOReturnSuccess ) && ( numBytesRemaining > 0 ) && continuePlaying )
	{
	    // Pause? Dann im Sekundentakt nachschauen, ob internalCmd noch Pause anzeigt.
	    while ( internalCmd == CMD_PAUSE)
	    {
		sleep(1);
	    }

            // sendMP3Data wurde mit buffer+1 aufgerufen
            buffer--;

            // Block als MP3-Daten kennzeichnen
            buffer[0] = MP3_DATA;

            // Anzahl der zu bertragenden Bytes bestimmen, mehr als channelMTU geht nicht pro Durchgang
	    numBytesToSend = ( ( numBytesRemaining > channelMTU ) ? channelMTU :  numBytesRemaining );

	    // Daten synchron an BlueMP3 bertragen, d. h. es geht erst nach der bertragung weiter
	    // Asynchrone bertragung bringt nix, da Funktion sowieso in einem separatem Thread luft
	    result = (IOReturn)[mL2CAPChannel writeSync:buffer length:numBytesToSend];

	    // Puffer-Ptr und Lngenmerker aktualisieren
	    numBytesRemaining -= numBytesToSend;
	    buffer += numBytesToSend;
	}

        // Alles erfolgreich bertragen? 
        if ( ( numBytesRemaining == 0 ) && ( result == kIOReturnSuccess ) )
	{
            return TRUE;
	}
    }

    // Im Fehlerfall
    return FALSE;
}

- (void)pausePlaying
{
    // wird in der Senderoutine abgefangen
    internalCmd = ( internalCmd == CMD_PAUSE ) ? NO_CMD : CMD_PAUSE;
}

- (void)stopPlaying
{
    // dem Thread ber Variable Bescheid sagen
    continuePlaying = FALSE;
    isPlaying = FALSE;
}

- (void)previousSong
{
    // wird im playThread: abgefangen
    continuePlaying = FALSE;
    internalCmd = CMD_PREV;
}

- (void)nextSong
{
    // wird im playThread: abgefangen
    continuePlaying = FALSE;
    internalCmd = CMD_NEXT;
}

- (void)setLoopMode:(BOOL)mode
{
    // schaltet den Endlosmodus an
    loopMode = mode;
}

- (BOOL)getLoopMode
{
    return loopMode;
}

- (BOOL)isPlaying
{
    // steht auf TRUE, wenn noch Lieder zu bertragen sind
    return isPlaying;
}

- (NSString *)songBeingPlayed
{
    return [NSString stringWithString:currentSong];
}

- (unsigned int)songBeingPlayedByIndex
{
    return (currentSongIndex + 1);
}

- (void)resetBlueMP3:(BOOL)hardReset
{
    // Puffer mit Daten fllen: hardReset = FALSE bedeutet Soft-Reset
    unsigned char buffer[2] = { MP3_CMD_RESET, hardReset };

    IOReturn result = (IOReturn)[mL2CAPChannel writeSync:buffer length:2];
}

- (void)setVolumeOfLeftChannel:(char)left ofRightChannel:(char)right
{
    // Puffer mit Daten fllen
    unsigned char buffer[3] = { MP3_CMD_VOLUME, 100 - left, 100 - right };

    IOReturn result = (IOReturn)[mL2CAPChannel writeSync:buffer length:3];
}

- (void)setVolume:(char)value
{
    // beide Kanle auf gleiche Lautstrke einstellen
    [self setVolumeOfLeftChannel:value ofRightChannel:value];
    blueMP3Volume = (unsigned int)value;
    [self informAppleScriptAbout:@"volumeChanged" withData:(int)value];
}

- (char)getVolume
{
    return (char)blueMP3Volume;
}

- (void)volumeFadeIn
{
    int tmpVol = 0;

    // Lautstrke schrittweise auf ursprngliches Level erhhen
    while ( tmpVol < blueMP3Volume )
    {
	tmpVol++;
	[self setVolume:tmpVol];
    }
}

- (void)volumeFadeOut
{
    int tmpVol = blueMP3Volume;

    // Lautstrke schrittweise auf 0 reduzieren
    while ( tmpVol > 0 )
    {
        tmpVol--;
        [self setVolume:tmpVol];
    }
}

#include <Carbon/Carbon.h>

- (void)setAppleScriptCallback:(NSString *)handler inScript:(NSString *)script
{
    NSDictionary	*errScript;
    NSString 		*compiledScript;
    NSURL		*scriptURL;

    if ( (handler != nil) && (script != nil) )
    {
        // Namen des AppleScript-Handler (zwingend in Kleinbuchstaben) merken
        asCallbackHandler = [[NSString stringWithString:handler] retain];

        // Pfad auf Skriptdatei im BlueMacP3-Package suchen
	compiledScript = [[NSBundle mainBundle] pathForResource:script ofType:@"scpt" inDirectory:@"Scripts"];

        // in URL umwandeln, Skript kann schlielich auch im Netz sein
	scriptURL = [NSURL fileURLWithPath:compiledScript];

	// fr Fehler ein Sammler anlegen
        errScript = [NSDictionary dictionary];

	// ein AppleScript erschaffen
        asCallbackScript = [[NSAppleScript alloc] initWithContentsOfURL:scriptURL error:&errScript];
    }
}

- (void)informAppleScriptAbout:(NSString *)msg withData:(int)data
{
    NSAppleEventDescriptor	*targetAddress;
    NSAppleEventDescriptor	*subroutineDescriptor;
    NSAppleEventDescriptor	*arguments;
    NSAppleEventDescriptor	*event;
    NSAppleEventDescriptor	*result;
    NSDictionary		*errExec = [NSDictionary dictionary];
    int 			pid;

    if ( asCallbackScript != nil )
    {
        pid = [[NSProcessInfo processInfo] processIdentifier];

        // "Interne AppleScript-Adresse" fr unser Programm bestimmen
        targetAddress = [NSAppleEventDescriptor descriptorWithDescriptorType:typeKernelProcessID bytes:&pid length:sizeof(pid)];

	// Dieses AppleEvent geht an unser Programm.
        event = [NSAppleEventDescriptor appleEventWithEventClass:kASAppleScriptSuite
                                        eventID:kASSubroutineEvent
                                        targetDescriptor:targetAddress
                                        returnID:kAutoGenerateReturnID
                                        transactionID:kAnyTransactionID];

        // AppleScript-kompatible Beschreibung fr Callback-Routine ...
        subroutineDescriptor = [NSAppleEventDescriptor descriptorWithString:asCallbackHandler];

        // ... and das Callback-Event anhngen
	[event setParamDescriptor:subroutineDescriptor forKeyword:keyASSubroutineName];

        // Variablenliste fr Event anlegen
        arguments = [NSAppleEventDescriptor listDescriptor];

        // String ist 1. Parameter in der Liste
        [arguments insertDescriptor:[NSAppleEventDescriptor descriptorWithString:msg] atIndex:0];

        // Integer ist 2. Parameter in der Liste
        [arguments insertDescriptor:[NSAppleEventDescriptor descriptorWithInt32:(SInt32)data] atIndex:0];

        // Parameter an das Event anhngen
        [event setParamDescriptor:arguments forKeyword:keyDirectObject];
        
        // AppleScript-Handler aufrufen
        result = [asCallbackScript executeAppleEvent:event error:&errExec];
    }
}

// Der Bluetooth-Stack ruft diese Routine auf, wenn der BlueMP3 Daten ber den L2CAP-Kanal sendet.
- (void)l2capChannelData:(IOBluetoothL2CAPChannel*)l2capChannel data:(void *)dataPtr length:(size_t)dataLength
{
    switch ( ((unsigned char *)dataPtr)[0] )
    {
	// Uns interessieren nur Daten zu den Tasten des BlueMP3.
	case MP3_CMD_BUTTONS:
	    if ( ((unsigned char *)dataPtr)[1] & 1 )
	    {
		// Lautstrke um 5 Punkte erhhen
		blueMP3Volume = ( ( blueMP3Volume < 95 ) ? blueMP3Volume += 5 :  100 );
		[self setVolume:blueMP3Volume];
	    }

	    if ( ((unsigned char *)dataPtr)[1] & 2 )
	    {
		// zum nchsten Titel in der zum Abspielen vorgesehenen Liste springen
		[self nextSong];
	    }

	    if ( ((unsigned char *)dataPtr)[1] & 4 )
	    {
		// Lautstrke um 5 Punkte reduzieren
		blueMP3Volume = ( ( blueMP3Volume > 5 ) ? blueMP3Volume -= 5 :  0 );
		[self setVolume:blueMP3Volume];
	    }

	    if ( ((unsigned char *)dataPtr)[1] & 8 )
	    {
		if ( batteryLow != TRUE )
                {
                    batteryLow = TRUE;
                    [self informAppleScriptAbout:@"batteryIsLow" withData:0];
                }
	    }

	    break;
	default:
	    NSLog( @"Fehler: Unbekannter Tasten-Code (0x%hx) vom BlueMP3\n", ((unsigned char *)dataPtr)[0] );
	    break;
    }
}

- (void)l2capChannelClosed:(IOBluetoothL2CAPChannel*)l2capChannel
{
    isPlaying = FALSE;
    continuePlaying = FALSE;
    (BOOL)[self disconnectFromBlueMP3];
}

@end