import java.io.*;
import java.util.*;
import javax.swing.*;

public class TKPDManager {
    
    private Properties config;
    private Vector repository;
    private String propertiesFilename, repositoryFilename;
    private Hashtable typeManagers;
    
    public static final String USER_HOME = System.getProperty("user.home");
    public static final String FILE_SEPARATOR = System.getProperty("file.separator");
    
    public static final String PROPERTIES_HEADER = "TKPDManager.properties by Th. Knneth";
    public static final String DEFAULT_TM = "Notiz,OutlookNotiz";
    public static final String CHOOSE_DIR = "Verzeichnis fr Eintrge des Typs ";
    public static final String FATAL_ERROR = "Fataler Fehler";
    public static final String EXCEPTION = "Exception aufgetreten";
    public static final String INFO_CHOOSE_DIR = "Sie mssen ein Verzeichnis auf dem tragbaren Gert angeben,\n in dem Bilddateien fr ein Modul abgelegt werden";
    
    public TKPDManager() {
        /*
         * falls vorhanden, laden der Datei TKPDManager.properties im
         * Heimatverzeichnis des Benutzers; der Zugriff auf alle
         * Programmeinstellungen erfolgt ber die Variable config
         * durch die Methode getProperty()
         */
        propertiesFilename = USER_HOME + FILE_SEPARATOR + "TKPDManager.properties";
        loadProperties(propertiesFilename);
        
        /*
         * laden aller Module (Notiz, Aufgabe, ...); Referenzen auf sie
         * werden in der Hashtable typeManagers abgelegt, auf die ber
         * den Modulnamen (z. B. Notiz) zugegriffen wird
         */
        loadTypeManagers();
        
        /*
         * im sog. repository werden alle dem System bekannten Eintrge
         * eingetragen; auf diese Weise kann festgestellt werden, falls
         * auf dem tragbaren Gert eine Datei gelscht wird; in der
         * Methode load findet der komplette Datenaustausch statt
         */
        repository = new Vector();
        repositoryFilename = USER_HOME + FILE_SEPARATOR + "TKPDManager.repository";
        load(repositoryFilename);
        
        /*
         * vor dem Programmende alle nderungen am repository und
         * and Programmeinstellungen speichern
         */
        saveProperties(propertiesFilename);
        save(repositoryFilename);
        System.exit(0);
    }
    
    /*
     * laden der Konfigurationsdatei
     */
    private void loadProperties(String filename) {
        config = new Properties();
        FileInputStream is = null;
        try {
            is = new FileInputStream(filename);
            config.load(is);
        } catch (IOException e) {
        } finally {
            if (is != null) {
                try {
                    is.close();
                } catch (IOException e) {
                }
            }
        }
    }
    
    /*
     * speichern der Konfigurationsdatei
     */
    private void saveProperties(String filename) {
        FileOutputStream os = null;
        try {
            os = new FileOutputStream(filename);
            config.store(os, PROPERTIES_HEADER);
        } catch (IOException e) {
        } finally {
            if (os != null) {
                try {
                    os.close();
                } catch (IOException e) {
                }
            }
        }
    }
    
    /*
     * falls in TKPDManager.properties ein bestimmter Schlssel
     * nicht gefunden wird, wird er explizit mit einem
     * Default-Wert eingetragen; die Datei wird vor dem
     * Programmende geschrieben
     */
    private String getProperty(String key, String defaultValue) {
        if (config.containsKey(key) == false)
            config.setProperty(key, defaultValue);
        return config.getProperty(key);
    }
    
    /*
     * diese Methode ld alle Module; diese mssen das Interface
     * TypeManager implementieren
     */
    private void loadTypeManagers() {
        typeManagers = new Hashtable();
        StringTokenizer st = new StringTokenizer(getProperty("TYPEMANAGERS", DEFAULT_TM), ",");
        String type;
        Class c;
        Object o;
        TypeManager tm;
        while (st.hasMoreTokens()) {
            type = st.nextToken();
            try {
                c = Class.forName(type);
                o = c.newInstance();
                if (o instanceof TypeManager)
                    typeManagers.put(type, o);
            } catch (IllegalAccessException e) {
                showExceptionDialog(e);
            } catch (ClassNotFoundException e) {
                showExceptionDialog(e);
            } catch (InstantiationException e) {
                showExceptionDialog(e);
            }
        }
    }
    
    /*
     * in dieser Methode wird das repository zeilenweise gelesen
     * und ausgewertet; sie wickelt den kompletten Datenaustausch
     * ab
     */
    private void load(String filename) {
        FileReader fr = null;
        BufferedReader br = null;
        String line;
        Entry entry;
        boolean addToVector;
        try {
            fr = new FileReader(filename);
            br = new BufferedReader(fr);
            while ((line = br.readLine()) != null) {
                addToVector = true;
                entry = new Entry(line);
                /*
                 * wenn die Datei auf dem Gert ist,
                 * prfen, ob sie gelscht wurde
                 */
                if (entry.getStatus().equals(Entry.STATUS_DEVICE))
                    if (checkIfPresent(entry) == false)
                        addToVector = fileDeleted(entry);
                if (addToVector == true)
                    repository.addElement(entry);
            }
        } catch (NoSuchElementException e) {
            // falls die eingelesene Zeile nicht dem erwarteten Format entspricht
        } catch (IOException e) {
        } finally {
            if (br != null) {
                try {
                    br.close();
                } catch (IOException e) {
                }
            }
            if (fr != null) {
                try {
                    fr.close();
                } catch (IOException e) {
                }
            }
            getNewEntries();
            moveNewFiles();
        }
    }
    
    /*
     * speichern des repository
     */
    private void save(String filename) {
        FileWriter fw = null;
        BufferedWriter bw = null;
        try {
            fw = new FileWriter(filename);
            bw = new BufferedWriter(fw);
            for (int i = 0; i < repository.size(); i++) {
                bw.write(repository.elementAt(i).toString());
                bw.newLine();
            }
        } catch (IOException e) {
        } finally {
            if (bw != null) {
                try {
                    bw.close();
                } catch (IOException e) {
                }
            }
            if (fw != null) {
                try {
                    fw.close();
                } catch (IOException e) {
                }
            }
        }
    }
    
    /*
     * anhand des Typs das Verzeichnis auf dem Gert ermitteln
     */
    private String getBaseDirForType(String type) {
        String key = "PATH_" + type;
        String result = config.getProperty(key);
        if (result == null) {
            result = chooseDirectory(type);
            if (result == null)
                fatalError(INFO_CHOOSE_DIR);
            config.setProperty(key, result);
        }
        return result;
    }
    
    /*
     * Prfung, ob die zu einem Eintrag gehrende Bilddatei auf
     * dem tragbaren Gert noch vorhanden ist
     */
    private boolean checkIfPresent(Entry e) {
        String dir = getBaseDirForType(e.getType());
        File f = new File(dir, e.getFilename());
        boolean isPresent = f.exists();
        if (isPresent == false)
            e.setStatus(Entry.STATUS_DELETED);
        return isPresent;
    }
    
    /*
     * eine Hilfsmethode, die einen leichten Zugriff auf
     * ein bestimmtes Modul gestattet
     */
    private TypeManager getTypeManager(String type) {
        return (TypeManager) typeManagers.get(type);
    }
    
    /*
     * ermitteln, welches Modul fr den Eintrag zustndig ist
     * und es darber informieren, dass die zum Eintrag
     * gehrende Bilddatei gelscht wurde
     */
    private boolean fileDeleted(Entry e) {
        TypeManager tm = getTypeManager(e.getType());
        boolean addToVector = true;
        if (tm != null)
            addToVector = tm.fileDeleted(e);
        return addToVector;
    }
    
    /*
     * alle Eintrge eines bestimmten Typs ermitteln
     */
    private Vector getEntries(String type) {
        Vector entries = new Vector();
        Entry e;
        for (int i = 0; i < repository.size(); i++) {
            e = (Entry) repository.elementAt(i);
            if (e.getType().equals(type))
                entries.addElement(e);
        }
        return entries;
    }
    
    /*
     * bei allen Modulen fragen, ob sich an ihrem
     * Datenbestand nderungen ergeben haben
     */
    private void getNewEntries() {
        String type;
        TypeManager tm;
        Vector entries;
        for (Enumeration e = typeManagers.keys(); e.hasMoreElements(); ) {
            type = (String) e.nextElement();
            tm = getTypeManager(type);
            entries = getEntries(type);
            entries = tm.getNewEntries(this, entries);
            for (int i = 0; i < entries.size(); i++) {
                Entry entry = (Entry) entries.elementAt(i);
                File f = new File(getDeviceFilename(entry));
                f.delete();
                repository.removeElement(entry);
            }
        }
    }
    
    /*
     * wird von Modulen aufgerufen, um dem repository einen neuen
     * Eintrag hinzuzufgen
     */
    public void addEntry(TKPDCanvas canvas, TypeManager caller, String name, String intern) {
        String type = caller.getClass().getName();
        String filename = name + ".jpg";
        Entry e = new Entry(filename + "," + type + "," + Entry.STATUS_LOCAL + "," + intern);
        canvas.save(getLocalFilename(e));
        repository.addElement(e);
    }
    
    /*
     * ermittelt den kompletten Pfadnamen fr
     * die Speicherung lokal am PC
     */
    private String getLocalFilename(Entry e) {
        return getProperty("LOCALDIR", USER_HOME) + FILE_SEPARATOR + e.getType() + "_" + e.getFilename();
    }
    
    /*
     * ermittelt den kompletten Pfadnamen fr
     * die Speicherung auf dem tragbaren Gert
     */
    private String getDeviceFilename(Entry e) {
        return getBaseDirForType(e.getType()) + FILE_SEPARATOR + e.getFilename();
    }
    
    /*
     * diese Methode ermittelt alle Eintrge, die noch nicht auf das
     * tragbare Gert verschoben wurden und veranlasst dies
     */
    private void moveNewFiles() {
        Entry e;
        for (int i = 0; i < repository.size(); i++) {
            e = (Entry) repository.elementAt(i);
            if (e.getStatus().equals(Entry.STATUS_LOCAL)) {
                if (moveTo(getLocalFilename(e), getDeviceFilename(e)) == true)
                    e.setStatus(Entry.STATUS_DEVICE);
            }
        }
    }
    
    /*
     * bringt eine Dateiauswahlbox auf den Bildschirm;
     * es knnen nur Verzeichnisse ausgewhlt werden
     */
    private String chooseDirectory(String type) {
        JFileChooser fc = new JFileChooser();
        fc.setDialogTitle(CHOOSE_DIR + type);
        fc.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
        int result = fc.showOpenDialog(null);
        if (result == JFileChooser.APPROVE_OPTION)
            return fc.getSelectedFile().getAbsolutePath();
        return null;
    }
    
    /*
     * diese Methode verschiebt die Bilddateien auf das
     * tragbare Gert
     */
    private boolean moveTo(String src, String dst) {
        File fSrc, fDst;
        fSrc = new File(src);
        boolean result = false;
        fDst = new File(dst);
        FileInputStream fin = null;
        FileOutputStream fout = null;
        long srcLen = fSrc.length();
        long totalRead;
        int bytesRead;
        byte [] buffer = new byte [512];
        try {
            fin = new FileInputStream(fSrc);
            fout = new FileOutputStream(fDst);
            totalRead = 0;
            for (;;) {
                bytesRead = fin.read(buffer);
                if (bytesRead == -1)
                    break;
                totalRead += bytesRead;
                fout.write(buffer, 0, bytesRead);
            }
            result = true;
        } catch (IOException e) {
            showExceptionDialog(e);
        } finally {
            if (fin != null)
                try {
                    fin.close();
                } catch (IOException e) {
                }
            if (fout != null)
                try {
                    fout.close();
                } catch (IOException e) {
                }
            // wenn die Datei vollstndig kopiert wurde, Quelldatei lschen
            if (result == true)
                fSrc.delete();
        }
        return result;
    }
    
    /**
     * diese Hilfsmethode trgt ein Zeichen in einer Ersatzdarstellung
     * in ein char Feld ein
     */
    private static int insertSemiVowel(char ch, char [] chars, int pos) {
        int max = chars.length;
        String replacement = " ";
        if (ch == '')
            replacement = "ae";
        else if (ch == '')
            replacement = "oe";
        else if (ch == '')
            replacement = "ue";
        else if (ch == '')
            replacement = "ss";
        for (int i = 0; i < replacement.length(); i++) {
            if (pos == max)
                break;
            chars[pos++] = replacement.charAt(i);
        }
        return pos;
    }
    
    /**
     * generiert aus einem bergebenen Text einen
     * Dateinamen
     */
    public String createFilenameFromText(String text) {
        int len = Integer.parseInt(getProperty("MAX_FILENAME_LENGTH", "15"));
        char ch;
        int pos;
        char [] chars = new char [len];
        text = text.toLowerCase();
        pos = 0;
        for (int i = 0; i < text.length(); i++) {
            ch = text.charAt(i);
            if (((ch >= 'a') && (ch <= 'z')) || ((ch >= '0') && (ch <= '9')))
                chars[pos++] = ch;
            else
                pos = insertSemiVowel(ch, chars, pos);
            if (pos == len)
                break;
        }
        return new String(chars, 0, pos);
    }
    
    /*
     * erzeugt ein Bild in der fr das tragbare Gert
     * passenden Gre
     */
    public TKPDCanvas createTKPDCanvas(int lines) {
        int w = Integer.parseInt(getProperty("IMAGE_WIDTH", "660"));
        int h = Integer.parseInt(getProperty("IMAGE_HEIGHT", "468"));
        return new TKPDCanvas(w, h, lines);
    }
    
    /*
     * wenn eine Exception aufgetreten ist,
     * lt sich mit dieser Methode leicht
     * ein entsprechender Hinweis ausgeben;
     * Achtung: das Programm wird NICHT
     * beendet
     */
    public static void showExceptionDialog(Exception e) {
        String info = e.getClass().getName();
        String msg = e.getMessage();
        if (msg != null)
            info += '\n' + msg;
        JOptionPane.showMessageDialog(null, info, EXCEPTION, JOptionPane.ERROR_MESSAGE);
    }
    
    /*
     * kann aufgerufen werden, wenn eine
     * weitere Programmausfhrung nicht mglich ist;
     * es werden KEINE Dateien gespeichert
     */
    public static void fatalError(String msg) {
        JOptionPane.showMessageDialog(null, msg, FATAL_ERROR, JOptionPane.ERROR_MESSAGE);
        System.exit(1);
    }
    
    /*
     * hier beginnt die Programmausfhrung
     */
    public static void main(String [] args) {
        new TKPDManager();
    }
}
