/**
 * Copyright 2005 The Apache Software Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;

import org.apache.lucene.document.DateField;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;

/**
 * Einfache Dokumentenklasse um ein Lucene-Dokument aus einer Text-Datei zu erzeugen.
 * 
 * @author Bernhard Messer
 * @version $rcs = ' $Id: PlainTextDocument.java,v 1.6 2005/03/12 23:38:04 dnaber Exp $ ' ;
 */
public class PlainTextDocument {

  /**
   * Statische Methode zum Erzeugen eines Lucene Dokumentes aus
   * einer einfachen Text-Datei.
   * 
   * @param file  die Text-Datei die verarbeitet werden soll.
   * @return org.apache.lucene.document.Document das Lucene-Dokument mit Inhalt.
   *  
   * @throws java.io.IOException
   */
  public static Document getDocument(File file) throws IOException {
    
    // Leeres Lucene-Dokument anlegen 
    Document document = new Document();

    // Der Dateiname wird im Feld "url" gespeichert. Das Feld wird nicht indexiert,
    // aber im Index gespeichert. Es ist also nicht durchsuchbar, kann aber später
    // zur Anzeige aus dem Index geholt werden.
    document.add(new Field("url",
                           file.getAbsolutePath().replace(File.separatorChar, '/'),
                           true, false, false));

    // Das Dateidatum wird im Feld "modified" abgelegt. Das Feld wird indexiert und
    // gespeichert und ist später als Ganzes suchbar. Der eingetragene Feldinhalt wird
    // jedoch nicht "tokenized" und in seine Bestandteile zerlegt.
    document.add(new Field("modified",
                           DateField.timeToString(file.lastModified()),
                           true, true, false));

    // Erstellen einer eindeutigen Dokumenten-ID anhand des Dateinamens
    // und des Datums der letzten Änderung.
    String uid = file.getPath().replace(File.separatorChar, '\u0000')
        + "\u0000" + DateField.timeToString(file.lastModified());

    // Die uid wird als eigenes Feld eingefügt um den Index später inkrementell
    // aktualisieren zu können (nicht Teil dieser Demo, siehe offizielle Lucene-Demo).
    // Das Feld wird nicht im Index gespeichert, es wird indexiert, aber vor der
    // Indexierung nicht zerlegt (tokenized)
    document.add(new Field("uid", uid, false, true, false));

    // Der Inhalt des Dokumentes wird als Reader-Wert an das Feld angehängt.
    document.add(Field.Text("contents", new FileReader(file)));

    // Die ersten 500 Zeichen werden zusätzlich gespeichert, damit sie angezeigt werden
    // können.
    FileReader fr = null;
    BufferedReader br = null;
    try {
      fr = new FileReader(file);
      br = new BufferedReader(fr);
      StringBuffer sb = new StringBuffer();
      String line;
      while((line = br.readLine()) != null) {
        sb.append(line);
        if (sb.length() > 500)
          break;
      }
      int summarySize = Math.min(sb.length(), 500);
      document.add(new Field("summary", sb.toString().substring(0, summarySize),
          true, false, false));
    } finally {
      if (br != null) br.close();
      if (fr != null) fr.close();
    }

    // Document zurückgeben
    return document;
  }

}
