#include "ExpenseModel.h"
#include "ExchangeRateMutex.h"
#include <QVector>
#include <QModelIndex>
#include <QDate>

enum ExpenseColumns {
	Description = 0,
	OriginalAmount,
    Currency,
    Date,
	ConversionRate,
	ConvertedAmount,
	numColumns
};

struct Expense {
	QString description;
	float originalAmount;
    QString currency;
    QDate conversionDate;
	float conversionRate;
	float convertedAmount;
};

static QVector<Expense*> expenseRecords;

ExpenseModel::ExpenseModel( const ExchangeRates* rates ) : mExchangeRates( rates )
{
}

ExpenseModel::~ExpenseModel()
{
}

QModelIndex ExpenseModel::index( int row, int column, const QModelIndex & parent ) const
{
	// No child indexes and out of bounds indexes
	if( parent.isValid() ||
		row < 0 || row >= rowCount( QModelIndex() ) ||
		column < 0 || column >= columnCount( QModelIndex() ) )
		return QModelIndex();

	return createIndex( row, column, 0 );
}

QModelIndex ExpenseModel::parent ( const QModelIndex & ) const
{
	return QModelIndex();
}

int ExpenseModel::columnCount( const QModelIndex & parent ) const
{
	if( !parent.isValid() ) // root index
		return numColumns;
	else
		return 0;
}

int ExpenseModel::rowCount ( const QModelIndex & parent ) const
{
	if( !parent.isValid() ) // root index
		return expenseRecords.size();
	else
		return 0;
}

QVariant ExpenseModel::data( const QModelIndex & index, int role ) const
{
	if( !index.isValid() || index.model() != this )
		return QVariant();

	const Expense * curExpense = expenseRecords.at( index.row() );

	switch( role )
	{
	case Qt::DisplayRole:
	case Qt::EditRole:
		switch( index.column() )
		{
		case Description:
			return QVariant( curExpense->description );
		case OriginalAmount:
			return QVariant( curExpense->originalAmount );
        case Currency:
            return QVariant( curExpense->currency );
        case Date:
            return QVariant( curExpense->conversionDate );
		case ConversionRate:
			return QVariant( curExpense->conversionRate );
		case ConvertedAmount:
            if( role == Qt::DisplayRole )
                return QString::number( curExpense->convertedAmount, 'f', 2 );
            else
                return QVariant( curExpense->convertedAmount );
		default:
			return QVariant();
		}
	default:
		return QVariant();
	}
}

Qt::ItemFlags ExpenseModel::flags ( const QModelIndex & index ) const
{
	Qt::ItemFlags empty;
	if( !index.isValid() || index.model() != this )
		return empty;

	Qt::ItemFlags basicFlags = Qt::ItemIsSelectable | Qt::ItemIsEnabled;
	if( index.column() == ConvertedAmount || index.column() == ConversionRate )
		return basicFlags; // Read-only
	else
		return basicFlags | Qt::ItemIsEditable;

}

QVariant ExpenseModel::headerData ( int section, Qt::Orientation orientation, int role ) const
{
	if( orientation != Qt::Horizontal )
		return QVariant();

	switch( role )
	{
		case Qt::DisplayRole:
			switch( section )
			{
			case Description:
				return QVariant( tr( "Description" ) );
			case OriginalAmount:
				return QVariant( tr( "Original Amount" ) );
            case Currency:
                return QVariant( tr( "Currency" ) );
			case Date:
				return QVariant( tr( "Date" ) );
			case ConversionRate:
				return QVariant( tr( "Conversion Rate" ) );
			case ConvertedAmount:
				return QVariant( tr( "Converted Amount" ) );
			default:
				return QVariant();
			}
		default:
			return QVariant();
	}
}

bool ExpenseModel::setData ( const QModelIndex & idx, const QVariant & value, int role )
{
	if( !idx.isValid() || idx.model() != this )
		return false;

	Expense * curExpense = expenseRecords[ idx.row() ];

	bool conversionOK = true;
	switch( idx.column() ) {
	case Description:
		curExpense->description = value.toString();
		break;
	case OriginalAmount:
		curExpense->originalAmount = value.toDouble( &conversionOK );
		break;
    case Currency:
        curExpense->currency = value.toString();
        break;
    case Date:
        curExpense->conversionDate = value.toDate();
        break;
	default:
		return false;
	}

    // If we have the currency and the date, we can retrieve the conversion rate and store it.
    if( ( idx.column() == Currency || idx.column() == Date ) &&
        !curExpense->currency.isEmpty() &&
        curExpense->conversionDate.isValid() ) {
        if( ratesMutex.tryLock() ) {
            curExpense->conversionRate = (*mExchangeRates)[qMakePair(curExpense->conversionDate,
                                                                     curExpense->currency)];
            ratesMutex.unlock();
            QModelIndex cellIndex = index( idx.row(), ConversionRate, QModelIndex() );
            emit dataChanged( cellIndex, cellIndex );
        } else
            qDebug( "Couldn't get lock, no conversion performed." );
    }

    // If we have enough data for a conversion, perform it.
	if( conversionOK ) {
		emit dataChanged( idx, idx );
		// re-compute converted amount
		curExpense->convertedAmount = curExpense->originalAmount / curExpense->conversionRate;
		QModelIndex cellIndex = index( idx.row(), ConvertedAmount, QModelIndex() );
		emit dataChanged( cellIndex, cellIndex );
	}
	return conversionOK;
}

bool ExpenseModel::setHeaderData ( int section, Qt::Orientation orientation, const QVariant & value, int role )
{
	// Headers can't be modified
	return false;
}

bool ExpenseModel::insertRows( int row, int count, const QModelIndex & parentIndex ) {
	if( parentIndex.isValid() || row < 0 || row > rowCount( QModelIndex() ) )
		return false;
    beginInsertRows( parentIndex, row, row + count - 1 );
	for( int i = 0 ; i < count ; i++ ) {
		Expense * e = new Expense();
		e->description = tr( "Enter Description" );
		e->originalAmount = 0;
        e->conversionDate = QDate::currentDate();
		e->conversionRate = 0;
		e->convertedAmount = 0;
		expenseRecords.insert( row + i, e );
	}
    endInsertRows();
    return true;
}

bool ExpenseModel::removeRows( int row, int count, const QModelIndex & parentIndex ) {
	if ( parentIndex.isValid() || row < 0 || ( row + count ) > rowCount( QModelIndex() ) )
        return false;
    beginRemoveRows( parentIndex, row, row + count - 1 );
	for( int i = 0; i < count ; i++ ) {
		delete expenseRecords[ row + i ];
		expenseRecords.remove( row + i );
	}
    endRemoveRows();
	return true;
}

