#include "ExpenseModel.h"
#include "ExchangeRateMutex.h"
#include <QModelIndex>
#include <QDate>
#include <QSqlRecord>
#include <QtDebug>

enum ExpenseColumns {
	Description = 0,
	OriginalAmount,
    Currency,
    Date,
	ConversionRate,
	ConvertedAmount,
	numColumns
};

struct Expense {
	QString description;
	float originalAmount;
    QString currency;
    QDate conversionDate;
	float conversionRate;
	float convertedAmount;
};

//static QVector<Expense*> expenseRecords;

ExpenseModel::ExpenseModel( const ExchangeRates* rates ) : mExchangeRates( rates )
{
    // CHANGED
    mSqlModel.setTable( "expenses" );
    mSqlModel.select();
}

ExpenseModel::~ExpenseModel()
{
}

QModelIndex ExpenseModel::index( int row, int column, const QModelIndex & parent ) const
{
	// No child indexes and out of bounds indexes
	if( parent.isValid() ||
		row < 0 || row >= rowCount( QModelIndex() ) ||
		column < 0 || column >= columnCount( QModelIndex() ) )
		return QModelIndex();

	return createIndex( row, column, 0 );
}

QModelIndex ExpenseModel::parent ( const QModelIndex & ) const
{
	return QModelIndex();
}

int ExpenseModel::columnCount( const QModelIndex & parent ) const
{
	if( !parent.isValid() ) // root index
		return numColumns;
	else
		return 0;
}

int ExpenseModel::rowCount ( const QModelIndex& ) const
{
    // CHANGED
    return mSqlModel.rowCount();
}

QVariant ExpenseModel::data( const QModelIndex & index, int role ) const
{
	if( !index.isValid() || index.model() != this )
		return QVariant();

    // CHANGED
    QSqlRecord curExpense = mSqlModel.record( index.row() );

	switch( role )
	{
	case Qt::DisplayRole:
	case Qt::EditRole:
		switch( index.column() )
		{
            // CHANGED
		case Description:
			return curExpense.value( "description" );
		case OriginalAmount:
			return curExpense.value( "original_amount" );
        case Currency:
            return curExpense.value( "currency" );
        case Date:
            return curExpense.value( "date" );
		case ConversionRate:
			return curExpense.value( "exchange_rate" );
		case ConvertedAmount:
            if( role == Qt::DisplayRole )
                return QString::number( curExpense.value( "euro_amount" ).toDouble(), 'f', 2 );
            else
                return curExpense.value( "euro_amount" );
		default:
			return QVariant();
		}
	default:
		return QVariant();
	}
}

Qt::ItemFlags ExpenseModel::flags ( const QModelIndex & index ) const
{
	Qt::ItemFlags empty;
	if( !index.isValid() || index.model() != this )
		return empty;

	Qt::ItemFlags basicFlags = Qt::ItemIsSelectable | Qt::ItemIsEnabled;
	if( index.column() == ConvertedAmount || index.column() == ConversionRate )
		return basicFlags; // Read-only
	else
		return basicFlags | Qt::ItemIsEditable;

}

QVariant ExpenseModel::headerData ( int section, Qt::Orientation orientation, int role ) const
{
	if( orientation != Qt::Horizontal )
		return QVariant();

	switch( role )
	{
		case Qt::DisplayRole:
			switch( section )
			{
			case Description:
				return QVariant( tr( "Description" ) );
			case OriginalAmount:
				return QVariant( tr( "Original Amount" ) );
            case Currency:
                return QVariant( tr( "Currency" ) );
			case Date:
				return QVariant( tr( "Date" ) );
			case ConversionRate:
				return QVariant( tr( "Conversion Rate" ) );
			case ConvertedAmount:
				return QVariant( tr( "Converted Amount" ) );
			default:
				return QVariant();
			}
		default:
			return QVariant();
	}
}

bool ExpenseModel::setData ( const QModelIndex & idx, const QVariant & value, int )
{
	if( !idx.isValid() || idx.model() != this )
		return false;

    // CHANGED
    // Retrieve current values.
    QSqlRecord curExpense = mSqlModel.record( idx.row() );

	bool conversionOK = true;
	switch( idx.column() ) {
	case Description:
		curExpense.setValue( "description", value );
		break;
	case OriginalAmount:
		curExpense.setValue( "original_amount", value );
		break;
    case Currency:
        curExpense.setValue( "currency", value );
        break;
    case Date:
        curExpense.setValue( "date", value );
        break;
	default:
		return false;
	}

    // If we have the currency and the date, we can retrieve the conversion rate and store it.
    if( ( idx.column() == Currency || idx.column() == Date ) &&
        !curExpense.value( "currency" ).toString().isEmpty() &&
        curExpense.value( "date" ).toDate().isValid() ) {
        if( ratesMutex.tryLock() ) {
            curExpense.setValue( "exchange_rate", (*mExchangeRates)[qMakePair(curExpense.value( "date" ).toDate(),
                                                                              curExpense.value( "currency" ).toString() )]);
            ratesMutex.unlock();
            QModelIndex cellIndex = index( idx.row(), ConversionRate, QModelIndex() );
            // Write back the data before the dataChanged() signal is emitted.
            bool success = mSqlModel.setRecord( idx.row(), curExpense );
            Q_ASSERT( success );
            emit dataChanged( cellIndex, cellIndex );
        } else
            qDebug( "Couldn't get lock, no conversion performed." );
    }

    // If we have enough data for a conversion, perform it.
	if( conversionOK ) {
		emit dataChanged( idx, idx );
		// re-compute converted amount
		curExpense.setValue( "euro_amount", curExpense.value( "original_amount" ).toDouble() / curExpense.value( "exchange_rate" ).toDouble() );
		QModelIndex cellIndex = index( idx.row(), ConvertedAmount, QModelIndex() );
        // Write back the data before the dataChanged() signal is emitted.
        bool success = mSqlModel.setRecord( idx.row(), curExpense );
        Q_ASSERT( success );
		emit dataChanged( cellIndex, cellIndex );
	}

    // Make sure data is written in any case.
    bool success = mSqlModel.setRecord( idx.row(), curExpense );
    Q_ASSERT( success );
	return conversionOK;
}

bool ExpenseModel::setHeaderData ( int section, Qt::Orientation orientation, const QVariant & value, int )
{
	// Headers can't be modified
	return false;
}

bool ExpenseModel::insertRows( int row, int count, const QModelIndex & parentIndex ) {
	if( parentIndex.isValid() || row < 0 || row > rowCount( QModelIndex() ) )
		return false;

    beginInsertRows( parentIndex, row, row + count - 1 );

	for( int i = 0 ; i < count ; i++ ) {
        // CHANGED
        QSqlRecord expense = mSqlModel.record();
        expense.setValue( "description", tr( "Enter Description" ) );
        expense.setValue( "original_amount", 0 );
        expense.setValue( "currency", QVariant( QString::null ) );
        expense.setValue( "date", QDate::currentDate() );
        expense.setValue( "exchange_rate", 0 );
        expense.setValue( "euro_amount", 0 );
        mSqlModel.insertRecord( row, expense );
    }

    endInsertRows();

    return true;
}

bool ExpenseModel::removeRows( int row, int count, const QModelIndex & parentIndex ) {
	if ( parentIndex.isValid() || row < 0 || ( row + count ) > rowCount( QModelIndex() ) )
        return false;
    beginRemoveRows( parentIndex, row, row + count - 1 );

    mSqlModel.removeRows( row, count, parentIndex );

    endRemoveRows();
	return true;
}


bool ExpenseModel::submit()
{
    return mSqlModel.submitAll();
}
