#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <pthread.h>
#include "helper.h"
#include <sys/times.h>
#include <unistd.h>

#define SLOW 0
#define FAST 1

#ifndef VERSION
#define VERSION FAST
#endif

#ifndef MAX_THREADS
#define MAX_THREADS   2
#endif

#ifndef N_ITERATIONS
#define N_ITERATIONS 10000000
#endif

#define LOG if (0)

typedef struct _threadparam {
  int id;
} THREADPARAM;


typedef struct _barrier_t {
  pthread_mutex_t mutex;
  pthread_cond_t  cond;
  int counter;
} barrier_t;

static barrier_t barrier =
  { PTHREAD_MUTEX_INITIALIZER, PTHREAD_COND_INITIALIZER, 0 };


void *aThread(void *params)
{
  int rc, it;
  int id = ((THREADPARAM *)params)->id;

	for (it = 0; it < N_ITERATIONS; ++it) {
	  LOG printf("Thread %2d geht auf die Schranke zu ...\n", id);

	  /* Mutex sperren, um Zugriff auf Zhler zu schtzen */
	  rc = pthread_mutex_lock(&barrier.mutex);
	  if (rc != 0)
	    err_abort(rc, "pthread_mutex_lock() fehlgeschlagen");

	  /* durch Hochzhlen der globalen Variable signalisieren, 
	     dass ein weiterer Thread gestartet wurde */
	  ++barrier.counter;

	  // sched_yield();

	  if (barrier.counter < MAX_THREADS) {
	    /* es sind noch nicht alle Threads an der Barriere angekommen,
	       also warten ... */
	    LOG printf("Thread %2d wartet vor der Schranke ...\n", id);
	    rc = pthread_cond_wait(&barrier.cond, &barrier.mutex);
	    if (rc != 0)
	      err_abort(rc, "pthread_cond_wait() fehlgeschlagen");
#if VERSION == FAST
		  pthread_mutex_unlock(&barrier.mutex);
#endif
	  }
	  else {
	    /* der als letzter an der Barriere angelangte Thread
	       ffnet die Schranke */
	    LOG printf("Thread %2d hebt Schranke ...\n", id);
#if VERSION == FAST
		  pthread_mutex_unlock(&barrier.mutex);
#endif
	    rc = pthread_cond_broadcast(&barrier.cond);
	    if (rc != 0)
	      err_abort(rc, "pthread_cond_broadcast() fehlgeschlagen");
	  }

	  /* pthread_cond_wait() und pthread_cond_broadcast() kehren mit
	     gesperrtem Mutex zurck, weshalb der Mutex hier entsperrt
	     werden muss */
#if VERSION == SLOW
	  pthread_mutex_unlock(&barrier.mutex);
#endif
	  LOG printf("Thread %2d ist durch die Schranke gegangen.\n", id);
	}

  return NULL;
}


int main(int argc, char *argv[])
{
	const char version[][5] = {"slow", "fast"};
  pthread_t tid[MAX_THREADS];
  THREADPARAM param[MAX_THREADS];
  int i;
  int rc;
  clock_t startTime, stopTime;

	startTime = times(NULL);

  /* Threads starten */
  for (i = 0; i < MAX_THREADS; ++i) {
    param[i].id = i + 1;
    rc = pthread_create(&tid[i], NULL, aThread, &param[i]);
    if (rc != 0)
      err_abort(rc, "pthread_create() fehlgeschlagen");
  }

  /* auf Beendigung der Threads warten */
  for (i = 0; i < MAX_THREADS; ++i) {
    rc = pthread_join(tid[i], NULL);
    if (rc != 0)
      err_abort(rc, "pthread_join() fehlgeschlagen");
  }

	stopTime = times(NULL);
	printf("nThreads: %5i ; nIterations: %8i ; version: %s ; time: %5.2f\n", MAX_THREADS, N_ITERATIONS, version[VERSION], (double)(stopTime-startTime)/sysconf(_SC_CLK_TCK));

  pthread_cond_destroy(&barrier.cond);
  pthread_mutex_destroy(&barrier.mutex);

  return 0;
}
