#!/usr/bin/perl -w
#
# CD rippen, Tracks simultan MP3-kodieren.
#
# Copyright (C) 2006 Heise Zeitschriften Verlag
#                    Oliver Lau <ola@ctmagazin.de>
# Alle Rechte vorbehalten.
#
# $Id: simulame.pl 967 2006-08-01 08:46:50Z olau $

use strict;
use warnings;
use Audio::CD();
use Getopt::Std;
use Config;

######################################################################
# Konfigurationsparameter
######################################################################

# Pfad zum MP3-Encoder und Parameter
my $LAME = '/usr/local/bin/lame --silent -m s -V 7 -q 9 --ta "%s" --tl "%s" --ty "%s" --tt "%s" --tn "%s" --tg "%s" --tc "%s" --add-id3v2 "%s" "%s"';

# Pfad zum CD-Ripper cdparanoia und Parameter 
my $CDPARANOIA = '/usr/bin/cdparanoia --batch --output-wav';

# Format der Dateinamen, unter denen cdparanoia die von der CD
# gelesenen Audio-Tracks ablegt
my $CDPARANOIA_FILENAME_FORMAT = 'track%02d.cdda.wav';

BEGIN {
    if ($Config{useithreads}) {
		use threads;
		use threads::shared;
    }
    else {
		die "Ihr Perl beherrscht leider keine Interpreter-Threads";
    }
}

use constant {
    YES       => 1,
    NO        => 0
};

use constant DEFAULT_NUM_THREADS => 2;

use constant PROGRAMNAME => 'simulame.pl';

my %option = ();
getopts("vdrt:e", \%option); 

# Debugging-Ausgaben: ein/aus
my $DEBUG = ($option{'d'})? YES : NO;

# im CD-Laufwerk liegende Audio-CD rippen: ja/nein
my $DO_RIP = ($option{'r'})? YES : NO;

# maximale Anzahl simultan laufenden LAMEs
my $MAX_THREADS = DEFAULT_NUM_THREADS;
BEGIN {
    if (eval 'require Sys::CPU') {
	$MAX_THREADS = Sys::CPU::cpu_count();
	print 'Sys::CPU::cpu_count() = ', Sys::CPU::cpu_count(), "\n";
    }
}
$MAX_THREADS = $option{'t'} if $option{'t'};

# Ausgabedetails einstellen
my $VERBOSITY = ($option{'v'})? YES : NO;

# CD auswerfen oder nicht
my $DO_EJECT = ($option{'e'})? YES : NO;

# gemeinsam genutzte Variablen initialisieren
my %trackData : shared;
my $dataReady : shared = NO;


# HELP_MESSAGE() wird automatisch von getopts() aufgerufen, 
# wenn "--help" in den Kommandozeilenparametern auftaucht
sub HELP_MESSAGE() {
    printf "%s - CDs rippen, Tracks simultan MP3-kodieren.\n", PROGRAMNAME;
    print "Copyright (C) 2006 Heise Zeitschriften Verlag, " .
	"Oliver Lau <ola\@ctmagazin.de>\n" .
	"Aufruf:\n";
    printf "  %s [-t n] [-r] [-e] [-d]\n", PROGRAMNAME;
    print "Optionen:\n";
    printf "  -t n    n Threads starten (Standard: %d)\n",
           DEFAULT_NUM_THREADS; 
    print "  -r      vor dem Kodieren rippen (mit cdparanoia)\n" .
	"  -e      nach der CDDB-Abfrage bzw. nach dem Rippen CD auswerfen\n" .
	"  -d      Debug-Ausgaben einschalten\n\n";
    print "Eine ausführlichere Hilfe gibt mit 'perldoc simulame.pl'.\n"; 
    exit;
}


# MP3-Encoder-Thread
sub encoder {
    my $id = threads->self->tid;
    my ($filename, $title, $pos, $artist, $album);
    my $year = ''; my $genre = ''; my $comment = '';
    LOOP: while (1) {
	 	{
		    lock(%trackData);
		    next LOOP unless $dataReady == YES;
		    # Trackdaten abholen
		    printf "\nThread %02d: Arbeit abholen ...\n", $id;
		    $filename = $trackData{'filename'};
		    last LOOP if $filename eq '';
		    $title    = $trackData{'title'};
		    $pos      = $trackData{'pos'};
		    $artist   = $trackData{'artist'};
		    $album    = $trackData{'album'};
		    # Manager informieren, dass Daten abgeholt wurden
		    $dataReady = NO;
		    printf "Thread %02d: Manager informieren, " .
			"dass Arbeit abgeholt wurde ...\n", $id if $DEBUG;
		    cond_signal(%trackData);
		}
		printf "Thread %02d: kodieren der Datei %s (%s) ...\n",
		       $id, $filename, $title;
		$artist =~ s/\s/_/g; # Leerzeichen in Unterstriche wandeln
		$title  =~ s/\s/_/g; # Leerzeichen in Unterstriche wandeln
		my $outFile = sprintf '%02d-%s-%s.mp3', $pos, $artist, $title;
		my $lamecall = sprintf $LAME, $artist, $album,
		                       $year, $title, $pos,
		                       $genre, $comment, $filename,
		                       $outFile;
		system($lamecall) == 0
		    or die 'kann LAME nicht ausführen: $?';
    };
    printf "Thread %02d: fertig.\n", $id;
}


######################################################################
#
# hier gehts los ...
#
print "CD-Info einlesen ...\n" if $VERBOSITY > 0;
my $cd = Audio::CD->init;
my $info = $cd->stat;

# Die dem Perl-Modul zur CDDB-Abfrage (Audio::CD) zugrundeliegende
# Audio-Bibliothek libcdaudio erwartet die Zugangsdaten
# zum CDDB-Server in der Datei ~/.cdserverrc.
print "CDDB abfragen ...\n" if $VERBOSITY > 0;
my $cddb = $cd->cddb;
$cddb->verbose(1) if $DEBUG;
printf "CDDB id = %lx, Anzahl Tracks = %d\n",
       $cddb->discid, $info->total_tracks if $DEBUG;
my $data = $cddb->lookup or die 'CDDB-Abfrage gescheitert';
printf "%s - %s (%s)\n", $data->artist, $data->title, $data->genre;
$trackData{'artist'} = $data->artist;
$trackData{'album'}  = $data->title;
$trackData{'genre'}  = $data->genre;

# CD rippen
if ($DO_RIP) {
    system($CDPARANOIA) == 0
	or die "kann cdparanoia nicht ausführen: $?";
}
$cd->eject if ($DO_EJECT);

# Threads starten
my @threads;
print "$MAX_THREADS Threads starten ...\n" if $DEBUG;
for (my $i = 1; $i <= $MAX_THREADS; $i++) {
    $threads[$i] = threads->new(\&encoder);
}

# Tracks sammeln ...
my $trackNames = $data->tracks($info);
my $tracksUnsorted = $info->tracks;
my @audioTracks;
for (my $pos = 1, my $i = 0; $i < $info->total_tracks; $i++) {
    my %track;
    my $t = @$tracksUnsorted[$i];
    next if $t->is_data; # Datentracks ignorieren
    my ($mins, $secs) = $t->length;
    $track{'length'}   = 60 * $mins + $secs;
    $track{'pos'}      = $pos; 
    $track{'filename'} = sprintf $CDPARANOIA_FILENAME_FORMAT, $pos;
    $track{'title'}    = @$trackNames[$i]->name;
    $track{'artist'}   = $data->artist;
    push @audioTracks, \%track;
    $pos++;
}

# ... und nach Länge absteigend sortieren
my @tracks = sort { $b->{'length'} <=> $a->{'length'} } @audioTracks;

# Tracks nach und nach zum Kodieren austeilen
foreach my $track (@tracks) {
    printf "Nächster Track: %02d - %s - %s (%d secs)\n",
           $track->{'pos'}, $track->{'artist'},
           $track->{'title'}, $track->{'length'} if $DEBUG;
    {
		# Trackdaten bereitlegen
		lock(%trackData);
		$trackData{'title'}    = $track->{'title'};
		$trackData{'pos'}      = $track->{'pos'};
		$trackData{'filename'} = $track->{'filename'};
		$dataReady = YES;
		# warten, bis Trackdaten abgeholt wurden
		cond_wait(%trackData)
		    until $dataReady == NO; # cond_wait() gibt %trackData frei
    }
}

print "Manager läutet Feierabendglocke ...\n" if $DEBUG;
{
    lock(%trackData);
    # leerer Dateiname signalisiert, dass keine Arbeit mehr vorliegt
    $trackData{'filename'} = '';
    $dataReady = YES;
}

print "Manager wartet, bis Threads fertig ...\n" if $DEBUG;
for (my $i = 1; $i <= $MAX_THREADS; $i++) {
    $threads[$i]->join();
}

print "ENDE.\n" if $DEBUG;
0;


__END__

=head1 NAME

simulame.pl

=head1 BESCHREIBUNG

CD rippen, Tracks simultan MP3-kodieren.

Eine detailliertere Beschreibung der Funktionsweise dieses Programms ist im
Artikel "Perl Parallel" der c't 17/06 auf Seite 206 zu finden.

=head1 AUFRUF

simulame.pl [-t n] [-r] [-d] [-v]

=head1 OPTIONEN

=over

=item -t n

Mit n Threads kodieren, das heißt n LAMEs gleichzeitig kodieren
lassen. Voreinstellung: zwei Threads.

=item -r 

Vor dem Kodieren die Audio-Tracks von der eingelegte CD als Wave-Dateien im
aktuellen Verzeichnis ablegen (rippen).

=item -d

Debug-Informationen ausgeben.

=item -v

Noch mehr Meldungen ausgeben.

=back

=head1 INSTALLATION

Für den Zugriff auf die FreeDB-Datenbank benötigt das Programm das Perl-Modul
Audio::CD. Das Modul bedient sich der (leider derzeit nur für Unix/Linux
verfügbaren) Bibliothek libcdaudio. Diese erwartet eine Konfigurationsdatei
namens .cdserverrc im Home-Verzeichnis des aufrufenden Benutzers mit etwa
folgendem Inhalt:

 ACCESS=REMOTE
 PROXY=http://proxy.example.com:8080
 SERVER=cddbp://de.freedb.org:8880/ CDDB
 SERVER=http://de.freedb.org:80/~cddb/cddb.cgi CDDB
 SERVER=cddbp://at.freedb.org:8880 CDDB
 SERVER=cddbp://freedb.freedb.org:8880/ CDDB
 SERVER=http://freedb.freedb.org:80/~cddb/cddb.cgi CDDB
 SERVER=cddbp://at.freedb.org:8880/ CDDB
 SERVER=http://at.freedb.org:80/~cddb/cddb.cgi CDDB
 SERVER=cddbp://ca.freedb.org:8880/ CDDB
 SERVER=http://ca.freedb.org:80/~cddb/cddb.cgi CDDB
 SERVER=cddbp://cz.freedb.org:8880/ CDDB
 SERVER=http://cz.freedb.org:80/~cddb/cddb.cgi CDDB
 SERVER=cddbp://no.freedb.org:8880/ CDDB
 SERVER=http://no.freedb.org:80/~cddb/cddb.cgi CDDB
 SERVER=cddbp://uk.freedb.org:8880/ CDDB
 SERVER=http://uk.freedb.org:80/~cddb/cddb.cgi CDDB
 SERVER=cddbp://us.freedb.org:8880/ CDDB
 SERVER=http://us.freedb.org:80/~cddb/cddb.cgi CDDB

Die mit C<PROXY> beginnende Zeile kann mit C<#> auskommentiert werden, falls
der Zugriff aufs Web nicht über einen Proxy vonstatten geht.

=head1 VORAUSSETZUNGEN

=over

=item *

Perl 5.8.8 oder neuer

=item *

Perl-Module:

=over

=item -

Audio::CD

=item -

Sys::CPU

=item -

Getopt::Std

=item - 

Config

=back

=item *

CDDB-Bibliothek: libcdaudio, L<http://libcdaudio.sourceforge.net/>

=item *

MP3-Encoder: LAME, L<http://lame.sourceforge.net/>

Oder ein anderer Encoder wie OggEnc, L<http://www.xiph.org/>

=item *

CD-Ripper: cdparanoia, L<http://www.xiph.org/>

=back

=head1 BEKANNTE PROBLEME

=over

=item 

Wegen Implementierungslücken im Perl-Modul Audio::CD lassen sich die
Tracks zusammengestellter CDs (Compilations) nur unzureichend ID3-taggen: Das
Feld "artist" trägt stets den Namen "Various" o.ä.

=item

Jahr und Genre des Titels bzw. Albums werden nicht ausgewertet.

=item

Die Bibliothek libcdaudio ist derzeit nur für Unix/Linux verfügbar. Dieses
Perl-Skript läuft demnach nicht unter Windows.

=back

=head1 COPYRIGHT

Copyright (C) 2006 Heise Zeitschriften Verlag, L<http://www.heise.de/ct>,
Oliver Lau L<mailto:ola@ctmagazin.de>

Alle Rechte vorbehalten.

=head1 HAFTUNGSAUSSCHLUSS

Dieses Programm wurde zu Lehrzwecken erstellt und darf kostenlos verwendet
werden. Modifikationen sind erlaubt, wenn das modifizierte Programm die o.a. Copyright-Hinweis wiedergeben kann.

Weder der Autor noch der Heise Zeitschriften Verlag übernehmen Gewährleistung
für das Programm, einschließlich -- aber nicht begrenzt auf -- Marktreife oder
Verwendbarkeit für einen bestimmten Zweck. Das volle Risiko bezüglich Qualität
und Leistungsfähigkeit des Programms liegt bei Ihnen. Sollte sich das Programm
als fehlerhaft herausstellen, liegen die Kosten für etwaig notwendig
gewordenen Service oder Reparaturen sowie Korrekturen bei Ihnen.

$Date: 2006-08-01 10:46:50 +0200 (Di, 01 Aug 2006) $
