namespace Sudoku
{
    public enum SudokuStatus
    {
        NoSolution, OneSolution, MoreSolutions, Timeout
    }

    public partial class SudokuField
    {
        public int Iterations;
        private int MaxIterations;
        private SudokuField Solution, SecondSolution;
        private DataObject[] O;
        private ColumnObject h;

        /// <summary>
        /// Solves actual SudokuField using the DLX algorithm described in Donald E. Knuth's 
        /// paper "Dancing Links" http://xxx.lanl.gov/PS_cache/cs/pdf/0011/0011047.pdf.
        /// </summary>
        /// <returns>NoSolution, OneSolution, MoreSolutions, Timeout</returns>
        public SudokuStatus Solve()
        {
            SudokuStatus Status = GetStatus(int.MaxValue);
            if (Status == SudokuStatus.OneSolution || Status == SudokuStatus.MoreSolutions)
                Copy(Solution);
            return Status;
        }

        /// <summary>
        /// Calculates the status of the actual SudokuField without changing its Digits and Lists.
        /// </summary>
        /// <param name="MaxNumberOfIterations">Max. number of Iterations</param>
        /// <returns>NoSolution, OneSolution, MoreSolutions, Timeout (result is in Solution)</returns>
        private SudokuStatus GetStatus(int MaxNumberOfIterations)
        {
            CreateMatrix();
            Iterations = 0;
            MaxIterations = MaxNumberOfIterations;
            return Search(SudokuStatus.NoSolution, 0);
        }

        private void CreateMatrix()
        {
            O = new DataObject[nn * nn];
            DataObject l = h = new ColumnObject(null, Constraint.Head, -1);
            ColumnObject[,] c1 = new ColumnObject[nn, nn];
            for (int i = 0; i < nn; i++)
                for (int j = 0; j < nn; j++)
                    l = c1[i, j] = new ColumnObject(l, Constraint.OneDigitInEachCell, -1);
            ColumnObject[,] c2 = new ColumnObject[nn, nn];
            for (int d = 1; d <= nn; d++)
                for (int i = 0; i < nn; i++)
                    l = c2[d - 1, i] = new ColumnObject(l, Constraint.EachDigitInRowOnce, i);
            ColumnObject[,] c3 = new ColumnObject[nn, nn];
            for (int d = 1; d <= nn; d++)
                for (int j = 0; j < nn; j++)
                    l = c3[d - 1, j] = new ColumnObject(l, Constraint.EachDigitInColumnOnce, j);
            ColumnObject[,] c4 = new ColumnObject[nn, nn];
            for (int d = 1; d <= nn; d++)
                for (int b = 0; b < nn; b++)
                    l = c4[d - 1, b] = new ColumnObject(l, Constraint.EachDigitInBlockOnce, d);

            for (int i = 0; i < nn; i++)
                for (int j = 0; j < nn; j++)
                    for (int d = 1; d <= nn; d++)
                        if (Digits[i, j] == d || Digits[i, j] == 0) // enumerate all possibilities, create
                        {                                           // new matrix row for every possibility,
                            int b = n * (i / n) + (j / n);          // and put constraints into matrix columns:
                            l = new DataObject(null, c1[i, j]);     // 1) cell i,j contains d
                            l = new DataObject(l, c2[d - 1, i]);    // 2) d occurs in row i exactly once
                            l = new DataObject(l, c3[d - 1, j]);    // 3) d occurs in column j exactly once
                            l = new DataObject(l, c4[d - 1, b]);    // 4) d occurs in block b exactly once
                        }
        }

        private SudokuStatus Search(SudokuStatus Status, int k)
        {
            if (h.R == h)                           // DLX matrix empty ->
            {                                       // selected Rows are the solution
                SecondSolution = Solution;
                Solution = new SudokuField(O, k);
                return Status + 1;
            }
            if (++Iterations >= MaxIterations)
                return SudokuStatus.Timeout;
            ColumnObject C = GetMinColumnObject();  // choose the DLX column with the minimum size
            C.CoverColumn();                        // and perform Donald E. Knuth's DLX algorithm
            for (DataObject i = C.D; i != C; i = i.D)
            {
                O[k] = i;                           // add row i to the solution
                for (DataObject j = i.R; j != i; j = j.R)
                    j.C.CoverColumn();               
                Status = Search(Status, k + 1);     // try to solve the reduced matrix
                if (Status >= SudokuStatus.MoreSolutions)
                    return Status;                  // terminate if second solution found or timeout
                for (DataObject j = i.L; j != i; j = j.L)
                    j.C.UncoverColumn();
            }
            C.UncoverColumn();
            return Status;
        }

        private SudokuField(DataObject[] SelectedRows, int k)
        {
            Digits = new int[nn, nn];
            Lists = new SudokuList[nn, nn];
            Clear();
            for (int i = 0; i < k; i++)
            {
                DataObject j = SelectedRows[i];
                while (j.C.T != Constraint.EachDigitInRowOnce)
                    j = j.R;
                Digits[j.C.X, j.R.C.X] = j.R.R.C.X;
            }
        }

        private ColumnObject GetMinColumnObject()
        {
            ColumnObject MinC = (ColumnObject)h.R;
            int MinS = MinC.S;
            for (ColumnObject j = (ColumnObject)MinC.R; j != h; j = (ColumnObject)j.R)
                if (j.S < MinS)
                {
                    MinC = j;
                    MinS = MinC.S;
                }
            return MinC;
        }

        private enum Constraint
        {
            Head, OneDigitInEachCell, EachDigitInRowOnce, EachDigitInColumnOnce, EachDigitInBlockOnce
        }

        private class ColumnObject : DataObject
        {
            public int S, X;
            public Constraint T;

            public ColumnObject(DataObject l, Constraint t, int x)
                : base(l, null)
            {
                S = 0;
                T = t;
                X = x;
            }

            public void CoverColumn()
            {
                R.L = L;                                // remove myself from ColumnHeader list
                L.R = R;
                for (DataObject i = D; i != this; i = i.D)
                    for (DataObject j = i.R; j != i; j = j.R)
                    {
                        j.D.U = j.U;                    // remove every DataObject of other
                        j.U.D = j.D;                    // columns which are in the same row as
                        j.C.S--;                        // the DataObjects in my column
                    }
            }

            public void UncoverColumn()
            {
                for (DataObject i = U; i != this; i = i.U)
                    for (DataObject j = i.L; j != i; j = j.L)
                    {
                        j.C.S++;                        // unremove every DataObject of other
                        j.D.U = j;                      // columns which were in the same row as
                        j.U.D = j;                      // the DataObjects in my column
                    }
                R.L = this;
                L.R = this;                             // unremove myself
            }
        }

        private class DataObject
        {
            public DataObject L, R, U, D;
            public ColumnObject C;

            public DataObject(DataObject l, ColumnObject c)
            {
                if (c == null)   // first object in a column
                {
                    C = null;
                    U = this;
                    D = this;
                }
                else
                {
                    C = c;
                    C.S++;       // count and
                    D = C;       // double-link it
                    U = C.U;
                    U.D = this;
                    D.U = this;
                }
                if (l == null)   // first object in a row
                {
                    L = this;
                    R = this;
                }
                else
                {
                    L = l;       // double-link it
                    R = L.R;
                    L.R = this;
                    R.L = this;
                }
            }
        }
    }
}
