/*
 * dedupl.c
 * --------
 *
 * "Skeleton" program for searching duplicates in a database.
 *
 * Copyright (c):
 * 2007:  Joerg MICHAEL, Adalbert-Stifter-Str. 11, 30655 Hannover, Germany
 *
 * SCCS: @(#) dedupl.c  1.1  2007-11-15
 *
 * This file is subject to the GNU Lesser General Public License (LGPL)
 * (formerly known as GNU Library General Public Licence)
 * as published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 * This file is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this file; if not, write to the
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Actually, the LGPL is __less__ restrictive than the better known GNU General
 * Public License (GPL). See the GNU Library General Public License or the file
 * LIB_GPLA.TXT for more details and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * There is one important restriction: If you modify this program in any way
 * (e.g. modify the underlying logic or translate this program into another
 * programming language), you must also release the changes under the terms
 * of the LGPL.
 * (However, since __this__ program is intended to be customized, all changes
 * covered by TO-DO comments are free.)
 *
 * That means you have to give out the source code to your changes,
 * and a very good way to do so is mailing them to the address given below.
 * I think this is the best way to promote further development and use
 * of this software.
 *
 * If you have any remarks, feel free to e-mail to:
 *     ct@ct.heise.de
 *
 * The author's email address is:
 *    astro.joerg@googlemail.com
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "addr_ext.h"



/****  TO-DO: If you want to use this program      ****/
/****  in a production environment, delete the     ****/
/****  following macro:                            ****/
#define  FILE_FOR_STANDALONE_TEST  "samples.txt"

/****  TO-DO:  If you want to use this program     ****/
/****  in a library, delete the following macro:   ****/
  #define DEDUPL_EXECUTABLE
/****  (and do not forget to call the function     ****/
/****  "cleanup_addr()" when you exit the program) ****/


/****  TO-DO:  In a non-German environment, delete        ****/
/****         "| COMPARE_LANGUAGE_GERMAN"  from RUN_MODE  ****/
#define RUN_MODE    (COMPARE_NORMAL | COMPARE_LANGUAGE_GERMAN | TRACE_ERRORS)

/****  TO-DO: values between 84 and 94 are recommended ****/
/****         for "MIN_POINTS_DUPLICATES"              ****/
/****  (values less than 82 points don't make sense)   ****/
#define MIN_POINTS_DUPLICATES   86


/****  macros for the unload files       ****/
/****  TO-DO: change them, if necessary  ****/
#define MAX_UNLOAD_FILES         7
#define IGNORE_THIS_UNLOAD_FILE  "-"

/****  macros and database variables for the database select  ****/
/****  TO-DO: change them, if necessary   ****/
#define SQL_SUCCESS       0L
#define SQL_NOT_FOUND    100L



static long sql_err_code;    /****  SQL result (error code)  ****/

static long db_matchcode;    /****  database matchcodes  ****/
static long my_matchcode;    /****  (MUST be unique !!)  ****/



/****  External variables for address comparison.  ****/
/****  TO-DO: change them, if necessary, e.g.:     ****/
/****    extern char xyz_name[LENGTH_FAM_NAME+1];  ****/
/****    #define db_fam_name xyz_name              ****/

static char my_gender [LENGTH_GENDER+1];
static char my_first_name [LENGTH_FIRST_NAME +1];
static char my_fam_name [LENGTH_FAM_NAME +1];
static char my_c_o_name [LENGTH_C_O_NAME +1];
static char my_street [LENGTH_STREET +1];
static char my_first_name_ph [LENGTH_FIRST_NAME +1];
static char my_fam_name_ph [LENGTH_FAM_NAME +1];
static char my_street_ph [LENGTH_STREET +1];
static char my_zip_code [LENGTH_ZIP_CODE +1];
static char my_city [LENGTH_CITY +1];
static char my_country [LENGTH_COUNTRY +1];
static char my_b_day [LENGTH_BIRTH_DAY +1];
static char my_b_month [LENGTH_BIRTH_MONTH +1];
static char my_b_year [LENGTH_BIRTH_YEAR +1];
static char my_phone_number [LENGTH_PHONE_NUMBER +1];
static char my_mobile_number [LENGTH_MOBILE_NUMBER +1];
static char my_iban_code [LENGTH_IBAN_CODE +1];
static char my_bank_account [LENGTH_BANK_ACCOUNT +1];
static char my_cust_number [LENGTH_CUST_NUMBER +1];

static char db_gender [LENGTH_GENDER+1];
static char db_first_name [LENGTH_FIRST_NAME +1];
static char db_fam_name [LENGTH_FAM_NAME +1];
static char db_c_o_name [LENGTH_C_O_NAME +1];
static char db_street [LENGTH_STREET +1];
static char db_first_name_ph [LENGTH_FIRST_NAME +1];
static char db_fam_name_ph [LENGTH_FAM_NAME +1];
static char db_street_ph [LENGTH_STREET +1];
static char db_zip_code [LENGTH_ZIP_CODE +1];
static char db_city [LENGTH_CITY +1];
static char db_country [LENGTH_COUNTRY +1];
static char db_b_day [LENGTH_BIRTH_DAY+1];
static char db_b_month [LENGTH_BIRTH_MONTH +1];
static char db_b_year [LENGTH_BIRTH_YEAR +1];
static char db_phone_number [LENGTH_PHONE_NUMBER +1];
static char db_mobile_number [LENGTH_MOBILE_NUMBER +1];
static char db_iban_code [LENGTH_IBAN_CODE +1];
static char db_bank_account [LENGTH_BANK_ACCOUNT +1];
static char db_cust_number [LENGTH_CUST_NUMBER +1];


static struct MAIL_ADDR my_addr[] =
   { { my_gender,        IS_GENDER      },
     { my_first_name,    IS_FIRST_NAME  },
     { my_fam_name,      IS_FAM_NAME    },
     { my_c_o_name,      IS_C_O_NAME    },
     { my_street,        IS_STREET      },
     { my_first_name_ph, IS_FIRST_NAME_PHONET },
     { my_fam_name_ph,   IS_FAM_NAME_PHONET },
     { my_street_ph,     IS_STREET_PHONET },
     { my_zip_code,      IS_ZIP_CODE    },
     { my_city,          IS_CITY        },
     { my_country,       IS_COUNTRY     },
     { my_b_day,         IS_BIRTH_DAY   },
     { my_b_month,       IS_BIRTH_MONTH },
     { my_b_year,        IS_BIRTH_YEAR  },
     { my_phone_number,  IS_PHONE_NUMBER },
     { my_mobile_number, IS_MOBILE_NUMBER },
     { my_iban_code,     IS_IBAN_CODE    },
     { my_bank_account,  IS_BANK_ACCOUNT },
     { my_cust_number,   IS_CUST_NUMBER },
     {   NULL,              0           }
   };

static struct MAIL_ADDR db_addr[] =
   { { db_gender,        IS_GENDER      },
     { db_first_name,    IS_FIRST_NAME  },
     { db_fam_name,      IS_FAM_NAME    },
     { db_c_o_name,      IS_C_O_NAME    },
     { db_street,        IS_STREET      },
     { db_first_name_ph, IS_FIRST_NAME_PHONET },
     { db_fam_name_ph,   IS_FAM_NAME_PHONET },
     { db_street_ph,     IS_STREET_PHONET },
     { db_zip_code,      IS_ZIP_CODE    },
     { db_city,          IS_CITY        },
     { db_country,       IS_COUNTRY     },
     { db_b_day,         IS_BIRTH_DAY   },
     { db_b_month,       IS_BIRTH_MONTH },
     { db_b_year,        IS_BIRTH_YEAR  },
     { db_phone_number,  IS_PHONE_NUMBER },
     { db_mobile_number, IS_MOBILE_NUMBER },
     { db_iban_code,     IS_IBAN_CODE    },
     { db_bank_account,  IS_BANK_ACCOUNT },
     { db_cust_number,   IS_CUST_NUMBER },
     {   NULL,              0           }
   };



/****  macros and variables for formatting an address  ****/

static char *chars_to_delete = ",;.:-_# +*<>^!@$%/)=?'`";


/****  international company identificators  ****/
/****  TO-DO: extend this list, if need be   ****/
static char *COMPANY_NAMES_MID[]
   = { " & ",
       NULL
     };

static char *COMPANY_NAMES_TAIL[]
   = { " Inc.",
       NULL
     };


/****  German company identificators  ****/
static char *FIRMA_GERMAN_MID[]
   =  { "berater",
        "bro",
        "buero",
        "Co.KG",
        "Genossenschaft",
        "genossenschaft",
        " Ges.",
        "-Ges.",
        "Geschft",
        "geschft",
        "Geschaeft",
        "geschaeft",
        "GmbH",
        " GMBH",
        " gmbh",
        "Institut",
        "institut",
        "kanzlei",
        NULL
      };

static char *FIRMA_GERMAN_TAIL[]
   =  { " mbH",
        ". mbH",
        ".mbH",
        " mbh",
        "sges.",
        " GbR",
        " OHG",
        " eG",
        " AG",
        " KG",
        ".KG",
        " KGaA",
        ".KGaA",
        " & Co",
        " Co.",
        "CoKG",
        " e.V.",
        " eV",
        NULL
      };


#define  ADEL_GERMAN_COUNT    16
#define  NAME_PREFIXES_COUNT  38

static int adel_german_init = 0;
static int name_prefixes_init = 0;

static int adel_german_len [ADEL_GERMAN_COUNT];
static int name_prefixes_len [NAME_PREFIXES_COUNT];

static char *adel_german [ADEL_GERMAN_COUNT]
   =  { "Freiherr",  "Frhr.",
        "Freifrau",  "Frfr.",
        "Graf",      "Grfin",
        "Markgraf",  "Markgrfin",
        "Frst",     "Frstin",
        "Ritter",    "Baron",
        "Baronin",   "Baronesse",
        "Prinz",     "Prinzessin"
      };

static char *name_prefixes [NAME_PREFIXES_COUNT]
   =  {
        /****  "?.",  "?'",  ****/
        "a",     "an",
        "auf",   "aus",
        "bei",   "da",
        "de",    "del",
        "dell'",  "della",
        "dem",   "den",
        "der",   "des",
        "di",    "du",
        "el",    "genannt",
        "im",    "in",
        "la",    "las",
        "le",    "les",
        "op",    "'t",
        "te",    "ten",
        "ter",   "to",
        "und",   "van",
        "vom",   "von",
        "vor",   "zu",
        "zum",   "zur"
      };

/****  German family names which may be found in combination  ****/
/****  with "name_prefixes"   (e.g. "Meyer zur Heide")        ****/
static char *common_german_family_names[]
   =  { "Maier",   "Mayer",
        "Meier",   "Meyer",
        "Mller",  "Mueller",
        "Schmid",  "Schmidt",
        NULL
      };




/************************************************************/
/****  database functions  **********************************/
/************************************************************/

#ifdef FILE_FOR_STANDALONE_TEST

   /****  standalone test  ****/
   static char db_full_birthday [LENGTH_FULL_BIRTHDAY +1];
   static char *s1_fam_name = "";
   static char *s2_fam_name = "";
   static char *s3_fam_name = "";
   static char *s4_fam_name = "";
   static char *s_zip_code  = "";
   static char *s_cust_number = "";

   #define SPLIT_FULL_BIRTHDAY
   #include "dummy_db.h"

#else


/****  we are working on a real database  ****/

static void open_database (void)
{
 /**** TO-DO: Add "open" command for your database
  *           (e.g. similar to):
  *
  *  open_db (my_database);
  *
  ****/
}


static void close_database (void)
{
 /**** TO-DO: Add "close" command for your database
  *           (e.g. similar to):
  *
  *  close_db (my_database);
  *
  ****/
}


static void declare_cursor (void)
{
 /**** TO-DO: Add your SQL "declare" and "open"-statements
  *           here (no "where"-clause, e.g. similar to):
  *
  **** TO-DO: Delete unused fields
  *
  *  declare cursor_unl cursor for
  *  select
  *     [database fields] matchcode, customer_number,
  *     gender, title, first_name,
  *     fam_name, c_o_name, street, zip_code, country,
  *     birth_day, birth_month, birth_year, phone_number,
  *     mobile_number, iban_code, bank_account,
  *  into
  *     [strings] db_matchcode, db_cust_number,
  *     db_gender, db_title, db_first_name, db_fam_name,
  *     db_c_o_name, db_street, db_zip_code, db_country,
  *     db_b_day, db_b_month, db_b_year, db_phone_number,
  *     db_mobile_number, db_iban_code, db_bank_account,
  *   from my_table;
  *
  ****/
}


static void fetch_cursor (void)
{
 /**** TO-DO: Add your SQL "fetch"-command here
  *           (e.g. similar to):
  *
  *  fetch (cursor_unl);
  *
  ****/

 /**** TO-DO: If you use birthdays in full format,
  ****        split them (e.g. similar to):
  *
  * split_birthday (db_full_birthday, NULL,
  *              db_b_day, db_b_month, db_b_year);
  *
  ****/
}


static void close_cursor (void)
{
 /**** TO-DO: Add your SQL "close"-command here
  *           (e.g. similar to):
  *
  *  close (cursor_unl);
  *
  ****/
}

#endif




/************************************************************/
/****  other private (static) functions  ********************/
/************************************************************/


static void format_string (char src[])
/****  delete forbidden chars  ****/
{
 char *s;

 while ((s = strchr (src,'\n')) != NULL)
   {
    *s = ' ';
   }
 while ((s = strchr (src,'\r')) != NULL)
   {
    *s = ' ';
   }
 while ((s = strchr (src,'\t')) != NULL)
   {
    *s = ' ';
   }

 s = src;
 /****  delete leading blanks  ****/
 while (*s == ' ')
   {
    s++;
   }
 /****  delete leading "forbidden" chars  ****/
 while (*s != '\0'  &&  strchr (chars_to_delete,*s) != NULL)
   {
    s++;
   }
 if (s != src)
   {
    strcpy (src,s);
    s = src;
   }

 /****  mask '|' and go to end of string  ****/
 while (*s != '\0')
   {
    if (*s == '|')
      {
       *s = '!';
      }
    s++;
   }
 /****  search for end of word  ****/
 while (s > src  &&  *(s-1) == ' ')
   {
    s--;
   }
 if (s >= src+3  &&  strcmp (s-3,"...") == 0)
   {
    /****  delete trailing "..."  ****/
    while (s > src  &&  (*(s-1) == '.'  ||  *(s-1) == ' '))
      {
       s--;
      }
    *s = '\0';
   }
}



static void split_title_from_name
      (char name[], int len, char title[], int len_t, int run_mode)
/****  len, len_t = string sizes incl. '\0'  ****/
{
 /****  This function splits an academic title from a name  ****/
 /****  (current version works for German only)             ****/
 int i,k;

 if (strcmp (name,"Prof") == 0
 ||  strcmp (name,"Professor") == 0)
   {
    strcpy (name+4,".");
   }
 if (strncmp (name,"Professor ",10) == 0)
   {
    strcpy (name+5, name+9);
    name[4] = '.';
   }

 if (strcmp (name,"Dr") == 0
 ||  strcmp (name,"Doctor") == 0
 ||  strcmp (name,"Doktor") == 0)
   {
    strcpy (name,"Dr.");
   }
 if (strncmp (name,"Doctor ",7) == 0
 ||  strncmp (name,"Doktor ",7) == 0)
   {
    strcpy (name+3, name+6);
    name[2] = '.';
   }

 if (run_mode & COMPARE_LANGUAGE_GERMAN)
   {
    if (strncmp (name,"Prof ",5) == 0)
      {
       name[4] = '.';
      }
    if (strncmp (name,"Dr ",3) == 0)
      {
       name[2] = '.';
      }

    if (strncmp (name,"Dr. Ing ",8) == 0
    ||  strncmp (name,"Dr. Ing.",8) == 0
    ||  strncmp (name,"Dr. ing ",8) == 0
    ||  strncmp (name,"Dr. ing.",8) == 0
    ||  strncmp (name,"Dr. med ",8) == 0
    ||  strncmp (name,"Dr. med.",8) == 0)
      {
       name[3] = '-';
      }

    i = -1;

    do{
       /****  look for one or more titles  ****/
       i++;
       if (name[i] != '\0'  &&   name[i] == ' ')
         {
          /****  ignore blanks before title  ****/
          i++;
         }

       k = i-1;
       while (IS_LETTER (name[i]))
         {
          /****  search for end of word  ****/
          i++;
         }

       /****  check, if length of word matches a title  ****/
      }while (i >= k+3   &&   i <= k+5   &&   name[i] == '.');

    k++;

    if ((k > 0  &&  strncmp (name,"Dr",2) == 0)
    ||  strncmp (name,"Prof",4) == 0
    ||  strncmp (name,"Dipl",4) == 0)
      {
       if (strncmp (name,"Dr.Ing ",7) == 0
       ||  strncmp (name,"Dr.Ing.",7) == 0
       ||  strncmp (name,"Dr.ing ",7) == 0
       ||  strncmp (name,"Dr.ing.",7) == 0
       ||  strncmp (name,"Dr.med ",7) == 0
       ||  strncmp (name,"Dr.med.",7) == 0)
         {
          k = 7;
         }

       if (strncmp (name,"Dr.-", 4) == 0
       ||  strncmp (name,"Dr. -",5) == 0
       ||  strncmp (name,"Dipl", 4) == 0
       ||  strncmp (name,"Prof. Dr.-", 10) == 0
       ||  strncmp (name,"Prof. Dr. -",11) == 0)
         {
          k = 3;
          if (strncmp (name, "Prof. Dr.",9) == 0)
            {
             k = 9;
            }

          while (IS_LETTER (name[k]))
            {
             /****  look for end of "Dipl*"  ****/
             k++;
            }
          while (name[k] != '\0'  &&  ! IS_LETTER (name[k]))
            {
             /****  search begin of word  ****/
             k++;
            }

          while (name[k] != '\0'
          &&  name[k] != ' '  &&  name[k] != '.')
            {
             /****  look for end of word  ****/
             k++;
            }
          if (name[k] == '.')
            {
             k++;
            }
         }

       /****  get title  ****/
       if (title != NULL)
         {
          i = (k < len_t) ?  (k+1) : len_t;
          StrNCpy (title,name,i);
          blank_cut (title,len_t);
         }

       while (name[k] == ' ')
         {
          k++;
         }
       strcpy (name, name+k);
      }
   }

 name [len-1] = '\0';
}



static void split_first_name_from_fam_name
      (char name[], int len, char first_name[], int len_f, int run_mode)
/****  len, len_f = string sizes incl. '\0'  ****/
{
 /****  This function splits first name from whole name,  ****/
 /****   e.g.:   [B.] [Dieter] [Meyer] von der Heyde      ****/
 char *s;
 int i,k,n,x;

 k = 0;
 for (i=0; name[i] != '\0'; i++)
   {
    if (i == 0
    ||  name[i-1] == ' '  ||  name[i-1] == '.')
      {
       /****  begin of word  ****/
       k = i;
       n = 0;

       if (run_mode & COMPARE_LANGUAGE_GERMAN)
         {
          for (x=0; common_german_family_names[x] != NULL; x++)
            {
             s = common_german_family_names[x];
             if (name[i] == *s)
               {
                n = (int) strlen (s);

                if (strncmp (name+i, s,n) == 0
                &&  strchr (" -", name[i+n]) != NULL)
                  {
                   /****  family name found  ****/
                   n = -1;
                   break;
                  }
               }
            }
         }

       if (n < 0)
         {
          /****  family name found  ****/
          break;
         }

       if (IS_LOWER (name[i])
       ||  name[i] == '\''  ||  name[i] == '`')
         {
          /****  noble name (e.g. "von", "'t") found  ****/
          break;
         }
      }
   }

 if (k > 0)
   {
    if (k > len_f)
      {
       k = len_f;
      }
    /****  get first name  ****/
    while (i > 0  &&  name[i-1] == ' ')
      {
       i--;
      }
    StrNCpy (first_name, name,i+1);

    /****  get family name  ****/
    while (name[k] == ' ')
      {
       k++;
      }
    strcpy (name, name+k);
   }

 name [len-1] = '\0';
}



static void format_gender_to_fam_name (char gender[], char first_name[],
    int len_first, char fam_name[], int len_fam, int run_mode)
{
 int  i,k,len;
 char *s, temp [LENGTH_WHOLE_NAME+1];

 if (gender[0] != IS_COMPANY)
   {
    len = (int) strlen (fam_name);

    for (i=0; (s=COMPANY_NAMES_MID[i]) != NULL; i++)
      {
       if (strstr (first_name,s) != NULL
       ||  strstr (fam_name,s) != NULL)
         {
          gender[0] = IS_COMPANY;
          gender[1] = '\0';
         }
      }

    for (i=0; (s=COMPANY_NAMES_TAIL[i]) != NULL; i++)
      {
       k = (int) strlen (s);
       if (len > k+1  &&  strcmp (fam_name+len-k, s) == 0)
         {
          gender[0] = IS_COMPANY;
          gender[1] = '\0';
         }
      }

    if (run_mode & COMPARE_LANGUAGE_GERMAN)
      {
       for (i=0; (s=FIRMA_GERMAN_MID[i]) != NULL; i++)
         {
          if (strstr (first_name,s) != NULL
          ||  strstr (fam_name,s) != NULL)
            {
             gender[0] = IS_COMPANY;
             gender[1] = '\0';
            }
         }

       for (i=0; (s=FIRMA_GERMAN_TAIL[i]) != NULL; i++)
         {
          k = (int) strlen (s);
          if (len > k  &&  strcmp (fam_name+len-k, s) == 0)
            {
             gender[0] = IS_COMPANY;
             gender[1] = '\0';
            }
         }
      }
   }

 if (gender[0] == IS_COMPANY  &&  first_name[0] != '\0')
   {
    if ((int) strlen (first_name) + (int) strlen (fam_name) < len_fam-1)
      {
       /****  "compress" company name  ****/
       StrNCpy (temp, fam_name, LENGTH_WHOLE_NAME+1);
       sprintf (fam_name, "%s %s", first_name, temp);
       blank_cut (fam_name, len_fam);
       first_name[0] = '\0';
      }
   }

 if (db_gender[0] == IS_MALE  ||  db_gender[0] == IS_FEMALE)
   {
    split_title_from_name (db_first_name,LENGTH_FIRST_NAME+1,
            NULL,0, run_mode);
    split_title_from_name (db_fam_name, LENGTH_FAM_NAME+1,
            NULL,0, run_mode);

    if (first_name[0] == '\0')
      {
       split_first_name_from_fam_name
            (fam_name, len_fam, first_name, len_first, run_mode);
      }

    if (run_mode & COMPARE_LANGUAGE_GERMAN)
      {
       if (adel_german_init == 0)
         {
           for (i=0; i< ADEL_GERMAN_COUNT; i++)
             {
               k = -1;
               if ((s = adel_german[i]) != NULL)
                 {
                   k = (int) strlen (s);
                 }
               adel_german_len[i] = k;
             }
           adel_german_init = 1;
         }

       for (i=0; i< ADEL_GERMAN_COUNT; i++)
         {
          /****  delete "adel" from first name and family name  ****/
          k = adel_german_len[i];

          if (k > 0  &&  strncmp (first_name, adel_german[i],k) == 0
          &&  first_name[k] == ' '
          &&  IS_LETTER (first_name[k+1]))
            {
              strcpy (first_name, first_name+k+1);
            }

          if (k > 0  &&  strncmp (fam_name, adel_german[i],k) == 0
          &&  fam_name[k] == ' '
          &&  IS_LETTER (fam_name[k+1]))
            {
             strcpy (fam_name, fam_name+k+1);
            }
         }
      }

    if (name_prefixes_init == 0)
      {
        for (i=0; i< NAME_PREFIXES_COUNT; i++)
          {
            k = -1;
            if ((s = name_prefixes[i]) != NULL)
              {
                k = (int) strlen (s);
              }
           name_prefixes_len[i] = k;
          }
        name_prefixes_init = 1;
      }

    len = (int) strlen (first_name);
    i = -1;
    k = 0;
    while (len > 0  &&  i < NAME_PREFIXES_COUNT)
      {
       /****  move all prefixes from first name to family name  ****/
       if (i < 0)
         {
          /****  look for prefix like (e.g.) "v.", "v.d." or "d'"  ****/
          k = 2;
          s = first_name + len -2;

          if (len > 4  &&  IS_LOWER (*s)
          && (*(s+1) == '.'  ||  *(s+1) == '\''))
            {
             if (*(s-1) == '.'  &&  len > 6
             &&  *(s-3) == ' '  &&  IS_LOWER (*(s-2)))
               {
                 s -= 2;
                 k = 4;
               }
             else if (*(s-1) != ' ')
               {
                 i = 0;
               }
            }
         }

       if (i >= 0)
         {
           s = name_prefixes[i];
           k = name_prefixes_len[i];
         }

       if (k > 0  &&  len > k+1
       &&  first_name [len-k-1] == ' '
       &&  strcmp (first_name+len-k, s) == 0)
         {
          if (k + (int) strlen (fam_name) < LENGTH_WHOLE_NAME)
            {
             sprintf (temp, "%s %s", s,fam_name);
             StrNCpy (fam_name, temp, len_fam);
             blank_cut (fam_name, len_fam);
            }
          first_name [len-k-1] = '\0';
          blank_cut (first_name, len_first);
          len = (int) strlen (first_name);

          /****  look for more prefixes  ****/
          i = -1;
          continue;
         }

       i++;
      }
   }
}



static void format_db_fields (int run_mode)
{
 int  i;
 char temp_first [LENGTH_FIRST_NAME+1];
 char temp_fam [LENGTH_FAM_NAME+1];

 /****  format database address  ****/
 for (i=0; db_addr[i].info != 0; i++)
   {
    format_string (db_addr[i].text);
    blank_cut (db_addr[i].text, 0);
   }

 format_street (db_street, db_street, run_mode);

 /****  TO-DO: If your database contains a "title"-field,
  ****  (for academic titles), activate the following lines of code
  ****  (and correct the database entry, if necessary):
  *
  *  if (db_gender[0] == IS_MALE  ||  db_gender[0] == IS_FEMALE)
  *    {
  *     if (db_title[0] == '\0')
  *       {
  *        split_title_from_name (db_first_name, LENGTH_FIRST_NAME+1,
  *              db_title, LENGTH_TITLE+1, run_mode);
  *       }
  *     if (db_title[0] == '\0')
  *       {
  *        split_title_from_name (db_fam_name, LENGTH_FAM_NAME+1,
  *              db_title, LENGTH_TITLE+1, run_mode);
  *       }
  *    }
  *
  ****/

 format_gender_to_fam_name (db_gender, db_first_name, LENGTH_FIRST_NAME+1,
       db_fam_name, LENGTH_FAM_NAME+1, run_mode);

 /****  TO-DO: If you do not want to search company names
  ****  for duplicates, activate the following lines of code:
  *
  *  if (db_gender[0] == IS_COMPANY)
  *    {
  *     return;
  *    }
  ****/

 if (db_gender[0] == IS_COMPANY  &&  db_first_name[0] == '\0')
   {
     if (strncmp (db_c_o_name,"C/O ",4) == 0
     ||  strncmp (db_c_o_name,"C/o ",4) == 0
     ||  strncmp (db_c_o_name,"c/o ",4) == 0)
       {
        strcpy (db_c_o_name, db_c_o_name+4);
       }

     if (run_mode & COMPARE_LANGUAGE_GERMAN)
       {
        if (strncmp (db_c_o_name,"Z.H.",4) == 0
        ||  strncmp (db_c_o_name,"z.H.",4) == 0)
          {
           strcpy (db_c_o_name, db_c_o_name+4);
          }
        if (strncmp (db_c_o_name,"Z.Hd.",5) == 0
        ||  strncmp (db_c_o_name,"z.Hd.",5) == 0)
          {
           strcpy (db_c_o_name, db_c_o_name+5);
          }
        if (strncmp (db_c_o_name,"Z. Hd.",6) == 0
        ||  strncmp (db_c_o_name,"z. Hd.",6) == 0)
          {
           strcpy (db_c_o_name, db_c_o_name+6);
          }
       }

     StrNCpy (db_first_name, db_c_o_name, LENGTH_FIRST_NAME+1);
     db_c_o_name[0] = '\0';
   }

 i = run_mode | COMPRESS_MULTIPLE_CHARS;
 (void) up_expand (db_first_name, my_first_name, LENGTH_FIRST_NAME+1, i);
 (void) up_expand (db_fam_name, my_fam_name, LENGTH_FAM_NAME+1, i);
 (void) up_expand (db_c_o_name, my_c_o_name, LENGTH_C_O_NAME+1, i);
 (void) up_expand (db_street,   my_street,   LENGTH_STREET+1,   i);
 (void) up_expand (db_country,  my_country,  LENGTH_COUNTRY+1,  i);

 i = run_mode & ~COMPRESS_MULTIPLE_CHARS;    /** !! **/
 (void) up_expand (db_zip_code,   my_zip_code,  LENGTH_ZIP_CODE+1, i);
 (void) up_expand (db_phone_number, my_phone_number, LENGTH_PHONE_NUMBER +1, i);
 (void) up_expand (db_mobile_number,my_mobile_number,LENGTH_MOBILE_NUMBER +1, i);
 (void) up_expand (db_iban_code,    my_iban_code,    LENGTH_IBAN_CODE +1, i);
 (void) up_expand (db_bank_account, my_bank_account, LENGTH_BANK_ACCOUNT +1, i);
 (void) up_expand (db_cust_number,  my_cust_number,  LENGTH_CUST_NUMBER+1, i);

 /****  delete suffixes ("jun." etc.) for phonetic name  ****/
 StrNCpy (temp_first, db_first_name, LENGTH_FIRST_NAME +1);
 StrNCpy (temp_fam, db_fam_name, LENGTH_FAM_NAME +1);
 (void) delete_suffix_jun (temp_first, temp_fam);

 /****  same formatting for MacGregor, McGregor and M'Gregor  ****/
 if (strncmp (temp_fam, "Mac",3) == 0
 &&  IS_UPPER (temp_fam[3]))
   {
    strcpy (temp_fam+1, temp_fam+2);
   }
 if ((strncmp (temp_fam, "M'",2) == 0
  ||  strncmp (temp_fam, "M",2) == 0)
 &&  IS_UPPER (temp_fam[2]))
   {
    temp_fam[1] = 'c';
   }

 /****  same formatting for "van den Berg" and "Vandenberg"  ****/
 if (strncmp (temp_fam, "van den ",8) == 0
 &&  IS_UPPER (temp_fam[8]))
   {
    strncpy (temp_fam, "Vanden",6);
    strcpy (temp_fam+6, temp_fam+8);
   }

 if (run_mode & COMPARE_LANGUAGE_GERMAN)
   {
    i = PHONET_GERMAN | PHONET_FIRST_RULES;
    (void) phonet (temp_first, db_first_name_ph, LENGTH_FIRST_NAME+1, i);
    (void) phonet (temp_fam, db_fam_name_ph, LENGTH_FAM_NAME+1, i);
    (void) phonet (db_street, db_street_ph, LENGTH_STREET+1, i);
   }
 else
   {
    (void) up_expand (temp_first, db_first_name_ph, LENGTH_FIRST_NAME+1, i);
    (void) up_expand (temp_fam, db_fam_name_ph, LENGTH_FAM_NAME+1, i);
    (void) up_expand (db_street, db_street_ph, LENGTH_STREET+1, i);
   }
}




static void compress_fam_name (char *text)
{
 char c;
 int i,k;

 k = 0;
 for (i=0; (c=text[i]) != '\0'; i++)
   {
    if (IS_XLETTER (c)
    || (c != '\''  &&  c != ''  &&  c != '`'))
      {
       text[k] = text[i];
       k++;
      }
   }
 text[k] = '\0';
}




static void compress_number (char *text)
{
 char c;
 int i,k;

 k = 0;
 for (i=0; (c=text[i]) != '\0'; i++)
   {
    if (IS_XLETTER (c))
      {
       text[k] = text[i];
       k++;
      }
   }
 text[k] = '\0';
}




static void write_line (FILE *fw, int step_no,
   char *fam_name_ph, char *first_name_ph, char *fam_name,
   char *first_name, char *phone_or_mobile)
{
 char prefix[80];
 char text[31],text2[21],text3[21],text4[21];
 char date_gender[13];

 StrNCpy (text2, db_b_day,  3);
 StrNCpy (text3, db_b_month,3);
 StrNCpy (text4, db_b_year, 5);
 StrNCpy (text, db_gender, 2);

 sprintf (date_gender, "%-4s-%2s-%2s%1s",
     text4, text3, text2, text);

 strcpy (prefix,"");

 /****  create 79-byte prefix  ****/
 switch (step_no)
   {
    case 1 :  StrNCpy (text, fam_name_ph, 31);
              compress_fam_name (text);
              StrNCpy (text2, first_name_ph, 21);
              StrNCpy (text3, date_gender, 12);
              StrNCpy (text4, my_c_o_name, 19);

              sprintf (prefix, "%-30s%-20s%-11s%-18s",
                   text, text2, text3, text4);
         break;

    case 2 :  StrNCpy (text2, my_zip_code, 16);
              compress_number (text2);
              StrNCpy (text, db_street_ph, 31);
              StrNCpy (text3, date_gender, 15);
              StrNCpy (text4, first_name_ph, 21);

              sprintf (prefix, "%-15s%-30s%-14s%-20s",
                   text2, text, text3, text4);
         break;

    case 3 :  StrNCpy (text2, my_zip_code, 10);
              compress_number (text2);
              StrNCpy (text3, fam_name_ph, 21);
              compress_fam_name (text3);
              StrNCpy (text, db_street_ph, 28);
              StrNCpy (text4, first_name_ph, 16);
              StrNCpy (text2+10, my_c_o_name, 9);

              sprintf (prefix, "%-9s%-20s%-27s%-15s%-8s",
                   text2, text3, text, text4, text2+10);
         break;

    case 4 :  StrNCpy (text2, date_gender, 12);
              StrNCpy (text3, first_name_ph, 21);
              StrNCpy (text, my_zip_code, 10);
              StrNCpy (text4, fam_name_ph, 21);
              StrNCpy (text+10, db_street_ph, 20);

              sprintf (prefix, "%-11s%-20s%-9s%-20s%-19s",
                   text2, text3, text, text4, text+10);

              date_gender[10] = '\0';
              if (strchr (date_gender+4,' ') != NULL)
                {
                 /****  birthday is (partially) empty  ****/
                 return;
                }

              if (! IS_XLETTER (text3[0])
              ||  ! IS_XLETTER (text3[1])
              ||  ! IS_XLETTER (text3[2]))
                {
                 /****  first name contains too few letters  ****/
                 return;
                }
          break;

    case 5 :  StrNCpy (text, my_cust_number, 30);
              compress_number (text);
              StrNCpy (text2, first_name_ph, 21);
              StrNCpy (text3, fam_name_ph, 21);
              StrNCpy (text4, my_zip_code, 11);

              sprintf (prefix, "%-29s%-20s%-20s%-10s",
                   text, text2, text3, text4);
         break;

    case 6 :  /***  phone number  ****/
              StrNCpy (text, phone_or_mobile, 31);
              compress_number (text);

              sprintf (prefix, "%-79s", text);
         break;

    case 7 :  /****  IBAN number + bank account  ****/
              StrNCpy (text, my_iban_code, 31);
              compress_number (text);
              StrNCpy (text2, my_bank_account, 21);
              compress_number (text2);

              sprintf (prefix, "%-35s%-44s", text, text2);

              if (! IS_XLETTER (text2[0]))
                {
                 /****  bank_account is empty  ****/
                 return;
                }
         break;

    default:  return;
   }

 if (prefix[0] != ' ')
   {
    fprintf (fw, "%s%d|%s|%s|%s|%s|%s|%s|%s|%s|%s|%s|%s|%s|%s|%s|%s|%ld|\n",
        prefix, step_no, fam_name_ph, first_name_ph, date_gender,
        fam_name, first_name, db_c_o_name, db_zip_code, db_country,
        db_street_ph, db_street, db_phone_number, db_mobile_number,
        db_iban_code, db_bank_account, db_cust_number, db_matchcode);
   }
}



static void write_to_unload_file (FILE *fw, int step_no, int result_get_gender)
{
 int  i;
 char temp_first [LENGTH_FIRST_NAME+1];

 write_line (fw, step_no, db_fam_name_ph, db_first_name_ph,
        db_fam_name, db_first_name, my_phone_number);

 /****  check for possible name inversions like   ****/
 /****  (e.g.) "Samuel, Paul" vs. "Paul, Samuel"  ****/
 if (db_first_name[0] != '\0'
 &&  strcmp (db_fam_name, db_first_name) != 0
 &&  result_get_gender != 0)
   {
    /****  Is it necessary to look for "name inversions" ?  ****/
    i = 0;
    while (IS_LETTER (db_first_name[i]))
      {
       i++;
      }
    if (i > 4  ||  (i > 2  &&  db_first_name[i] == '\0'))
      {
       /****  search for "name inversions"   ****/
       /****  (i.e. "first_name, fam_name")  ****/
       write_line (fw, step_no, db_first_name_ph, db_fam_name_ph,
              db_first_name, db_fam_name, my_phone_number);
      }
   }

 /****  check for first names like "John Carl"  ****/
 i = 0;
 while (db_first_name_ph[i] != '\0'
 &&  db_first_name_ph[i] != '-'  &&  db_first_name_ph[i] != ' ')
   {
    i++;
   }

 if (db_first_name_ph[i] != '\0')
   {
    /****  save first name  ****/
    StrNCpy (temp_first, db_first_name_ph, LENGTH_FIRST_NAME +1);

    if (i >= 4)
      {
       /****  add first part of phonetic name to unload file  ****/
       db_first_name_ph[i] = '\0';

       write_line (fw, step_no, db_fam_name_ph, db_first_name_ph,
              db_fam_name, db_first_name, my_phone_number);
      }

    i++;
    while (db_first_name_ph[i] == '-'  ||  db_first_name_ph[i] == ' ')
      {
       i++;
      }

    if (db_first_name_ph[i] != '\0')
      {
       /****  add second part of phonetic name to unload file  ****/
       strcpy (db_first_name_ph, db_first_name_ph+i);

       write_line (fw, step_no, db_fam_name_ph, db_first_name_ph,
              db_fam_name, db_first_name, my_phone_number);
      }

    /****  reset first name  ****/
    StrNCpy (db_first_name_ph, temp_first, LENGTH_FIRST_NAME +1);
   }

 /****  check for family names like "Miller-Smith"  ****/
 i = 0;
 while (db_fam_name_ph[i] != '\0'
 &&  db_fam_name_ph[i] != '-'  &&  db_fam_name_ph[i] != ' ')
   {
    i++;
   }

 if (db_fam_name_ph[i] != '\0')
   {
    if (i >= 4)
      {
       /****  add first part of phonetic name to unload file  ****/
       db_fam_name_ph[i] = '\0';

       write_line (fw, step_no, db_fam_name_ph, db_first_name_ph,
              db_fam_name, db_first_name, my_phone_number);
      }

    i++;
    while (db_fam_name_ph[i] == '-'  ||  db_fam_name_ph[i] == ' ')
      {
       i++;
      }

    if (db_fam_name_ph[i] != '\0')
      {
       /****  add second part of phonetic name to unload file  ****/
       strcpy (db_fam_name_ph, db_fam_name_ph+i);

       write_line (fw, step_no, db_fam_name_ph, db_first_name_ph,
              db_fam_name, db_first_name, my_phone_number);
      }
   }

 if (step_no == 6  &&  strcmp (my_mobile_number,"") != 0)
   {
    write_line (fw, step_no, db_fam_name_ph, db_first_name_ph,
           db_fam_name, db_first_name, my_mobile_number);
   }
}



static int read_unload_line (char text[])
/****  read family name, first name, c/o name, birthday, gender,  ****/
/****  street, zip code, country, customer number and matchcode   ****/
{
 int  i,k,n,len;
 char *s,date_gender[15];
 char dummy[3];

 /****  write command was:
    fprintf (fw, "%s%d|%s|%s|%s|%s|%s|%s|%s|%s|%s|%s|%s|%s|%s|%s|%s|%ld|\n",
        prefix, step_no, fam_name_ph, first_name_ph, date_gender,
        fam_name, first_name, db_c_o_name, db_zip_code, db_country,
        db_street_ph, db_street, db_phone_number, db_mobile_number,
        db_iban_code, db_bank_account, db_cust_number, db_matchcode);
 ****/

 struct MAIL_ADDR addr_line[] =
   { {   NULL,               2             },
     { db_fam_name_ph,   LENGTH_FAM_NAME   },
     { db_first_name_ph, LENGTH_FIRST_NAME },
     {   NULL,              14             },
     { db_fam_name,      LENGTH_FAM_NAME   },
     { db_first_name,    LENGTH_FIRST_NAME },
     { db_c_o_name,      LENGTH_C_O_NAME   },
     { db_zip_code,      LENGTH_ZIP_CODE   },
     { db_country,       LENGTH_COUNTRY    },
     { db_street_ph,     LENGTH_STREET     },
     { db_street,        LENGTH_STREET     },
     { db_phone_number,  LENGTH_PHONE_NUMBER },
     { db_mobile_number, LENGTH_MOBILE_NUMBER },
     { db_iban_code,     LENGTH_IBAN_CODE    },
     { db_bank_account,  LENGTH_BANK_ACCOUNT },
     { db_cust_number,   LENGTH_CUST_NUMBER  },
     {   NULL,               0             }
   };

 addr_line[0].text = dummy;
 addr_line[3].text = date_gender;

 i = 0;
 for (n=0; addr_line[n].info > 0; n++)
   {
    s = addr_line[n].text;
    len = addr_line[n].info;
    k = 0;
    while (text[i] != '|'  &&  text[i] != '\0')
      {
       if (k < len)
         {
          *(s+k) = text[i];
          k++;
         }
       i++;
      }
    *(s+k) = '\0';

    if (text[i] == '|')
      {
       i++;
      }
   }

 /****  read matchcode  ****/
 if (! IS_DIGIT (text[i]))
   {
    db_matchcode = -1L;
   }
 else
   {
    db_matchcode = atol (text+i);
   }

 if (strcmp (db_first_name,"") == 0)
   {
    StrNCpy (db_first_name, db_first_name_ph, LENGTH_FIRST_NAME+1);
   }
 if (strcmp (db_fam_name,"") == 0)
   {
    StrNCpy (db_fam_name, db_fam_name_ph, LENGTH_FAM_NAME+1);
   }
 if (strcmp (db_street,"") == 0)
   {
    StrNCpy (db_street, db_street_ph, LENGTH_STREET+1);
   }

 if ((int) strlen (date_gender) >= 11
 &&  date_gender[4] == '-'  &&  date_gender[7] == '-')
   {
    StrNCpy (db_b_year, date_gender,   LENGTH_BIRTH_YEAR+1);
    StrNCpy (db_b_month,date_gender+5, LENGTH_BIRTH_MONTH+1);
    StrNCpy (db_b_day,  date_gender+8, LENGTH_BIRTH_DAY+1);
    StrNCpy (db_gender, date_gender+10,LENGTH_GENDER+1);

    blank_cut (db_b_year, LENGTH_BIRTH_YEAR+1);
    blank_cut (db_b_month,LENGTH_BIRTH_MONTH+1);
    blank_cut (db_b_day,  LENGTH_BIRTH_DAY+1);
    blank_cut (db_gender, LENGTH_GENDER+1);

    /****  convert one-digit strings  into       ****/
    /****  two-digit strings (e.g. "5" -> "05")  ****/
    s = db_b_year;
    if (IS_DIGIT (*s)  &&  *(s+1) == '\0')
      {
       *(s+1) = *s;
       *s = '0';
       *(s+2) = '\0';
      }
    s = db_b_month;
    if (IS_DIGIT (*s)  &&  *(s+1) == '\0')
      {
       *(s+1) = *s;
       *s = '0';
       *(s+2) = '\0';
      }
    s = db_b_day;
    if (IS_DIGIT (*s)  &&  *(s+1) == '\0')
      {
       *(s+1) = *s;
       *s = '0';
       *(s+2) = '\0';
      }
   }
 else
   {
    strcpy (db_b_year, "");
    strcpy (db_b_month,"");
    strcpy (db_b_day,  "");
    strcpy (db_gender, "");
   }

 n = -1;
 if (i > 80)
   {
    n = atoi (text+79);
   }
 return (n);
}





/************************************************************/
/****  "external" functions  ********************************/
/************************************************************/

int do_unload (int file_count, char *file_list[], int run_mode)
{
 FILE *fw[MAX_UNLOAD_FILES];
 int  i,result_get_gender;

 if (! (conv_strings_initialized & CONV_STRINGS_ARE_INITIALIZED))
   {
    initialize_conv_strings (run_mode);

    if (! (conv_strings_initialized & CONV_STRINGS_ARE_INITIALIZED))
      {
       /****  internal error  ****/
       if (run_mode & (TRACE_ADDR | TRACE_ERRORS))
         {
           printf ("Internal error: Initialization failed\n");
         }
       return (ADDR_INTERNAL_ERROR);
      }
   }

 if (file_count > MAX_UNLOAD_FILES)
   {
    file_count = MAX_UNLOAD_FILES;
   }

 for (i=0; i < file_count; i++)
   {
    if (strcmp (file_list[i], "") == 0
    ||  strcmp (file_list[i], IGNORE_THIS_UNLOAD_FILE) == 0)
      {
       /****  skip unload for this file  ****/
       fw[i] = NULL;
       continue;
      }

    if ((fw[i] = fopen (file_list[i],"w")) == NULL)
      {
       if (run_mode & (TRACE_ADDR | TRACE_ERRORS))
         {
          printf ("Error: cannot create destination file '%s'\n", file_list[i]);
         }
       return (ADDR_CANNOT_CREATE_FILE);
      }
   }

 declare_cursor();

 if (sql_err_code != SQL_SUCCESS)
   {
    if (run_mode & (TRACE_ADDR | TRACE_ERRORS))
      {
       printf ("Error when doing SQL 'declare'-command\n");
      }
    return (ADDR_SQL_DECLARE_ERROR);
   }

 /****  unload all entries in table  ****/
 for (;;)
   {
    fetch_cursor();

    if (sql_err_code == SQL_SUCCESS)
      {
       format_db_fields (run_mode);

       result_get_gender = 0;

       if (db_first_name[0] != '\0'
       &&  check_valid_first_name (db_fam_name) > 0        /**  > 0  !! **/
       &&  check_valid_first_name (db_first_name) >= 0)    /**  >= 0  !! **/
         {
          result_get_gender = 1;
         }

       for (i=0; i < file_count; i++)
         {
          if (fw[i] != NULL)
            {
             write_to_unload_file (fw[i], i+1, result_get_gender);
            }
         }
      }
    else if (sql_err_code == SQL_NOT_FOUND)
      {
       break;
      }
    else
      {
       if (run_mode & (TRACE_ADDR | TRACE_ERRORS))
         {
          printf ("Error when doing SQL 'fetch'-command\n");
         }
       return (ADDR_SQL_FETCH_ERROR);
      }
   }

 for (i=0; i < file_count; i++)
   {
    if (fw[i] != NULL)
      {
       fclose (fw[i]);
      }
   }

 close_cursor();

 if (sql_err_code != SQL_SUCCESS)
   {
    if (run_mode & (TRACE_ADDR | TRACE_ERRORS))
      {
       printf ("Error when doing SQL 'close'-command\n");
      }
    return (ADDR_SQL_CLOSE_ERROR);
   }

 return (0);
}





static char db_text[601];
static char my_text[601];

int search_duplicates (char *unload_file,
         char *dest_file, int min_points, int run_mode)
{
 FILE *fr,*fw;
 int  i,n,points;

 if (! (conv_strings_initialized & CONV_STRINGS_ARE_INITIALIZED))
   {
    initialize_conv_strings (run_mode);

    if (! (conv_strings_initialized & CONV_STRINGS_ARE_INITIALIZED))
      {
       /****  internal error  ****/
       if (run_mode & (TRACE_ADDR | TRACE_ERRORS))
         {
           printf ("Internal error: Initialization failed\n");
         }
        return (ADDR_INTERNAL_ERROR);
      }
   }

 if ((fr = fopen (unload_file,"r")) == NULL)
   {
     if (run_mode & (TRACE_ADDR | TRACE_ERRORS))
       {
         printf ("Error: cannot open unload file '%s'\n", unload_file);
       }
     return (ADDR_CANNOT_READ_FILE);
   }

 fw = stdout;
 if (dest_file != NULL  &&  (fw = fopen (dest_file,"w")) == NULL)
   {
    if (run_mode & (TRACE_ADDR | TRACE_ERRORS))
      {
        printf ("Error: cannot create destination file '%s'\n", dest_file);
      }
    fclose (fr);
    return (ADDR_CANNOT_CREATE_FILE);
   }

 if (min_points < MIN_POINTS_DUPLICATES - 3)
   {
    min_points = MIN_POINTS_DUPLICATES - 3;
    if (run_mode & TRACE_ADDR)
      {
        printf ("Note: min_points has been raised to %d\n", min_points);
      }
   }
 if (min_points >= MAX_POINTS)
   {
     min_points = MAX_POINTS -1;
   }

 /****  look for duplicates  ****/
 points = 0;
 strcpy (db_text,"");
 db_matchcode = -1L;

 strcpy (db_gender, "");
 strcpy (db_first_name,"");
 strcpy (db_fam_name,"");
 strcpy (db_c_o_name,"");
 strcpy (db_street,  "");
 strcpy (db_zip_code,"");
 strcpy (db_country, "");
 strcpy (db_b_day,   "");
 strcpy (db_b_month, "");
 strcpy (db_b_year,  "");
 strcpy (db_phone_number,"");
 strcpy (db_mobile_number,"");
 strcpy (db_iban_code,   "");
 strcpy (db_bank_account,"");
 strcpy (db_cust_number, "");

 while (! feof (fr))
   {
     StrNCpy (my_text, db_text, 601);
     my_matchcode = db_matchcode;

     StrNCpy (my_gender,    db_gender,    LENGTH_GENDER +1);
     StrNCpy (my_first_name,db_first_name,LENGTH_FIRST_NAME +1);
     StrNCpy (my_fam_name,  db_fam_name,  LENGTH_FAM_NAME +1);
     StrNCpy (my_c_o_name,  db_c_o_name,  LENGTH_C_O_NAME +1);
     StrNCpy (my_street,    db_street,    LENGTH_STREET +1);
     StrNCpy (my_zip_code,  db_zip_code,  LENGTH_ZIP_CODE +1);
     StrNCpy (my_country,   db_country,   LENGTH_COUNTRY +1);
     StrNCpy (my_b_day,     db_b_day,     LENGTH_BIRTH_DAY +1);
     StrNCpy (my_b_month,   db_b_month,   LENGTH_BIRTH_MONTH +1);
     StrNCpy (my_b_year,    db_b_year,    LENGTH_BIRTH_YEAR +1);
     StrNCpy (my_phone_number, db_phone_number, LENGTH_PHONE_NUMBER +1);
     StrNCpy (my_mobile_number,db_mobile_number,LENGTH_MOBILE_NUMBER +1);
     StrNCpy (my_iban_code,    db_iban_code,    LENGTH_IBAN_CODE +1);
     StrNCpy (my_bank_account, db_bank_account, LENGTH_BANK_ACCOUNT +1);
     StrNCpy (my_cust_number,  db_cust_number,  LENGTH_CUST_NUMBER +1);

    /****  read data  ****/
    if (fgets (db_text,600,fr) != NULL)
      {
        i = (int) strlen (db_text);
        if (i > 0  &&  db_text[i-1] == '\n')
          {
           /****  important  ****/
           db_text[i-1] = '\0';

           if (i <= 82)
             {
               /****  line is too short  ****/
               continue;
             }
         }

       /****  read "db_*"-variables  ****/
       i = read_unload_line (db_text);

       /****  search for duplicates  ****/
       if (my_matchcode < 0L  ||  db_matchcode < 0L
       ||  my_matchcode == db_matchcode
       || (int)strlen (my_text) < 10  ||  (int)strlen (db_text) < 10)
         {
           points = 0;
           continue;
         }

       n = 0;
       if (i == 4)
         {
          /****  raise "min_points" for unload file 4  ****/
          n = 93 - min_points;

          if (min_points >= 92) n = 94 - min_points;
          if (min_points >= 94) n = 0;
         }

       /****  compare addresses  ****/
       points = compare_addr (my_addr, db_addr,
                     min_points+n, (run_mode | SKIP_BLANKCUT));

       if (points >= min_points)
         {
          /****  a duplicate has been found  ****/
          if (my_matchcode < db_matchcode)
            {
              fprintf (fw, "%3d points (min = %2d):  '%s'   '%s'\n",
                   points, min_points, my_text+81, db_text+81);
            }
          else
            {
              fprintf (fw, "%3d points (min = %2d):  '%s'   '%s'\n",
                   points, min_points, db_text+81, my_text+81);
            }
         }
      }
   }

 if (fw != stdout)
   {
     fclose (fw);
   }
 fclose (fr);
 return (0);
}





#ifdef DEDUPL_EXECUTABLE

int main (int argc, char *argv[])
{
 if (argc < 3
 ||  strcmp (argv[1], "-?") == 0
 ||  strcmp (argv[1], "-h") == 0
 ||  strcmp (argv[1], "-help") == 0)
   {
    printf ("Usage:  %s  -unload  <dest_file>  [ <dest_file2> .. [<dest_file7>] ]\n",
        argv[0]);
    printf (" or  :  %s  -search_duplicates  <sorted_unload_file>  <result_file>  [ <min_points> ]\n",
        argv[0]);
    printf ("            (default for \"min_points\" is %d)\n", MIN_POINTS_DUPLICATES);
    printf ("\n");
    printf ("Program for searching duplicates in a database\n");
    printf ("\n");
    return (1);
   }

 if (strcmp (argv[1], "-unload") == 0)
   {
    if (argc <= 2)
      {
       printf ("Error: wrong # of arguments.\n");
       return (1);
      }

    open_database();
    if (sql_err_code != SQL_SUCCESS)
      {
       printf ("Error when opening database\n");
       cleanup_addr();
       return (1);
      }

    (void) do_unload (argc-2, argv+2, RUN_MODE);

    close_database();
    if (sql_err_code != SQL_SUCCESS)
      {
       printf ("Error when closing database\n");
       cleanup_addr();
       return (1);
      }
   }

 if (strcmp (argv[1], "-search_duplicates") == 0)
   {
    int i = MIN_POINTS_DUPLICATES;
    if (argc > 4)
      {
       i = atoi (argv[4]);
      }

    if (argc <= 3)
      {
        printf ("Error: wrong # of arguments.\n");
        return (1);
      }

    (void) search_duplicates (argv[2], argv[3], i, RUN_MODE);
   }

 cleanup_addr();
 return (0);
}

#endif

/************************************************************/
/****  end of file "dedupl.c"  ******************************/
/************************************************************/
