// Copyright (c) 2008 Oliver Lau <ola@ctmagazin.de>
// Heise Zeitschriften Verlag, Hannover, Germany

#ifndef __THREADPOOL_H_
#define __THREADPOOL_H_

#include <QObject>
#include <QRunnable>
#include <QThread>
#include <QMutex>
#include <QSemaphore>
#include <QQueue>

#if _MSC_VER >= 1500
#define HAS_MM_MONITOR
#include <intrin.h>
#endif

class WorkerThread;

typedef QRunnable Job;
typedef QQueue< Job* > JobQueue;
typedef QList< WorkerThread* > WorkerThreadList;

class ThreadPool : public QObject
{
    Q_OBJECT

public:
    /// ThreadPool constructor.
    /// Create a thread pool with the given number of worker threads.
    ThreadPool(int numThreads = 1, QThread::Priority p = QThread::InheritPriority);

    /// ThreadPool destructor.
    /// Wait for all worker threads to return, clear job queue.
    /// This method is thread-safe.
    ~ThreadPool();

    void setMaxThreadCount(int);

    /// Add a job to the job queue.
    /// This method is thread-safe.
    void enqueue(Job*);

    /// Add a list of jobs to the job queue.
    /// This method is thread-safe.
	void enqueue(const JobQueue&);

    /// Wait for worker threads to finish work
    void waitForFinished(void);

    /// Wait for worker threads to send ready event.
    void waitForTerminated(void);

    /// Start worker threads unless already running (\see start()).
    /// Then signal worker threads to take up their jobs (\see signalAll()).
    void run(void);

    /// Signal worker threads to quit (\see quitAll()),
    /// then trigger worker threads to process the job queue (\see signalAll()).
    void cancel(void);

	/// Return true if ThreadPool ended unexpectedly.
	bool isCanceled(void);

	/// Return next job in queue (returns NULL if queue is empty).
    Job* nextJob(void);

	/// This function is being called from a worker thread. 
	/// Calling this function tells the thread pool that the
	/// worker thread has finished *all* jobs, i.e. the job
	/// queue is empty.
	void jobsDone(void);

	/// This function is being called from a worker thread. 
	/// Calling this function tells the thread pool that the
	/// worker thread has finished a *single* job.
	void jobDone(WorkerThread*);

	/// Set priority of all threads in pool.
	void setPriority(QThread::Priority);

signals:
	void finished(void);
	void progressRangeChanged(int, int);
	void progressValueChanged(int);

private:
    WorkerThreadList pool;
    QSemaphore ready;
	int requestedPoolSize;
	int numThreadsCompleted;
    QMutex jobMutex;
	QMutex poolMutex;
    QQueue< Job *> jobs;
	int numJobs0;
	bool canceled;
	bool finishedSignalSent;
	QThread::Priority priority;
	
    /// Trigger worker threads to process the job queue.
    void signalAll(void);

    /// Signal worker threads to quit by setting the doQuit flag.
    void quitAll(void);

	/// Terminate thread pointed to by WorkerThreadList::iterator
	/// and remove it from the pool. 
	/// This function is thread-safe.
	void eraseThread(WorkerThreadList::iterator i);
};

#endif // __THREADPOOL_H_
