#!/usr/bin/perl -w
#
# Aus GPS-Tracks (GPX) und korrespondierenden JPG-Fotos in sich
# geschlossene virtuelle Sightseeing-Touren fuer Google Earth
# im KMZ-Format erzeugen.
#
# Copyright (c) 2008 Oliver Lau <ola@ctmagazin.de>
# Copyright (c) 2008 Heise Zeitschriften Verlag
# Alle Rechte vorbehalten.


#########################################################
#
# GPS::Helper
#
#########################################################

package GPS::Helper;

use warnings;
use strict;
use DateTime::Format::Strptime;
use POSIX qw( floor );

use constant LATITUDE => 'lat';
use constant LONGITUDE => 'lon';

sub get_single_value($)
{
    my $v = shift;
    return $v->[0] if ref($v) eq 'ARRAY';
    return $v;
}

=head2 normalize_duration

C<DateTime::Duration> doesn't return hours, which is a pity. C<normalize_duration()> transforms minutes into hours if necessary, and adjusts
higher order values (days ...) accordingly:

    $t0 = DateTime->new( year   => 2008,
                         month  => 2,
                         day    => 23,
                         hour   => 12,
                         minute => 12,
                         second => 47,
                     );
    $t1 = DateTime->new( year   => 2008,
                         month  => 2,
                         day    => 23,
                         hour   => 17,
                         minute => 23,
                         second => 59,
                     );
    $duration = $t1->subtract($t0);
    my %dur;
    normalize_duration(\%dur, $duration);
    printf( '%d:%02d:%02d', $dur{'hours'}, $dur{'minutes'}, $dur{'seconds'} );

prints

    5:11:12

=cut

sub normalize_duration($$)
{
    my ( $d, $res ) = @_;
    my %d = $d->deltas;
    while (my ($key, $val) = each %d)
    {
        $res->{$key} = $val;
    }
    $res->{minutes} += int( $res->{seconds} / 60 );
    $res->{seconds} %= 60;
    $res->{hours}   += int( $res->{minutes} / 60 );
    $res->{minutes} %= 60;
    $res->{days}    += int( $res->{hours} / 24 );
    $res->{hours}   %= 24;
}

=head2 dec_to_dms

Convert a decimal GPS position to "degrees minutes seconds" representation:

    @lat = dec_to_dms(  28.259221, 'lat' );
    @lon = dec_to_dns( -16.669854, 'lon' );
    print join ' ', @lat, "\n";
    print join ' ', @lon, "\n";

prints

    N 28 15 33.19596
    W 16 40 11.47296

The second argument is optional. When omitting it, the first element in the
list contains the sign:

    @lat = dec_to_dms(  28.259221, 'lat' );
    @lon = dec_to_dns( -16.669854, 'lon' );
    print join ' ', @lat, "\n";
    print join ' ', @lon, "\n";

prints

    1 28 15 33.19596
    -1 16 40 11.47296

=cut

sub dec_to_dms($;$)
{
    my ( $dec, $ref ) = @_;
    my $sign;
    if ($ref)
    {
        $sign = ( $ref =~ /^lat/i )? (( $dec < 0 ) ? 'S' : 'N') : (( $dec < 0 ) ? 'W' : 'E');
    }
    else
    {
        $sign = ( $dec < 0 ) ? -1 : 1;
    }
    $dec = abs($dec);
    my $deg = floor($dec);
    my $min_ = ( $dec - $deg ) * 60;
    my $min = floor($min_);
    my $sec = sprintf( '%.5f', ( $min_ - $min ) * 60 );
    return ( $sign, $deg, $min, $sec );
}

=head2 new_datetime

Construct a C<DateTime> object from an ISO 8601 formatted date/time string:

    my $dt = new_datetime('2008-02-12T11:45:17Z');

Valid dates are:

=over

=item C<YYYY-MM-DDThh:mmTZD>

e.g. 1997-07-16T19:20+01:00

=item C<YYYY-MM-DDThh:mm:ssTZD>

e.g. 1997-07-16T19:20:30+01:00

=item C<YYYY-MM-DDThh:mm:ss.sTZD>

e.g. 1997-07-16T19:20:30.45+01:00

=back

where:

    YYYY = four-digit year
    MM   = two-digit month (01=January, etc.)
    DD   = two-digit day of month (01 through 31)
    hh   = two digits of hour (00 through 23) (am/pm NOT allowed)
    mm   = two digits of minute (00 through 59)
    ss   = two digits of second (00 through 59)
    s    = one or more digits representing a decimal fraction of a second
    TZD  = time zone designator (Z or +hh:mm or -hh:mm)

See http://www.w3.org/TR/NOTE-datetime

=cut

sub new_datetime($)
{
    my ( $dt ) = @_;
    my ( $year, $month, $day, $h, $m, $s, $sfrac, $tzd ) =
      $dt =~ /(\d{4})-(\d{2})-(\d{2})T(\d{2}):(\d{2}):(\d{2})(\.\d+)?(Z|([+-]\d{2}:\d{2}))?/;
    return DateTime->new(
                year       => $year,
                month      => $month,
                day        => $day,
                hour       => $h,
                minute     => $m,
                second     => $s,
                nanosecond => (defined($sfrac) ? $sfrac * 1000000000 : 0),
                time_zone  => ((defined($tzd) && $tzd =~ /[+-]\d{2}:\d{2}/) ? $tzd : 'Europe/Berlin'),
                formatter  => DateTime::Format::Strptime->new(pattern => '%d.%m.%Y %H:%M:%S'),
            );
}

#########################################################
#
# GPS::Trackpoint
#
#########################################################

package GPS::Trackpoint;

use warnings;
use strict;

use Math::Trig;
use Carp;

our $VERSION = '0.1';

our $ellipsoid = 'WGS84';

our %ellipsoids = (
    'AIRY'               => [ 6377563.396, 1 / 299.3249646     ],
    'AIRY-MODIFIED'      => [ 6377340.189, 1 / 299.3249646     ],
    'AUSTRALIAN'         => [ 6378160.0,   1 / 298.25          ],
    'BESSEL-1841'        => [ 6377397.155, 1 / 299.1528128     ],
    'CLARKE-1880'        => [ 6378249.145, 1 / 293.465         ],
    'EVEREST-1830'       => [ 6377276.345, 1 / 300.8017        ],
    'EVEREST-MODIFIED'   => [ 6377304.063, 1 / 300.8017        ],
    'FISHER-1960'        => [ 6378166.0,   1 / 298.3           ],
    'FISHER-1968'        => [ 6378150.0,   1 / 298.3           ],
    'GRS80'              => [ 6378137.0,   1 / 298.25722210088 ],
    'HOUGH-1956'         => [ 6378270.0,   1 / 297.0           ],
    'HAYFORD'            => [ 6378388.0,   1 / 297.0           ],
    'IAU76'              => [ 6378140.0,   1 / 298.257         ],
    'KRASSOVSKY-1938'    => [ 6378245.0,   1 / 298.3           ],
    'NAD27'              => [ 6378206.4,   1 / 294.9786982138  ],
    'NWL-9D'             => [ 6378145.0,   1 / 298.25          ],
    'SOUTHAMERICAN-1969' => [ 6378160.0,   1 / 298.25          ],
    'SOVIET-1985'        => [ 6378136.0,   1 / 298.257         ],
    'WGS72'              => [ 6378135.0,   1 / 298.26          ],
    'WGS84'              => [ 6378137.0,   1 / 298.257223563   ],
);

sub new
{
    my $class = shift;
    my $self = { };
    bless $self, $class;
    $self->_init(@_);
    return $self;
}

sub _init
{
    my $self = shift;
    if (@_)
    {
        my %extra = @_;
        @$self{keys %extra} = values %extra;
    }
    $self->{Distance} = 0 unless exists $self->{Distance};
}

sub latitude()
{
    my $self = shift;
    return $self->{Latitude};
}

sub longitude()
{
    my $self = shift;
    return $self->{Longitude};
}

sub elevation()
{
    my $self = shift;
    return $self->{Elevation};
}

sub distance()
{
    my $self = shift;
    return $self->{Distance};
}

sub set_distance($)
{
    my $self = shift;
    $self->{Distance} = shift;
}

sub timestamp()
{
    my $self = shift;
    return $self->{Timestamp};
}

sub set_timestamp($)
{
    my $self = shift;
    $self->{Timestamp} = shift;
}

sub range_to($)
{
    my $self = shift;
    my $other = shift;
    my ( $radius, $flattening ) = @{ $ellipsoids{$ellipsoid} };
    my $F = deg2rad( ( $self->{Latitude} + $other->{Latitude} ) / 2 );
    my $G = deg2rad( ( $self->{Latitude} - $other->{Latitude} ) / 2 );
    my $l = deg2rad( ( $self->{Longitude} - $other->{Longitude} ) / 2 );
    my $S =
      sin($G) * sin($G) * cos($l) * cos($l) + cos($F) * cos($F) * sin($l) *
      sin($l);
    my $C =
      cos($G) * cos($G) * cos($l) * cos($l) + sin($F) * sin($F) * sin($l) *
      sin($l);
    return -1 if $S == 0 || $C == 0;
    my $o  = atan( sqrt( $S / $C ) );
    my $R  = sqrt( $S / $C ) / $o;
    my $D  = 2 * $o * $radius;
    my $H1 = ( 3 * $R - 1 ) / ( 2 * $C );
    my $H2 = ( 3 * $R + 1 ) / ( 2 * $S );
    my $s  =
      $D * ( 1 + $flattening * $H1 * sin($F) * sin($F) * cos($G) * cos($G) -
          $flattening * $H2 * cos($F) * cos($F) * sin($G) * sin($G) );
    return $s;
}


#########################################################
#
# GPS::Track
#
#########################################################

package GPS::Track;

use strict;
use warnings;

use Carp;
use XML::Simple;
use DateTime::Format::Strptime;
use Data::Dumper qw( Dumper );

our $VERSION = '0.1';

sub new
{
    my $class = shift;
    my $self = { };
    bless $self, $class;
    $self->_init;
    my %args;
    if (ref $_[0] eq 'HASH') { %args = %{$_[0]}; }
    elsif (not ref $_[0])    { %args = @_; }
    else {
        carp "Usage: $self->new( { key => values, } )";
        return undef;
    }
    $self->load_gpx_file($args{FROM_GPX_FILE}) if $args{FROM_GPX_FILE};
    return $self;
}

sub _init
{
    my $self = shift;
    $self->{Name} = '';
    $self->{Trackpoints} = [];
    $self->{MinHeight} = +1e37;
    $self->{MaxHeight} = -1e37;
}

sub points()
{
    my $self = shift;
    return $self->{Trackpoints};
}

=head2 min_elevation

Return minimum elevation along track

    $track->min_elevation

=cut

sub min_elevation()
{
    my $self = shift;
    return $self->{MinHeight};
}

=head2 max_elevation

Return maximum elevation along track

    $track->max_elevation

=cut

sub max_elevation()
{
    my $self = shift;
    return $self->{MaxHeight};
}

=head2 name

Get name of track

    $name = $track->name;

=cut

sub name()
{
    my $self = shift;
    return $self->{Name};
}

=head2 set_name

Set name of track

    $track->set_name($name);

=cut

sub set_name($)
{
    my $self = shift;
    $self->{Name} = shift;
}

=head2 distance

Get distance of track.

    $distance = $track->distance;

=cut

sub distance()
{
    my $self = shift;
    return $self->{Trackpoints}->[ scalar @{$self->{Trackpoints}} - 1 ]->distance;
}

sub _calculate_distances()
{
    my $self = shift;
    my $other = $self->{Trackpoints}->[0];
    foreach my $i (2 .. scalar @{$self->{Trackpoints}} - 1)
    {
        $self->{Trackpoints}->[$i]->set_distance($self->{Trackpoints}->[$i-1]->distance + $self->{Trackpoints}->[$i - 1]->range_to($self->{Trackpoints}->[$i]));
    }
}

=head2 start_datetime

Get C<DateTime> at start of track.

    $dt = $track->start_datetime;

=cut

sub start_datetime()
{
    my $self = shift;
    return $self->{Trackpoints}->[0]->timestamp;
}

=head2 end_datetime

Get C<DateTime> at end of track.

    $dt = $track->end_datetime;

=cut

sub end_datetime()
{
    my $self = shift;
    return $self->{Trackpoints}->[ scalar @{$self->{Trackpoints}} - 1 ]->timestamp;
}

=head2 duration

Get duration as C<DateTime::Duration> object.

    $duration = $track->duration;

=cut

sub duration()
{
    my $self = shift;
    my $t0 = GPS::Helper::new_datetime($self->start_datetime);
    my $t1 = GPS::Helper::new_datetime($self->end_datetime);
    return $t1->subtract_datetime($t0);
}


sub _bin_search_timestamp($$$)
{
    my $self = shift;
    my ( $value, $low, $high ) = @_;
    my $mid = int( ( $low + $high ) / 2 );
    return $mid if $high < $low;
    if ( $self->{Trackpoints}->[$mid]->timestamp gt $value )
    {
        return $self->_bin_search_timestamp( $value, $low, $mid - 1 );
    }
    elsif ( $self->{Trackpoints}->[$mid]->timestamp lt $value )
    {
        return $self->_bin_search_timestamp( $value, $mid + 1, $high );
    }
    else
    {
        return $mid;
    }
}

=head2 get_matching_point

Find a track point whose timestamp matches closest a given one

    $matched_point = get_matching_point(
        Timestamp => '2008-01-26T13:17:04Z'
    );

=cut

sub get_matching_point(%)
{
    my $self = shift;
    my %args;
    if (ref $_[0] eq 'HASH') { %args = %{$_[0]}; }
    elsif (not ref $_[0])    { %args = @_; }
    else {
        carp "Usage: $self->new( { key => values, } )";
        return undef;
    }
    # must do a range check because bin_search() does a
    # nearest neighbor match but not an exact match
    return undef if ( $args{Timestamp} lt $self->start_datetime ) || ( $args{Timestamp} gt $self->end_datetime );
    my $idx = $self->_bin_search_timestamp( $args{Timestamp}, 0, scalar @{ $self->{Trackpoints} } - 1 );
    return $self->{Trackpoints}->[$idx];
}


sub _append_trackpoint
{
    my $self = shift;
    my $trackpoint = shift;
    push @{$self->{Trackpoints}}, $trackpoint;
}

sub _append_gpx_trkpts
{
    my $self = shift;
    my $trkpts = shift;
    foreach ( @{$trkpts} )
    {
        my $ele = sprintf( '%.0f', GPS::Helper::get_single_value($_->{ele}) );
        $self->{MaxHeight} = $ele if $ele > $self->{MaxHeight};
        $self->{MinHeight} = $ele if $ele < $self->{MinHeight};
        $self->_append_trackpoint(
            new GPS::Trackpoint(
                    Distance  => 0,
                    Elevation => $ele,
                    Latitude  => GPS::Helper::get_single_value($_->{lat}),
                    Longitude => GPS::Helper::get_single_value($_->{lon}),
                    Timestamp => GPS::Helper::get_single_value($_->{time}),
                )
        );
    }
}

=head2 append_gpx_file

Append the tracks of a GPX file to the trackpoint list.

    $track->append_gpx_file('/path/to/file.gpx');

=cut

sub append_gpx_file($)
{
    my $self = shift;
    my $gpx_file = shift;
    my $gpx = XMLin($gpx_file, ForceArray => 1, KeyAttr => []);
    $gpx->{trk} or croak "GPX file does not contain a track";
    foreach my $trk ( @{ $gpx->{trk} } )
    {
        foreach my $trkseg ( @{ $trk->{trkseg} } )
        {
            $self->_append_gpx_trkpts( $trkseg->{trkpt} );
        }
    }
    $self->_calculate_distances;
    return 0;
}

=head2 load_gpx_file

Load the tracks of a GPX file into the trackpoint list.
Existing trackpoints will be deleted in front.

    $track->load_gpx_file('/path/to/file.gpx');

=cut

sub load_gpx_file($)
{
    my $self = shift;
    my $gpx_file = shift;
    $self->_init;
    $self->append_gpx_file($gpx_file);
    return 0;
}


#########################################################
#
# main
#
#########################################################

package main;

#
# Module
#
use strict;
use warnings;
use XML::Simple qw( :strict );
use Data::Dumper;
use Carp;
use DateTime;
use DateTime::Format::Duration;
use DateTime::Format::Strptime;
use Math::Trig;
use Getopt::Long;
use Archive::Zip qw( :ERROR_CODES :CONSTANTS );
use Image::ExifTool qw( :Public );
use Image::Thumbnail;
use Term::ReadKey;
use Config::Auto;

use constant TRUE  => 1;
use constant FALSE => 0;

#
# Globale Variablen
#
my $config_file = 'geotag.conf';
my $config = Config::Auto::parse( $config_file, format => 'equal' ) if -f $config_file;

my $DEBUG    = $config->{debug}   || FALSE;
my $img_dir  = $config->{imgdir}  || './images';
my $gps_file = $config->{gpsfile} || './track.gpx';
my $doc_name = $config->{docname} || '<unnamed>';
$doc_name =~ tr/_/ /;
my $time_zone         = $config->{timezone}         || 'Europe/Berlin';
my $t_offset          = $config->{toffset}          || '+00:00:00';
my $max_dim           = $config->{maxdim}           || 640;
my $create_thumbnails = $config->{createthumbnails} || FALSE;
my $rough_track       = $config->{rough_track}      || FALSE;
my $kmz_file          = $config->{kmzfile}          || 'tour.kmz';
my $thumb_dir         = $config->{thumbdir}         || './thumbs';
my $tmp_kml_file      = $config->{tmpkmlfile}       || './tour.kml';
my $QUIET             = $config->{quiet}            || FALSE;
my $HELP              = FALSE;
my $dt_formatter      = DateTime::Format::Strptime->new(pattern => '%d.%m.%Y %H:%M:%S');
my $dt_isoformatter   = DateTime::Format::Strptime->new(pattern => '%Y-%m-%dT%H:%M:%SZ');

sub disclaimer();
sub USAGE();

#
# Und los gehts ...
#
GetOptions(
            'timeoffset=s'     => \$t_offset,
            'imgdir=s'         => \$img_dir,
            'gpsfile=s'        => \$gps_file,
            'kmzfile=s'        => \$kmz_file,
            'thumbdir=s'       => \$thumb_dir,
            'description=s'    => \$doc_name,
            'createthumbnails' => \$create_thumbnails,
            'maxdim=i'         => \$max_dim,
            'quiet'            => \$QUIET,
            'debug'            => \$DEBUG,
            'help'             => \$HELP,
);

$DEBUG = $DEBUG && !$QUIET;

disclaimer() unless $QUIET;

USAGE() if $HELP;
USAGE() unless $img_dir && $gps_file && $kmz_file;

my ( $offset_sign, $h_offset, $m_offset, $s_offset ) =
  $t_offset =~ /([+-])(\d{2}).(\d{2}).(\d{2})/;
USAGE() unless $offset_sign;

if ( ( $doc_name eq '' ) && !$QUIET )
{
    print "Eine Beschreibung der Sightseeing-Tour fehlt.\n",
      "Letzte Moeglichkeit, eine einzugeben.\n",
      "(Zum Fortsetzen ohne Beschreibung [ENTER] druecken): ";
    $doc_name = ReadLine(0);
}
$doc_name =~ s/[\n\r]//g;

unlink $tmp_kml_file if -e $tmp_kml_file;
unlink $kmz_file     if -e $kmz_file;
unlink $thumb_dir    if -f $thumb_dir;
mkdir $thumb_dir unless -d $thumb_dir;

-d $img_dir or croak "Bildverzeichnis '$img_dir' existiert nicht: $!";

-f $gps_file or croak "Kann GPS-Daten nicht aus '$gps_file' lesen: $!";

print "Lesen der GPS-Daten aus '$gps_file' ..\n" unless $QUIET;

my $track = GPS::Track->new( FROM_GPX_FILE => $gps_file );

if ($track->start_datetime && $track->end_datetime)
{
    print "  Zeit: ", $track->start_datetime, "\n",
          "      - ", $track->end_datetime, "\n";
    my %dur;
    GPS::Helper::normalize_duration($track->duration, \%dur);
    print "  Dauer: ",
      (
          ( $dur{days} > 0 )
        ? ( sprintf( '%d Tag%s ', $dur{days}, ( $dur{days} == 1 ) ? '' : 'e' ) )
        : ''
      ),
      sprintf( '%d:%02d:%02d', $dur{hours}, $dur{minutes}, $dur{seconds} ),
      "\n";
}

print '  min./max. Hoehe: ', $track->min_elevation, ' m - ', $track->max_elevation, " m\n"
  if $track->min_elevation && $track->max_elevation;
printf "  Entfernung: %.2f km\n", 1e-3 * $track->distance;


opendir( DIR, $img_dir ) or croak "Kann Verzeichnis '$img_dir' nicht oeffnen: $!";
my @jpgs = sort grep { m/\.jpg$/i && -f "$img_dir/$_" } readdir(DIR);
closedir DIR;

my @photos;
my @unmatched;
my $i = 0;
foreach my $jpg_file (@jpgs)
{
    my $file_name = "$img_dir/$jpg_file";
    -f $file_name or croak "Kann JPG-Datei '$file_name' nicht lesen: $!";
    print "Verarbeiten von $file_name ..\n" unless $QUIET;
    my $exif = new Image::ExifTool;
    my %options = ( CoordFormat => '%f', PrintConv => 1 );
    $exif->ExtractInfo( $file_name, \%options ) or next;
    my $info       = $exif->GetInfo;
    my $thumb_file = sprintf('%s/%04d.jpg', $thumb_dir, $i);

    my $dt         = $info->{CreateDate} || $info->{DateTimeOriginal};
    my ( $year, $month, $day, $h, $m, $s ) =
      $dt =~ /(\d{4}):(\d{2}):(\d{2}) (\d{2}):(\d{2}):(\d{2})/;

    my $timeshift = sprintf( '%d:%02d', $h_offset, $m_offset );
    my $timeshift_direction = ($offset_sign eq '-')? -1 : 1;
    $exif->SetNewValue(DateTimeOriginal => $timeshift, Shift => $timeshift_direction);
    $exif->SetNewValue(CreateDate => $timeshift, Shift => $timeshift_direction);

    my $datetime = DateTime->new(
                                  year   => $year,
                                  month  => $month,
                                  day    => $day,
                                  hour   => $h,
                                  minute => $m,
                                  second => $s,
                                  formatter => $dt_isoformatter,
    );
    my %offset = (hours => $h_offset, minutes => $m_offset, seconds => $s_offset);
    if ( $offset_sign eq '-' ) {
        $datetime->subtract(%offset);
    }
    else {
        $datetime->add(%offset);
    }
    $dt = "$datetime";

    my $trkpt = $track->get_matching_point(Timestamp => $dt);
    push( @unmatched, $file_name ) && next unless $trkpt;

    my @longitude = GPS::Helper::dec_to_dms( $trkpt->longitude, GPS::Helper::LONGITUDE );
    my @latitude = GPS::Helper::dec_to_dms( $trkpt->latitude, GPS::Helper::LATITUDE );

    my $lon_ref = $longitude[0];
    my $lat_ref = $latitude[0];

    my $lon     = abs( $trkpt->longitude );
    my $lat     = abs( $trkpt->latitude );

    printf( "  Passender Laengen- und Breitengrad gefunden: %s%f / %s%f\n", $lat_ref, $lat, $lon_ref, $lon ) if $DEBUG;

    $exif->SetNewValuesFromFile($file_name);
    $exif->SetNewValue( 'GPS:GPSLongitude',    $lon );
    $exif->SetNewValue( 'GPS:GPSLatitude',     $lat );
    $exif->SetNewValue( 'GPS:GPSLongitudeRef', $lon_ref );
    $exif->SetNewValue( 'GPS:GPSLatitudeRef',  $lat_ref );
    $exif->SetNewValue( 'GPS:GPSAltitude',     $trkpt->elevation );
    $exif->SetNewValue( 'GPS:GPSMapDatum',     'WGS-84' );

    if ($create_thumbnails)
    {
        printf("  Erzeugen des Thumbnail in '%s' ..\n", $thumb_file) if $DEBUG;
        my $t = new Image::Thumbnail(
                                      size       => $max_dim,
                                      create     => TRUE,
                                      input      => $file_name,
                                      outputpath => $thumb_file,
        );
        $exif->SetNewValue( 'ImageWidth',  $t->{x} );
        $exif->SetNewValue( 'ImageHeight', $t->{y} );
    }

    my $name    = $info->{FileNumber};
    my $caption = $info->{'Caption-Abstract'}  # Picasa
      || $info->{Title}                        # generic
      || $info->{XPTitle}                      # Vista, XP(?)
      || '';
    my $comment = $info->{UserComment}         # generic
      || $info->{XPComment}                    # Vista, XP(?)
      || '';
    my $author = $info->{Artist}               # Vista
      || $info->{Creator}                      # Vista
      || $info->{XPAuthor}                     # Vista, XP(?)
      || '';
    my $description = '<![CDATA[ ';
    $description .=
      $caption ? '<strong>' . $caption . '</strong><br />' . "\n" : '';
    $description .= $comment ? $comment . '<br />' . "\n" : '';
    $description .=
      '<img src="' . $thumb_file . '" />' . "\n" . '<table>' . "\n";
    $description .=
        '  <tr>' . "\n"
      . '    <th>Name der Originaldatei</th>' . "\n"
      . '    <td>'
      . $jpg_file . '</td>' . "\n"
      . '  </tr>' . "\n";
    $description .=
        '  <tr>' . "\n"
      . '    <th>Datum/Uhrzeit</th>' . "\n"
      . '    <td>'
      . $dt . '</td>' . "\n"
      . '  </tr>' . "\n";
    $description .=
        '  <tr>' . "\n"
      . '    <th>Laengengrad/Breitengrad</th>' . "\n"
      . '    <td>'
      . "$lon_ref $lon / $lat_ref $lat" . '</td>' . "\n"
      . '  </tr>' . "\n";
    $description .=
        '  <tr>' . "\n"
      . '    <th>Hoehe ueber NN</th>' . "\n"
      . '    <td>'
      . $trkpt->elevation
      . 'm</td>' . "\n"
      . '  </tr>' . "\n";
    $description .=
        '  <tr>' . "\n"
      . '    <th>Kamera</th>' . "\n"
      . '    <td>'
      . ( $info->{Model} || '?' ) . '</td>' . "\n"
      . '  </tr>' . "\n";
    $description .=
        '  <tr>' . "\n"
      . '    <th>Belichtungszeit</th>' . "\n"
      . '    <td>'
      . ( $info->{ExposureTime} || '?' ) . '</td>' . "\n"
      . '  </tr>' . "\n";
    $description .=
        '  <tr>' . "\n"
      . '    <th>Blendenoeffnung</th>' . "\n"
      . '    <td>'
      . ( $info->{FNumber} || '?' ) . '</td>' . "\n"
      . '  </tr>' . "\n";
    $description .=
        '  <tr>' . "\n"
      . '    <th>Brennweite</th>' . "\n"
      . '    <td>'
      . ( $info->{FocalLength} || '?' ) . '</td>' . "\n"
      . '  </tr>' . "\n";
    $description .=
        '  <tr>' . "\n"
      . '    <th>Belichtungskorrektur</th>' . "\n"
      . '    <td>'
      . ( $info->{ExposureCompensation} || '?' ) . '</td>' . "\n"
      . '  </tr>' . "\n";
    $description .=
        '  <tr>' . "\n"
      . '    <th>Blitz</th>' . "\n"
      . '    <td>'
      . ( $info->{Flash} || '?' ) . '</td>' . "\n"
      . '  </tr>' . "\n";
    $description .=
        '  <tr>' . "\n"
      . '    <th>Weissabgleich</th>' . "\n"
      . '    <td>'
      . ( $info->{WhiteBalance} || '?' ) . '</td>' . "\n"
      . '  </tr>' . "\n";
    $description .=
        '  <tr>' . "\n"
      . '    <th>ISO</th>' . "\n"
      . '    <td>'
      . ( $info->{ISO} || '?' ) . '</td>' . "\n"
      . '  </tr>' . "\n";
    $description .= '</table>' . "\n" . ' ]]>';
    push @photos,
      {
        file        => $thumb_file,
        trkpt       => $trkpt,
        name        => $name,
        description => $description,
      };

    $exif->SetNewValue( 'Comment',
                        'Created with GeoTag.pl - (C) 2008 Oliver Lau' );
    $exif->SetNewValue( 'XMP:CreatorTool', 'geotag.pl' );
    $exif->WriteInfo($thumb_file);

    ++$i;
}

print "Schreiben der KML-Daten in temporaere Datei ..\n" if $DEBUG;
open KML, ">$tmp_kml_file";
print KML qq{<?xml version="1.0" encoding="UTF-8"?>
<kml xlmns="http://earth.google.com/kml/2.2">
  <Document>
  <name>$doc_name</name>
  <Style id="camera">
    <Icon>
      <href>http://maps.google.com/mapfiles/kml/shapes/camera.png</href>
    </Icon>
  </Style>
  <Style id="track">
  <LineStyle>
    <color>f0f0c8c8</color>
    <width>3</width>
  </LineStyle>
  </Style>
};

foreach my $photo (@photos)
{
    print KML qq{
     <Placemark>
        <name>$photo->{'name'}</name>
        <description>$photo->{'description'}</description>
        <Point>
    };
    print KML '          <coordinates>', $photo->{trkpt}->longitude, ',', $photo->{trkpt}->latitude, ',', $photo->{trkpt}->elevation, "</coordinates>\n";
    print KML qq{
        </Point>
        <styleUrl>#camera</styleUrl>
     </Placemark>
    };
}

print KML qq{
    <Placemark>
      <name>Track</name>
      <LineString>
        <tesselate>1</tesselate>
        <coordinates>
},
map ('          ' . $_->longitude . ',' . $_->latitude . ',' . ( $_->elevation || '0' ) . "\n", @{ $track->points }),
qq{
        </coordinates>
      </LineString>
      <styleUrl>#track</styleUrl>
    </Placemark>
  </Document>
</kml>
};
close KML;

print "Nicht zugeordnete Fotos: \n",
    ( map " * $_\n", @unmatched ) if @unmatched && !$QUIET;

print "Generieren der KMZ-Datei ..\n" unless $QUIET;
my $zip = Archive::Zip->new;
$zip->addFile($tmp_kml_file);
$zip->addDirectory($thumb_dir);
foreach (@photos)
{
    $zip->addFile( $_->{file} );
}
$zip->writeToFileNamed($kmz_file) == AZ_OK
  or croak "Fehler beim Schreiben der KMZ-Datei: $!";

unlink $tmp_kml_file unless $DEBUG;

0;

##################################################################
#
# subroutines
#
##################################################################

sub disclaimer()
{
    print "\ngeotag.pl - JPGs anhand ihrer Zeitstempel mit den Zeitstempeln eines\n",
      "            GPS-Tracks verorten und daraus eine in sich geschlossene\n",
      "            KMZ-Datei erzeugen, die ausser dem Track auch saemtliche\n",
      "            Foto-Dateien enthaelt, die dem Track zugeordnet werden\n",
      "            konnten.\n\n",
      "Copyright (c) 2008 Oliver Lau <ola\@ctmagazin.de>,\n",
      "Heise Zeitschriften Verlag. Alle Rechte vorbehalten.\n\n";
}

sub USAGE()
{
    print "Aufruf mit: geotag.pl <Argumente> [Optionen]\n",
      "\nArgumente:\n",
      "  --imgdir=/Pfad/zu/den/JPGs\n",
      "      Name des Verzeichnisses, dass die zu verortenden Fotos enthaelt\n",
      "  --gpsfile=/Pfad/zur/GPX/Datei\n",
      "      Pfad zur GPX-Datei mit dem/den Track(s)\n",
      "  --kmzfile=/path/to/file.kmz\n",
      "      Pfad der resultierenden KMZ-Datei\n",
      "\nOptions:\n",
      "  --timeoffset=[+-]hh:mm:ss\n",
      "      den EXIF-Zeitstempel um ein Offset korrigieren\n",
      "      (Vorgabe: $t_offset)\n",
      "  --maxdim=n\n",
      "      Die Thumbnails sollen nicht breiter und nicht hoeher sein als n Pixel\n",
      "      (Vorgabe: $max_dim)\n",
      "  --thumbdir=relativer/Pfad/zu/den/Thumbnails\n",
      "      relativer Pfad zu dem (temporaeren) Verzeichnis, das die Thumbnails\n",
      "      aufnehmen soll\n",
      "      (Vorgabe: \"$thumb_dir\")\n",
      "  --description=kurze_Beschreibung_der_Tour\n",
      "  --createthumbnails\n",
      "      Thumbnails neu erzeugen\n",
      "  --debug\n",
      "      Debug-Informationen ausgeben\n",
      "  --quiet\n",
      "      Jegliche Ausgabe unterdruecken\n",
      "  --help\n",
      "      Diese Hilfe anzeigen\n",
      "\n";
    exit;
}
