﻿// PNRP Beacon - Ein PNRP Beispielprogramm
// benoetigt .NET Runtime 3.5 SP1 oder neuer
// Windows 2008 Server oder Vista mit PNRP 
// IPv6 Netzwerkanbindung
//
// 12/2008 Carsten Strotmann
// (c) 2008 c't Magazin, Heise Verlag
//

using System;
using System.Collections;
using System.Collections.Generic;
using System.Text;
using System.Net;
using System.Net.PeerToPeer;
using System.Threading;
using System.Windows.Threading;

namespace PnrpBeacon
{
    // die Klasse PnrpSearch beinhaltet die PNRP-Registrations-
    // und Such-Routinen
    internal class PnrpSearch
    {
        // Flag ob die Suche gestoppt wurde
        bool stop = false;
        
        // Datenstruktur mit den Topics
        PnrpTopics topics = new PnrpTopics();
        
        // Datenstruktur der PNRP-Registrierungen
        PnrpRegistrations regs = new PnrpRegistrations();
        
        // Datenstruktur fuer Logeintraege
        PnrpLog log = new PnrpLog();
        
        // der Hintergrund-Thread fuer die PNRP-Suche
        Thread searchThread = null;

        // neuen Such-Thread starten
        internal void startSearch()
        {
            ThreadStart search;
            search = new ThreadStart(searchPnrp);
            searchThread = new Thread(search);
            searchThread.Start();
        }

        // ein neues Topic hinzufuegen und
        // eine neue PNRP-Registrerung starten
        internal void addTopic(String topicname)
        {
            // ein neuer Peer-Name wird aus dem Topic-Namen erstellt
            // der Name wird nicht ueber kryptografische Schluessel 
            // geschuetzt
            PeerName peerName = new PeerName(topicname, PeerNameType.Unsecured);

            // eine neue PNRP-Registrierung erstellen
            PeerNameRegistration pnReg = new PeerNameRegistration();
            pnReg.PeerName = peerName;
            
            // Port der Applikation, hier Port 80 (= htttp)
            pnReg.Port = 80;
            
            // Kommentar (hier der Topicname, wird im PNRP-Beacon Programm
            // nicht verwendet
            pnReg.Comment = topicname;
            
            // Data (=PNRP Payload). 
            // im PNRP-Beacon wird hier ein Unicode-String gespeichert, welcher
            // den Maschinennamen, das Betriebssystem und die Anzahl der Prozessoren
            // der lokalen Maschine beschreibt
            pnReg.Data = System.Text.Encoding.UTF8.GetBytes(System.Environment.MachineName + ", " + System.Environment.OSVersion + ", CPU Count:" + System.Environment.ProcessorCount);
            
            // PNRP-Registrierung starten, der neue PNRP-Name wird veroeffendlicht
            pnReg.Start();

            // die PNRP-Registrierung wird lokal gespeichert
            regs.addRegistration(topicname, pnReg);

            // Informationen in das Log schreiben
            log.append("Registration for: " + topicname);
            log.append("PNRP ID of Service: " + peerName.ToString());
            log.append("DNS Name of Service: " + peerName.PeerHostName);

            // Topic der lokalen Datenstruktur hinzufuegen
            topics.Add(new PnrpTopic(topicname));
        }

        // Suche nach PNRP-Topics im PNRP-System
        internal void searchPnrp()
        {
            log.append("PNRP Search started....");
            
            while (!this.stop)
            {
                // eine Kopie der aktuellen Topics herstellen
                // um eine Aenderung der Datenstruktur durch
                // einen anderen Thread verhindern
                PnrpTopics loop_topics = topics.clone();

                // fuer jedes angemeldete Topic 
                // eine PNRP Suche durchfuehren
                IEnumerator enumerator = loop_topics.GetEnumerator();
                while (enumerator.MoveNext())
                {
                    if (!this.stop)
                    {
                        PnrpTopic topic = (PnrpTopic)enumerator.Current;
                        String topicname = topic.Name;
                        System.Diagnostics.Debug.WriteLine("Topic Name to search:" + topicname);
                        
                        // einen neuen PNRP-Resovler erstellen
                        PeerNameResolver resolver = new PeerNameResolver();
                        
                        // Peer-Name setzen (=Topic-Name)
                        PeerName peerName = new PeerName(topicname, PeerNameType.Unsecured);

                        // gefundene Hosts werden zurueckgeliefert
                        PeerNameRecordCollection results = resolver.Resolve(peerName);
                        
                        // fuer jeden gefunden Hosts ...
                        foreach (PeerNameRecord record in results)
                        {
                            String comment = "";

                            // Payload ermitteln
                            if (record.Data != null)
                            {
                                comment = System.Text.Encoding.ASCII.GetString(record.Data);
                            }

                            // Endpoints (IPv6 Adressen) ermittlen
                            // eine Maschinen kann mehrere IPv6
                            // Adressen registriert haben
                            if (record.EndPointCollection != null)
                            {
                                Hashtable old_endpoints = (Hashtable)topic.Peers.Clone();
                                
                                // Endpoints in der Topic-Datenstruktur speichern
                                foreach (EndPoint endpoint in record.EndPointCollection)
                                {
                                    // pruefen, ob der Endpoint schon bekannt war
                                    if (!topic.hasEndpoint(endpoint))
                                    {
                                        if (record.Comment != null)
                                        {
                                            log.append("Topic: " + record.Comment);
                                        }
                                        log.append("New Endpoint found: " + endpoint.ToString());
                                        topic.addEndpoint(endpoint, comment);
                                    }
                                }
                            }
                        }
                    }
                }
                if (!stop)
                    Thread.Sleep(2);
            }
            log.append("PNRP Search stopped!");
        }

        // ein Topic entfernen und die PNRP-Registrierung stoppen
        internal void removeTopic(String topic)
        {
            // PNRP-Registrierung fuer Topic ermitteln
            PeerNameRegistration pnReg = (PeerNameRegistration)regs.getRegistration(topic);
            log.append("Stopping registration for " + pnReg.PeerName.ToString());
            
            // Registrierung stoppen
            pnReg.Stop();
            
            // Topic aus den Datenstrukturen entfernen
            regs.removeRegistration(topic);
            topics.Remove(topics.getTopicbyName(topic));
        }

        // PNRP-Suche stoppen
        internal void stopSearch()
        {
            log.append("PNRP Search stop signal recieved....");
            
            // "stop"-Flag setzen
            stop = true;

            // warten das sich der Such-Thread beendet
            searchThread.Join(2000);

            // alle PNRP-Registrierungen stoppen
            IDictionaryEnumerator enumerator = regs.getRegistrations().GetEnumerator();
            while (enumerator.MoveNext())
            {
                string topicname = (string)enumerator.Key;
                PeerNameRegistration pnReg = (PeerNameRegistration)enumerator.Value;
            
                log.append("Stopping registration for " + pnReg.PeerName.ToString());
                pnReg.Stop();
            }
            log.append("De-registration complete.");
        }

        // Datenstruktur der Topics
        internal PnrpTopics getTopics()
        {
            return topics;
        }

        // Datenstruktur der Log-Eintraege
        internal PnrpLog getLog()
        {
            return log;
        }
    }
}
