﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.IO;
using System.Linq;
using System.Net;
using System.Xml.Linq;

namespace PhoneRssReader
{
  // INotifyPropertyChanged sorgt dafür, dass die Liste im Feed-Editor
  // automatisch aktualisiert wird.
  public class RssFeed : INotifyPropertyChanged
  {
    public RssFeed() { }

    private string _title;
    public string Title
    {
      get { return _title; }
      set
      {
        if(value != _title) {
          _title = value;
          NotifyPropertyChanged("Title");
          NotifyPropertyChanged("TitleOrUrl");
        }
      }
    }

    private string _description;
    public string Description
    {
      get { return _description; }
      set
      {
        if(value != _description) {
          _description = value;
          NotifyPropertyChanged("Description");
        }
      }
    }

    private string _url;
    public string Url
    {
      get { return _url; }
      set
      {
        string url = value;
        if(!url.Contains(@"://"))
          url = @"http://" + url;
        if(url != _url) {
          _url = url;
          NotifyPropertyChanged("Url");
          NotifyPropertyChanged("TitleOrUrl");
        }
      }
    }

    [System.Xml.Serialization.XmlIgnoreAttribute]
    public String TitleOrUrl
    {
      get
      {
        return string.IsNullOrEmpty(Title) ? Url : Title;
      }
    }

    private List<RssItem> _items = null;
    [System.Xml.Serialization.XmlIgnoreAttribute]
    public List<RssItem> Items
    {
      get
      {
        if(_items == null)
          _items = new List<RssItem>();
        return _items;
      }
      private set
      {
        if(value != _items) {
          _items = value;
          NotifyPropertyChanged("Items");
        }
      }
    }

    public event PropertyChangedEventHandler PropertyChanged;
    private void NotifyPropertyChanged(String propertyName)
    {
      if(null != PropertyChanged) {
        PropertyChanged(this, new PropertyChangedEventArgs(propertyName));
      }
    }

    private HttpWebRequest loadFeedRequest = null;

    public void LoadFeed()
    {
      if(string.IsNullOrEmpty(Url))
        throw new InvalidOperationException("Can't load feed without URL.");
      if(loadFeedRequest != null)
        throw new InvalidOperationException("Already loading.");

      loadFeedRequest = (HttpWebRequest)WebRequest.Create(Url);
      IAsyncResult ar = loadFeedRequest.BeginGetResponse(ContinueLoadFeed, this);
    }

    private void ContinueLoadFeed(IAsyncResult ar)
    {
      XDocument xml = null;
      try {
        using(HttpWebResponse resp = (HttpWebResponse)loadFeedRequest.EndGetResponse(ar)) {
          using(Stream strm = resp.GetResponseStream()) {
            xml = XDocument.Load(strm);
          }
        }

        Title = xml.Descendants("channel").Select(c => c.Element("title").Value).First();
        Description = xml.Descendants("channel").Select(c => c.Element("description").Value).First();
        var query = from item in xml.Descendants("item")
                    select new RssItem() {
                      Title = item.Element("title").Value,
                      Description = item.Element("description").Value,
                      Url = item.Element("link").Value,
                      PublicationDate = DateTime.Parse(item.Element("pubDate").Value)
                    };
        Items = query.ToList();
      }
      catch(WebException) {
        // Fehler beim Zugriff auf die Url.
        Items = null;
      }
      catch(Exception) {
        // Url enthält keinen gültigen Feed.
        Items = null;
      }
      finally {
        loadFeedRequest = null;
      }
      NotifyDoneLoading();
    }

    public event EventHandler DoneLoading;
    private void NotifyDoneLoading()
    {
      if(null != DoneLoading)
        DoneLoading(this, EventArgs.Empty);
    }

    public override string ToString()
    {
      return Title;
    }
  }
}
