package com.linkesoft.bbingo;

import java.io.File;

import android.app.AlertDialog;
import android.app.ListActivity;
import android.content.Intent;
import android.database.Cursor;
import android.os.Bundle;
import android.os.Environment;
import android.view.ContextMenu;
import android.view.Menu;
import android.view.MenuItem;
import android.view.View;
import android.view.ContextMenu.ContextMenuInfo;
import android.widget.CursorAdapter;
import android.widget.ListView;
import android.widget.SimpleCursorAdapter;
import android.widget.Toast;
import android.widget.AdapterView.AdapterContextMenuInfo;

/**
 * Verwalte beliebig viele Wortlisten in einem ListView: Auswahl, Anlegen und Lschen, 
 * Importieren und Exportieren. 
 */
public class WordLists extends ListActivity {

	public static final String ID="id"; // Parameter zur bergabe der ausgewhlten ID 
	private static final int ADD = Menu.FIRST;
	private static final int DELETE = Menu.FIRST + 1;
	private static final int IMPORT = Menu.FIRST + 2;
	private static final int EXPORT = Menu.FIRST + 3;
	private static final int EDIT = Menu.FIRST + 4;
	private static final int FILESELECTOR_REQUEST = 0;
	private BBingoDB db;

	@Override
	public void onCreate(Bundle savedInstanceState) {
	    super.onCreate(savedInstanceState);
	
	    db = new BBingoDB(this);
	    Cursor cursor=db.getWordListsCursor(); // Cursor ber alle Eintrge, wird vom ListAdapter verwaltet und geschlossen
	    setListAdapter(
	    		new SimpleCursorAdapter(this, 
	    			android.R.layout.simple_list_item_1, // Layout fr Listeneintrag, hier ein einfacher TextView
	    			cursor, 
	    			new String[]{BBingoDB.TITLE}, new int[]{android.R.id.text1}) // Mapping zwischen ID im Eintragslayout und Feld in der Datenbank
	    		);
	    registerForContextMenu(getListView()); // zeige popup men fr Liste
	}
	
	@Override
	protected void onResume() {
		((CursorAdapter)getListAdapter()).getCursor().requery(); // frische Liste auf    		
		super.onResume();
	}
	
	@Override
	protected void onDestroy() {
		db.close();
		super.onDestroy();
	}

/**
 * Benutzer klickt auf einen Eintrag in der Liste
 */
	@Override
	protected void onListItemClick(final ListView l, final View v, final int position, final long id) {
		super.onListItemClick(l, v, position, id);
		Intent intent = new Intent();
		intent.putExtra(ID, id);
		setResult(RESULT_OK, intent);
		finish(); // beende Aktivitt erfolgreich
	}

/**
 * Erzeuge Context(Popup)-Men, durch langen Tap auf einen Listeintrag ausgewhlt
 */
	@Override
	public void onCreateContextMenu(ContextMenu menu, View v, ContextMenuInfo menuInfo) {
		if (v.equals(getListView())) {
			AdapterContextMenuInfo info = (AdapterContextMenuInfo) menuInfo;
			if (info.id >= 0) {
				menu.setHeaderTitle(db.getWordList(info.id).title);
				menu.add(0, DELETE, 0, R.string.Delete);
				menu.add(0, EDIT, 0, R.string.Edit);
				MenuItem add = menu.add(0, EXPORT, 0, R.string.Export);
				if (!Environment.getExternalStorageState().equals(Environment.MEDIA_MOUNTED))
					add.setEnabled(false); // s.u.
				return;
			}
		}
		super.onCreateContextMenu(menu, v, menuInfo);
	}

/**
 * Context-Men fr List-Eintrag wurde durch Tap-and-Hold ausgewhlt
 */
	@Override
	public boolean onContextItemSelected(MenuItem item) {
		AdapterContextMenuInfo info = (AdapterContextMenuInfo) item.getMenuInfo();
		switch(item.getItemId()) {
    	case DELETE:
    		// hier ggf. noch Besttigungs-Popup per AlertBuilder zeige
    		db.removeWordList(info.id); // lsche Wortliste
    		((CursorAdapter)getListAdapter()).getCursor().requery(); // frische Liste auf 
    		break;
    	case EDIT:
			Intent intent = new Intent(this, EditWordList.class);
			intent.putExtra(EditWordList.ID, info.id); // bergebe aktuelle ID
			startActivity(intent);
    		break;
    	case EXPORT:
			// speichere Datei auf der SD-Karte, ins Verzeichnis bbingo
			File dir = new File(Environment.getExternalStorageDirectory(), "bbingo");
			dir.mkdir(); // lege Verzeichnis an, ignoriere Fehler
			if (db.exportWordList(info.id, dir)) {
				String msg = getString(R.string.Exported, dir.getPath());
				Toast.makeText(this, msg, Toast.LENGTH_LONG).show();
			} else {
				// Fehlermeldung mit Dateipfad als Argument (Platzhalter %s im
				// Format-String)
				String msg = getString(R.string.ExportError, dir.getPath());
				new AlertDialog.Builder(this).setTitle(R.string.Error).setMessage(msg).setPositiveButton(android.R.string.ok, null).show();
			}
			break;		
		}
		return super.onContextItemSelected(item);
	}
// normales Men zum Anlegen neuer Wortlisten und zum Importieren	
	@Override
	public boolean onCreateOptionsMenu(Menu menu) {
    	menu.add(0, ADD, 0, R.string.Add).setIcon(android.R.drawable.ic_menu_add);
    	menu.add(0, IMPORT, 0, R.string.Import);
		return super.onCreateOptionsMenu(menu);
	}
	
	@Override
	public boolean onPrepareOptionsMenu(Menu menu) {
		// Men-Eintrag deaktivieren wenn keine SD-Karte verfgbar ist, 
		// oder das Dateisystem z.B. wegen USB-Verbindung nicht lesbar ist
		if(!Environment.getExternalStorageState().equals(Environment.MEDIA_MOUNTED))
			menu.findItem(IMPORT).setEnabled(false); 
		return super.onPrepareOptionsMenu(menu);
	}
	
	@Override
	public boolean onOptionsItemSelected(MenuItem item) {
		switch(item.getItemId()) {
    	case ADD:
    		startActivity(new Intent(this,EditWordList.class)); // keine ID bergeben = neue Wortliste
    		break;
    	case IMPORT:
        	Intent i=new Intent(this,FileSelectorActivity.class);
        	// starte eigene Dateiauswahl fr *.txt Dateien, 
        	// Ergebnis in onActivityResult unten
        	i.putExtra(FileSelectorActivity.EXTENSIONS, "txt"); //$NON-NLS-1$
        	startActivityForResult(i, FILESELECTOR_REQUEST);
    		break;
		}
		return super.onOptionsItemSelected(item);
	}
	
    @Override
	protected void onActivityResult(int requestCode, int resultCode, Intent data) {
		super.onActivityResult(requestCode, resultCode, data);
		if (requestCode == FILESELECTOR_REQUEST && resultCode == RESULT_OK) {
			// Benutzer hat eine Datei zum Import ausgewhlt
			File file = new File(data.getExtras().getString(FileSelectorActivity.PATH));
			if (db.importWordList(file))
				((CursorAdapter) getListAdapter()).getCursor().requery(); // Liste
																			// auffrischen
			else {
				// Fehlermeldung mit Dateipfad als Argument (Platzhalter %s im
				// Format-String)
				String msg = getString(R.string.ImportError, file.getPath());
				new AlertDialog.Builder(this).setTitle(R.string.Error).setMessage(msg).setPositiveButton(android.R.string.ok, null).show();
			}

		} // FILESELECTOR_REQUEST
	} // onActivityResult	
}
