<isol>

package main

import (
  "os"
  "fmt"
  "bufio"
  "flag"
  t "time"
)

// Konstanten für die Datumsformatierung
const (
  outputTimeFormat = "02.01.2006, 15:04"
  syslogTimeFormat = "Jan  2 15:04:05"
)

// Protokolleintrag
type logLine struct {
  time int64 // Sekunden seit 1970
  msg string
  src *string
}

// String wandelt eine Protokollzeile wieder in ein Textformat um.
// Hierbei wird allerdings ein deutsches Datumsformat verwendet.
func (this logLine) String() string {
  return t.SecondsToLocalTime(this.time).Format(outputTimeFormat) +
       " [" + *this.src + "] " + this.msg
}

// logError fügt die Fehlermeldung an der aktuellen Position in den
// Ausgabestrom ein und reicht dann alle eingehenden Zeilen durch.
func logError(msg os.Error, name string, in <-chan *logLine,
        out chan<- *logLine) {
  out <- &logLine{0, msg.String()+"\n", &name}
  for line := range in {
    out <- line
  }
  close(out)
}

// parseLog liest eine Logdatei im syslog-Format, fügt die Zeilen
// chronologisch korrekt in den Datenstrom ein, der über channel "in"
// gelesen wird und gibt alles über channel "out" aus.
func parseLog(name string, in <-chan *logLine, out chan<- *logLine) {
  fh, err := os.Open(name)
  if err != nil {
    logError(err, name, in, out)
    return
  }
  defer fh.Close()
  file := bufio.NewReader(fh)
  incoming := <-in
  var current logLine
  current.src = &name
  now := t.LocalTime()
  for {
    line, err := file.ReadString('\n')
    if err != nil {
      if incoming != nil { out <- incoming }
      logError(err, name, in, out)
      return
    }
    time, err := t.Parse(syslogTimeFormat, line[:len(syslogTimeFormat)])
    if (err == nil) {
      time.Year = now.Year
      current.time = time.Seconds()
      current.msg = line[len(syslogTimeFormat)+1:]
    } else {
      fmt.Println(err)
      // letzter Zeitstempel wird wiederverwendet
      current.msg = line
    }
    for incoming != nil && current.time > incoming.time {
      out <- incoming
      incoming = <- in
    }
    out <- &current
  }
}

// startParser startet eine Goroutine, die die angegebene Datei liest
// und mit dem Datenstrom aus channel "in" verknüpft. Zurückgegeben
// wird der Ausgabechannel.
func startParser(name string, in <-chan *logLine) <-chan *logLine {
  out := make(chan *logLine)
  go parseLog(name, in, out)
  return out
}

// main startet für jedes Kommandozeilenargument einen Parser und
// gibt den Ausgabedatenstrom aus.
func main() {
  var out <-chan *logLine = make(chan *logLine)
  close(out)
  for _, name := range flag.Args() {
    out = startParser(name, out)
  }
  for line := range out {
    fmt.Print(line)
  }
}
