// Copyright (c) 2011 Oliver Lau <oliver@von-und-fuer-lau.de>
// All rights reserved.
// $Id: 3dwidget.cpp f9a43276afac 2011/11/03 10:35:31 Oliver Lau <oliver@von-und-fuer-lau.de> $


#include <QColor>
#include <QImage>
#include <QPixmap>
#include <QtCore/QDebug>
#include <QtCore/QTimer>
#include <qmath.h>

#include "3dwidget.h"


const float ThreeDWidget::DefaultZoom = -7.0f;


ThreeDWidget::ThreeDWidget(QWidget* parent) : QGLWidget(parent)
        , mXRot(0.0f)
        , mYRot(0.0f)
        , mXTrans(0.0f)
        , mYTrans(-1.0f)
        , mZTrans(0.0f)
        , mZoom(DefaultZoom)
        , mRoom(0)
        , mBody(0)
        , mKinect(0)
        , mLastFloorLevel(0)
        , mLastTilt(0)
        , mTextureHandle(NULL/*new GLuint[NumHandles]*/)
        , mHotspot(0)
        , mHotspotEnabled(false)
        , mHotspotTimer(0)
        , mHotspotRotation(0.0f)
{
    setFocus(Qt::OtherFocusReason);
    setCursor(Qt::OpenHandCursor);
}


ThreeDWidget::~ThreeDWidget(void)
{
    if (mBody)
        glDeleteLists(mBody, 1);
    if (mRoom)
        glDeleteLists(mRoom, 1);
    if (mTextureHandle)
        delete [] mTextureHandle;
}


void ThreeDWidget::loadTexture(const QString& name, GLuint handle)
{
    QPixmap bitmap(name);
    mTextureImg = bitmap.toImage();
    glBindTexture(GL_TEXTURE_2D, handle);
    switch (mTextureImg.depth()) {
    case 24:
        glTexImage2D(GL_TEXTURE_2D, 0, 3, mTextureImg.width(), mTextureImg.height(), 0, GL_RGB, GL_UNSIGNED_BYTE, (const GLvoid*)mTextureImg.scanLine(0));
        break;
    case 32:
        glTexImage2D(GL_TEXTURE_2D, 0, 4, mTextureImg.width(), mTextureImg.height(), 0, GL_RGBA, GL_UNSIGNED_BYTE, (const GLvoid*)mTextureImg.scanLine(0));
        break;
    default:
        qFatal("texture image has invalid depth: %d", mTextureImg.depth());
        break;
    }
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
}


void ThreeDWidget::initializeGL(void)
{
    // glGenTextures(NumHandles, mTextureHandle);
    glClearColor(0.0f, 0.0f, 0.0f, 0.0f);
    glShadeModel(GL_SMOOTH);
    glEnable(GL_ALPHA_TEST);
    glEnable(GL_DEPTH_TEST);
    glEnable(GL_BLEND);
    glEnable(GL_POINT_SMOOTH);
    glEnable(GL_LINE_SMOOTH);
    glDepthFunc(GL_LEQUAL);
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
    glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);
    glHint(GL_POINT_SMOOTH_HINT, GL_NICEST);
    glDisable(GL_TEXTURE_2D);
    glCullFace(GL_BACK);
    makeRoom();
    makeKinect();
}


void ThreeDWidget::resizeGL(int width, int height)
{
    glViewport(0, 0, width, height);
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    gluPerspective(38, (GLdouble)width/(GLdouble)height, 1, 100);
    glMatrixMode(GL_MODELVIEW);
}


const float ThreeDWidget::light0_diffuse[] = {0.8f, 0.8f, 0.8f, 1.0f};
const float ThreeDWidget::light0_pos[]     = {1.0f, -7.0f, 1.0f, 0.0f};
const float ThreeDWidget::mat_shininess[]  = {8.0f};
const float ThreeDWidget::mat_specular[]   = {1.0f, 1.0f, 1.0f, 1.0f};


void ThreeDWidget::paintGL(void)
{
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
    glLoadIdentity();
    glTranslatef(-0.0f, -0.0f, mZoom);
    glRotatef(mXRot, 1.0f, 0.0f, 0.0f);
    glRotatef(mYRot, 0.0f, 1.0f, 0.0f);
    glTranslatef(mXTrans, mYTrans, mZTrans);
    glMaterialfv(GL_FRONT_AND_BACK, GL_SHININESS, mat_shininess);
    glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, mat_specular);
    glColorMaterial(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE);
    glEnable(GL_COLOR_MATERIAL);
    glLightfv(GL_LIGHT0, GL_DIFFUSE, light0_diffuse);
    glLightfv(GL_LIGHT0, GL_POSITION, light0_pos);
    glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, GL_FALSE);
    glEnable(GL_LIGHTING);
    glEnable(GL_LIGHT0);
    glCallList(mRoom);
    if (mKinect)
        glCallList(mKinect);
    if (mBody)
        glCallList(mBody);
    if (mHotspotEnabled && mHotspot != 0) {
        glTranslatef(0.0f, 0.0f, NUI::HotspotDistance);
        glRotatef(mHotspotRotation, 0.0f, 1.0f, 0.0f);
        glCallList(mHotspot);
    }
    if (!mMessage.isEmpty()) {
        glColor3f(1.0f, 1.0f, 1.0f);
        renderText(10, 40, mMessage, QFont("Arial", 32));
    }
}


void ThreeDWidget::timerEvent(QTimerEvent* e)
{
    if (e->timerId() == mHotspotTimer) {
        mHotspotRotation += 9.3f;
        updateGL();
    }
}


void ThreeDWidget::keyPressEvent(QKeyEvent* e)
{
    qDebug() << "ThreeDWidget::keyPressEvent() type =" << e->type();
    const double incrTrans = ((e->modifiers() & Qt::ShiftModifier) == Qt::ShiftModifier)? 1 : 10;
    switch(e->key()) {
    case Qt::Key_Left:
        mXTrans -= incrTrans;
        updateGL();
        break;
    case Qt::Key_Right:
        mXTrans += incrTrans;
        updateGL();
        break;
    case Qt::Key_Down:
        mYTrans -= incrTrans;
        updateGL();
        break;
    case Qt::Key_Up:
        mYTrans += incrTrans;
        updateGL();
        break;
    case Qt::Key_PageDown:
        mZTrans -= incrTrans;
        updateGL();
        break;
    case Qt::Key_PageUp:
        mZTrans += incrTrans;
        updateGL();
        break;
    case Qt::Key_Escape:
        mZoom = DefaultZoom;
        mXRot = 0;
        mYRot = 0;
        mXTrans = 0.0;
        mYTrans = 0.0;
        mZTrans = 0.0;
        updateGL();
        break;
    default:
        break;
    }
}


void ThreeDWidget::mousePressEvent(QMouseEvent* e)
{
    setCursor(Qt::ClosedHandCursor);
    mLastPos = e->pos();
}


void ThreeDWidget::mouseReleaseEvent(QMouseEvent*)
{
    setCursor(Qt::OpenHandCursor);
}


void ThreeDWidget::wheelEvent(QWheelEvent* e)
{
    if (e->delta() < 0) mZoom -= 0.2f;
    else mZoom += 0.2f;
    updateGL();
}


void ThreeDWidget::mouseMoveEvent(QMouseEvent* e)
{
    if ((e->buttons() & Qt::LeftButton) == Qt::LeftButton) {
        setXRotation(mXRot + (e->y() - mLastPos.y()) / 2);
        setYRotation(mYRot + (e->x() - mLastPos.x()) / 2);
    }
    mLastPos = e->pos();
}


void ThreeDWidget::showHotspot(bool enabled)
{
    mHotspotEnabled = enabled;
    if (mHotspotEnabled)
        mHotspotTimer = startTimer(30);
    else if (mHotspotTimer)
        killTimer(mHotspotTimer);
    makeHotspot();
    updateGL();
}


void ThreeDWidget::displayMessage(QString msg, int timeout)
{
    mMessage = msg;
    updateGL();
    if (timeout > 0)
        QTimer::singleShot(timeout, this, SLOT(clearMessage()));
}


void ThreeDWidget::clearMessage(void)
{
    displayMessage(QString());
}


void ThreeDWidget::setXRotation(int angle)
{
    angle %= 360;
    if (angle != mXRot) {
        mXRot = angle;
        updateGL();
    }
}


void ThreeDWidget::setYRotation(int angle)
{
    angle %= 360;
    if (angle != mYRot) {
        mYRot = angle;
        updateGL();
    }
}


void ThreeDWidget::makeRoom(void)
{
    static const GLfloat L = -4.0f;
    static const GLfloat R =  4.0f;
    if (mRoom)
        glDeleteLists(mRoom, 1);
    mRoom = glGenLists(1);
    glNewList(mRoom, GL_COMPILE);
    glBegin(GL_TRIANGLE_STRIP);
    glColor3f(0.8f, 0.0f, 0.0f);
    glVertex3f(L, 0, 0);
    glVertex3f(L, 0, 0.8f);
    glVertex3f(0, 0, 0);
    glVertex3f(-0.53333333333333333f, 0, 0.8f);
    glEnd();
    glBegin(GL_TRIANGLE_STRIP);
    glColor3f(0.95f, 0.1f, 0.1f);
    glVertex3f(0, 0, 0);
    glVertex3f(-0.53333333333333333f, 0, 0.8f);
    glVertex3f(0.53333333333333333f, 0, 0.8f);
    glEnd();
    glBegin(GL_TRIANGLE_STRIP);
    glColor3f(0.8f, 0.0f, 0.0f);
    glVertex3f(0, 0, 0);
    glVertex3f(0.53333333333333333f, 0, 0.8f);
    glVertex3f(R, 0, 0);
    glVertex3f(R, 0, 0.8f);
    glEnd();
    glBegin(GL_TRIANGLE_STRIP);
    glColor3f(0.8f, 0.8f, 0.0f);
    glVertex3f(L, 0, 0.8f);
    glVertex3f(L, 0, 1.2f);
    glVertex3f(-0.53333333333333333f, 0, 0.8f);
    glVertex3f(-0.8f, 0, 1.2f);
    glEnd();
    glBegin(GL_TRIANGLE_STRIP);
    glColor3f(0.95f, 0.95f, 0.1f);
    glVertex3f(-0.53333333333333333f, 0, 0.8f);
    glVertex3f(-0.8f, 0, 1.2f);
    glVertex3f(0.53333333333333333f, 0, 0.8f);
    glVertex3f(0.8f, 0, 1.2f);
    glEnd();
    glBegin(GL_TRIANGLE_STRIP);
    glColor3f(0.8f, 0.8f, 0.0f);
    glVertex3f(0.53333333333333333f, 0, 0.8f);
    glVertex3f(0.8f, 0, 1.2f);
    glVertex3f(R, 0, 0.8f);
    glVertex3f(R, 0, 1.2f);
    glEnd();
    glBegin(GL_TRIANGLE_STRIP);
    glColor3f(0.4f, 0.8f, 0.0f);
    glVertex3f(L, 0, 1.2f);
    glVertex3f(L, 0, 2.1f);
    glVertex3f(-0.8f, 0, 1.2f);
    glVertex3f(-1.4f, 0, 2.1f);
    glEnd();
    glBegin(GL_TRIANGLE_STRIP);
    glColor3f(0.55f, 0.95f, 0.1f);
    glVertex3f(-0.8f, 0, 1.2f);
    glVertex3f(-1.4f, 0, 2.1f);
    glVertex3f(0.8f, 0, 1.2f);
    glVertex3f(1.4f, 0, 2.1f);
    glEnd();
    glBegin(GL_TRIANGLE_STRIP);
    glColor3f(0.4f, 0.8f, 0.0f);
    glVertex3f(0.8f, 0, 1.2f);
    glVertex3f(1.4f, 0, 2.1f);
    glVertex3f(R, 0, 1.2f);
    glVertex3f(R, 0, 2.1f);
    glEnd();
    glBegin(GL_TRIANGLE_STRIP);
    glColor3f(0.0f, 0.8f, 0.0f);
    glVertex3f(L, 0, 2.1f);
    glVertex3f(L, 0, 3.5f);
    glVertex3f(-1.4f, 0, 2.1f);
    glVertex3f(-2.33333333333f, 0, 3.5f);
    glEnd();
    glBegin(GL_TRIANGLE_STRIP);
    glColor3f(0.1f, 0.95f, 0.1f);
    glVertex3f(-1.4f, 0, 2.1f);
    glVertex3f(-2.33333333333f, 0, 3.5f);
    glVertex3f(1.4f, 0, 2.1f);
    glVertex3f(2.33333333333f, 0, 3.5f);
    glEnd();
    glBegin(GL_TRIANGLE_STRIP);
    glColor3f(0.0f, 0.8f, 0.0f);
    glVertex3f(1.4f, 0, 2.1f);
    glVertex3f(2.33333333333f, 0, 3.5f);
    glVertex3f(R, 0, 2.1f);
    glVertex3f(R, 0, 3.5f);
    glEnd();
    glBegin(GL_TRIANGLE_STRIP);
    glColor3f(0.4f, 0.8f, 0.0f);
    glVertex3f(L, 0, 3.5f);
    glVertex3f(L, 0, 4.0f);
    glVertex3f(-2.33333333333f, 0, 3.5f);
    glVertex3f(-2.66666666667f, 0, 4.0f);
    glEnd();
    glBegin(GL_TRIANGLE_STRIP);
    glColor3f(0.55f, 0.95f, 0.1f);
    glVertex3f(-2.33333333333f, 0, 3.5f);
    glVertex3f(-2.66666666667f, 0, 4.0f);
    glVertex3f(2.33333333333f, 0, 3.5f);
    glVertex3f(2.66666666667f, 0, 4.0f);
    glEnd();
    glBegin(GL_TRIANGLE_STRIP);
    glColor3f(0.4f, 0.8f, 0.0f);
    glVertex3f(2.33333333333f, 0, 3.5f);
    glVertex3f(2.66666666667f, 0, 4.0f);
    glVertex3f(R, 0, 3.5f);
    glVertex3f(R, 0, 4.0f);
    glEnd();
    glEndList();
}


static void rgbFromWaveLength(__IN float wave, __OUT float& R, __OUT float& G, __OUT float& B)
{
    float r, g, b;
    if (wave >= 380 && wave <= 440) {
        r = -1.0f * (wave - 440) / (440 - 380);
        g = 0.0f;
        b = 1.0f;
    }
    else if (wave >= 440 && wave <= 490) {
        r = 0.0f;
        g = (wave - 440) / (490 - 440);
        b = 1.0f;
    }
    else if (wave >= 490 && wave <= 510) {
        r = 0.0f;
        g = 1.0f;
        b = - (wave - 510) / (510 - 490);
    }
    else if (wave >= 510 && wave <= 580) {
        r = (wave - 510) / (580 - 510);
        g = 1.0;
        b = 0.0f;
    }
    else if (wave >= 580 && wave <= 645) {
        r = 1.0f;
        g = - (wave - 645) / (645 - 580);
        g = 0.0f;
    }
    else if (wave >= 645 && wave <= 780) {
        r = 1.0f;
        g = 0.0f;
        g = 0.0f;
    }
    else {
        r = 0.0f;
        g = 0.0f;
        g = 0.0f;
    }
    float s = 1.0f;
    if (wave > 700)
        s = 0.3f + 0.7f * (780 - wave) / (780 - 700);
    else if (wave <  420)
        s = 0.3f + 0.7f * (wave - 380) / (420 - 380);
    R = pow(s*r, 0.8f);
    G = pow(s*g, 0.8f);
    B = pow(s*b, 0.8f);
}


void ThreeDWidget::makeHotspot(void)
{
    if (mHotspot)
        glDeleteLists(mHotspot, 1);
    mHotspot = glGenLists(1);
    glNewList(mHotspot, GL_COMPILE);
    glBegin(GL_TRIANGLE_FAN);
    static const float pi = 3.14159265358979323846f;
    static const float r = 0.24f;
    static const float cz = 0.001f;
    glColor3f(1.0f, 1.0f, 1.0f);
    glVertex3f(0.0f, cz, 0.0f);
    for (float i = 0.0f; i <= 2*pi; i += pi/18) {
        float R, G, B;
        rgbFromWaveLength(380.0f + (i * 400.0f / (2*pi)), R, G, B);
        glColor4f(R, G, B, 0.9f);
        glVertex3f(r*cos(i), cz, r*sin(i));
    }
    glEnd();
    glEndList();
}


void ThreeDWidget::makeKinect(void)
{
    if (mKinect)
        glDeleteLists(mKinect, 1);
    static const float maxDist = 4.0f;
    static const float verticalApertureAngle = 43.0f;
    const float aperture = deg2rad(.5f*verticalApertureAngle + NUI::instance()->tilt());
    const float ah = maxDist * sin(aperture);
    const float h = -NUI::instance()->floorLevel();
    const float b = h / tan(aperture);
    mKinect = glGenLists(1);
    glNewList(mKinect, GL_COMPILE);
#if 0
    glLineWidth(1.0f);
    glBegin(GL_LINES);
    glColor4f(1.0f, 1.0f, 1.0f, 0.5f);
    // head line
    glVertex3f(0, h, 0);
    glVertex3f(0, h+ah, maxDist);
    // floor line
    glVertex3f(0, h, 0);
    glVertex3f(0, 0, b);
    glEnd();
#else
    glLineWidth(1.0f);
    glBegin(GL_TRIANGLE_STRIP);
    glColor4f(1.0f, 1.0f, 1.0f, 0.5f);
    glVertex3f(0, h, 0);
    glVertex3f(0, 0, b);
    glVertex3f(0, h+ah, maxDist);
    glVertex3f(0, 0, maxDist);
    glEnd();
#endif
    glEndList();
}


void ThreeDWidget::setSkeletons(const SkeletonList& skeletons)
{
    Q_ASSERT(skeletons.size() > 0);
    if (mBody)
        glDeleteLists(mBody, 1);
    mBody = glGenLists(1);
    glNewList(mBody, GL_COMPILE);
    glLineWidth(2.2f);
    glPointSize(5.1f);
    const NUI* nui = NUI::instance();
    const LimbList& limbs = nui->limbs();
    for (SkeletonList::const_iterator i = skeletons.constBegin(); i != skeletons.constEnd(); ++i) {
        const Skeleton& skeleton = *i;
        // generate limbs
        for (LimbList::const_iterator j = limbs.constBegin(); j != limbs.constEnd(); ++j) {
            glColor3f(0.2f, 0.5f, 0.6f);
            const JointIndexList& vertices = *j;
            glBegin(GL_LINE_STRIP);
            for (JointIndexList::const_iterator k = vertices.constBegin(); k != vertices.constEnd(); ++k) {
                const int jointIndex = *k;
                const Point& p = skeleton[jointIndex];
                glVertex3f(p.x(), p.y(), p.z());
            }
            glEnd();
        }
        // generate joints
        glBegin(GL_POINTS);
        for (int j = 0; j < skeleton.size(); ++j) {
            const Point& p = skeleton[j];
            const QColor& c = NUI::JointColorTable[j];
            glColor4f(c.redF(), c.greenF(), c.blueF(), (p.state() == NUI_SKELETON_POSITION_TRACKED)? 1.0f : 0.5f);
            glVertex3f(p.x(), p.y(), p.z());
        }
        glEnd();
    }
    glEndList();
    const int currentFloorLevelCm = int(100*NUI::instance()->floorLevel());
    if (currentFloorLevelCm != mLastFloorLevel || NUI::instance()->tilt() != mLastTilt)
        makeKinect();
    mLastFloorLevel = currentFloorLevelCm;
    mLastTilt = (int)NUI::instance()->tilt();
    updateGL();
}
