// Copyright (c) 2011 Oliver Lau <oliver@von-und-fuer-lau.de>
// All rights reserved.
// $Id: nui.cpp dd19e64be22a 2011/10/24 07:57:51 Oliver Lau <oliver@von-und-fuer-lau.de> $


#include <QtCore>
#include <QtCore/QString>
#include <QtCore/QDebug>
#include <QtGui/QMessageBox>

#include "nui.h"


NUI* NUI::mInstance = NULL;

#if defined(NUI_ENSURE_REENTRANCE)
QMutex NUI::mInstanceMutex;
#endif


const float NUI::HotspotDistance = 2.26f;


NUI* NUI::instance(void)
{
#if defined(NUI_ENSURE_REENTRANCE)
    mInstanceMutex.lock();
#endif
    if (mInstance == NULL)
        mInstance = new NUI;
#if defined(NUI_ENSURE_REENTRANCE)
    mInstanceMutex.unlock();
#endif
    return mInstance;
}


NUI::NUI(QObject* parent) : QObject(parent)
        , mMetersAboveFloor(0.0f)
        , mSmoothTransform(false)
        , mWaveDetection(false)
        , mLastWaveDirectionChangePositionRightX(0.0f)
        , mLastWaveDirectionChangePositionLeftX(0.0f)
        , mWaveDirectionRight(NONE)
        , mWaveDirectionLeft(NONE)
        , mLastRightHandPositionX(0.0f)
        , mLastLeftHandPositionX(0.0f)
        , mWaveCountRight(0)
        , mWaveCountLeft(0)
#if defined(NUI_WITH_AUDIO)
        , mDMO(NULL)
        , mSoundSourceLocalizer(NULL)
#endif
{
    mResult = NuiInitialize(NUI_INITIALIZE_FLAG_USES_DEPTH_AND_PLAYER_INDEX |
                            NUI_INITIALIZE_FLAG_USES_COLOR |
                            NUI_INITIALIZE_FLAG_USES_SKELETON);
    if (FAILED(mResult))
        QMessageBox::critical(NULL, QObject::tr("NuiInitialize() fehlgeschlagen"), QObject::tr("Kinect angeschlossen?"));
    BSTR devName;
    MSR_NuiGetPropsBlob(MsrNui::INDEX_UNIQUE_DEVICE_NAME, &devName, NULL);
    mDeviceName = QString((QChar*)devName);
    SysFreeString(devName);
    JointIndexList core, leftArm, rightArm, leftLeg, rightLeg;
    core << NUI_SKELETON_POSITION_HIP_CENTER << NUI_SKELETON_POSITION_SPINE << NUI_SKELETON_POSITION_SHOULDER_CENTER << NUI_SKELETON_POSITION_HEAD;
    leftArm << NUI_SKELETON_POSITION_SHOULDER_CENTER << NUI_SKELETON_POSITION_SHOULDER_LEFT << NUI_SKELETON_POSITION_ELBOW_LEFT << NUI_SKELETON_POSITION_WRIST_LEFT << NUI_SKELETON_POSITION_HAND_LEFT;
    rightArm << NUI_SKELETON_POSITION_SHOULDER_CENTER << NUI_SKELETON_POSITION_SHOULDER_RIGHT << NUI_SKELETON_POSITION_ELBOW_RIGHT << NUI_SKELETON_POSITION_WRIST_RIGHT << NUI_SKELETON_POSITION_HAND_RIGHT;
    leftLeg << NUI_SKELETON_POSITION_HIP_CENTER << NUI_SKELETON_POSITION_HIP_LEFT << NUI_SKELETON_POSITION_KNEE_LEFT << NUI_SKELETON_POSITION_ANKLE_LEFT << NUI_SKELETON_POSITION_FOOT_LEFT;
    rightLeg << NUI_SKELETON_POSITION_HIP_CENTER << NUI_SKELETON_POSITION_HIP_RIGHT << NUI_SKELETON_POSITION_KNEE_RIGHT << NUI_SKELETON_POSITION_ANKLE_RIGHT << NUI_SKELETON_POSITION_FOOT_RIGHT;
    mLimbs << core << leftArm << rightArm << leftLeg << rightLeg;

#if defined(NUI_WITH_AUDIO)
    initAudio();
#endif
}


NUI::~NUI()
{
    NuiShutdown();
}


#if defined(NUI_WITH_AUDIO)
void NUI::initAudio(void)
{
    HRESULT hr = S_OK;
    CoInitializeEx(NULL, COINIT_APARTMENTTHREADED | COINIT_SPEED_OVER_MEMORY);
    mDMO = NULL;
    IPropertyStore* pPS = NULL;
    SetPriorityClass (GetCurrentProcess(), HIGH_PRIORITY_CLASS);
    CoCreateInstance(CLSID_CMSRKinectAudio, NULL, CLSCTX_INPROC_SERVER, IID_IMediaObject, (PVOID*)&mDMO);
    mDMO->QueryInterface(IID_IPropertyStore, (PVOID*)&pPS);
    PROPVARIANT pvSysMode;
    PropVariantInit(&pvSysMode);
    pvSysMode.vt = VT_I4;
    pvSysMode.lVal = 4;
    pPS->SetValue(MFPKEY_WMAAECMA_SYSTEM_MODE, pvSysMode);
    PropVariantClear(&pvSysMode);
    hr = mDMO->QueryInterface(IID_ISoundSourceLocalizer, (PVOID*)&mSoundSourceLocalizer);
    if (FAILED(hr))
        mSoundSourceLocalizer = NULL;
}
#endif


const QColor NUI::JointColorTable[NUI_SKELETON_POSITION_COUNT] =
{
    QColor(169, 176, 155), // HIP CENTER
    QColor(169, 176, 155), // SPINE
    QColor(168, 230,  29), // SHOULDER CENTER
    QColor(200,   0,   0), // HEAD
    QColor( 79,  84,  33), // SHOULDER LEFT
    QColor( 84,  33,  42), // ELBOW LEFT
    QColor(255, 126,   0), // WRIST LEFT
    QColor(215,  86,   0), // HAND LEFT
    QColor( 33,  79,  84), // SHOULDER RIGHT
    QColor( 33,  33,  84), // ELBOW RIGHT
    QColor( 77, 109, 243), // WRIST RIGHT
    QColor( 37,  69, 243), // HAND RIGHT
    QColor( 77, 109, 243), // HIP LEFT
    QColor( 69,  33,  84), // KNEE LEFT
    QColor(229, 170, 122), // ANKLE LEFT
    QColor(255, 126,   0), // FOOT LEFT
    QColor(181, 165, 213), // HIP RIGHT
    QColor( 71, 222,  76), // KNEE RIGHT
    QColor(245, 228, 156), // ANKLE RIGHT
    QColor( 77, 109, 243)  // FOOT RIGHT
};


long NUI::tilt(void)
{
    long angle;
    mResult = NuiCameraElevationGetAngle(&angle);
    if (FAILED(mResult))
        qDebug() << QObject::tr("NuiCameraElevationGetAngle() fehlgeschlagen");
    return angle;
}


void NUI::setTilt(long angle)
{
    mResult = NuiCameraElevationSetAngle(angle);
    if (FAILED(mResult))
        qDebug() << QObject::tr("NuiCameraElevationSetAngle() fehlgeschlagen");
}


void NUI::enableTracking(HANDLE event)
{
    mResult = NuiSkeletonTrackingEnable(event, 0);
    if (FAILED(mResult))
        QMessageBox::critical(NULL, QObject::tr("NuiSkeletonTrackingEnable() fehlgeschlagen"), QObject::tr("Kinect angeschlossen?"));
}


void NUI::disableTracking(void)
{
    mResult = NuiSkeletonTrackingDisable();
    if (FAILED(mResult))
        QMessageBox::critical(NULL, QObject::tr("NuiSkeletonTrackingDisable() fehlgeschlagen"), QObject::tr("Kinect angeschlossen?"));
}


HANDLE NUI::openStream(NUI_IMAGE_TYPE eImageType, NUI_IMAGE_RESOLUTION eResolution, HANDLE event)
{
    HANDLE stream;
    mResult = NuiImageStreamOpen(eImageType, eResolution, 0, 2, event, &stream);
    if (FAILED(mResult))
        QMessageBox::critical(NULL, QObject::tr("NuiImageStreamOpen() fehlgeschlagen"), QObject::tr("Kinect angeschlossen?"));
    return stream;
}


inline QRgb NUI::depthToRGB(quint16 s)
{
    const int person = s & 7;
    const int b = 256 - ((s >> 7) & 0xff);
    switch (person) {
    case 0: // no person
        return qRgb(b, b, b);
    case 1: // person 1
        return qRgb(b, 0, 0);
    case 2: // person 2
        return qRgb(0, b, 0);
    case 3: // person 3
        return qRgb(0, 0, b);
    case 4: // person 4
        return qRgb(b, b, 0);
    case 5: // person 5
        return qRgb(0, b, b);
    case 6: // person 6
        return qRgb(b/4, b, b);
    case 7: // person 7
        return qRgb(b, b/2, b/2);
    }
    return 0; // just to please the compiler
}


QImage NUI::getStreamDepthFrame(HANDLE stream)
{
    const NUI_IMAGE_FRAME* pImageFrame = NULL;
    HRESULT hr = NuiImageStreamGetNextFrame(stream, 0, &pImageFrame);
    if (FAILED(hr))
        return QImage();
    NuiImageBuffer* const pTexture = pImageFrame->pFrameTexture;
    KINECT_LOCKED_RECT LockedRect;
    pTexture->LockRect(0, &LockedRect, NULL, 0);
    KINECT_SURFACE_DESC SurfaceDesc;
    pTexture->GetLevelDesc(0, &SurfaceDesc);
    QImage frame(SurfaceDesc.Width, SurfaceDesc.Height, QImage::Format_RGB32);
    if (LockedRect.Pitch != 0) {
        const quint16* buf = reinterpret_cast<const quint16*>(LockedRect.pBits);
        for (UINT y = 0; y < SurfaceDesc.Height; ++y) {
            QRgb* scanLine = reinterpret_cast<QRgb*>(frame.scanLine(y));
            for (UINT x = 0; x < SurfaceDesc.Width; ++x)
                *scanLine++ = depthToRGB(*buf++);
        }
    }
    NuiImageStreamReleaseFrame(stream, pImageFrame);
    return frame;
}


QImage NUI::getStreamVideoFrame(HANDLE stream)
{
    const NUI_IMAGE_FRAME* pImageFrame = NULL;
    HRESULT hr = NuiImageStreamGetNextFrame(stream, 0, &pImageFrame);
    if (FAILED(hr))
        return QImage();
#if 0
    NuiImageBuffer* pTexture = pImageFrame->pFrameTexture;
    KINECT_LOCKED_RECT LockedRect;
    pTexture->LockRect(0, &LockedRect, NULL, 0);
    if (FAILED(hr))
        return QImage();
    KINECT_SURFACE_DESC SurfaceDesc;
    pTexture->GetLevelDesc(0, &SurfaceDesc);
    QImage frame = (LockedRect.Pitch > 0) ?
                   QImage(reinterpret_cast<const uchar*>(LockedRect.pBits), SurfaceDesc.Width, SurfaceDesc.Height, QImage::Format_RGB32):
                   QImage();
#else
    // etwas schlankere Fassung, die die Ausmae des Frames
    // auf 640 x 480 Pixel festlegt und annimmt, dass der Framebuffer
    // mit entsprechend vielen Bytes gefllt ist
    NuiImageBuffer* pTexture = pImageFrame->pFrameTexture;
    KINECT_LOCKED_RECT LockedRect;
    hr = pTexture->LockRect(0, &LockedRect, NULL, 0);
    if (FAILED(hr))
        return QImage();
    QImage frame((const uchar*)LockedRect.pBits, 640, 480, QImage::Format_RGB32);
#endif
    NuiImageStreamReleaseFrame(stream, pImageFrame);
    return frame;
}


bool NUI::detectWaving(__IN const Skeleton& skeleton,
                       __IN int userId,
                       __IN FlagSemaphore::Side side,
                       __IN __OUT QTime& lastWaveDirectionChange,
                       __IN __OUT float& lastWaveDirectionChangePositionX,
                       __IN __OUT float& lastHandPositionX,
                       __IN __OUT NUI::WaveDirection& waveDirection,
                       __IN __OUT int& waveCount) {
    Q_ASSERT_X(skeleton.size() != 0, "NUI::detectWaving()", "Skeleton must not be empty");
    if (lastWaveDirectionChange.isNull())
        lastWaveDirectionChange = QTime::currentTime();
    static const float MinimumRequiredDistance = 0.2f;
    static const int MininumMoveTimeMs = 120;
    static const int MaximumMoveTimeMs = 1500;
    const float currentHandPositionX = (side == FlagSemaphore::LEFT)? skeleton.leftHand().x() : skeleton.rightHand().x();
    const int timeElapsed = lastWaveDirectionChange.elapsed();
    if (skeleton.handAboveElbow(side)) {
        if (currentHandPositionX < lastHandPositionX) {
            if (waveDirection == RIGHT) {
                const float dist = fabs(currentHandPositionX - lastWaveDirectionChangePositionX);
                if (dist > MinimumRequiredDistance && timeElapsed > MininumMoveTimeMs && timeElapsed < MaximumMoveTimeMs)
                    ++waveCount;
                lastWaveDirectionChange.start();
                lastWaveDirectionChangePositionX = currentHandPositionX;
            }
            waveDirection = LEFT;
        }
        else if (currentHandPositionX > lastHandPositionX) {
            if (waveDirection == LEFT) {
                const float dist = fabs(currentHandPositionX - lastWaveDirectionChangePositionX);
                if (dist > MinimumRequiredDistance && timeElapsed > MininumMoveTimeMs && timeElapsed < MaximumMoveTimeMs)
                    ++waveCount;
                lastWaveDirectionChange.start();
                lastWaveDirectionChangePositionX = currentHandPositionX;
            }
            waveDirection = RIGHT;
        }
        lastHandPositionX = currentHandPositionX;
        if (waveCount > 3) {
            emit waving(userId, side);
            waveCount = 0;
            return true;
        }
    }
    else {
        lastWaveDirectionChange.start();
        lastWaveDirectionChangePositionX = 0.0f;
        lastHandPositionX = 0.0f;
        waveDirection = NONE;
        waveCount = 0;
    }
    return false;
}


void NUI::getSkeletons(__OUT SkeletonList& skeletons, __OUT SkeletonList& skeletonsOnImage, int trackingState)
{
    Q_ASSERT_X(skeletons.size() == 0, "NUI::getSkeletons()", "SkeletonList skeletons must be empty");
    Q_ASSERT_X(skeletonsOnImage.size() == 0, "NUI::getSkeletons()", "SkeletonList skeletonsOnImage must be empty");
    NUI_SKELETON_FRAME skeletonFrame;
    mResult = NuiSkeletonGetNextFrame(0, &skeletonFrame);
    if (FAILED(mResult))
        return;
    mMetersAboveFloor = -skeletonFrame.vFloorClipPlane.w;
    if (mSmoothTransform) {
        static const NUI_TRANSFORM_SMOOTH_PARAMETERS smoothingParams = {
            0.5f,  // smoothing
            0.5f,  // correction
            0.5f,  // prediction
            0.05f, // jitter radius
            0.04f  // max deviation radius
        };
        mResult = NuiTransformSmooth(&skeletonFrame, &smoothingParams);
        if (FAILED(mResult))
            return;
    }
    for (int i = 0; i < NUI_SKELETON_COUNT; ++i) {
        const NUI_SKELETON_DATA& d = skeletonFrame.SkeletonData[i];
        if (d.eTrackingState == trackingState) {
#if 0
            if (d.dwQualityFlags & NUI_SKELETON_QUALITY_CLIPPED_RIGHT)
                qDebug() << "Player" << i << "NUI_SKELETON_QUALITY_CLIPPED_RIGHT" << skeletonFrame.dwFrameNumber;
            if (d.dwQualityFlags & NUI_SKELETON_QUALITY_CLIPPED_LEFT)
                qDebug() << "Player" << i << "NUI_SKELETON_QUALITY_CLIPPED_LEFT" << skeletonFrame.dwFrameNumber;
            if (d.dwQualityFlags & NUI_SKELETON_QUALITY_CLIPPED_TOP)
                qDebug() << "Player" << i << "NUI_SKELETON_QUALITY_CLIPPED_TOP" << skeletonFrame.dwFrameNumber;
            if (d.dwQualityFlags & NUI_SKELETON_QUALITY_CLIPPED_BOTTOM)
                qDebug() << "Player" << i << "NUI_SKELETON_QUALITY_CLIPPED_BOTTOM" << skeletonFrame.dwFrameNumber;
#endif
            Skeleton skeleton(NUI_SKELETON_POSITION_COUNT);
            Skeleton skeletonOnImage(NUI_SKELETON_POSITION_COUNT);
            for (int j = 0; j < NUI_SKELETON_POSITION_COUNT; ++j) {
                const int state = (int)d.eSkeletonPositionTrackingState[j];
                const Vector4& v = d.SkeletonPositions[j];
                const float xd = skeletonFrame.vFloorClipPlane.x * v.y;
                const float yd = mMetersAboveFloor;
                const float zd = skeletonFrame.vFloorClipPlane.z * v.y;
                skeleton[j] = Point(v.x - xd, v.y - yd, v.z - zd, state);
                FLOAT fx, fy; USHORT fz;
                NuiTransformSkeletonToDepthImageF(v, &fx, &fy, &fz);
                skeletonOnImage[j] = Point(fx, fy, (float)fz, state);
            }
            skeletons.append(skeleton);
            skeletonsOnImage.append(skeletonOnImage);
        }
        if (mWaveDetection && !skeletons.empty()) {
            const Skeleton& skeleton = skeletons.first();
            detectWaving(skeleton, i, FlagSemaphore::RIGHT,
                         mLastWaveDirectionChangeRight,
                         mLastWaveDirectionChangePositionRightX,
                         mLastRightHandPositionX,
                         mWaveDirectionRight,
                         mWaveCountRight);
            detectWaving(skeleton, i, FlagSemaphore::LEFT,
                         mLastWaveDirectionChangeLeft,
                         mLastWaveDirectionChangePositionLeftX,
                         mLastLeftHandPositionX,
                         mWaveDirectionLeft,
                         mWaveCountLeft);
        }
    }
}
