// Copyright (c) 2011 Oliver Lau <oliver@von-und-fuer-lau.de>
// All rights reserved.
// $Id: speechrecognizer.cpp c5f95eadb0c7 2011/11/03 10:12:40 Oliver Lau <oliver@von-und-fuer-lau.de> $

#include "speechrecognizer.h"

#include <QtCore/QtDebug>


SpeechRecognizer::SpeechRecognizer(QObject* parent) : QObject(parent)
{
    HRESULT hr;
    CComPtr<ISpRecognizer> cpRecoEngine;
    hr = cpRecoEngine.CoCreateInstance(CLSID_SpInprocRecognizer);
    if (FAILED(hr))
        qFatal("Cannot create SAPI Speech Recognizer (ISpRecognizer)");

    hr = cpRecoEngine->CreateRecoContext(&mRecognitionCtx);
    if (FAILED(hr))
        qFatal("Cannot create SAPI Recognition Context (ISpRecoContext)");

    hr = mRecognitionCtx->SetNotifyCallbackFunction(&sapiCallback, 0, LPARAM(this));
    if (FAILED(hr))
        qFatal("Cannot set notify callback function. (SetNofifyCallbackFunction)");

    hr = mRecognitionCtx->SetInterest(SPFEI(SPEI_RECOGNITION), SPFEI(SPEI_RECOGNITION));
    if (FAILED(hr))
        qFatal("Cannot set interest (SetInterest)");

    CComPtr<ISpAudio> cpAudio;
    SpCreateDefaultObjectFromCategoryId(SPCAT_AUDIOIN, &cpAudio);

    hr = cpRecoEngine->SetInput(cpAudio, TRUE);
    if (FAILED(hr))
        qFatal("Cannot set input (SetInput)");

    hr = cpRecoEngine->SetRecoState(SPRST_ACTIVE);
    if (FAILED(hr))
        qFatal("Cannot activate recognition engine");

    hr = mRecognitionCtx->CreateGrammar(0, &mDictationGrammar);
    if (FAILED(hr))
        qFatal("Cannot create grammar (CreateGrammar)");

    hr = mDictationGrammar->LoadCmdFromFile(L"grammar-de.xml", SPLO_STATIC);
    if (FAILED(hr))
        qFatal("Cannot load command file (LoadCmdFromFile)");

    hr = mDictationGrammar->SetGrammarState(SPGS_ENABLED);
    if (FAILED(hr))
        qFatal("Cannot enable grammar (SetGrammarState)");

    hr = mDictationGrammar->SetRuleState(NULL, NULL, SPRS_ACTIVE);
    if (FAILED(hr))
        qFatal("Cannot enable rule (SetRuleState)");
}


SpeechRecognizer::~SpeechRecognizer()
{
    mRecognitionCtx.Release();
    mDictationGrammar.Release();
}


void __stdcall SpeechRecognizer::sapiCallback(WPARAM wParam, LPARAM lParam)
{
    Q_ASSERT_X(lParam != NULL, "SpeechRecognizer", "sapiCallback() received null object");
    Q_UNUSED(wParam);
    reinterpret_cast<SpeechRecognizer*>(lParam)->sapiEventReceived();
}


void SpeechRecognizer::sapiEventReceived(void)
{
    CSpEvent event;
    while (event.GetFrom(mRecognitionCtx) == S_OK) {
        if (event.eEventId == SPEI_RECOGNITION) {
            SPPHRASE* pPhrase = NULL;
            HRESULT hr = event.RecoResult()->GetPhrase(&pPhrase);
            if (FAILED(hr))
                continue;
            if (pPhrase->Rule.Confidence == SP_LOW_CONFIDENCE)
                continue;
            QString phrase;
            const SPPHRASEELEMENT* el = pPhrase->pElements;
            for (ULONG i = pPhrase->Rule.ulFirstElement; i < pPhrase->Rule.ulCountOfElements; ++i) {
                phrase.append(QString::fromStdWString(el->pszDisplayText));
                if (i < pPhrase->Rule.ulCountOfElements-1)
                    phrase.append(" ");
                ++el;
            }
            emit phraseRecognized(phrase, QString::fromStdWString(pPhrase->Rule.pszName), pPhrase->Rule.SREngineConfidence);
        }
    }
}
