//
//	tsp.h
//	=====
//
//	Software License Agreement (BSD License)
//	----------------------------------------
//	Copyright (c) 2013 Thorsten Radde (thorstenr@idealsoftware.com). All rights reserved.
//	Source code: www.IdealSoftware.com
//
//	Redistribution and use in source and binary forms, with or without modification,
//	are permitted provided that the following conditions are met:
//
//	* Redistributions of source code must retain the above copyright notice, this 
//	  list of conditions and the following disclaimer.
//
//	* Redistributions in binary form must reproduce the above copyright notice, this
//	  list of conditions and the following disclaimer in the documentation and/or
//	  other materials provided with the distribution.
//
//	* Neither the names Thorsten Radde or IDEAL Software GmbH, nor the names of contributors
//	  may be used to endorse or promote products derived from this software without
//	  specific prior written permission of Thorsten Radde.
//
//	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
//	ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//	WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
//	DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
//	ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
//	(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
//	LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
//	ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
//	(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
//	SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


// For this simplified model, we store the location of each city based on an x/y coordinate system, and
// the distances between all cities are computed as a simple point-to-point connection
class CCity
{
public:
	const TCHAR *Name;
	int			xPos;	// x-position of the city on map
	int			yPos;	// y-position of the city on map
};


#define PROBLEM_Bier127
//#define PROBLEM_a280


#if defined PROBLEM_Bier127

// TSPLib Bier127 (127 Biergaerten in Augsburg (Juenger/Reinelt))
#define MAP_UPSIDE_DOWN				// to be compatible to Concorde, the map is painted upside down
const int nOptimum = 118282;
CCity Cities[] =
{
	_T(""),	9860,	14152,
	_T(""),	9396,	14616,
	_T(""),	11252,	14848,
	_T(""), 11020,	13456,
	_T(""), 9512,	15776,
	_T(""), 10788,	13804,
	_T(""), 10208,	14384,
	_T(""), 11600,	13456,
	_T(""), 11252,	14036,
	_T(""), 10672,	15080,
	_T(""), 11136,	14152,
	_T(""), 9860,	13108,
	_T(""), 10092,	14964,
	_T(""), 9512,	13340,
	_T(""), 10556,	13688,
	_T(""), 9628,	14036,
	_T(""), 10904,	13108,
	_T(""), 11368,	12644,
	_T(""), 11252,	13340,
	_T(""), 10672,	13340,
	_T(""), 11020,	13108,
	_T(""), 11020,	13340,
	_T(""), 11136,	13572,
	_T(""), 11020,	13688,
	_T(""), 8468,	11136,
	_T(""), 8932,	12064,
	_T(""), 9512,	12412,
	_T(""), 7772,	11020,
	_T(""), 8352,	10672,
	_T(""), 9164,	12876,
	_T(""), 9744,	12528,
	_T(""), 8352,	10324,
	_T(""), 8236,	11020,
	_T(""), 8468,	12876,
	_T(""), 8700,	14036,
	_T(""), 8932,	13688,
	_T(""), 9048,	13804,
	_T(""), 8468,	12296,
	_T(""), 8352,	12644,
	_T(""), 8236,	13572,
	_T(""), 9164,	13340,
	_T(""), 8004,	12760,
	_T(""), 8584,	13108,
	_T(""), 7772,	14732,
	_T(""), 7540,	15080,
	_T(""), 7424,	17516,
	_T(""), 8352,	17052,
	_T(""), 7540,	16820,
	_T(""), 7888,	17168,
	_T(""), 9744,	15196,
	_T(""), 9164,	14964,
	_T(""), 9744,	16240,
	_T(""), 7888,	16936,
	_T(""), 8236,	15428,
	_T(""), 9512,	17400,
	_T(""), 9164,	16008,
	_T(""), 8700,	15312,
	_T(""), 11716,	16008,
	_T(""), 12992,	14964,
	_T(""), 12412,	14964,
	_T(""), 12296,	15312,
	_T(""), 12528,	15196,
	_T(""), 15312 ,	6612,
	_T(""), 11716,	16124,
	_T(""), 11600,	19720,
	_T(""), 10324,	17516,
	_T(""), 12412,	13340,
	_T(""), 12876,	12180,
	_T(""), 13688,	10904,
	_T(""), 13688,	11716,
	_T(""), 13688,	12528,
	_T(""), 11484,	13224,
	_T(""), 12296,	12760,
	_T(""), 12064,	12528,
	_T(""), 12644,	10556,
	_T(""), 11832,	11252,
	_T(""), 11368,	12296,
	_T(""), 11136,	11020,
	_T(""), 10556,	11948,
	_T(""), 10324,	11716,
	_T(""), 11484 ,	9512,
	_T(""), 11484 ,	7540,
	_T(""), 11020 ,	7424,
	_T(""), 11484 ,	9744,
	_T(""), 16936,	12180,
	_T(""), 17052,	12064,
	_T(""), 16936,	11832,
	_T(""), 17052,	11600,
	_T(""), 13804,	18792,
	_T(""), 12064,	14964,
	_T(""), 12180,	15544,
	_T(""), 14152,	18908,
	_T(""), 5104,	14616,
	_T(""), 6496,	17168,
	_T(""), 5684,	13224,
	_T(""), 15660,	10788,
	_T(""), 5336,	10324,
	_T(""), 812 ,	6264,
	_T(""), 14384,	20184,
	_T(""), 11252,	15776,
	_T(""), 9744 ,	3132,
	_T(""), 10904 ,	3480,
	_T(""), 7308,	14848,
	_T(""), 16472,	16472,
	_T(""), 10440,	14036,
	_T(""), 10672,	13804,
	_T(""), 1160,	18560,
	_T(""), 10788,	13572,
	_T(""), 15660,	11368,
	_T(""), 15544,	12760,
	_T(""), 5336,	18908,
	_T(""), 6264,	19140,
	_T(""), 11832,	17516,
	_T(""), 10672,	14152,
	_T(""), 10208,	15196,
	_T(""), 12180,	14848,
	_T(""), 11020,	10208,
	_T(""), 7656,	17052,
	_T(""), 16240 ,	8352,
	_T(""), 10440,	14732,
	_T(""), 9164,	15544,
	_T(""), 8004,	11020,
	_T(""), 5684,	11948,
	_T(""), 9512,	16472,
	_T(""), 13688,	17516,
	_T(""), 11484 ,	8468,
	_T(""), 3248,	14152,
};

const int StartCity = 0;
const int CityCount = 127;

#elif defined PROBLEM_a280

// TSPLib a280
#define MAP_UPSIDE_DOWN				// to be compatible to Concorde, the map is painted upside down
const int nOptimum = 2579;

CCity Cities[] =
{
	#include "a280.tsp.h"
};

const int StartCity = 0;
const int CityCount = 280;

#endif


// Holds the pre-computed distances from each city to each other city
int Distance[CityCount][CityCount];

void ComputeDistances()
{
	for (int start = 0; start < CityCount; start++)
	{
		for (int end = 0; end < CityCount; end++)
		{
			// Pythagoras
			double a = Cities[start].xPos - Cities[end].xPos;
			double b = Cities[start].yPos - Cities[end].yPos;
			Distance[start][end] = (int)(sqrt(a * a + b * b) + 0.5);
		}
	}
}


// The type of a chromosome.
// Note: The unsigned short type is usable for max. 65536 cities.
typedef unsigned short TChromType;


// Definition of the genome for the tsp-problem
class CTspGenome
{
protected:
	TChromType	m_Genome[CityCount];	// size of the genome
	double		m_dblFitness;			// fitness value

public:
	static const size_t m_nNumChromosomes = CityCount;
	static const size_t m_nChromosomeScale = CityCount;

	CTspGenome(const TChromType *genome)
	{
		memcpy(m_Genome, genome, sizeof(m_Genome));
		m_dblFitness = 0;
	}

	CTspGenome()
	{
		m_dblFitness = 0;
	}

	void Reset()
	{
		m_dblFitness = 0;
	}

	const TChromType	*GetGenome() const { return m_Genome; }
	void				SetGenome(const TChromType *genome) { memcpy(m_Genome, genome, sizeof(m_Genome)); }

	double		GetFitness() const { return m_dblFitness; }
	void		SetFitness(double val) { m_dblFitness = val; }

	TChromType	GetChromosome(size_t n) const { return m_Genome[n]; }

	void RandomSwap()
	{
		size_t a = gen_rand32() % m_nNumChromosomes;
		size_t b = gen_rand32() % m_nNumChromosomes;
		TChromType swap = m_Genome[a];
		m_Genome[a] = m_Genome[b];
		m_Genome[b] = swap;
	}


	// initializes the genome with random chromosomes
	void Randomize()
	{
		// setup the array of cities and shuffle it
		int q = 0;
		for (size_t j = 0; j < m_nNumChromosomes; j++)
			m_Genome[j] = q++;

		// shuffle
		for (size_t j = 0; j < m_nNumChromosomes * 8; j++)
			RandomSwap();

		Normalize();
	}


	// Normalization means that we want each genome to have the start city as first chromosome.
	// This leads to an anchor for the crossover operation - and better visualization.
	void Normalize()
	{
		size_t i;
		for (i = 0; i < m_nNumChromosomes && m_Genome[i] != StartCity; i++)
			;	// NOP

		ASSERT(m_Genome[i] == StartCity);
		
		if (i > 0)
		{
			TChromType genome[CityCount];
			memcpy(genome, m_Genome, m_nNumChromosomes * sizeof(TChromType));

			for (size_t j = 0; j < m_nNumChromosomes; j++)
			{
				m_Genome[j] = genome[i++];
				if (i >= m_nNumChromosomes)
					i = 0;
			}
		}
	}


	double ComputeRouteLength(const TChromType *genome) const
	{
		double len = 0;

		for (size_t i = 0; i < m_nNumChromosomes - 1; i++)
			len += Distance[genome[i]][genome[i + 1]];

		len += Distance[genome[m_nNumChromosomes - 1]][genome[0]];

		return len;
	}


	double GetRouteLength() const
	{
		return ComputeRouteLength(m_Genome);
	}


	// computes the fitness-value, the bigger the better
	double ComputeFitness()
	{
		m_dblFitness = GetRouteLength();

		#define MAX_VAL 10000000
		m_dblFitness = MAX_VAL - m_dblFitness;

		return m_dblFitness;
	}


#ifdef _DEBUG
	bool VerifyGenome(size_t num_chromosomes = 0) const
	{
		if (num_chromosomes == 0)
			num_chromosomes = m_nNumChromosomes;

		for (size_t j = 0; j < num_chromosomes; j++)
		{
			// check, if all chromosomes are valid
			// (for unsigned types the < 0 check is useless, but if someone decides to use a signed type,
			// it is there for safety)
			if (m_Genome[j] < 0 || m_Genome[j] >= m_nNumChromosomes)
				return false;

			// check, if there is no doubled chromosome
			for (size_t i = j + 1; i < num_chromosomes; i++)
				if (m_Genome[j] == m_Genome[i])
					return false;
		}

		return true;
	}
#endif


	bool NotInGenome(const TChromType *genome, const TChromType *end, TChromType chromosome) const
	{
		while (genome != end && chromosome != *genome)
			genome++;

		return genome == end;
	}


	// Crossover
	void Crossover(const CTspGenome &father, const CTspGenome &mother)
	{
		// Enhanced Fast GSX Algorithm ("Greedy Subtour Crossover", Hiroaki SENGOKU and Ikuo YOSHIHARA)
		//
		// Enhancement by me. It leads earlier to better results.
		// Remaining unset chromosomes are not filled by random. They are filled to the left and right
		// by father / mother chromosomes in their original order.
		const TChromType *father_genome = father.GetGenome();
		const TChromType *mother_genome = mother.GetGenome();

		bool fa = true;
		bool fb = true;
		size_t x = gen_rand32() % m_nNumChromosomes;
		size_t org_x = x;
		TChromType chromosome = father_genome[x];
		size_t y;
		for (y = 0; y < m_nNumChromosomes && chromosome != mother_genome[y]; y++)
			; // NOP

		ASSERT(y < m_nNumChromosomes);
		size_t org_y = y;

		// we use a big array for assembling the new genome, so we can eliminate memory-shift operations
		const size_t array_size = m_nNumChromosomes * 2 - 1;
		TChromType genome[array_size];
		TChromType *left = genome + array_size / 2;
		*left = chromosome;
		TChromType *right = left + 1;
		size_t n = 1;	// number of chromosomes stored in new genome
		do
		{
			if (fa)
			{
				if (x == 0)
					x = m_nNumChromosomes;
				x--;

				chromosome = father_genome[x];
				if (NotInGenome(left, right, chromosome))
				{
					// insert chromosome at beginning
					*--left = chromosome;
					n++;
				}
				else
					fa = false;
			}

			if (fb)
			{
				y++;
				if (y >= m_nNumChromosomes)
					y = 0;

				chromosome = mother_genome[y];
				if (NotInGenome(left, right, chromosome))
				{
					// append chromosome at end
					*right++ = chromosome;
					n++;
				}
				else
					fb = false;
			}
		}
		while (fa || fb);

		if (n < m_nNumChromosomes)
		{
			do
			{
				if (x != org_x)
				{
					if (x == 0)
						x = m_nNumChromosomes;
					x--;

					chromosome = father_genome[x];
					if (NotInGenome(left, right, chromosome))
					{
						// insert chromosome at beginning
						*--left = chromosome;
						n++;
					}
				}

				if (y != org_y)
				{
					y++;
					if (y >= m_nNumChromosomes)
						y = 0;

					chromosome = mother_genome[y];
					if (NotInGenome(left, right, chromosome))
					{
						// append chromosome at end
						*right++ = chromosome;
						n++;
					}
				}
			}
			while (n < m_nNumChromosomes);
		}

		// Copy the assembled genome over and normalize
		TChromType *p = left;
		while (p != right && *p != StartCity)
			p++;

		ASSERT(*p == StartCity);
		
		for (size_t y = 0; y < m_nNumChromosomes; y++)
		{
			m_Genome[y] = *p++;
			if (p == right)
				p = left;
		}

		ASSERT(VerifyGenome());
	}


	// This is a modified 2opt algorithm. It works much better than the original 2opt algorithm.
	// I don't remember who invented it. Can't find anything about it on Google.
	//
	// Original 2opt algorithm:
	// If length(ab, cd) > length(ac, bd), it changes (ab, cd) ==> (ac, bd).
	// Then the whole route length of the new genome is computed, and if shorter than the original
	// route length, the change is applied.
	//
	// The Xopt algorithm works like the original 2opt, but it also reverses the order of all cities between b and c.
	//
	// Example, the current genome is:
	// A B C D E F G H I
	//
	// The route AB + HI is longer than AH + BI.
	// Therefore B is swapped with H.
	// A H C D E F G B I
	//
	// And the order of cities in between is reversed:
	// A H G F E D C B I
	//
	// Therefore, all previous links between B and H are kept intact.
	// I.e. the subtour between B and H is the same as before, but reversed.
	//
	// This involves two optimizations compared to the original 2opt algorithm:
	// 1) The subtour is not destroyed, which might already be optimal.
	// 2) We do not need to compute the whole route length of the new genome, to compare it
	//    with the original route length. It is clear that the new route is shorter, if:
	//    (delta = AB + CD - AC - BD) > 0
	//    Therefore current_route_length = current_route_length - delta.
	//
	// As a result, the Xopt algorithm leads in less generations to a better result and
	// it is much faster.
	bool MutateXOpt()
	{
		bool did_exchange = false;
		double current_route_len = GetRouteLength();

		for (int a = 0; a < (int)m_nNumChromosomes - 4; a++)
		{
			int b = a + 1;
			TChromType city_a = m_Genome[a];
			TChromType city_b = m_Genome[b];
			int distance_ab = Distance[city_a][city_b];

			for (int c = b + 1; c < (int)m_nNumChromosomes - 1; c++)
			{
				int d = c + 1;
				TChromType city_c = m_Genome[c];
				TChromType city_d = m_Genome[d];
				int delta =	distance_ab + Distance[city_c][city_d] - 
							Distance[city_a][city_c] - Distance[city_b][city_d];

				if (delta > 0)	// the new route is shorter
				{
					// reverse the order of cities from B to C
					int bi = b;
					for (int ci = c; ci > bi; ci--)
					{
						TChromType tmp = m_Genome[bi];
						m_Genome[bi] = m_Genome[ci];
						m_Genome[ci] = tmp;
						bi++;
					}
					
					current_route_len -= delta;
					did_exchange = true;

					// reflect changes
					city_b = m_Genome[b];
					distance_ab = Distance[city_a][city_b];
				}
			}
		}

		return did_exchange;
	}


	// mutation
	void Mutate()
	{
		ASSERT(VerifyGenome());

		while (MutateXOpt())
			;	// NOP

		ASSERT(VerifyGenome());
	}
};
