package com.linkesoft.ctlongitude;

import java.text.*;
import java.util.*;

import org.apache.http.*;
import org.apache.http.client.HttpClient;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.params.*;
import org.apache.http.protocol.*;
import org.apache.http.util.EntityUtils;
import org.json.*;

import android.app.Activity;
import android.location.*;
import android.os.AsyncTask;
import android.util.Log;
import android.widget.Toast;

import com.google.android.gms.auth.*;
import com.google.android.gms.maps.model.*;

public class Friend {
	String userid;
	String name;
	LatLng position;
	Date timestamp;
	Address address;
	Marker marker;

	public void setFromJSON(JSONObject friendobj) throws JSONException {
		userid = friendobj.getString("userid");
		name = friendobj.getString("name");
		position=new LatLng(friendobj.getDouble("lat"), friendobj.getDouble("lng"));
		timestamp = new Date(friendobj.getInt("timestamp")*1000L); // Date expects ms, API gives sec
	}

	@Override
	public String toString() {
		StringBuffer str = new StringBuffer();
		str.append("Friend");
		if (name != null)
			str.append(" " + name);
		if(timestamp!=null)
			str.append(" "+getFormattedDate());
		if (address != null)
			str.append(" " + getFormattedAddress(null));
		return str.toString();
	}

	public String getFormattedDate() {
		Calendar cal=Calendar.getInstance(); // aktueller Zeitpunkt
		cal.set(Calendar.HOUR_OF_DAY, 0);
		cal.set(Calendar.MINUTE, 0);
		cal.set(Calendar.SECOND,0);
		// heute? -> nur Uhrzeit
		if(timestamp.after(cal.getTime()))
		{
			// time
			DateFormat sdf = SimpleDateFormat.getTimeInstance(DateFormat.SHORT, Locale.getDefault());
			return  sdf.format(timestamp.getTime());
		}
		else {
			// date
			DateFormat sdf = SimpleDateFormat.getDateInstance();
			return  sdf.format(timestamp.getTime());
		}
	}
	/**
	 * kompakte Adressanzeige
	 * wenn homeAddress angegeben, zeige nur Strae falls in derselben Stadt+Land
	 * Strae+Ort falls in anderer Stadt
	 * Ort+Land falls in anderem Land
	 * wenn keine homeAddress angegeben, nutze Standard-Formatierung 
	 * @param homeAddress Adresse des Nutzers selbst (falls bekannt)
	 * @return formatierte Adresse
	 */
	public String getFormattedAddress(Address homeAddress) {
		// Address-Struktur-Namen:
		// http://www.oasis-open.org/committees/ciq/ciq.html#6
		String town = address.getLocality();
		String street = address.getThoroughfare();
		String country = address.getCountryName();
		if (address.getSubThoroughfare() != null && street != null)
			street += " " + address.getSubThoroughfare();
		StringBuffer buf = new StringBuffer();
		if (homeAddress != null) {
			if (country != null && homeAddress.getCountryName() != null && country.equals(homeAddress.getCountryName())) {
				// gleiches Land
				if (town != null && homeAddress.getLocality() != null && town.equals(homeAddress.getLocality())) {
					// gleiche Stadt
					if (street != null)
						buf.append(street);
				} else {
					// andere Stadt
					if (street != null)
						buf.append(street + "\n");
					if (town != null)
						buf.append(town);
				}
			} else {
				// anderes Land
				if (town != null)
					buf.append(town + "\n");
				if (country != null)
					buf.append(country);
			}
		}
		if (buf.length() == 0) {
			// construct standard address string
			for (int i = 0; i < address.getMaxAddressLineIndex(); i++) {
				if (buf.length() != 0)
					buf.append("\n");
				buf.append(address.getAddressLine(i));
			}
		}
		return buf.toString();

	}
	
	// callback fr GetFriendsTask
	public static interface GotFriendsListener {
		void gotFriends(List<Friend> friends,Address ownAddress);
	}

	public static class GetFriendsTask extends AsyncTask<Void, Void, Boolean> {
		private GotFriendsListener gotFriendsListener;
		private Activity activity;

		public GetFriendsTask(Activity activity,GotFriendsListener gotFriendsListener)
		{
			this.activity = activity;
			this.gotFriendsListener = gotFriendsListener;
			
		}
		@Override
		protected Boolean doInBackground(Void... params) {

			String oauthToken;
			try {
				oauthToken = GoogleAuthUtil.getToken(activity, MainActivity.getAccountName(activity),
						MainActivity.OAUTH_CLIENT_SCOPE);
			} catch (UserRecoverableAuthException e) {
				Log.v(getClass().getSimpleName(), "UserRecoverableAuthException", e);
				activity.startActivityForResult(e.getIntent(), MainActivity.REQUEST_CODE_SIGN_IN);
				return Boolean.TRUE;
			} catch (Exception e) {
				Log.e(getClass().getSimpleName(), "Could not get OAuth token", e);
				Toast.makeText(activity, e.getLocalizedMessage(), Toast.LENGTH_LONG).show();
				return Boolean.TRUE;
			}
			// eigentlicher Request
			// API erwartet UrlEncodedFormEntity
			HttpContext httpContext = new BasicHttpContext();
			HttpParams httpParameters = new BasicHttpParams();
			HttpClient httpClient = new DefaultHttpClient(httpParameters);
			HttpPost httpPost = new HttpPost(MainActivity.FRIENDS_URL);
			List<NameValuePair> nameValuePairs = new ArrayList<NameValuePair>();
			nameValuePairs.add(new BasicNameValuePair("as_array", "true"));
			nameValuePairs.add(new BasicNameValuePair("oauth[clientId]", MainActivity.CLIENT_ID));
			nameValuePairs.add(new BasicNameValuePair("oauth[token]", oauthToken));
			try {
				httpPost.setEntity(new UrlEncodedFormEntity(nameValuePairs));
				HttpResponse response = httpClient.execute(httpPost, httpContext);
				HttpEntity entity = response.getEntity();
				String result = EntityUtils.toString(entity);
				// JSON parsen
				JSONObject obj = new JSONObject(result);
				Log.v(getClass().getSimpleName(), "Result JSON: " + obj);
				if (!obj.getString("status").equals("ok")) {
					Toast.makeText(activity, "Error, server returned " + obj.getString("status"),
							Toast.LENGTH_LONG).show();
					return Boolean.TRUE;
				}
				// fill friends array
				final List<Friend> friends = new ArrayList<Friend>();
				// look up addresses with reverse geo-coding
				Geocoder geocoder = new Geocoder(activity);
				Address ownAddress=null;
				for (int i = 0; i < obj.getJSONArray("users").length(); i++) {
					JSONObject friendobj = obj.getJSONArray("users").getJSONObject(i);
					Friend friend = new Friend();
					try {
						friend.setFromJSON(friendobj);
						if (Geocoder.isPresent()) // nicht berall verfgbar
						{
							try {
								List<Address> address = geocoder.getFromLocation(friend.position.latitude,
										friend.position.longitude, 1);
								if (address != null && address.size() != 0)
									friend.address = address.get(0);

							} catch (Exception e) {
								Log.e(getClass().getSimpleName(), "Error getting friend location for "+friend.name, e);
							}
						}
						if (!friend.userid.equals(MainActivity.getAccountUserId(activity)))
							friends.add(friend);
						else
							ownAddress = friend.address;
						// TODO find address for real current location (our
						// address on ctLongitude could be outdated)
					} catch (JSONException e) {
						Log.e(getClass().getSimpleName(), "Invalid JSON Object" + friendobj, e);
					}
				} // for

				// sortiere nach Name, mit Umlaut-Untersttzung ber Collator
				final Collator collator = Collator.getInstance();
				collator.setStrength(Collator.PRIMARY);
				Collections.sort(friends, new Comparator<Friend>() {
					@Override
					public int compare(Friend f1, Friend f2) {
						// locale aware compare
						return collator.compare(f1.name, f2.name);
					}
				});
				for (Friend friend : friends) {
					Log.v(getClass().getSimpleName(), friend.toString());
				}
				if (gotFriendsListener != null && !isCancelled()) {
					final Address address=ownAddress; // vars must be final
					activity.runOnUiThread(new Runnable() {
						public void run() {
							gotFriendsListener.gotFriends(friends,address);
						}
					});
				}
			} catch (final Exception e) {
				Log.e(getClass().getSimpleName(), "Could not get friends", e);
				activity.runOnUiThread(new Runnable() {
					public void run() {
						Toast.makeText(activity, e.getLocalizedMessage(), Toast.LENGTH_LONG).show();
					}
				});
			}
			return Boolean.TRUE;
		}
	}

}
