/*
* ----------------------------------------------------------------------------
* "THE BEER-WARE LICENSE" (Revision 42):
* David Wischnjak wrote this file in the hope it will be useful.
* As long as you retain this notice you can do whatever you want with this stuff.
* If we meet some day, and you think this stuff is worth it, you can buy me a beer in return.
* ----------------------------------------------------------------------------
*/

/*
* MidMouse
* Middle mouse clicks for Microsoft Precision Touchpads (PTP).
*
* Once started, this small app fires middle mouse click events when you tap the
* touchpad with three fingers. Alternatively, holding the left ALT-key while
* performing a normal left click will also result in a middle click.
* (ALT-key behaviour can be turned off with "-a" argument.)
*
* Since we have no access to the internal PTP data, we have to rely on the keyboard
* events Windows fires when a three finger tap is performed. (L_SHIFT + L_WIN + L_CTRL + F22)
* When MidMouse catches such a keypress event, it fires a middle mouse click event
* for the current mouse position.
*/


#define _WIN32_WINNT 0x500
#include <Windows.h>
#include <string>

bool useAltKey = true;

bool controlKeyDown = false;
bool winKeyDown = false;
bool shiftKeyDown = false;
bool altKeyDown = false;

bool readyForClick = false;
bool middleButtonDown = false;


// Fires middle click events. Pass true for both parameters to perform a "click".
void mouseMiddleClick (bool down, bool up) {
    INPUT Input = {0}; //initialize INPUT struct.

    if(down){
        Input.type = INPUT_MOUSE; //mouse event with...
        Input.mi.dwFlags = MOUSEEVENTF_MIDDLEDOWN;  //...middle button down
        //No coordinates specified, current mouse position will be used
        SendInput(1,&Input,sizeof(INPUT)); //fire event
        middleButtonDown = true; //remember button state
    }

    if(up){
        ZeroMemory(&Input,sizeof(INPUT)); //re-initialize event
        Input.type = INPUT_MOUSE;
        Input.mi.dwFlags = MOUSEEVENTF_MIDDLEUP;
        SendInput(1,&Input,sizeof(INPUT));
        middleButtonDown = false;
    }
}

// Get mouse click events. Fire middle click when pressing ALT + L_MOUSE
LRESULT CALLBACK LowLevelMouseProc(int nCode, WPARAM wParam, LPARAM lParam){

    if (nCode == HC_ACTION) {

        // Get ALT-key state
        altKeyDown = GetAsyncKeyState (VK_LMENU) >> ((sizeof(SHORT) * 8) - 1); // shift some bits to get a bool.

        switch(wParam){

            case WM_LBUTTONDOWN: //left mouse button down
                if(altKeyDown){  //If ALT-Key is pressed...
                    mouseMiddleClick(true, false); //...fire middle mouse button down...
                    return(1); //...and consume the left click, the system just gets the middle click.
                }
            break;

            case WM_LBUTTONUP: //left mouse button up
                if(middleButtonDown){ //If we still keep pressing the middle mouse button...
                    mouseMiddleClick(false, true); //...we release it now.
                    return(1); //consume...
                }
            break;
        }
    }
    //We don't need this event, so let's pass it instead of consuming it.
    return(CallNextHookEx(NULL, nCode, wParam, lParam));
}

// Get keypress events. Fire middle click when getting L_SHIFT + L_WIN + L_CTRL + F22 (sic!)
LRESULT CALLBACK LowLevelKeyboardProc(int nCode, WPARAM wParam, LPARAM lParam) {

    if (nCode == HC_ACTION) {
        PKBDLLHOOKSTRUCT p = (PKBDLLHOOKSTRUCT)lParam;

        //The following is a little bit tricky...
        //Those 4 keys can't be recognized at the same time, so they are triggered one after another.
        //Can't remember the exact order, however L_Win is the first and F22 the last one "pressed".
        //We only want to fire a middle click when no other key is "pressed" by windows, so we wait
        //until Windows "releases" all these keys, while L_WIN key is the last one.

        //Get current key states...
        controlKeyDown = GetAsyncKeyState (VK_LCONTROL) >> ((sizeof(SHORT) * 8) - 1);
        winKeyDown = GetAsyncKeyState (VK_LWIN) >> ((sizeof(SHORT) * 8) - 1);
        shiftKeyDown = GetAsyncKeyState (VK_LSHIFT) >> ((sizeof(SHORT) * 8) - 1);

        if (p->vkCode == VK_F22 && controlKeyDown && winKeyDown && shiftKeyDown)  { //F22 + 3 other keys
            switch(wParam){

                case WM_KEYUP: //F22 key released
                case WM_SYSKEYUP:
                    readyForClick = true; //middle click event unlocked and ready...
                break;
            }
        } else if (p->vkCode == VK_LWIN && readyForClick){
            switch(wParam){

                case WM_KEYUP: //L_WIN key released
                case WM_SYSKEYUP:
                    mouseMiddleClick(true, true); //...fire middle mouse click event!
                    readyForClick = false;
                break;
            }
        }
    }
    //Better not consume anything here...
    return(CallNextHookEx(NULL, nCode, wParam, lParam));
}

// main sweet main
int main(int argc, char** argv){

    //deactivate ALT-key feature when passing -a
    if(argc > 1){
        if(std::string(argv[1]) == "-a")
            useAltKey = false;
    }

    // install low-level keyboard & mouse hooks
    HHOOK LLKBHook = SetWindowsHookEx(WH_KEYBOARD_LL, LowLevelKeyboardProc, GetModuleHandle("kernel32.dll"), 0);
    HHOOK LLMouseHook;
    if(useAltKey)
       LLMouseHook = SetWindowsHookEx(WH_MOUSE_LL, LowLevelMouseProc, GetModuleHandle("kernel32.dll"), 0);

    //MidMouse must not be killed but exits gracefully when Windows shuts down.
    MSG msg;
    while (GetMessage(&msg, NULL, NULL, NULL)) {
        TranslateMessage(&msg);
        DispatchMessage(&msg);
    }

    //Unhook stuff when shutting down
    UnhookWindowsHookEx(LLKBHook);
    if(useAltKey)
        UnhookWindowsHookEx(LLMouseHook);

    //This is the end...
    return(0);
}
