package com.linkesoft.bbingo;

import android.content.Context;
import android.content.Intent;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.os.Bundle;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.text.TextUtils;
import android.util.Log;
import android.widget.Toast;

import com.google.android.gms.common.ConnectionResult;
import com.google.android.gms.common.api.GoogleApiClient;
import com.google.android.gms.common.api.ResultCallback;
import com.google.android.gms.common.api.Status;
import com.google.android.gms.nearby.Nearby;
import com.google.android.gms.nearby.connection.AdvertisingOptions;
import com.google.android.gms.nearby.connection.ConnectionInfo;
import com.google.android.gms.nearby.connection.ConnectionLifecycleCallback;
import com.google.android.gms.nearby.connection.ConnectionResolution;
import com.google.android.gms.nearby.connection.Connections;
import com.google.android.gms.nearby.connection.ConnectionsStatusCodes;
import com.google.android.gms.nearby.connection.DiscoveredEndpointInfo;
import com.google.android.gms.nearby.connection.DiscoveryOptions;
import com.google.android.gms.nearby.connection.EndpointDiscoveryCallback;
import com.google.android.gms.nearby.connection.Payload;
import com.google.android.gms.nearby.connection.PayloadCallback;
import com.google.android.gms.nearby.connection.PayloadTransferUpdate;
import com.google.android.gms.nearby.connection.Strategy;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Sammelt alle Funktionen rund um das Nearby Connection API
 */

public class NearbyConnections {

    private final Context context;
    private final String userName;
    private final GoogleApiClient googleApiClient;
    private static final String SERVICE_ID = "com.linkesoft.bbingo";
    // broadcast intents
    public static final String CONNECTIONS_UPDATED = SERVICE_ID + ".connectionsUpdated";
    public static final String SELECT_WORDLIST = SERVICE_ID + ".selectWordList";
    public static final String WINNER_FOUND = SERVICE_ID + ".winnerFound";
    public static final String USER_NAME = "winnerName";

    private static final int PAYLOAD_TYPE_WINNER = 1;
    private static final int PAYLOAD_TYPE_WORDLIST = 2;

    private static final String TAG="NearbyConnections"; // for logging

    // lesbare Namen der verbundenen Geräte
    private final Map<String,String> pendingPeerNames = new HashMap<>();
    private final Map<String,String> connectedPeerNames = new HashMap<>();
    // File Payloads
    private final Map<Long,Payload> pendingPayloads = new HashMap<>();
    public Bitmap lastWinnerBitmap;

    private String lastErrorMessage = null;

    public NearbyConnections(final Context context, String userName) {
        this.context = context;
        this.userName = userName;

        googleApiClient =
                new GoogleApiClient.Builder(context)
                        .addApi(Nearby.CONNECTIONS_API)
                        .addConnectionCallbacks(new GoogleApiClient.ConnectionCallbacks() {
                            @Override
                            public void onConnected(@Nullable Bundle bundle) {
                                Log.v(TAG,"connected");
                                lastErrorMessage=null;
                                startAdvertising();
                                startDiscovering();
                            }

                            @Override
                            public void onConnectionSuspended(int reason) {
                                Log.i(TAG,"suspended");

                            }
                        })
                        .addOnConnectionFailedListener(new GoogleApiClient.OnConnectionFailedListener() {
                            @Override
                            public void onConnectionFailed(@NonNull ConnectionResult connectionResult) {
                                lastErrorMessage = connectionResult.getErrorMessage();
                                Log.i(TAG,"failed "+lastErrorMessage);
                                Toast.makeText(context, lastErrorMessage, Toast.LENGTH_LONG).show();
                                refreshStatus();
                            }
                        })
                        .build();
        googleApiClient.connect();
    }

    public void disconnect() {
        if(googleApiClient.isConnected()) {
            // sauberes Abräumen der Verbindung
            Nearby.Connections.stopAdvertising(googleApiClient);
            Nearby.Connections.stopDiscovery(googleApiClient);
            Nearby.Connections.stopAllEndpoints(googleApiClient);
            googleApiClient.disconnect();
        }
        pendingPeerNames.clear();
        connectedPeerNames.clear();
        pendingPayloads.clear();
        refreshStatus();
    }

    public void sendWordList(BBingoDB.WordList wordList) {
        // konvertiere in Bytes
        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        DataOutputStream dos = new DataOutputStream(bos);
        try {
            dos.writeInt(PAYLOAD_TYPE_WORDLIST);
            dos.writeUTF(wordList.title);
            dos.writeUTF(wordList.words);
            dos.close();
            byte[] bytes = bos.toByteArray();
            // sende an alle Empfänger
            List<String> endPoints = new ArrayList<>(connectedPeerNames.keySet());
            if(endPoints.size()!=0) {
                Nearby.Connections.sendPayload(googleApiClient,endPoints,Payload.fromBytes(bytes));
            }
        } catch(IOException e) {
            Log.e(TAG,"Could not create payload",e);
        }
    }

    public void sendWinner(Bitmap bitmap) {
        // sende zuerst Screenshot Bitmap als File
        sendBitmap(bitmap);
        // sende "Win"-Info
        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        DataOutputStream dos = new DataOutputStream(bos);
        try {
            dos.writeInt(PAYLOAD_TYPE_WINNER);
            dos.close();
            byte[] bytes = bos.toByteArray();
            // sende an alle Empfänger
            List<String> endPoints = new ArrayList<>(connectedPeerNames.keySet());
            if (endPoints.size() != 0) {
                Nearby.Connections.sendPayload(googleApiClient, endPoints, Payload.fromBytes(bytes));
            }
        } catch (IOException e) {
            Log.e(TAG, "Could not create payload", e);
        }

    }
    private void sendBitmap(Bitmap bitmap) {
        FileOutputStream out = null;
        File tmpDir = context.getCacheDir();
        tmpDir.mkdirs();
        try {
            // erzeuge temporäre Datei
            File uploadFile = File.createTempFile("winnerScreen", ".png", tmpDir);
            out = new FileOutputStream(uploadFile);
            bitmap.compress(Bitmap.CompressFormat.PNG, 100, out); // 100 wird für PNG ignoriert
            out.close();
            out = null;
            // sende als Datei an alle Empfänger
            List<String> endPoints = new ArrayList<>(connectedPeerNames.keySet());
            if (endPoints.size() != 0) {
                Nearby.Connections.sendPayload(googleApiClient, endPoints, Payload.fromFile(uploadFile));
            }
        } catch (Exception e) {
            Log.e(TAG,"Could not send bitmap",e);
        } finally {
            try {
                if (out != null) {
                    out.close();
                }
            } catch (IOException e) {
                Log.e(TAG, "Could not send bitmap", e);
            }
        }
    }

    private void handlePayload(String endPointId, byte[] bytes) {
        ByteArrayInputStream bis = new ByteArrayInputStream(bytes);
        DataInputStream dis = new DataInputStream(bis);
        try {
            int tag = dis.readInt();
            if(tag == PAYLOAD_TYPE_WINNER) {
                String winnerName = connectedPeerNames.get(endPointId);
                Intent intent = new Intent(WINNER_FOUND);
                intent.putExtra(USER_NAME,winnerName);
                context.sendBroadcast(intent);
            } else if(tag == PAYLOAD_TYPE_WORDLIST) {
                String wordListTitle = dis.readUTF();
                String wordListWords = dis.readUTF();
                Intent intent = new Intent(SELECT_WORDLIST);
                intent.putExtra(BBingoDB.TITLE,wordListTitle);
                intent.putExtra(BBingoDB.WORDS,wordListWords);
                String senderUserName = connectedPeerNames.get(endPointId);
                intent.putExtra(USER_NAME,senderUserName);
                context.sendBroadcast(intent);
            }
			dis.close();

        } catch(IOException e) {
            Log.e(TAG,"Could not decode payload of size "+bytes.length,e);
        }
    }

    /**
     * Beschreibe aktuellen Zustand (z.B. "Suche nach Partnern")
     * @return menschenlesbarer Status
     */
    public @Nullable String status() {
        if(lastErrorMessage != null)
            return lastErrorMessage;
        // lässt sich in Java 8 (Android 7+) eleganter mit stream() und map() lösen
        if(connectedPeerNames.size() != 0) {
            List<String> peerNames = new ArrayList<>();
            for(String userName: connectedPeerNames.values())
                peerNames.add(userName);
            return TextUtils.join(", ", peerNames);
        } else if(pendingPeerNames.size() != 0) {
            List<String> peerNames = new ArrayList<>();
            for(String userName: pendingPeerNames.values())
                peerNames.add(userName);
            return context.getString(R.string.ConnectingTo,TextUtils.join(", ", peerNames));
        } else if(googleApiClient.isConnected()) {
            return context.getString(R.string.LookingForPeers);
        } else {
            return null;
        }
    }

    public boolean hasConnectedPeers() {
        return connectedPeerNames.size() != 0;
    }

    private void startAdvertising() {
        Nearby.Connections.startAdvertising(
                googleApiClient,
                userName,
                SERVICE_ID,
                connectionLifecycleCallback,
                new AdvertisingOptions(Strategy.P2P_CLUSTER))
                .setResultCallback(
                        new ResultCallback<Connections.StartAdvertisingResult>() {
                            @Override
                            public void onResult(@NonNull Connections.StartAdvertisingResult result) {
                                if (result.getStatus().isSuccess()) {
                                    // We're advertising!
                                    lastErrorMessage = null;
                                    Log.i(TAG, "Advertising "+ result.getLocalEndpointName());
                                    // weiter geht's im connectionLifecycleCallback
                                } else {
                                    // We were unable to start advertising.
                                    lastErrorMessage =  ConnectionsStatusCodes.getStatusCodeString(result.getStatus().getStatusCode());
                                    Log.e(TAG, "Could not start advertising " + lastErrorMessage);
                                    refreshStatus();
                                }
                            }
                        });
    }
    private void startDiscovering() {
        Nearby.Connections.startDiscovery(
                googleApiClient,
                SERVICE_ID,
                endpointDiscoveryCallback,
                new DiscoveryOptions(Strategy.P2P_CLUSTER))
                .setResultCallback(
                        new ResultCallback<Status>() {
                            @Override
                            public void onResult(@NonNull Status status) {
                                if (status.isSuccess()) {
                                    // We're discovering!
                                    Log.i(TAG, "Discovering ");
                                    // weiter geht's im endpointDiscoveryCallback
                                    lastErrorMessage = null;
                                    refreshStatus();
                                } else {
                                    // We were unable to start discovering.
                                    lastErrorMessage = ConnectionsStatusCodes.getStatusCodeString(status.getStatusCode());
                                    Log.e(TAG, "Could not start discovering " + lastErrorMessage);
                                    refreshStatus();
                                }
                            }
                        });
    }

    private void refreshStatus() {
        context.sendBroadcast(new Intent(CONNECTIONS_UPDATED));
    }


    private final EndpointDiscoveryCallback endpointDiscoveryCallback =
            new EndpointDiscoveryCallback() {
                @Override
                public void onEndpointFound(
                        final String endpointId, DiscoveredEndpointInfo discoveredEndpointInfo) {
                    // ein EndPoint von startAdvertising() wurde gefunden
                    Log.d(TAG, "Found endpoint " + endpointId+" "+discoveredEndpointInfo.getEndpointName() +", connecting");
                    Toast.makeText(context, context.getString(R.string.FoundNearby,discoveredEndpointInfo.getEndpointName()), Toast.LENGTH_SHORT).show();
                    pendingPeerNames.put(endpointId,discoveredEndpointInfo.getEndpointName());
                    refreshStatus();
                    // starte Verbindung
                    Nearby.Connections.requestConnection(
                            googleApiClient, userName, endpointId, connectionLifecycleCallback)
                            .setResultCallback(
                                    new ResultCallback<Status>() {
                                        @Override
                                        public void onResult(@NonNull Status status) {
                                            // weiter geht's im connectionLifecycleCallback
                                            Log.i(TAG, "requestConnection with " + endpointId+": "+ConnectionsStatusCodes.getStatusCodeString(status.getStatusCode()));
                                        }
                                    });
                }

                @Override
                public void onEndpointLost(String endpointId) {
                    // A previously discovered endpoint has gone away.
                    Log.i(TAG, "Lost endpoint " + endpointId);
                    pendingPeerNames.remove(endpointId);
                    connectedPeerNames.remove(endpointId);
                    refreshStatus();
                }
            };

    private final ConnectionLifecycleCallback connectionLifecycleCallback =
            new ConnectionLifecycleCallback() {
                @Override
                public void onConnectionInitiated(final String endpointId, ConnectionInfo connectionInfo) {
                    // ein Endpoint hat eine Verbindung angefragt
                    Log.v(TAG, "connection with " + endpointId + " " + connectionInfo.getEndpointName() + " initiated, authentication code "+connectionInfo.getAuthenticationToken());
                    pendingPeerNames.put(endpointId,connectionInfo.getEndpointName());
                    Nearby.Connections.acceptConnection(googleApiClient, endpointId, payloadCallback)
                            .setResultCallback(new ResultCallback<Status>() {
                        @Override
                        public void onResult(@NonNull Status status) {
                            // Verbindung wird akzeptiert, weiter geht's in onConnectionResult
                            if(!status.isSuccess())
                                Log.e(TAG, "acceptConnection with " + endpointId +": " + ConnectionsStatusCodes.getStatusCodeString(status.getStatusCode()));
                        }
                    });
                }

                @Override
                public void onConnectionResult(String endpointId, ConnectionResolution result) {
                    String userName = pendingPeerNames.remove(endpointId);
                    if(result.getStatus().isSuccess()) {
                        // Verbindung steht, ab jetzt können Daten gesendet und empfangen (siehe payloadCallback) werden
                        Log.i(TAG, "Connected to " + endpointId);
                        if (userName != null) {
                            connectedPeerNames.put(endpointId, userName);
                            Toast.makeText(context, context.getString(R.string.ConnectedToNearby, userName), Toast.LENGTH_SHORT).show();
                        }
                        lastErrorMessage = null;
                        refreshStatus();
                    } else {
                            // Verbindungsversuch fehlgeschlagen
                            lastErrorMessage = ConnectionsStatusCodes.getStatusCodeString(result.getStatus().getStatusCode());
                            refreshStatus();
                            Log.e(TAG, "Could not connect to " + endpointId+": "+lastErrorMessage);
                            Toast.makeText(context, context.getString(R.string.FailedToConnectToNearby,userName), Toast.LENGTH_SHORT).show();
                    }
                }

                @Override
                public void onDisconnected(String endpointId) {
                    Log.i(TAG, "Disconnected from " + endpointId);
                    String userName = connectedPeerNames.remove(endpointId);
                    Toast.makeText(context, context.getString(R.string.DisconnectedFromNearby,userName), Toast.LENGTH_SHORT).show();
                    refreshStatus();
                }
            };

    private final PayloadCallback payloadCallback = new PayloadCallback() {
        @Override
        public void onPayloadReceived(String endPointId, Payload payload) {
            Log.v(TAG, "Payload received from " + endPointId);
            if(payload.getType() == Payload.Type.BYTES) {
                // byte-Payload (bis Nearby.Connections.MAX_BYTES_DATA_SIZE), vollständig empfangen
                handlePayload(endPointId,payload.asBytes());
            } else if(payload.getType() == Payload.Type.FILE) {
                // Datei, speichere Payload bis Datei vollständig geladen
                pendingPayloads.put(payload.getId(),payload);
            } else {
                //noinspection StatementWithEmptyBody
                // Stream
            }
        }

        @Override
        public void onPayloadTransferUpdate(String endPointId, PayloadTransferUpdate payloadTransferUpdate) {
            Log.v(TAG, "Payload update from " + endPointId);
            if(payloadTransferUpdate.getStatus() == PayloadTransferUpdate.Status.SUCCESS) {
                Payload payload = pendingPayloads.remove(payloadTransferUpdate.getPayloadId());
                if(payload!=null) {
                    // eine Datei ist fertig heruntergeladen
                    Log.v(TAG,"File downloaded "+payloadTransferUpdate.getTotalBytes()+ " B");
                    File bitmapFile = payload.asFile().asJavaFile();
                    if(bitmapFile!=null) {
                        lastWinnerBitmap = BitmapFactory.decodeFile(bitmapFile.getPath());
                        bitmapFile.delete(); // Datei kann gelöscht werden
                    } else {
                        // use file descriptor
                        Log.e(TAG,"Invalid file payload "+payload.asFile());
                    }
                }
            } else if(payloadTransferUpdate.getStatus() == PayloadTransferUpdate.Status.IN_PROGRESS) {
                Log.v(TAG,"File download progress "+payloadTransferUpdate.getBytesTransferred() + " of "+payloadTransferUpdate.getTotalBytes()+ " B");
            }
        }
    };
}
