/*******************************************************************************
 *                                                                             *
 *  MODULE      : COLORMAP.C                                                   *
 *                                                                             *
 *  DESCRIPTION : Routines for creating a resonable palette for a DIB          *
 *                                                                             *
 *  FUNCTIONS:                                                                 * 
 *                                                                             *
 *    OpenColorMapArray() - Returns a handle to an initialized color           *
 *                          histogram                                          *
 *                                                                             *
 *    AddColorToArray()   - Adds a color to a color histogram                  *
 *                                                                             *
 *    GetOptimizedPalette() - Returns a list of the most frequently added      *
 *                          colors in the histogram.                           *  
 *                                                                             *
 *    CloseColorMapArray() - Deletes a handle to a color histogram             * 
 *                                                                             *
 *  *Note:  Using this code to obtain a palette for 16,24, and 32BPP DIBs is   *
 *          not always going to give you the best looking palette. One example *
 *          where this method would crash and burn would be calculating a      *
 *          palette for a DIB that used an even sampling of the spectrum.      * 
 *          In this case, the sort would take a long time and the most         *
 *          frequently used colors would cover the spectrum unevenly.          *
 *                                                                             *  
 *******************************************************************************
 *                                                                             *
 *  Last modified: 1/20/95 by Mike Irvine                                      *
 *                                                                             *
 ******************************************************************************/  

#include "colormap.h"

typedef struct {
    WORD wFreq;
	WORD wPos;
} COLORMAP;
typedef COLORMAP COLORARR[MAXWORD+1];
typedef COLORMAP *LPCOLORMAP;

HANDLE OpenColorMapArray() 
{
  LPCOLORMAP cp = (LPCOLORMAP)GlobalAlloc(GPTR, sizeof(COLORARR));
  WORD wPos;
  
  if (cp != NULL) {
    for (wPos = 0; wPos < MAXWORD; wPos++)
	  cp[wPos].wPos = wPos;
    cp[wPos].wPos = wPos; // Avoid rolling over in for-loop
  }

  return (cp); 
}

void AddColorToArray(HANDLE hColorMap, RGBTRIPLE *rgb) 
{
   LPCOLORMAP cp = (LPCOLORMAP)hColorMap;
   WORD wPos = MAKE565WORD(rgb->rgbtRed,rgb->rgbtGreen,rgb->rgbtBlue);
    
   if (cp[wPos].wFreq < MAXWORD) 
     cp[wPos].wFreq++;
}

int _cdecl Compare(const void *elem1, const void *elem2)
{ 
  return (int)((LPCOLORMAP)elem2)->wFreq - (int)((LPCOLORMAP)elem1)->wFreq;
}

void GetOptimizedPalette(HANDLE hColorMap, RGBQUAD *rgbt, int iNumColors)
{
   WORD wPos;
   LPCOLORMAP cp = (LPCOLORMAP)hColorMap;
  
   if (iNumColors < 1)
     return;
   
   qsort((void*)cp, (size_t)MAXWORD, sizeof(COLORMAP), Compare); 
   
   for (wPos = 0; wPos < iNumColors; wPos++)
     MAKE565RGB(cp[wPos].wPos, rgbt[wPos].rgbRed, 
                               rgbt[wPos].rgbGreen, 
                               rgbt[wPos].rgbBlue);
}

HANDLE CloseColorMapArray(HANDLE hColorMap)
{
  return GlobalFree(hColorMap);
}
