
/*******************************************************************************
 *                                                                             *
 *  PROGRAM     : Icompare.c                                                   *
 *                                                                             *
 *  PURPOSE     : Application to compare 2 Bitmaps                             *
 *                                                                             *
 *******************************************************************************
 *                                                                             *
 *  Last modified: 25081999 by A. Eschenburg								   *
 *                                                                             *
 ******************************************************************************/  

#include <windows.h>
#include <io.h>
#include <stdio.h>
#include "icompare.h"
#include "dib.h"

sADIB   sImage1,sImage2,sDiff;

/******************************************************************************
 *                                                                            *
 *  FUNCTION   : ApplyBlurr(sADIB *hsADIB, INT iBwid)                         *
 *                                                                            *
 *  PURPOSE    : Given a Pointer to a DIB structure it will apply a blurr     *
 *               Filter to the DIB. NOTE : Only 24Bit DIBS are supported.     *
 *                                                                            *
 *  RETURNS    : The maximum brightness in the resulting DIB                  *
 *               Negative Results mark errors.								  *
 *                                                                            *
 *****************************************************************************/
INT ApplyBlurr (sADIB *hsADIB, INT iBwid)
{
	LPBITMAPINFOHEADER       lpbih;
	LPBITMAPINFOHEADER       lpbiht;
	sADIB                    hsTADIB;

    if (!hsADIB)
        return -10;

	if (!(iBwid & 1)) iBwid+=1;

	if (iBwid > 15) iBwid=15; 
	
	lpbih = (VOID FAR *)GlobalLock(hsADIB->hbi);

	if (hsADIB->hbi) hsTADIB.hbi = CopyDIB(hsADIB->hbi);

	lpbiht = (VOID FAR *)GlobalLock(hsTADIB.hbi);

    if (!lpbih)
        return -9;

    if (!NEW_DIB_FORMAT(lpbih))  // Return if we dont have a BITMAPINFOHEADER
        return -9;

    if (lpbih->biBitCount == 24)  // only 24Bit
    { 
        INT       dwX, dwY;
        RGBTRIPLE *rgbt = (RGBTRIPLE *)((LPSTR)lpbiht + (WORD)lpbiht->biSize + ColorTableSize(lpbiht));
        RGBTRIPLE *rgbtT = (RGBTRIPLE *)((LPSTR)lpbih + (WORD)lpbih->biSize + ColorTableSize(lpbih));
	    INT       iNoIdx=iBwid*iBwid;
	    INT       i,j;
	    INT       iHalfBwid = iBwid/2;
	    long      rgbwtAve[3];
		int       maxVal=0;
		int       tVal;


	    rgbwtAve[0]=0;
	    rgbwtAve[1]=0;
	    rgbwtAve[2]=0;

        for (dwY = 0; dwY < lpbih->biHeight; dwY++) 
		{
            for (dwX = 0; dwX < lpbih->biWidth; dwX++) 
			{
				rgbwtAve[0]=0;
				rgbwtAve[1]=0;
				rgbwtAve[2]=0;
 	 			for (i=0;i<iBwid;i++)
					for (j=0;j<iBwid;j++) 
					{
						if ((dwX+i-iHalfBwid < 0) || (dwX+i-iHalfBwid >= lpbih->biWidth) ||
							(dwY+j-iHalfBwid < 0) || (dwY+j-iHalfBwid >= lpbih->biHeight)) 
						{
							rgbwtAve[0]+=127L;
							rgbwtAve[1]+=127L;
							rgbwtAve[2]+=127L;
						}
						else
						{
							rgbwtAve[0]+=(1L*rgbt[dwX+i-iHalfBwid+lpbih->biWidth*(dwY+j-iHalfBwid)].rgbtBlue);
							rgbwtAve[1]+=(1L*rgbt[dwX+i-iHalfBwid+lpbih->biWidth*(dwY+j-iHalfBwid)].rgbtGreen);
							rgbwtAve[2]+=(1L*rgbt[dwX+i-iHalfBwid+lpbih->biWidth*(dwY+j-iHalfBwid)].rgbtRed);
						}
				}
				rgbtT[dwX+lpbih->biWidth*(dwY)].rgbtBlue  = (rgbwtAve[0] / iNoIdx ) % 256;
				rgbtT[dwX+lpbih->biWidth*(dwY)].rgbtGreen = (rgbwtAve[1] / iNoIdx ) % 256;
				rgbtT[dwX+lpbih->biWidth*(dwY)].rgbtRed   = (rgbwtAve[2] / iNoIdx ) % 256;
				tVal=rgbtT[dwX+lpbih->biWidth*(dwY)].rgbtBlue+rgbtT[dwX+lpbih->biWidth*(dwY)].rgbtRed+rgbtT[dwX+lpbih->biWidth*(dwY)].rgbtGreen;
				if (tVal>maxVal) maxVal=tVal;
            }
		}
        FreeDIB(&hsTADIB);
        return maxVal;
	}
    FreeDIB(&hsTADIB);
    return -8;
}
              

/******************************************************************************
 *                                                                            *
 *  FUNCTION   : CalcDifference (sADIB *hsADIB1, sADIB *hsADIB2)              *
 *                                                                            *
 *  PURPOSE    : Given Pointers to 2 DIB Structures, it will generate a       *
 *               difference Image, which is stored in place of the second     *
 *               DIB given.													  *
 *                                                                            *
 *  RETURNS    : The maximum contrast in the difference Image                 *
 *               Negative Results mark errors.								  *
 *                                                                            *
 *****************************************************************************/
INT CalcDifference (sADIB *hsADIB1, sADIB *hsADIB2)
{
	LPBITMAPINFOHEADER       lpbih;
	LPBITMAPINFOHEADER       lpbiht;
	int                      maxDiff = 0;
	int                      minDiff = 1000;
	int                      tDiff;
	int                      actDiff;


    if ((!hsADIB1) || (!hsADIB2))
        return -10;
	
	lpbih = (VOID FAR *)GlobalLock(hsADIB1->hbi);
	lpbiht = (VOID FAR *)GlobalLock(hsADIB2->hbi);

    if ((!lpbih) || (!lpbiht))
        return -9;

    if (!NEW_DIB_FORMAT(lpbih))  // Return if we dont have a BITMAPINFOHEADER
        return -9;
    if (!NEW_DIB_FORMAT(lpbiht))  // Return if we dont have a BITMAPINFOHEADER
        return -9;

    if ((lpbih->biBitCount == 24) && (lpbiht->biBitCount == 24)) // Only 24-Bit Images
    { 
        INT       dwX, dwY;
        RGBTRIPLE *rgbt = (RGBTRIPLE *)((LPSTR)lpbiht + (WORD)lpbih->biSize + ColorTableSize(lpbiht));
        RGBTRIPLE *rgbtT = (RGBTRIPLE *)((LPSTR)lpbih + (WORD)lpbiht->biSize + ColorTableSize(lpbih));

		if ((lpbih->biWidth!=lpbiht->biWidth) || (lpbih->biHeight!=lpbiht->biHeight))
			return -6;

        for (dwY = 0; dwY < lpbih->biHeight; dwY++) 
            for (dwX = 0; dwX < lpbih->biWidth; dwX++) 
			{
			      actDiff=0;
				  tDiff=rgbtT[dwX+lpbih->biWidth*(dwY)].rgbtBlue-rgbt[dwX+lpbih->biWidth*(dwY)].rgbtBlue;
				  if (tDiff<0) actDiff-=tDiff; else actDiff+=tDiff;
				  tDiff=rgbtT[dwX+lpbih->biWidth*(dwY)].rgbtGreen-rgbt[dwX+lpbih->biWidth*(dwY)].rgbtGreen;
				  if (tDiff<0) actDiff-=tDiff; else actDiff+=tDiff;
				  tDiff=rgbtT[dwX+lpbih->biWidth*(dwY)].rgbtRed-rgbt[dwX+lpbih->biWidth*(dwY)].rgbtRed;
				  if (tDiff<0) actDiff-=tDiff; else actDiff+=tDiff;

				  if (actDiff<minDiff) minDiff=actDiff;
				  if (actDiff>maxDiff) maxDiff=actDiff;

				  rgbt[dwX+lpbih->biWidth*(dwY)].rgbtBlue=actDiff/3;
				  rgbt[dwX+lpbih->biWidth*(dwY)].rgbtRed=actDiff/3;
				  rgbt[dwX+lpbih->biWidth*(dwY)].rgbtGreen=actDiff/3;

            }
    return maxDiff-minDiff;
	}
	return -8;
}



/****************************************************************************
 *                                                                          *
 *  FUNCTION   : main(int argc, char* const argv[])                         *
 *                                                                          *
 ****************************************************************************/
int
main(int argc, char* const argv[])
{
	int i;
	char achS1[128]="";
	char achS2[128]="";
	char achDiff[128]="";
	int iWid=0;
	int iRes;
	BOOL SaveDiff=FALSE;
	BOOL BlurrDiff=FALSE;

	if (argc>1) 
		for (i=0;i<argc;i++)
	    {
			if ((argv[i][0]=='-') || (argv[i][0] == '/'))
	        {
		        switch (argv[i][1])
				{
				case '1': 
					strcpy(achS1,&(argv[i][3]));
					if (!strrchr(achS1,'.')) strcat(achS1,".bmp");
					break;
				case '2':
					strcpy(achS2,&(argv[i][3]));
					if (!strrchr(achS2,'.')) strcat(achS2,".bmp");
					break;
				case 'd':
					strcpy(achDiff,&(argv[i][3]));
					if (!strrchr(achDiff,'.')) strcat(achDiff,".bmp");
					SaveDiff=TRUE;
					break;
				case 'b':
					iWid=atoi(&(argv[i][3]));
					break;
				case 'm':
					BlurrDiff= argv[i][3]=='2';
					break;
				}
			}
	    }
	else 
	{
		printf("Icompare v0.9\n"
			   "Aufruf : icompare -1:<Bild1> -2:<Bild2> [-d:<Differenzbild>] [-b:<Filterbreite]\n\n"
			   "  <Bild1> und <Bild2> sind die Dateinamen der zu vergleichenden Bilder\n"			   
			   "  Als <Differenzbild> wird das gefilterte Differenzbild abgelegt\n"
			   "  <Filterbreite> gibt die Breite des Unschaerfefilters an.\n"
			   "  - Die Filterbreite muss ungerade sein, und zwischen 1 und 15 liegen.\n"
			   "  - Filterbreite und Differenzbild sind optionale Parameter.\n"
			   "  - Die Endung .BMP kann weggelassen werden, sie wird automatisch angehaengt.\n"
			   );
	    return (0);
	}

	iRes=InitDIB(&sImage1,achS1);
	if (iRes<0) return (-100+iRes);

	iRes=InitDIB(&sImage2,achS2);
	if (iRes<0) return (-200+iRes);

	if (iWid || BlurrDiff) { // Filter Images
	  iRes=ApplyBlurr(&sImage1,iWid);
	  if (iRes<0) return (-100+iRes);
	  iRes=ApplyBlurr(&sImage2,iWid);
	  if (iRes<0) return (-200+iRes);
	}

	iRes=CalcDifference (&sImage1,&sImage2);

	if (iRes<0) return  (-300+iRes);

	if (iWid) {
	  iRes=ApplyBlurr(&sImage2,iWid);
	  if (iRes<0) return (-400+iRes);
    }
	else iRes=-405;

	if (SaveDiff) WriteDIB(achDiff,sImage2.hbi);

    FreeDIB(&sImage1);
    FreeDIB(&sImage2);
    return iRes;
}



/****************************************************************************
 *                                                                          *
 *  FUNCTION   : BinaryOf(DWORD dw, LPSTR szNum)                            *
 *                                                                          *
 *  PURPOSE    : Returns the binary representation for a given DWORD as a   *
 *               string.                                                    *
 *                                                                          *
 ****************************************************************************/

void BinaryOf(DWORD dw, LPSTR szNum) 
{
  int i;
  
  for (i=0; i<32; i++) {
    if (((1 << i) & dw) > 0)
      szNum[31-i] = '1';
    else
      szNum[31-i] = '0';
  }
  szNum[32] = 0;       
}


/****************************************************************************
 *                                                                          *
 *  FUNCTION   : InitDIB(hWnd)                                              *
 *                                                                          *
 *  PURPOSE    : Reads a DIB from a file, obtains a handle to it's          *
 *               BITMAPINFO struct., sets up the palette and loads the DIB. *
 *                                                                          *
 *  RETURNS    : 0 - DIB loads ok                                           *
 *               int < 0 - otherwise                                        *
 *                                                                          *
 ****************************************************************************/
INT InitDIB(sADIB *hsADIB,const char* FileName)
{
    LPBITMAPINFOHEADER lpbi;
    BITMAPINFOHEADER   bi;
	HANDLE             thbi;
	char               achFileName[128];

    FreeDIB(hsADIB);

	strcpy(achFileName,FileName);

    /* Open the file and get a handle to it's BITMAPINFO */
    hsADIB->hbi = OpenDIB((LPSTR) achFileName);
    if (hsADIB->hbi == NULL) {
        return -10; // Illegal Image
    }

    DIBInfo(hsADIB->hbi,&bi);

    lpbi = (VOID FAR *)GlobalLock(hsADIB->hbi);

	if ((lpbi->biCompression != BI_RGB) || (lpbi->biBitCount != 24))
	{
		GlobalUnlock(hsADIB->hbi);
		hsADIB->hdib = ChangeDIBFormat(hsADIB->hbi, 24, BI_RGB);
		if (hsADIB->hdib) 
		{
			if (hsADIB->hbi) thbi=hsADIB->hbi;
			hsADIB->hbi=hsADIB->hdib;
			hsADIB->hdib=thbi;
		}
		else return -7;
	}
	GlobalUnlock(hsADIB->hbi);

    return 0;
}

/****************************************************************************
 *                                                                          *
 *  FUNCTION   : FreeDIB()                                                  *
 *                                                                          *
 *  PURPOSE    : Frees all currently active bitmap, DIB and palette objects *
 *               and initializes their handles.                             *
 *                                                                          *
 ****************************************************************************/
VOID FreeDIB(sADIB *hsADIB)
{
    if (hsADIB->hpal)
        DeleteObject(hsADIB->hpal);

    if (hsADIB->hdib)
        GlobalFree(hsADIB->hdib);

    if (hsADIB->hbi && hsADIB->hbi != hsADIB->hdib)
        GlobalFree(hsADIB->hbi);

    hsADIB->fPalColors  = FALSE;
    hsADIB->bLegitDraw  = FALSE;
    hsADIB->hpal = NULL;
    hsADIB->hdib = NULL;
    hsADIB->hbi  = NULL;
}
