#include <memory.h>
#include <string.h>
#include "idea.hpp"

// Klasse IDEAKey - Kapselung von IDEA-Schlsseln
IDEAKey::IDEAKey()
{
	memset((void*)UserKey, 0, 16);
//        strcpy(UserKey, "\x00\x00\xED\x27\x5C\x8F\x8B\x3E\x16\xAF\x0D\x56\xC9\x14\x43\x0B");
	generatePartKeys();
}

IDEAKey::IDEAKey(unsigned char *userkey)
{
	memcpy((void*)UserKey, (void*)userkey, 16);
	generatePartKeys();
}

IDEAKey::~IDEAKey()
{
	memset((void*)DecryptionKey, 0, sizeof(DecryptionKey));
	memset((void*)EncryptionKey, 0, sizeof(EncryptionKey));
	memset((void*)UserKey, 0, sizeof(UserKey));
}

void IDEAKey::generateKey_start()
{
	time(&time2);
	memset((void*)UserKey, 0, 16);
	byte_pos = 0;
	bit_pos = 0;
}

bool IDEAKey::generateKey_action()
{
	time1 = time2;
	time(&time2);
	UserKey[byte_pos] |= ((time2 - time1) & 3) << bit_pos;
	if(bit_pos == 6) {
		byte_pos++;
		bit_pos = 0;
	} else
		bit_pos += 2;
	return (byte_pos < 16 ? true : false);
}

void IDEAKey::generateKey_end()
{
	generatePartKeys();
}

void IDEAKey::setkey(const unsigned char aUserKey[16])
{
	memcpy(UserKey, aUserKey, 16);
	generatePartKeys();
}

unsigned short IDEAKey::mulInv(unsigned short x)
{
	unsigned short t0, t1, q, y;

	if (x <= 1)
		return x; // 0 und 1 sind selbstinvers
	t1 = 0x10001L / x;
	y = 0x10001L % x;
	if (y == 1)
		return 1 - t1;
	t0 = 1;
	do {
		q = x / y;
		x = x % y;
		t0 += q * t1;
		if (x == 1)
			return t0;
		q = y / x;
		y = y % x;
		t1 += q * t0;
	} while (y != 1);
	return 1 - t1;
}

void IDEAKey::generatePartKeys()
{
	// Schlssel expandieren: UserKey => EncryptionKey

	unsigned i, j;
	unsigned short *ek = EncryptionKey;
	unsigned char *userkey = UserKey;

	for (j = 0; j < 8; j++) {
		ek[j] = (userkey[0] << 8) + userkey[1];
		userkey += 2;
	}
	for (i = 0; j < IDEAKEYLEN; j++) {
		i++;
		ek[i + 7] = (ek[i & 7] << 9) | (ek[i + 1 & 7] >> 7);
		ek += i & 8;
		i &= 7;
	}

	// Schlssel invertieren: EncryptionKey => DecryptionKey

	unsigned short t1, t2, t3, *p = DecryptionKey + IDEAKEYLEN;
	ek = EncryptionKey;

	t1 = mulInv(*ek++);
	t2 = -*ek++;
	t3 = -*ek++;
	*--p = mulInv(*ek++);
	*--p = t3;
	*--p = t2;
	*--p = t1;

	for (i = 0; i < IDEAROUNDS - 1; i++) {
		t1 = *ek++;
		*--p = *ek++;
		*--p = t1;

		t1 = mulInv(*ek++);
		t2 = -*ek++;
		t3 = -*ek++;
		*--p = mulInv(*ek++);
		*--p = t2;
		*--p = t3;
		*--p = t1;
	}
	t1 = *ek++;
	*--p = *ek++;
	*--p = t1;

	t1 = mulInv(*ek++);
	t2 = -*ek++;
	t3 = -*ek++;
	*--p = mulInv(*ek++);
	*--p = t3;
	*--p = t2;
	*--p = t1;
}

// Klasse XNoIDEAKey - Exception wenn kein Schlssel spezifiziert
const char* XNoIDEAKey::what () const throw()
{
	return "Kein IDEA-Schlssel gesetzt!";
}

// Klasse IDEA - Kapselung von IDEA
IDEA::IDEA()
{
	setkey(0);
}

IDEA::IDEA(IDEAKey *aKey)
{
	setkey(aKey);
}

void IDEA::encrypt(const unsigned char *inbuf, IDEACipher& outObj, 
                   unsigned size)
{
	if(!key)
		throw XNoIDEAKey();

	unsigned n, m;
	unsigned char tmpbuf[8];

	outObj.setBufferSize(size + (size % 8 > 0 ? 8 - size % 8 : 0));

	for(n = 0 ; n < size - size % 8; n += 8)
		cipher(&inbuf[n], &outObj.buffer[n], key->getEncryptionKey());
	if(size % 8 > 0) {
		memset((void*)tmpbuf, 0, 8);
		for(m = 0 ; m < size % 8 ; m++)
			tmpbuf[m] = inbuf[n + m];
		cipher(tmpbuf, tmpbuf, key->getEncryptionKey());
		for(m = 0 ; m < 8 ; m++)
			outObj.buffer[n + m] = tmpbuf[m];
	}
}

void IDEA::decrypt(const IDEACipher& inObj, unsigned char *&outbuf)
{
	if(!key)
		throw XNoIDEAKey();

	unsigned n;

	outbuf = new unsigned char[inObj.size];

	for(n = 0 ; n < inObj.size ; n += 8)
		cipher(&inObj.buffer[n], &outbuf[n], key->getDecryptionKey());
}

void IDEA::cipher(unsigned char const inbuf[8], unsigned char outbuf[8], 
                  unsigned short const *key)
{
	register unsigned short x1, x2, x3, x4, s2, s3;
	unsigned short *in, *out;

	int r = IDEAROUNDS;

	in = (unsigned short *) inbuf;
	x1 = *in++;
	x2 = *in++;
	x3 = *in++;
	x4 = *in;

	x1 = (x1 >> 8) | (x1 << 8);
	x2 = (x2 >> 8) | (x2 << 8);
	x3 = (x3 >> 8) | (x3 << 8);
	x4 = (x4 >> 8) | (x4 << 8);

	do {
		mul(x1, *key++);
		x2 += *key++;
		x3 += *key++;
		mul(x4, *key++);

		s3 = x3;
		x3 ^= x1;
		mul(x3, *key++);
		s2 = x2;
		x2 ^= x4;
		x2 += x3;
		mul(x2, *key++);
		x3 += x2;

		x1 ^= x2;
		x4 ^= x3;

		x2 ^= s3;
		x3 ^= s2;
	} while (--r);
	mul(x1, *key++);
	x3 += *key++;
	x2 += *key++;
	mul(x4, *key);

	out = (unsigned short *) outbuf;
	*out++ = (x1 >> 8) | (x1 << 8);
	*out++ = (x3 >> 8) | (x3 << 8);
	*out++ = (x2 >> 8) | (x2 << 8);
	*out   = (x4 >> 8) | (x4 << 8);
}

// Klasse IDEACipher - Kapselung von IDEA-Chiffraten
IDEACipher::IDEACipher()
{
	buffer = NULL;
	size = 0;
}

IDEACipher::~IDEACipher()
{
	if(buffer)
		delete[] buffer;
}

void IDEACipher::setBufferSize(unsigned aSize)
{
	size = aSize;
	if(buffer)
		delete[] buffer;
	buffer = new unsigned char[aSize];
}