unit DragDrop;

interface

uses
  Windows, Classes, Controls, Forms, ShellApi, Messages, ActiveX;

type

  TDropEffect=(deCopy, deMove, deLink);
  TDropEffectSet = set of TDropEffect;

  TOnDragEnter = procedure(DataObj: IDataObject; grfKeyState: Longint;
    pt: TPoint; var dwEffect: longint; var Accept:boolean) of object;
  TOnDragLeave = procedure of object;
  TOnDragOver = procedure(grfKeyState: Longint; pt: TPoint;
     var dwEffect: longint) of object;
  TOnDrop = procedure(DataObj: IDataObject; grfKeyState: Longint;
    pt: TPoint; var dwEffect: longint) of object;
  TOnQueryContinueDrag = procedure(fEscapePressed: BOOL; grfKeyState:
    Longint; var Result: HResult) of object;
  TOnGiveFeedback = procedure(dwEffect: Longint; var Result: HResult)
    of object;
  TOnProcessDropped = procedure(Sender: TObject; grfKeyState: Longint;
    pt: TPoint; dwEffect: Longint) of object;

  TFormatEtcArray = array of TFormatEtc;

  TFormatEtcList = class
  private
     FCount:integer;
     FList:TFormatEtcArray;
     function Get(Index: Integer): TFormatEtc;
     procedure Put(Index: Integer; Item: TFormatEtc);
  public
     constructor Create;
     destructor Destroy; override;
     function Add(Item: TFormatEtc):integer;
     procedure Clear;
     procedure Delete(Index: Integer);
     function Clone:TFormatEtcList;
     property Count:integer read FCount;
     property Items[Index:integer]:TFormatEtc read get write put;
  end;

  TDDInterfacedObject = class(TInterfacedObject)
  public
     function QueryInterface(const IID: TGUID; out Obj): HResult; stdcall;
     function _AddRef: Integer; stdcall;
     function _Release: Integer; stdcall;
  end;

  TEnumFormatEtc = class(TDDInterfacedObject, IEnumFormatEtc)
  protected
    FFormatEtcList:TFormatEtcList;
    FIndex: integer;
  public
    constructor Create(FormatEtcList:TFormatEtcList);
    destructor Destroy; override;
    function Next(celt: Longint; out elt;
      pceltFetched: PLongint): HResult; stdcall;
    function Skip(celt: Longint): HResult; stdcall;
    function Reset: HResult; stdcall;
    function Clone(out Enum: IEnumFormatEtc): HResult; stdcall;
  end;

  TDataObject = class(TDDInterfacedObject, IDataObject)
  private
    FFormatEtcList:TFormatEtcList;
  public
    constructor Create;
    destructor Destroy; override;
    function GetData(const formatetcIn: TFormatEtc; out medium:
      TStgMedium): HResult; stdcall;
    function GetDataHere(const formatetc: TFormatEtc; out medium:
      TStgMedium): HResult; stdcall;
    function QueryGetData(const formatetc: TFormatEtc): HResult;
      stdcall;
    function GetCanonicalFormatEtc(const formatetc: TFormatEtc;
      out formatetcOut: TFormatEtc): HResult; stdcall;
    function SetData(const formatetc: TFormatEtc; var medium:
      TStgMedium; fRelease: BOOL): HResult; stdcall;
    function EnumFormatEtc(dwDirection: Longint; out enumFormatEtc:
      IEnumFormatEtc): HResult; stdcall;
    function DAdvise(const formatetc: TFormatEtc; advf: Longint;
      const advSink: IAdviseSink; out dwConnection: Longint): HResult;
      stdcall;
    function DUnadvise(dwConnection: Longint): HResult; stdcall;
    function EnumDAdvise(out enumAdvise: IEnumStatData): HResult;
      stdcall;
    function RenderData(FormatEtc:TFormatEtc;
       var StgMedium: TStgMedium): HResult; virtual; abstract;
  end;

  TDragDrop = class;

  TDropSource = class(TDDInterfacedObject, IDropSource)
  private
    FOwner: TDragDrop;
  public
    constructor Create(AOwner: TDragDrop);
    destructor Destroy; override;
    function QueryContinueDrag(fEscapePressed: BOOL;
      grfKeyState: Longint): HResult; stdcall;
    function GiveFeedback(dwEffect: Longint): HResult; stdcall;
  end;

  TDropTarget = class(TDDInterfacedObject, IDropTarget)
  private
    FAccept:boolean;
  protected
    FOwner: TDragDrop;
    procedure SuggestDropEffect(grfKeyState: Longint;
      var dwEffect: longint); virtual;
    procedure AcceptDataObject(DataObj: IDataObject;
      var Accept:boolean); virtual;
    procedure RenderDropped(DataObj: IDataObject; grfKeyState: Longint;
      pt: TPoint; var dwEffect: longint); virtual;
  public
    constructor Create(AOwner: TDragDrop);
    destructor Destroy; override;
    function DragEnter(const dataObj: IDataObject; grfKeyState: Longint;
      pt: TPoint; var dwEffect: Longint): HResult; stdcall;
    function DragOver(grfKeyState: Longint; pt: TPoint;
      var dwEffect: Longint): HResult; stdcall;
    function DragLeave: HResult; stdcall;
    function Drop(const dataObj: IDataObject; grfKeyState: Longint;
      pt: TPoint; var dwEffect: Longint): HResult; stdcall;
  end;

  TDragDrop = class(TComponent)
  private
    FDragDropControl: TWinControl;
    FRegistered: Boolean;
    FTargetEffectsSet: TDropEffectSet;
    FTargetEffects: longint;
    FOnQueryContinueDrag: TOnQueryContinueDrag;
    FOnGiveFeedback: TOnGiveFeedback;
    FOnDragEnter: TOnDragEnter;
    FOnDragLeave: TOnDragLeave;
    FOnDragOver: TOnDragOver;
    FOnDrop: TOnDrop;
    FSourceEffectsSet: TDropEffectSet;
    FSourceEffects: longint;
    FOnProcessDropped: TOnProcessDropped;
    OldWndProc:Pointer;
    WndProcPtr:Pointer;
    FgrfKeyState: Longint;
    Fpt: TPoint;
    FdwEffect: Longint;
    FMessageHooked:boolean;
    FIsSource:boolean;
    procedure WndMethod(var Msg: TMessage);
    procedure SetDragDropControl(WinControl: TWinControl);
    procedure SetSourceEffects(Values:TDropEffectSet);
    procedure SetTargetEffects(Values:TDropEffectSet);
  protected
    FDropTarget: TDropTarget;
    procedure Loaded; override;
    procedure Notification(AComponent: TComponent;
      Operation: TOperation); override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function RegisterTarget: Boolean;
    function UnRegisterTarget: Boolean;
    procedure HookMessageHandler;
    procedure UnhookMessageHandler(ForceUnhook:boolean);
    procedure Execute(DataObject:IDataObject);
    property Registered: Boolean read FRegistered default False;
  published
    property DragDropControl: TWinControl read FDragDropControl
      write SetDragDropControl;
    property SourceEffects: TDropEffectSet read FSourceEffectsSet
      write SetSourceEffects;
    property TargetEffects: TDropEffectSet read FTargetEffectsSet
      write SetTargetEffects;
    property OnDragEnter: TOnDragEnter read FOnDragEnter
      write FOnDragEnter;
    property OnDragLeave: TOnDragLeave read FOnDragLeave
      write FOnDragLeave;
    property OnDragOver: TOnDragOver read FOnDragOver
      write FOnDragOver;
    property OnDrop: TOnDrop read FOnDrop write FOnDrop;
    property OnQueryContinueDrag: TOnQueryContinueDrag
      read FOnQueryContinueDrag
             write FOnQueryContinueDrag;
    property OnGiveFeedback: TOnGiveFeedback read FOnGiveFeedback
             write FOnGiveFeedback;
    property OnProcessDropped: TOnProcessDropped read FOnProcessDropped
      write FOnProcessDropped;
  end;

procedure Register;

implementation

var DDM_ProcessDropped:DWord;

// TFormatEtcList ----------------------------------------------------

constructor TFormatEtcList.Create;
begin
  inherited Create;
  FCount:=0;
  SetLength(FList, 0);
end;

destructor TFormatEtcList.Destroy;
begin
  if (FCount>0) and (FList<>nil) then SetLength(FList, 0);
  inherited Destroy;
end;

function TFormatEtcList.Get(Index: Integer): TFormatEtc;
begin
  if (Index>=FCount) or (FList=nil) then
    raise EListError.Create('Invalid item index')
  else Result:=FList[Index];
end;

procedure TFormatEtcList.Put(Index: Integer; Item: TFormatEtc);
begin
  if (Index>=FCount) or (FList=nil) then
    raise EListError.Create('Invalid item index')
  else FList[Index]:=Item;
end;

function TFormatEtcList.Add(Item: TFormatEtc):integer;
begin
  SetLength(FList, Succ(FCount));
  FList[FCount]:=Item;
  Result:=FCount;
  inc(FCount);
end;

procedure TFormatEtcList.Clear;
begin
  SetLength(Flist, 0);
  FCount:=0;
end;

function TFormatEtcList.Clone:TFormatEtcList;
var FEL:TFormatEtcList;
begin
  FEL:=TFormatEtcList.Create;
  if FList<>nil then
  begin
    SetLength(FEL.FList, FCount);
    CopyMemory(FEL.FList,FList,FCount*SizeOf(TFormatEtc));
    FEL.FCount:=FCount;
  end;
  Result:=FEL;
end;

procedure TFormatEtcList.Delete(Index: Integer);
var movecount:integer;
begin
  if (Index>=FCount) or (FList=nil) then
     raise EListError.Create('Invalid item index')
  else
  begin
    movecount:=FCount-Index-1;
    System.move(FList[Index+1],FList[Index],movecount*sizeof(TFormatEtc));
    dec(FCount);
    SetLength(FList, FCount);
  end;
end;

// TDDInterfacedObject -----------------------------------------------

function TDDInterfacedObject.QueryInterface(const IID: TGUID; out Obj):
  HResult;
begin
  Result:=inherited QueryInterface(IID,Obj);
end;

function TDDInterfacedObject._AddRef: Integer;
begin
  Result:=inherited _AddRef;
end;

function TDDInterfacedObject._Release: Integer;
begin
  Result:=inherited _Release;
end;

// TEnumFormatEtc ----------------------------------------------------

constructor TEnumFormatEtc.Create(FormatEtcList:TFormatEtcList);
begin
  inherited Create;
  _AddRef;
  FFormatEtcList:=FormatEtcList;
end;

destructor TEnumFormatEtc.Destroy;
begin
  if Assigned(FFormatEtcList) then FFormatEtcList.Free;
  inherited Destroy;
end;

function TEnumFormatEtc.Next(celt: Longint; out elt;
      pceltFetched: PLongint): HResult;
var copycount:integer;
begin
  Result:=S_False;
  if pceltFetched<>nil then pceltFetched^:=0;
  if (celt<=0) or (FFormatEtcList.Count=0) or
     (FIndex>=FFormatEtcList.Count) or
     ((pceltFetched=nil) and (celt<>1)) then exit;
  copycount:=FFormatEtcList.Count-FIndex;
  if celt<copycount then copycount:=celt;
  if pceltFetched<>nil then pceltFetched^:=copycount;
  CopyMemory(@TFormatEtc(elt),@TFormatEtc(FFormatEtcList.FList[FIndex]),
    copycount*sizeof(TFormatEtc));
  inc(FIndex,copycount);
  Result:=S_OK;
end;

function TEnumFormatEtc.Skip(celt: Longint): HResult;
begin
  if (FIndex+celt<=FFormatEtcList.Count) then
  begin
    inc(FIndex,celt);
    Result:=S_Ok;
  end
  else Result:=S_False;
end;

function TEnumFormatEtc.Reset: HResult;
begin
  FIndex:=0;
  Result:=S_OK;
end;

function TEnumFormatEtc.Clone(out Enum: IEnumFormatEtc): HResult;
begin
  Result:=S_OK;
  try
    Enum:=TEnumFormatEtc.Create(FFormatEtcList);
    TEnumFormatEtc(Enum).FIndex := FIndex;
  except
    Result:=E_Fail;
  end;
end;

// TDataObject -------------------------------------------------------

constructor TDataObject.Create;
begin
  inherited Create;
  _AddRef;
  FFormatEtcList:=TFormatEtcList.Create;
end;

destructor TDataObject.Destroy;
begin
  FFormatEtcList.Free;
  inherited Destroy;
end;

function TDataObject.GetData(const formatetcIn: TFormatEtc;
  out medium: TStgMedium): HResult;
var i:integer;
begin
  try
    if FFormatEtcList.Count>0 then
       for i:=0 to FFormatEtcList.Count-1 do
         if (formatetcIn.tymed and FFormatEtcList.Items[i].tymed<>0) and
            (formatetcIn.lindex=FFormatEtcList.Items[i].lindex) and
            (formatetcIn.dwAspect=FFormatEtcList.Items[i].dwAspect) and
            (formatetcIn.cfFormat=FFormatEtcList.Items[i].cfFormat) then
         begin
           Result:=RenderData(formatetcIn,medium);
           exit;
         end;
    Result:=DV_E_FormatEtc;
  except
    medium.HGlobal:=0;
    Result:=E_Fail;
  end;
end;

function TDataObject.GetDataHere(const formatetc: TFormatEtc;
  out medium: TStgMedium): HResult;
begin
  Result:=E_NOTIMPL;
end;

function TDataObject.GetCanonicalFormatEtc(const formatetc: TFormatEtc;
      out formatetcOut: TFormatEtc): HResult;
begin
  Result:=E_NOTIMPL;
end;

function TDataObject.QueryGetData(const formatetc: TFormatEtc): HResult;
const DVError:array[0..3] of HResult=(DV_E_FORMATETC, DV_E_TYMED,
  DV_E_DVASPECT, DV_E_LINDEX);
var i,j:integer;
begin
  j:=0;
  if FFormatEtcList.Count>0 then
    for i:=0 to FFormatEtcList.Count-1 do
      if FormatEtc.cfFormat=FFormatEtcList.Items[i].cfFormat then
      begin
        if FormatEtc.tymed and FFormatEtcList.Items[i].tymed<>0 then
        begin
          if FormatEtc.dwAspect=FFormatEtcList.Items[i].dwAspect then
          begin
            if FormatEtc.lindex=FFormatEtcList.Items[i].lindex then
            begin
                 Result:=S_OK;
                 exit;
            end
            else if j<3 then j:=3;
          end
          else if j<2 then j:=2;
        end
        else if j<1 then j:=1;
      end;
  Result:=DVError[j];
end;

function TDataObject.EnumFormatEtc(dwDirection: Longint;
  out enumFormatEtc: IEnumFormatEtc): HResult;
begin
  Result:=E_Fail;
  if dwDirection=DATADIR_GET then
  begin
    EnumFormatEtc:=TEnumFormatEtc.Create(FFormatEtcList.Clone);
    Result:=S_OK;
  end
  else EnumFormatEtc:=nil;
  if EnumFormatEtc=nil then Result:=OLE_S_USEREG;
end;

function TDataObject.SetData(const formatetc: TFormatEtc;
  var medium: TStgMedium; fRelease: BOOL): HResult;
var i:integer;
    AddData:boolean;
begin
  Result:=E_Fail;
  if FRelease then exit;
  AddData:=true;
  if FFormatEtcList.Count>0 then
    for i:=0 to FFormatEtcList.Count-1 do
        if FFormatEtcList.Items[i].cfFormat=FormatEtc.cfFormat then
        begin
          AddData:=false;
          FFormatEtcList.Items[i]:=FormatEtc;
        end;
  if AddData then FFormatEtcList.Add(FormatEtc);
end;

function TDataObject.DAdvise(const formatetc: TFormatEtc; advf: Longint;
      const advSink: IAdviseSink; out dwConnection: Longint): HResult;
begin
  Result:=E_NOTIMPL;
end;

function TDataObject.DUnadvise(dwConnection: longint): HResult; stdcall;
begin
  Result:=E_NOTIMPL;
end;

function TDataObject.EnumDAdvise(out enumAdvise: IEnumStatData): HResult;
begin
  Result:=OLE_E_AdviseNotSupported;
end;

// TDropSource methods -----------------------------------------------

constructor TDropSource.Create(AOwner: TDragDrop);
begin
  inherited Create;
  _AddRef;
  FOwner:=AOwner;
end;

destructor TDropSource.Destroy;
begin
  inherited Destroy;
end;

function TDropSource.QueryContinueDrag(fEscapePressed: BOOL;
      grfKeyState: Longint): HResult; stdcall;
begin
  // drag-and-drop abgebrochen?
  if (((grfKeyState and MK_LBUTTON)<>0) and
     ((grfKeyState and MK_RBUTTON)<>0)) or fEscapePressed then
     Result:=DRAGDROP_S_CANCEL
  // drag-and-drop beendet?
  else if (((grfKeyState and MK_LBUTTON)=0) and
          ((grfKeyState and MK_RBUTTON)=0)) then Result:=DRAGDROP_S_DROP
       else Result:=S_OK;
  if assigned(FOwner.FOnQueryContinueDrag) then
    FOwner.FOnQueryContinueDrag(fEscapePressed,grfKeyState,Result);
  if screen.cursor=crNoDrop then messagebeep(0);
end;

function TDropSource.GiveFeedback(dwEffect: Longint): HResult; stdcall;
begin
     if Assigned(FOwner.FOnGiveFeedback) then
        FOwner.FOnGiveFeedback(dwEffect,Result);
     Result:=DRAGDROP_S_USEDEFAULTCURSORS
end;

// TDropTarget interface ---------------------------------------------

constructor TDropTarget.Create(AOwner: TDragDrop);
begin
  inherited Create;
  FOwner:=AOwner;
  _AddRef;
end;

destructor TDropTarget.Destroy;
begin
  inherited Destroy;
end;

procedure TDropTarget.SuggestDropEffect(grfKeyState: Longint;
  var dwEffect: longint);
begin
  if (grfKeyState and MK_CONTROL=0) and (grfKeyState and MK_SHIFT<>0) and
     (FOwner.FTargetEffects and DropEffect_Move<>0) then
     dwEffect:=DropEffect_Move
  else if (grfKeyState and MK_CONTROL<>0) and
          (grfKeyState and MK_SHIFT<>0) and
          (FOwner.FTargetEffects and DropEffect_Link<>0) then
          dwEffect:=DropEffect_Link
       else if (deCopy in FOwner.FTargetEffectsSet) and
               (dwEffect and DropEffect_Copy<>0) then
               dwEffect:=DropEffect_Copy
            else if (deMove in FOwner.FTargetEffectsSet) and
                    (dwEffect and DropEffect_Move<>0) then
                    dwEffect:=DropEffect_Move
                 else if (deLink in FOwner.FTargetEffectsSet) and
                         (dwEffect and DropEffect_Link<>0) then
                         dwEffect:=DropEffect_Link
                      else dwEffect:=DropEffect_None;
end;

procedure TDropTarget.AcceptDataObject(DataObj: IDataObject;
  var Accept:boolean);
begin
  Accept:=true;
end;

function TDropTarget.DragEnter(const dataObj: IDataObject;
  grfKeyState: Longint; pt: TPoint; var dwEffect: Longint): HResult;
begin
  SuggestDropEffect(grfKeyState,dwEffect);
  AcceptDataObject(DataObj, FAccept);
  if Assigned(FOwner.OnDragEnter) then
    FOwner.OnDragEnter(DataObj, grfKeyState, pt, dwEffect, FAccept);
  if FAccept=false then dwEffect:=DropEffect_None;
  Result:= NOERROR;
end;

function TDropTarget.DragOver(grfKeyState: Longint; pt: TPoint;
         var dwEffect: Longint): HResult;
begin
  if FAccept=false then dwEffect:=DropEffect_None;
  SuggestDropEffect(grfKeyState,dwEffect);
  if Assigned(FOwner.OnDragOver) then
     FOwner.OnDragOver(grfKeyState, pt, dwEffect);
  Result:=NOERROR;
end;

function TDropTarget.DragLeave: HResult;
begin
  if Assigned(FOwner.OnDragLeave) then FOwner.OnDragLeave;
  Result:=NOERROR;
end;

function TDropTarget.Drop(const dataObj: IDataObject;
  grfKeyState: Longint; pt: TPoint; var dwEffect: Longint): HResult;
begin
  Result:=E_Fail;
  if FAccept then SuggestDropEffect(grfKeyState,dwEffect)
  else dwEffect:=DropEffect_None;
  if Assigned(FOwner.OnDragOver) then
     FOwner.OnDragOver(grfKeyState, pt, dwEffect);
  if dwEffect<>DROPEFFECT_None then
  begin
    if assigned(FOwner.OnDrop) then
       FOwner.OnDrop(DataObj, grfKeyState, pt, dwEffect);
    TDragDrop(FOwner).FdwEffect:=dwEffect;
    TDragDrop(FOwner).FgrfKeyState:=grfKeyState;
    TDragDrop(FOwner).Fpt:=pt;
    if TDragDrop(FOwner).FIsSource=false then
       RenderDropped(DataObj, grfKeyState, pt, dwEffect);
    PostMessage(FOwner.DragDropControl.Handle,DDM_ProcessDropped,0,0);
    Result:=NOERROR;
  end;
end;

procedure TDropTarget.RenderDropped(DataObj: IDataObject;
  grfKeyState: Longint; pt: TPoint; var dwEffect: longint);
begin
  // bei Bedarf ueberschreiben
end;

// TDragDrop control -------------------------------------------------

constructor TDragDrop.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FDropTarget:=TDropTarget.Create(Self);
  FRegistered:=False;
  FDragDropControl:=nil;
  FMessageHooked:=false;
  FIsSource:=false;
end;

destructor TDragDrop.Destroy;
begin
  UnregisterTarget;
  UnhookMessageHandler(true);
  FDropTarget._Release;
  FDropTarget:=nil;
  FDragDropControl:=nil;
  inherited Destroy;
end;

procedure TDragDrop.WndMethod(var Msg: TMessage);
// message-hook, um "DDM_ProcessDropped" zu empfangen
begin
  with Msg do
  begin
    if (Msg=DDM_ProcessDropped) and assigned(FOnProcessDropped) then
       FOnProcessDropped(self, FgrfKeyState, Fpt, FdwEffect);
    if Msg=WM_Destroy then
    begin
      if FRegistered then
      begin
        CoLockObjectExternal(FDropTarget, false, false);
        if FDragDropControl.HandleAllocated then
           RevokeDragDrop(FDragDropControl.Handle);
           FRegistered:=false;
      end;
      FMessageHooked:=false;
    end;
    Result:=CallWindowProc(OldWndProc, DragDropControl.Handle, Msg,
      wParam, LParam);
  end;
end;

procedure TDragDrop.Loaded;
begin
  inherited Loaded;
  if (FDragDropControl<>nil) and
     (csDesigning in ComponentState=false) then RegisterTarget;
end;

procedure TDragDrop.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited Notification(AComponent,Operation);
  if (AComponent=FDragDropControl) and (Operation=opRemove) then
  begin
    UnregisterTarget;
    UnhookMessageHandler(true);
    FDragDropControl:=nil;
  end;
end;

function TDragDrop.RegisterTarget: Boolean;
begin
  Result:=false;
  try
    HookMessageHandler;
  finally
  end;
  if FRegistered or (FTargetEffects=0) or (FDragDropControl=nil) then
    exit;
  try
    // Ensure that drag-and-drop interface stays in memory
    CoLockObjectExternal(FDropTarget, True, False);
    if RegisterDragDrop(FDragDropControl.Handle,
       IDropTarget(FDropTarget))=S_OK then
    begin
      Result:=True;
      FRegistered:=True;
    end;
  except
    Result:=false;
    FRegistered:=false;
  end;
end;

function TDragDrop.UnRegisterTarget: Boolean;
begin
  Result:=false;
  if (FRegistered=false) or (FDragDropControl=nil) then exit;
  try
     UnHookMessageHandler(false);
     CoLockObjectExternal(FDropTarget, false, False);
     if (FDragDropControl.HandleAllocated=false) or
        (FDragDropControl.HandleAllocated and
        (RevokeDragDrop(FDragDropControl.Handle)=S_OK)) then
     begin
       FRegistered:=false;
       Result:=true;
     end;
  except
  end;
end;

procedure TDragDrop.HookMessageHandler;
begin
  if (FDragDropControl=nil) or (FDragDropControl.Handle=0) then exit;
  if (FMessageHooked=false) and
     ((FSourceEffects<>0) or (FTargetEffects<>0)) then
  begin
    WndProcPtr:=MakeObjectInstance(WndMethod);
    OldWndProc:=Pointer(SetWindowLong(FDragDropControl.Handle,
      GWL_WNDPROC, longint(WndProcPtr)));
    FMessageHooked:=true;
  end;
end;

procedure TDragDrop.UnhookMessageHandler(ForceUnhook:boolean);
begin
  if FMessageHooked and (ForceUnhook or
     ((FSourceEffects=0) and (FTargetEffects=0))) then
  begin
    begin
      SetWindowLong(FDragDropControl.Handle, GWL_WNDPROC,
        longint(OldWndProc));
      FreeObjectInstance(WndProcPtr);
      WndProcPtr:=nil;
      OldWndProc:=nil;
    end;
    FMessageHooked:=false;
  end;
end;

procedure TDragDrop.SetDragDropControl(WinControl: TWinControl);
begin
  if WinControl<>FDragDropControl then
  begin
    if FRegistered and (csDesigning in ComponentState=false) then
    begin
      UnhookMessageHandler(true);
      UnregisterTarget;
    end;
    FDragDropControl:=WinControl;
    if (csDesigning in ComponentState=false) then RegisterTarget;
  end;
end;

procedure TDragDrop.Execute(DataObject:IDataObject);
var dwEffect: Longint;
    DropSource: TDropSource;
begin
  try
     FIsSource:=true;
     DropSource:=TDropSource.Create(self);
     try
        if (DataObject<>nil) and (DragDropControl<>nil) and
           (FSourceEffects<>0) then DoDragDrop(IDataObject(DataObject),
           DropSource, FSourceEffects, dwEffect);
     finally
        FIsSource:=false;
        DropSource._Release;
     end;
  finally
     DataObject._Release;
  end;
end;

procedure TDragDrop.SetSourceEffects(Values:TDropEffectSet);
begin
  FSourceEffectsSet:=Values;
  FSourceEffects:=0;
  if deCopy in Values then inc(FSourceEffects,DROPEFFECT_COPY);
  if deMove in Values then inc(FSourceEffects,DROPEFFECT_MOVE);
  if deLink in Values then inc(FSourceEffects,DROPEFFECT_LINK);  
  if (csDesigning in ComponentState=false) and
     (csLoading in ComponentState=false) then
  begin
    if (csDesigning in ComponentState=false) and (FMessageHooked=false)
       and (FSourceEffects<>0) then HookMessageHandler;
    if (csDesigning in ComponentState=false) and (FMessageHooked=true)
       and (FSourceEffects=0) then UnhookMessageHandler(false);
  end;
end;

procedure TDragDrop.SetTargetEffects(Values:TDropEffectSet);
begin
  FTargetEffectsSet:=Values;
  FTargetEffects:=0;
  if deCopy in Values then inc(FTargetEffects,DROPEFFECT_COPY);
  if deMove in Values then inc(FTargetEffects,DROPEFFECT_MOVE);
  if deLink in Values then inc(FTargetEffects,DROPEFFECT_LINK);  
  if (csDesigning in ComponentState=false) and (FRegistered=false) and
    (FTargetEffects<>0) then RegisterTarget;
  if (FRegistered=true) and (FTargetEffects=0) then UnRegisterTarget;
end;

// Register method ---------------------------------------------------

procedure Register;
begin
  RegisterComponents('System', [TDragDrop]);
end;

// initialize/de-initialize the ole libary ---------------------------

initialization
begin
  OleInitialize(nil);
  DDM_ProcessDropped:=RegisterWindowMessage('DDM_ProcessDropped');
end;

finalization
begin
  OleUninitialize;
end;

end.
