
//
// Projekt Raycast-Engine
// 1995 by Stefan Becker
//
// Render.c++
// Haupt-Render/Zeichen-Routinen
// (teilweise Mac-spezifisch)
//

//
// ANSI-Includes:
//

#include	<stdio.h>
#include	<stdlib.h>
#include	<string.h>
#include	<math.h>

//
// MacOS-Includes:
//

#include	<Packages.h>
#include	<GestaltEqu.h>
#include	<QDOffscreen.h>
#include	<Controls.h>
#include	<ToolUtils.h>
#include	<Script.h>

//
// Eigene Includes:
//

#include	"Globals.h"				// Allg. Definitionen
#include	"Bresenham.h"			// Zugang zum Bresenham-Algorithmus
#include	"Render.pro"			// Prototypen dieses Moduls

//
// Externe Referenzen (teilweise Mac-spezifisch):
//

extern	WINKEL_TAB	x_winkel_tab,y_winkel_tab;	// Tabelle der Richtungsvektoren
extern	MAZE		maze;						// Umgebung
extern	GWorldPtr	texture_gworld;				// Geladene Textur als Offscreen-Bitmap (Mac)
extern	GWorldPtr	frame_buffer;				// Offscreen-Bitmap zum Aufbauen der Grafik (Mac)
extern	int			my3d_flag;					// 3D- oder Kartendarstellung gewnscht?

#ifdef _NO_BRESENHAM_							// Nur beim berspringen von Bresenham ntig

extern 	SKIP_TAB	hsc_dx_tab,					// dx/dy-Tabellen fr effizienteres Raycasting
					hsc_dy_tab;
extern 	SKIP_TAB	vsc_dx_tab,
					vsc_dy_tab;

#endif

//
// Eine skalierte senkrechte Texturlinie malen:
//
// Diese Routine verwendet den MacOS-Quickdraw-Befehl
// "CopyBits" zum Skalieren und Zeichnen. Da "CopyBits" zwar
// sehr mchtig, aber nicht sehr schnell ist, wird hier die
// Haupt-Rechenleistung des Programms "verbraten". Hier ist
// also der Haupt-Ansatzpunkt zur eventuellen Beschleunigung
// der Raycast-Engine. Eventuell mte man hier eine eigene
// Texturskalierung und direktes Zeichnen in den Bildschirm
// einfgen.
//
// (komplett Mac spezifisch)
//

void draw_scaled_line(WindowPtr window, int xpos, int txt_xpos, int hoehe)
{
	PixMapHandle	pixmap;					// Verweis auf Offscreen-Bitmap
	Rect			srect,drect;			// Rechtecke
	int				ypos;					// Oberkante der zu zeichnenden Textur.

	window = window;	// Nicht benutzt!

	// Hhe sicherheitshalber auf "vernnftige" Grenzen limitieren:
	hoehe = MIN(32000,hoehe);
	hoehe = MAX(1,hoehe);

	// Texturposition sicherheitshalber auch auf "vernnftige" Werte limitieren:
	txt_xpos = MAX(0,txt_xpos);
	txt_xpos = MIN(TEX_SIZE,txt_xpos);

	// Wo mu die Oberkante der Linie im Grafikfenster hin? (0,0) ist links oben.
	ypos = (YAUF/2) - (hoehe/2);	// virtueller Horizont bei YAUF/2

	// Quell- und Zielrechteck setzen:
	SetRect(&srect,txt_xpos,0,txt_xpos+1,TEX_SIZE-1);		// Eine Spalte aus der Textur
	SetRect(&drect,xpos,ypos,xpos+1,ypos+hoehe);			// Diese skaliert malen

	// Diese senkrechte Texturlinie skaliert malen:
	pixmap = GetGWorldPixMap(texture_gworld);			// Zeiger auf die Bitmap holen
	CopyBits((BitMapPtr)*pixmap,&qd.thePort->portBits,&srect,&drect,srcCopy,0L);
	// CopyBits kann Rasterblcke skalieren, was hier ausgiebigst genutzt wird!
	// Nachteil:
	// Obiger Copybits-Aufruf "verbrt" nahezu die gesamte Rechenzeit des Programms.
	// Wenn man es schneller haben will, mu man zu "schmutzigen" Methoden greifen,
	// die Linien selbst mit Bresenham skalieren, clippen und sie direkt in den Schirm
	// hineinzeichnen. Das geht evtl. mit Grafikkarten nicht. Daher hier langsam und
	// elegant, aber dafr "sauber".
}

//
// Berechnen der Entfernung mittels Streckfaktor der orthogonalen Projektion:
//

float calc_dist(float x, float y, float rx, float ry)
{
	return (x * rx) + (y * ry);		// Geht, da Richtungsvektor (rx,ry) normiert ist. (gerade Linien)
//	return (sqrt(x*x + y*y));		// Zum Testen einer euklidischen Entfernung (->krumme Linien!)
}

//
// Ist dieser Punkt eine legale Position im Spielfeld?
//

int legal_pos(int x, int y)
{
	int		xok,yok;

	xok = ((x>=0) && (x<MAZE_MAX_X));	// Legale X-Position?
	yok = ((y>=0) && (y<MAZE_MAX_Y));	// Legale Y-Position?

	// Legaler Punkt?
	return (xok && yok);	// X und Y legal?
}

//
// Koordinatentransformation fr Karten-Modus:
//

int tx(float x)		// Punkt aus virt. Welt in Kartenkoordinaten umsetzen
{
	int	erg;

	erg = (int)(((float)x / (float)MAZE_MAX_X) * (float)XAUF);
	erg = MAX(0,erg);	// Legale Position (sicherheitshalber)
	erg = MIN(XAUF-1,erg);
	return erg;
}

int ty(float y)		// Punkt aus virt. Welt in Kartenkoordinaten umsetzen
{
	int	erg;
	
	erg = (YAUF-1) - (int)(((float)y / (float)MAZE_MAX_Y) * (float)YAUF);
	erg = MAX(0,erg);	// Legale Position (sicherheitshalber)
	erg = MIN(YAUF-1,erg);
	return erg;
}

//
// Kartenmodus darstellen:
// (grtenteils Mac-spezifisch)
//

void do_map_raycasting(WindowPtr window, int plx, int ply, int plw)
{
	Rect		wrect;				// Zeichenrechteck
	Pattern		patt;				// Benutztes Fllmuster
	int			x,y;				// x/y-Laufvariablen fr Felder
	int			p1x,p1y,p2x,p2y;	// Variablen fr Spieler-x/y-Punkte
	char		str[80];			// String fr "sprintf"

	// Nicht benutzt:
	window = window;

	// Voreinstellungen fr die Grafik (Mac):
	ForeColor(blackColor);
	GetIndPattern(&patt,0,5);

	// Das Labyrinth hinmalen:
	for(x=0; x<MAZE_X; x++)			// Alle Spalten...
		for(y=0; y<MAZE_Y; y++)		// Alle Zeilen...
		{
			if(maze[x][y] != ' ')	// Ist da eine Wand?
			{
				// Ja: Kstchen zeichnen (Mac):
				SetRect(&wrect,tx(x*128),ty((y+1)*128),tx((x+1)*128),ty(y*128));
				FillRect(&wrect,&patt);	// Geflltes Rechteck
				FrameRect(&wrect);		// Rahmen fr das Rechteck
			}
		}

	// Den Spieler mit einzeichnen:
	p1x = tx(plx);								// Aktuelle Spielerposition
	p1y = ty(ply);
	p2x = tx(plx + x_winkel_tab[plw] * 100);	// Linie mit Lnge 100 in
	p2y = ty(ply + y_winkel_tab[plw] * 100);	// Blickrichtung einzeichnen.
	MoveTo(p1x,p1y);							// Linie von (p1x,p1y) nach
	LineTo(p2x,p2y);							// (p2x,p2y) zeichnen (Mac).

	// Schriftausgabe vorbereiten:
	TextFont(applFont);		// =1 Standard-Anwendungs-Font
	TextSize(0);			// Systemfont-Gre
	TextFace(0);			// Normale Ausgabe, d.h. nicht fett o..
	TextMode(srcCopy);		// Replace-Mode beim Malen
	sprintf(str,"X:%4d Y:%4d W:%4d",plx,ply,plw);	// Ausgabestring generieren
	MoveTo(5,14);					// Hierhin zeichnen (Mac)
	DrawText(str,0,strlen(str));	// Text zeichnen (Mac)
}

#ifdef	_NO_BRESENHAM_

//
// Horizontalen Scan durchfhren:
//

int hscan(int *lx, int *ly, int winkel)
{
	float	x,y,delta_x,delta_y;
	int		ix,iy,rx,ry;
	int		zy;

	// Ist dieser Winkel fr horizontales Scannen verwendbar?
	if((hsc_dx_tab[winkel]==0) && (hsc_dy_tab[winkel]==0))
		return false;	// Nein! Illegaler Winkel => Keine Wand!

	// Startposition bernehmen:
	x = ix = *lx; 	y = iy = *ly;

	// Sehen, da man zu Beginn auf einer horizontalen Liniengrenze landet:
	if((iy % TEX_SIZE) != 0)
	{
		if(y_winkel_tab[winkel] > 0.0)	// Die Linie darber oder darunter?
		{
			// Wir mssen auf die Linie darber:
			zy = ((iy / TEX_SIZE) + 1) * TEX_SIZE;	// Y-Koordinate der gewnschten Zeile	
		}
		else
		{
			// Wir mssen auf die Linie darunter:
			zy = (iy / TEX_SIZE) * TEX_SIZE;		// Y-Koordinate der gewnschten Zeile	
		}
	
		// Schritt auf die nchste Linie durchfhren:
		delta_y = zy - iy;													// So weit ist es dahin
		delta_x = delta_y * x_winkel_tab[winkel] / y_winkel_tab[winkel];	// So weit zur Seite gehen.
		x += delta_x;		// Da liegt der neue X-Wert
		y = zy;				// Da liegt der neue Y-Wert
	}	// if (Aufsetzen)

	// Aus den Real-Koordinaten Integer-Koordinaten machen:
	ix = (int)x; 	iy = (int)y;

	// Schnell scannen:
	while(legal_pos(ix,iy))	// So lange man sich noch in virt. Welt befindet...
	{
		// In Rasterposition umrechnen. Da Texturgre eine 2er-Potenz ist,
		// knnte diese Operation auch durch Rechts-Shiften ersetzt werden (schneller!)
		rx = ix / TEX_SIZE; ry = iy / TEX_SIZE;

		// Ist da etwas oberhalb und unterhalb der aktuellen hor. Linie?
		if((maze[rx][ry]!=' ') || (maze[rx][ry-1]!=' '))
		{
			// Da ist eine Wand!
			*lx = ix; *ly = iy;	// Wandposition.
			return true;		// Hier ist eine Wand!
		}	// if
		
		// Nchsten Punkt holen:
		x += hsc_dx_tab[winkel];	y += hsc_dy_tab[winkel];
		ix = (int)x; 				iy = (int)y;
	}	// while

	// Keine Wand gefunden. Strahl geht ins Leere!
	return false;
}

//
// Vertikalen Scan durchfhren:
//

int vscan(int *lx, int *ly, int winkel)
{
	float	x,y,delta_x,delta_y;
	int		ix,iy,rx,ry;
	int		zx;

	// Ist dieser Winkel fr horizontales Scannen verwendbar?
	if((vsc_dx_tab[winkel]==0) && (vsc_dy_tab[winkel]==0))
		return false;	// Nein! Illegaler Winkel => Keine Wand!

	// Startposition bernehmen:
	x = ix = *lx; 	y = iy = *ly;

	// Sehen, da man zu Beginn auf einer vertikalen Linie landet:
	if((ix % TEX_SIZE) != 0)
	{
		if(x_winkel_tab[winkel] > 0.0)	// Die Linie rechts oder links?
		{
			// Wir mssen auf die Linie rechts:
			zx = ((ix / TEX_SIZE) + 1) * TEX_SIZE;	// X-Koordinate der gewnschten Spalte	
		}
		else
		{
			// Wir mssen auf die Linie links:
			zx = (ix / TEX_SIZE) * TEX_SIZE;		// X-Koordinate der gewnschten Spalte	
		}

		// Schritt auf die nchste Linie durchfhren:
		delta_x = zx - ix;													// So weit ist es dahin
		delta_y = delta_x * y_winkel_tab[winkel] / x_winkel_tab[winkel];	// So weit nach oben gehen.
		x = zx;			// Da liegt der neue X-Wert
		y += delta_y;	// Da liegt der neue Y-Wert
	}	// if (Aufsetzen)

	// Aus den Real-Koordinaten Integer-Koordinaten machen:
	ix = (int)x; 	iy = (int)y;

	// Schnell scannen:
	while(legal_pos(ix,iy))	// So lange man sich noch in der vrt. Welt befindet...
	{
		// In Rasterposition umrechnen:
		// Division wre durch Rechts-Shift ersetzbar.
		rx = ix / TEX_SIZE; ry = iy / TEX_SIZE;

		// Ist da etwas linkt und rechts von der senkrechten Linie?
		if((maze[rx][ry]!=' ') || (maze[rx-1][ry]!=' '))
		{
			// Ja: Eine Wand ist hier.
			*lx = ix; *ly = iy;	// Wandposition.
			return true;		// Hier ist eine Wand!
		}	// if
		
		// Nchsten Punkt holen:
		x += vsc_dx_tab[winkel];	y += vsc_dy_tab[winkel];
		ix = (int)x; 				iy = (int)y;
	}	// while

	// Nichts gefunden. Strahl geht ins Leere!
	return false;
}

//
// Einen Strahl "Raycasten" und eventuellen Auftreffpunkt auf einer Mauer
// samit zugehriger Texturposition liefern:
// (Arbeitet ohne Bresenham mit der Delta-X/Delta-Y-Methode fr max. Geschwindigkeit)
//

int cast_one_ray(int plx, int ply, int plw, int winkel, int *wall_x, int *wall_y, int *pat_pos)
{
	int			hwall_hit,vwall_hit;			// Horizontales bzw. vertikales Auftreffen auf Wand?
	int			vx,vy;							// Auftreffpunkt auf eine vertikale Linie
	int			hx,hy;							// Auftreffpunkt auf eine horizontale Linie
	float		h_dist,v_dist;					// Entfernung des horizontalen und vert. Auftreffpunkts

	//
	// Horizontalen Schnellscan durchfhren:
	//
	hx = plx; hy = ply;					// Start bei Spielerposition
	hwall_hit = hscan(&hx,&hy,winkel);	// Treffer auf eine hor. Wand? Wenn ja: wo?

	//
	// Vertikalen Schnellscan durchfhren:
	//
	vx = plx; vy = ply;					// Start bei Spielerposition
	vwall_hit = vscan(&vx,&vy,winkel);	// Treffer auf eine vert. Wand? Wenn ja: wo?

	//
	// Gar kein Treffer?
	//
	if(!vwall_hit && !hwall_hit)
		return false;	// Da ist absolut garnichts!

	//
	// Vertikaler aber kein horizontaler Treffer?
	//
	if(vwall_hit && !hwall_hit)
	{
		// Ja: Vertikalen Treffer liefern:
		*wall_x = vx; *wall_y = vy;		// Da war die Wand.
		*pat_pos = vy % TEX_SIZE;		// Diese Texturzeile ist da.
		return true;					// Treffer!
	}

	//
	// Horizontaler aber kein vertikaler Treffer?
	//
	if(!vwall_hit && hwall_hit)
	{
		// Ja: horizontalen Treffer liefern:
		*wall_x = hx; *wall_y = hy;		// Da war die Wand.
		*pat_pos = hx % TEX_SIZE;		// Diese Texturzeile ist da.
		return true;					// Treffer!
	}

	//
	// Falls mehr als 1 Treffer, dann den zum Spieler Nheren nehmen:
	//
	h_dist = calc_dist(hx-plx,hy-ply,x_winkel_tab[plw],y_winkel_tab[plw]);	// "orthogonale Distanz" errechnen
	v_dist = calc_dist(vx-plx,vy-ply,x_winkel_tab[plw],y_winkel_tab[plw]);	// "orthogonale Distanz" errechnen
	if(h_dist < v_dist)	// Horizontaler oder vertikaler Treffer nher?
	{
		*wall_x = hx; *wall_y = hy;	// horizontal war nher
		*pat_pos = hx % TEX_SIZE;	// Diese Texturspalte wird genommen.
	}
	else
	{
		*wall_x = vx; *wall_y = vy;	// vertikal war nher
		*pat_pos = vy % TEX_SIZE;	// Diese Texturspalte wird genommen.
	}
	return true;	// Treffer. Nherer der beiden Schnittpunkte.
}

#else	// _NO_BRESENHAM_

//
// Mauerschnittpunkte mit der langsameren Bresenham-Methode suchen:
//
int cast_one_ray(int plx, int ply, int plw, int winkel, int *wall_x, int *wall_y, int *pat_pos)
{
	int			endx,endy;						// Verlngerter Richtungsvektor fr Bresenham
	int			bx,by;							// Laufvariablen auf Bresenham-Weg
	int			rx,ry;							// Position im Raster in Rasterkstchen
	int			mx,my;							// Position innerhalb eines Rasterkstchens
	Bresenham	bres;							// Instanz des Bresenham-Algorithmus

	// In dieser Variante unntiger Parameter. Ignorieren.
	plw = plw;

	//
	// Ausgehend von der Spielerposition einen Strahl im richtigen Winkel "abschieen".
	// Dazu braucht man erst einmal einen geeigneten Endpunkt fr den Bresenham-Algorithmus.
	//
	endx = (int)(x_winkel_tab[winkel] * MAX_SICHT);	// Endpunkt mit Winkeltabelle errechnen
	endy = (int)(y_winkel_tab[winkel] * MAX_SICHT);	// er liegt in maximal groer Entfernung.

	bres.set(endx,endy);	// Bresenham kann damit alle Punkte von (0,0) bis Endpunkt liefern.
	while(bres.next(&bx,&by) && legal_pos(bx+plx,by+ply))		// Suchen bis Raster oder bis ins endgltige "Nichts"
	{
		// In Rasterkoordinaten umrechnen. Ginge auch mit Shifts anstatt Divisionen:
		rx = (bx + plx) / TEX_SIZE; ry = (by + ply) / TEX_SIZE;

		// Ja: ist da etwas?
		if(maze[rx][ry] != ' ')	// Wand?
		{
			// Ergebnis liefern:
			*wall_x = bx + plx;										// Da ist die Wand!
			*wall_y = by + ply;
			mx = (bx + plx) % TEX_SIZE;								// Koordinaten im Kstchen...
			my = (by + ply) % TEX_SIZE;	// Maskieren wre schneller.
			*pat_pos = (((mx==0) || (mx==TEX_SIZE-1)) ? my : mx);	// ...fr Muster auslesen.
			return true;											// Wand wurde gefunden.
		}	// if Mauer
	}	// while Bresenham
	return false;	// Nix gefunden!
}

#endif	// _NO_BRESENHAM_

//
// Hhe eines Objekts aus seiner Entfernung berechnen.
// (Wird mit Konstante/Entfernung berechnet.)
//

int calc_hoehe(float dist)
{
	float	hoehe;		// Ergebnis der Berechnung

	// Hhe berechnen:
	dist = ABS(dist);									// sicherheitshalber!!!
	dist = MIN(dist,MAX_SICHT);							// sicherheitshalber!!!
	hoehe = ((float)PERSPEKTIVE) / (dist + 1.0);		// Zurodnung Entfernung->Wandhhe
														// +1.0 wegen berlauf-Vermeidung.
	// Ergebnis als int liefern:
	return ((int)hoehe);
}

//
// Haupt-Raycast-Routine (3D-Modus):
// (teilweise Mac spezifisch)
//

void do_3d_raycasting(WindowPtr window, int plx, int ply, int plw)
{
	int				wall_x,wall_y;					// Schnittpunkt mit der Wand
	int				x,winkel,startw,endw;			// Laufvariable, Start-, Endwinkel
	float			dist;							// Distanz der Wand.
	int				hoehe;							// Daraus errechnete Hhe der darzustellenden Wand in Pixeln
	int				bx,by;							// Laufkoordinaten fr Bresenham
	int				txt_pos;						// Zu skalierende Spalte der Textur.
	char			str[80];						// String fr Koordinaten-Ausgabe.

	// Nun die Textur-Offscreen-Bitmap locken, da diese bald verwendet wird (Mac-spezifisch):
	LockPixels(GetGWorldPixMap(texture_gworld));		// Bild ist jetzt nicht verschiebbar, da gezeichnet wird!

	// Start- und Endwinkel fr Spieler-Blickwinkel errechnen:
	startw = plw + (XAUF / 2);				// Start-Blickwinkel errechnen
	if(startw >= ANZ_WINKEL)				// Legal?
		startw -= ANZ_WINKEL;				// Nein, mu "modulo" korrigiert werden.
	endw = plw - (XAUF / 2);				// End-Blickwinkel errechnen
	if(startw < 0)							// Legal?
		startw += ANZ_WINKEL;				// Nein, mu "modulo" korrigiert werden.

	// Raycasting fr alle Winkel durchfhren:
	x = 0;									// Dieser Strahl gehrt zur X-Koordinate 0 im generierten Bild
	winkel = startw;						// Mit dem Startwinkel wird links angefangen
	while(x < XAUF)							// So lange bis das gesamte Bild berechnet ist
	{
		// Einen Strahl absetzen und ggf. Mauerschnittpunkte ermitteln:
		if(cast_one_ray(plx,ply,plw,winkel,&wall_x,&wall_y,&txt_pos))
		{
			// Ja, da ist eine Wand. Aus den Koordinaten die Entfernung und daraus berechnen,
			// wie gro diese dargestellt werden soll:
			bx = wall_x - plx; by = wall_y - ply;
			dist =  calc_dist(bx,by,x_winkel_tab[plw],y_winkel_tab[plw]);	// "orthogonale Distanz" errechnen
			hoehe = calc_hoehe(dist);	// Wie hoch mu das Objekt nun sein?

			// Skalierte Texturspalte senkrecht zeichnen:
			draw_scaled_line(window,x,txt_pos,hoehe);
		}

		// Schleifenende: (Hier mu x inkrementiert und der Winkel "modulo" erhht werden)
		x++;								// Nchste X-Koordinate
		winkel--;							// Nchster Winkel
		if(winkel < 0)						// Illegal bzw. Unterlauf?
			winkel += ANZ_WINKEL;			// Korrigieren!
	}	// while

	// Die Textur-Offscreen-Bitmap wieder verschiebbar, da diese nicht mehr verwendet wird:
	UnlockPixels(GetGWorldPixMap(texture_gworld));		// Jetzt wieder verschiebbar (Mac-spezifisch)

	// Zu Debug-Zwecken die Koordinaten ausgeben:
	// Schriftausgabe vorbereiten:
	TextFont(applFont);		// =1 Standard-Anwendungs-Font
	TextSize(0);			// Systemfont-Gre
	TextFace(0);			// Normale Ausgabe, d.h. nicht fett o..
	TextMode(srcCopy);		// Replace-Mode beim Malen
	sprintf(str,"X:%4d Y:%4d W:%4d",plx,ply,plw);
	MoveTo(10,15);					// Dahin ausgeben.
	DrawText(str,0,strlen(str));	// Text zeichnen.
}

//
// Routine verzweigt je nach Flagge "my_3d_flag" auf Karten- oder
// 3D-Modus und leitet Zeichenoperationen in eine Offscreen-Bitmap um.
// (Mac-spezifisch)
//

void do_raycasting(WindowPtr window, int plx, int ply, int plw)
{
	Rect			wrect;							// Rechteck des Fensters
	CGrafPtr		currPort;						// Port merken
	GDHandle		currDev;						// Device merken
	PixMapHandle	pixmap;							// Verweis auf Offscreen-Bitmap

	// Zeichenoperationen nun in die Offscreen-Bitmap umlenken (Mac-spezifisch):
	GetGWorld(&currPort,&currDev);					// Alte Gworld merken
	LockPixels(GetGWorldPixMap(frame_buffer));		// Offscreen-Bitmap wird nun benutzt und ist daher gesperrt!
	SetGWorld(frame_buffer,nil);					// Bitte nun dahin zeichnen.

	// Offscreen-Bitmap lschen (Mac-spezifisch):
	SetRect(&wrect,0,0,XAUF-1,YAUF-1);				// In diesem Bereich
	EraseRect(&wrect);								// alles lschen.

	// 3D- oder Kartensicht?
	if(my3d_flag)
		do_3d_raycasting(window,plx,ply,plw);	// 3D-Sicht
	else
		do_map_raycasting(window,plx,ply,plw);	// Karten-Sicht

	// Den Schirm wieder aktivieren (Mac-spezifisch):
	SetGWorld(currPort,currDev);

	// Viewport auf eigenes Fenster setzen (Mac-spezifisch):
	SetPort(window);

	// Daten aus dem Frame-Buffer ins Fenster malen (Mac-spezifisch):
	pixmap = GetGWorldPixMap(frame_buffer);			// Zeiger auf die Bitmap holen
	CopyBits((BitMapPtr)*pixmap,&qd.thePort->portBits,&wrect,&wrect,srcCopy,0L);

	// Frame-Buffer ist nun wieder verschiebbar (Mac-spezifisch):
	UnlockPixels(GetGWorldPixMap(frame_buffer));
}

//
// Komplett-Redraw mit dem aktuellen Frame-Buffer:
// (Mac-spezifisch)
//

void do_complete_redraw(WindowPtr window)
{
	Rect			wrect;							// Rechteck des Fensters
	PixMapHandle	pixmap;							// Verweis auf Offscreen-Bitmap

	// Viewport auf eigenes Fenster setzen (Mac-spezifisch):
	SetPort(window);

	// Daten aus dem Frame-Buffer ins Fenster malen (Mac-spezifisch):
	SetRect(&wrect,0,0,XAUF-1,YAUF-1);				// So gro ist das Fenster und der Frame-Buffer
	pixmap = GetGWorldPixMap(frame_buffer);			// Zeiger auf die Bitmap holen
	LockPixels(pixmap);								// Framebuffer nun gelockt, da in Verwendung
	CopyBits((BitMapPtr)*pixmap,&qd.thePort->portBits,&wrect,&wrect,srcCopy,0L);	
	UnlockPixels(GetGWorldPixMap(frame_buffer));	// Framebuffer nun wieder verschiebbar
}

//
// Render.c++
//
