// SEARCH.CC
// Mustersuche in Zeichenfolgen mit folgenden Algorithmen:
// Brute Force, Boyer-Moore, Quicksearch, Turbosearch
//
// Michael Tamm / c't 8/97

static int shift[256]; // die Shifttabelle fr BM, QS und TS

/////////////////
// Brute Force //
/////////////////

int bf_search(char*text,char*p,int m) {
  register int i;
  register char*t=text;

  for(;;) {
    i=m-1;
    while(t[i]==p[i]) {
      if(--i<0) return t-text; // Muster gefunden
    }
    ++t; // nchste Stelle im Text untersuchen
  }
}

/////////////////
// Boyer-Moore //
/////////////////

void bm_init(char *p,int m) {
  int i;

  for(i=0;i<256;i++)
    shift[i]=m;
  for(i=1;i<m;i++)
    shift[(unsigned char)p[i-1]]=m-i;
}

int bm_search(char*t,char*p,int m) {
  register int i,j,k;

  for(i=j=m-1;j>=0;i--,j--)
    while(t[i]!=p[j]) {
      k=shift[(unsigned char)t[i]];
      i+=(m-j>k)?m-j:k;
      j=m-1;
    }
  return i+1;
}

/////////////////
// Quicksearch //
/////////////////

void qs_init(char*p,int m) {
  int i;

  for(i=0;i<256;i++)
    shift[i]=m+1; // Default-Sprungweite
  for(i=0;i<m;i++)
    shift[(unsigned char)p[i]]=m-i; // weniger fr Zeichen des Musters
}

int qs_search(char*text,char*p,int m) {
  register int i;
  register char*t=text;

  for(;;) {
    i=m-1;
    while(t[i]==p[i]) {
      if(--i<0) return t-text; // Muster gefunden
    }
    t+=shift[(unsigned char)t[m]];
  }
}

//////////////
// T-Search //
//////////////

static int look_at; // "Verschiebung in Gedanken"
                    // nach einem erfolgreichen Vergleich

void ts_init(char*p,int m) {
  int i;

  for(i=0;i<256;i++)    // Initialisierung wie bei Quicksearch
    shift[i]=m+1;
  for(i=0;i<m;i++)
    shift[(unsigned char)p[i]]=m-i;

  look_at=0; // kleinste "Verschiebung in Gedanken", wenn p[m-2]==p[m-1]
  while(look_at<m-1) {
    if(p[m-1]==p[m-(look_at+2)]) break; //letztes Zeichen nochmal im Muster
    ++look_at;
  }
}

int ts_search(char*text,char*p,int m) {
  register int i;
  register char*t=text;
  for(;;) {
    i=m-1;
    while(t[i]!=p[i]) { // Suche wie Quicksearch
      t+=shift[(unsigned char)t[m]];
    }
    // erster Vergleich war erfolgreich -> restliche Zeichen vergleichen
    --i;
    while(t[i]==p[i]) {
      if(--i<0) return t-text; // Muster gefunden
    }
    // Muster nicht gefunden -> Sprung um maximal 2*m
    t+=look_at+shift[(unsigned char)t[m+look_at]];
  }
}

#include <iostream.h>
#include <iomanip.h>
#include <io.h>
#include <sys\types.h>
#include <sys\stat.h>
#include <fcntl.h>
#ifdef __GNUG__
#include <unistd.h>
#endif
#include <string.h>
#include <time.h>
#include <stdlib.h>

void main(int argc,char* argv[]) {
  if(argc!=4)
    cout<<"\nAufruf: SEARCH datei muster durchlufe\n"
        <<"\nz.B.    SEARCH beispiel.txt muster 1000\n";
  else {
    int f=open(argv[1],O_RDONLY|O_BINARY);
    if(f==-1)
      cout<<"Datei "<< argv[1]<<" nicht gefunden.\n";
    else {
      struct stat file_buf;
      stat(argv[1],&file_buf);
      char*text=new char[file_buf.st_size+strlen(argv[2])+1];
      if(text==NULL)
        cout<<"Nicht gengend Speicher\n";
      else {
        read(f,text,file_buf.st_size);
        char*pattern=argv[2];
        int m=strlen(pattern);
        strcpy(text+file_buf.st_size,pattern);
        int l,loops=atoi(argv[3]);
        clock_t start,end;

        cout<<"BF: "<<hex<<bf_search(text,pattern,m)<<"h ..."<<flush;
        start=clock();
        for(l=0;l<loops;l++)
          bf_search(text,pattern,m);
        end=clock();
        cout<<" "<<dec<<(end-start)*1000/CLK_TCK<<" ms\n"<<flush;

        bm_init(pattern,m);
        cout<<"BM: "<<hex<<bm_search(text,pattern,m)<<"h ..."<<flush;
        start=clock();
        for(l=0;l<loops;l++)
          bm_search(text,pattern,m);
        end=clock();
        cout<<" "<<dec<<(end-start)*1000/CLK_TCK<<" ms\n"<<flush;

        qs_init(pattern,m);
        cout<<"QS: "<<hex<<qs_search(text,pattern,m)<<"h ..."<<flush;
        start=clock();
        for(l=0;l<loops;l++)
          qs_search(text,pattern,m);
        end=clock();
        cout<<" "<<dec<<(end-start)*1000/CLK_TCK<<" ms\n"<<flush;

        ts_init(pattern,m);
        cout<<"TS: "<<hex<<ts_search(text,pattern,m)<<"h ..."<<flush;
        start=clock();
        for(l=0;l<loops;l++)
          ts_search(text,pattern,m);
        end=clock();
        cout<<" "<<dec<<(end-start)*1000/CLK_TCK<<" ms\n"<<flush;

      }
      close(f);
    }
  }
}
