

// includes
#include "CTagDir.hpp"


// constants
const ULONG	LOGFLAGS_NONE		= 0x0L;
const ULONG	LOGFLAG_FILE		= 0x1L;
const ULONG	LOGFLAG_ERRORS		= 0x2L;
const ULONG	LOGFLAG_DELETED		= 0x4L;
const ULONG	LOGFLAG_KEPT		= 0x8L;

const ULONG MAX_DAYS_AGE		= 365L;
const ULONG SEARCH_GRANULARITY	= 10L;


// implementations
DWORD CTagDir::SearchThread (const CTagDir* tag)
{
	SYSTEMTIME	sysTime;
	CLogStream	logStream;

	do {
		// logfile for append
		logStream.open(tag->logFileName);
		// log time of start if any logging enabled
		if (LOGFLAGS_NONE != tag->logFlags) {
			GetLocalTime(&sysTime);
			logStream << "\r\n ** Start cleaning " << tag->fullPath << " - " << sysTime << "\r\n";
		}
		// search for files
		tag->SearchDir(tag->fullPath, tag->minSecondsAge, tag->logFlags, tag->nonNTFS, tag->preserveDirs, logStream);
		// log time of end if any logging enabled
		if (LOGFLAGS_NONE != tag->logFlags) {
			GetLocalTime(&sysTime);
			logStream << " ** Finished - " << sysTime << "\r\n\r\n";
		}
		// close logfile to flush results
		logStream.close();
		// wait 1/10 the time the files are kept or until the thread is signalled to stop
	} while (WAIT_TIMEOUT == WaitForSingleObject(tag->hEvent, (1000L / SEARCH_GRANULARITY) * tag->minSecondsAge));
	return 0L;
} /* ThreadStart */


VOID CTagDir::SearchDir (const CHAR* fullPath, const ULONG minSecondsAge, const ULONG logFlags, const BOOL nonNTFS, const BOOL preserveDirs, CLogStream& logStream) const
{
	BOOL				localEmpty;
	CHAR				actFullPath[MAX_PATH];
	CHAR				tmpStr[MAX_PATH];
	CHAR				localStr[MAX_PATH];
	unsigned __int64	actLLTime;
	FILETIME            actTime;
	HANDLE				searchHdl;
	HANDLE				localHdl;
	WIN32_FIND_DATA		fileInfo;
	WIN32_FIND_DATA		localInfo;

	// get actual system time (UTC time!) as a filetime structure and convert to 64 bit unsigned integer
	GetSystemTimeAsFileTime(&actTime);
	actLLTime = (((unsigned __int64) actTime.dwHighDateTime) << 32) + ((unsigned __int64) actTime.dwLowDateTime);
	// subtract the number of days from the received filetime
	actLLTime -= UInt32x32To64(minSecondsAge, 10000000L);
	// fill local variable actFullPath appending '\' to pathname if necessary
	lstrcpy(actFullPath, fullPath);
	if ('\\' != actFullPath[strlen(actFullPath)-1]) {
		lstrcat(actFullPath, "\\");
	}
	lstrcpy(tmpStr, actFullPath);
	lstrcat(tmpStr, "*");
	// find first entry in directory tree
	searchHdl = FindFirstFile(tmpStr, &fileInfo);
	if (INVALID_HANDLE_VALUE != searchHdl) {
		do {
			// exclude "." and ".." from beeing treated
			if ((0 != strcmp(".", fileInfo.cFileName)) && 
				(0 != strcmp("..", fileInfo.cFileName))) {
				// build full filename and store in tmpStr
				lstrcpy(tmpStr, actFullPath);
				lstrcat(tmpStr, fileInfo.cFileName);
				// test if file is a directory and scan it first
				if (FILE_ATTRIBUTE_DIRECTORY == 
					(fileInfo.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)) {
					SearchDir(tmpStr, minSecondsAge, logFlags, nonNTFS, preserveDirs, logStream);
				}
				// check age of file or directory depending on option /F (read access will be treated later)
				if (((!nonNTFS) && 
					 ((actLLTime > ((((unsigned __int64) fileInfo.ftLastWriteTime.dwHighDateTime) << 32) +
									  (unsigned __int64) fileInfo.ftLastWriteTime.dwLowDateTime)) &&
					  (actLLTime > ((((unsigned __int64) fileInfo.ftCreationTime.dwHighDateTime) << 32) +
									  (unsigned __int64) fileInfo.ftCreationTime.dwLowDateTime)))) ||
					((nonNTFS) &&
					 (actLLTime > (((unsigned __int64) fileInfo.ftCreationTime.dwHighDateTime << 32) + 
									(unsigned __int64) fileInfo.ftCreationTime.dwLowDateTime)))) {
					// if directory and directories are not preserved
					if ((!preserveDirs) &&
						(FILE_ATTRIBUTE_DIRECTORY == (fileInfo.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY))) {
						// check if directory is empty
						localEmpty = TRUE;
						lstrcpy(localStr, tmpStr);
						lstrcat(localStr, "\\*");
						// find first entry
						localHdl = FindFirstFile(localStr, &localInfo);
						if (INVALID_HANDLE_VALUE != localHdl) {
							// loop to see wether there are more entries than "." and ".."
							do {
								if ((0 != strcmp(".", localInfo.cFileName)) && 
									(0 != strcmp("..", localInfo.cFileName))) {
									localEmpty = FALSE;
								}
							} while ((FindNextFile(localHdl, &localInfo)) && (localEmpty));
							// close handle
							FindClose(localHdl);
						}
						// if directory is empty remove it
						if (localEmpty) {
							// change mode to writable
							if (SetFileAttributes(tmpStr, FILE_ATTRIBUTE_NORMAL | FILE_ATTRIBUTE_DIRECTORY)) {
								// try to remove directory without checking last read access
								// this is important if /I is given, because scanning a directory as done above is a read access
								if (RemoveDirectory(tmpStr)) {
									if (LOGFLAG_DELETED == (logFlags & LOGFLAG_DELETED)) {
										logStream << "  - deleted directory   : " << tmpStr << "\r\n";
									}
								}
								else {
									if (LOGFLAG_ERRORS == (logFlags & LOGFLAG_ERRORS)) {
										logStream << "  * error deleting dir. : " << tmpStr << "\r\n";
									}
								}
							}
							else {
								if (LOGFLAG_ERRORS == (logFlags & LOGFLAG_ERRORS)) {
									logStream << "  * error deleting dir. : " << tmpStr << "\r\n";
								}
							}
						}
						else {
							if (LOGFLAG_KEPT == (logFlags & LOGFLAG_KEPT)) {
								logStream << "  + keeping directory   : " << tmpStr << "\r\n";
							}
						}
					}
					// if file
					else if (FILE_ATTRIBUTE_DIRECTORY != (fileInfo.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)) {
						// check last read access on the file depending on option /F
						if (((!nonNTFS) &&
							(actLLTime > ((((unsigned __int64) fileInfo.ftLastAccessTime.dwHighDateTime) << 32) +
								(unsigned __int64) fileInfo.ftLastAccessTime.dwLowDateTime))) ||
							(nonNTFS)) {
							// change mode to writable
							if (SetFileAttributes(tmpStr,FILE_ATTRIBUTE_NORMAL)) {
								// remove file
								if (DeleteFile(tmpStr)) {
									if (LOGFLAG_DELETED == (logFlags & LOGFLAG_DELETED)) {
										logStream << "  - deleted file        : " << tmpStr << "\r\n";
									}
								}
								else {
									if (LOGFLAG_ERRORS == (logFlags & LOGFLAG_ERRORS)) {
										logStream << "  * error deleting file : " << tmpStr << "\r\n";
									}
								}
							}
							else {
								if (LOGFLAG_ERRORS == (logFlags & LOGFLAG_ERRORS)) {
									logStream << "  * error deleting file : " << tmpStr << "\r\n";
								}
							}
						}
						else {
							if (LOGFLAG_KEPT == (logFlags & LOGFLAG_KEPT)) {
								logStream << "  + keeping file        : " << tmpStr << "\r\n";
							}
						}
					}
				}
				else {
					if (LOGFLAG_KEPT == (logFlags & LOGFLAG_KEPT)) {
						if ((!preserveDirs) &&
							(FILE_ATTRIBUTE_DIRECTORY == (fileInfo.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY))) {
							logStream << "  + keeping directory   : " << tmpStr << "\r\n";
						}
						else if (FILE_ATTRIBUTE_DIRECTORY != (fileInfo.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)) {
							logStream << "  + keeping file        : " << tmpStr << "\r\n";
						}
					}
				}
			}
		} while (FindNextFile(searchHdl, &fileInfo));
		// close handle
		FindClose(searchHdl);
	}
} /* SearchDir */


CTagDir::CTagDir (const CHAR* cmdLine)
{
	CHAR	tmpStr[MAX_PATH];
	INT		i, j;
	
	// initialize synchronisation
	hEvent = CreateEvent(NULL, TRUE, FALSE, NULL);
	hThread = NULL;
	// parse commandline
	i = 0;
	// read leading blanks
	while ((' ' == cmdLine[i]) && ('\0' != cmdLine[i])) {
		i++;
	}
	// 1. argument is the tag directory
	j = 0;
	while ((' ' != cmdLine[i]) && ('\0' != cmdLine[i])) {
		fullPath[j] = cmdLine[i];
		j++;
		i++;
	}
	fullPath[j] = '\0';
	// read blanks
	while ((' ' == cmdLine[i]) && ('\0' != cmdLine[i])) {
		i++;
	}
	// 2. argument is the age
	j = 0;
	while ((' ' != cmdLine[i]) && ('\0' != cmdLine[i])) {
		tmpStr[j] = cmdLine[i];
		j++;
		i++;
	}
	tmpStr[j] = '\0';
	minSecondsAge = (ULONG) atol(tmpStr);
	if (1 > minSecondsAge) {
		minSecondsAge = 1;
	}
	if (MAX_DAYS_AGE < minSecondsAge) {
		minSecondsAge = MAX_DAYS_AGE;
	}
	minSecondsAge *= (24L * 3600L);
	// option defauls
	nonNTFS = FALSE;
	preserveDirs = FALSE;
	logFlags = LOGFLAGS_NONE;
	lstrcpy(logFileName, "CONOUT$");
	// process options
	while ('\0' != cmdLine[i]) {
		// read blanks
		while ((' ' == cmdLine[i]) && ('\0' != cmdLine[i])) {
			i++;
		}
		// get next arg
		j = 0;
		while ((' ' != cmdLine[i]) && ('\0' != cmdLine[i])) {
			tmpStr[j] = cmdLine[i];
			j++;
			i++;
		}
		tmpStr[j] = '\0';
		// check args
		if (('e' == tmpStr[1]) || ('E' == tmpStr[1])) {
			logFlags = (logFlags | LOGFLAG_ERRORS);
		}
		else if (('d' == tmpStr[1]) || ('D' == tmpStr[1])) {
			logFlags = (logFlags | LOGFLAG_DELETED);
		}
		else if (('k' == tmpStr[1]) || ('K' == tmpStr[1])) {
			logFlags = (logFlags | LOGFLAG_KEPT);
		}
		else if (('f' == tmpStr[1]) || ('F' == tmpStr[1])) {
			nonNTFS = TRUE;
		}
		else if (('p' == tmpStr[1]) || ('P' == tmpStr[1])) {
			preserveDirs = TRUE;
		}
		else if (('l' == tmpStr[1]) || ('L' == tmpStr[1])) {
			if (':' == tmpStr[2]) {
				j = 0;
				while (('\0' != tmpStr[3+j]) && (MAX_PATH > j)) {
					logFileName[j] = tmpStr[3+j];
					j++;
				}
				logFileName[j] = '\0';
				logFlags = (logFlags | LOGFLAG_FILE);
			}
		}
	}
} /* CTagDir */


CTagDir::~CTagDir (VOID)
{
	StopSearch();
	CloseHandle(hEvent);
} /* ~CTagDir */


BOOL CTagDir::StartSearch (const INT threadPriority)
{
	// only if not yet started
	if (NULL == hThread) {
		// set not signalled
		ResetEvent(hEvent);
		// create suspended thread, set the thread's priority and get started
		hThread = CreateThread(NULL, 0L, (LPTHREAD_START_ROUTINE) SearchThread, this, CREATE_SUSPENDED, &idThread);
		if (NULL != hThread) {
			SetThreadPriority(hThread, threadPriority);
			ResumeThread(hThread);
		}
	}
	return (NULL != hThread);
} /* StartSearch */


BOOL CTagDir::StopSearch (VOID)
{
	if (NULL != hThread) {
		// signal thread to stop
		SetEvent(hEvent);
		// await its end and close the handle to free the thread object
		WaitForSingleObject(hThread, INFINITE);
		CloseHandle(hThread);
		hThread = NULL;
	}
	return (NULL == hThread);
} /* StopSearch */


VOID CTagDir::Search (VOID)
{
	SYSTEMTIME	sysTime;
	CLogStream	logStream;

	if (NULL == hThread) {
		// logfile for append
		logStream.open(logFileName);
		// log time of start if any logging enabled
		if (LOGFLAGS_NONE != logFlags) {
			GetLocalTime(&sysTime);
			logStream << "\r\n ** Start cleaning " << fullPath << " - " << sysTime << "\r\n";
		}
		// imediately search for files in a single pass
		SearchDir(fullPath, minSecondsAge, logFlags, nonNTFS, preserveDirs, logStream);
		// log time of end if any logging enabled
		if (LOGFLAGS_NONE != logFlags) {
			GetLocalTime(&sysTime);
			logStream << " ** Finished - " << sysTime << "\r\n\r\n";
		}
		// close logfile to flush results
		logStream.close();
	}
} /* Search */

