unit DXSoundAS;  // 09-AUG-98 as (Arne Schpers)
{ Delphi-Komponenten fr DirectSound mit Notifications, baut
  auf den von Erik Unger umgesetzten Interface-Deklarationen
  fr DirectX auf.

  Pro Anwendung: *ein* IDirectSound-Objekt (globale Variable
  DirectSound, ber DirectSoundInitialize besetzt), beliebige
  Zahl von TDSoundBuffer-Komponenten.

Besonderheiten im Handling von TDSound[Stream]Buffer:
- Zweistufiges Create: erst die Delphi-Komponente anlegen, dann
  BufferObject ber LoadFromFile/LoadFromStream besetzen.
  Destroy ist einstufig.
- Eigenschaft BufferObject (= IDirectSoundBuffer), ber die man an
  die diversen Funktionen (von Stop bis SetVolume) herankommt.
  Auer bei Play (mit den Komplikationen wg. DSERR_BUFFERLOST)
  und Stop (wg. Verwalterei) habe ich keinen Sinn in einer
  doppelten Verkapselung gesehen.

Grenzen: Maximal 63 aktive Notifications. Gezhlt werden
ausschlielich spielende Puffer.
}

interface
uses
  Windows, SysUtils, Classes, ActiveX, Dialogs, Messages,
  MMSystem,    // TWaveFormatEx
  DSound,      // Erik Ungers Umsetzung fr DX5, DX6
  DXWaveUtils; // Minimalversion von WAVE.C aus dem DX-SDK

// statische Puffer funktionieren dann auch mit NT 4.0,
// Notifications gibt's erst ab DirectX 5 (NT 5.0)
{$DEFINE DX3}

type
  TDSoundBuffer = class(TComponent)
    private // Stream und PositionTemp fr statische Sounds,
      IOCB: TWaveIOCB;
    protected
      FPlaying: Boolean;
      FEventNo: Integer;  // von DSoundThread gesetzt
      FOnEvent: TNotifyEvent;  // Rckmeldung
      NotifyPositions: TList; // PDSBPositionNotify
      FOnBufferLost: TNotifyEvent;
      SoundNotify: IDirectSoundNotify;  // Setzen von Positionen
      procedure SetOnEvent(Value: TNotifyEvent);
      function SetNotifications: HResult;  // TList -> Array
      function GetBufDesc(Flags: Integer): TDSBufferDesc; virtual;
      procedure LoadEntireBuffer;  // WAV-Daten -> Puffer
      procedure SetPlaying(Value: Boolean);
    public
      BufferObject: IDirectSoundBuffer;  // SetVolume, GetCaps...
      constructor Create(AOwner: TComponent); override;
      destructor Destroy; override;
      // Positionsrckmeldungen
      property OnEvent: TNotifyEvent read FOnEvent write SetOnEvent;
      property EventNo: Integer read FEventNo;
      function AddNotification(Offset: LongInt): Integer;
      procedure DeleteNotification(Index: Integer);
      procedure ClearNotifications;
      // Play, Stop und Bufferlost
      function Play(Flags: Integer): HResult;
      procedure Stop;
      property OnBufferLost: TNotifyEvent read FOnBufferLost
                                          write FOnBufferLost;
      property Playing: Boolean read FPlaying;
    public       // Flags = DSBCAPS_...
      procedure LoadFromStream(S: TStream; Flags: Integer);
      procedure LoadFromFile(FName: TFileName;
         Flags: Integer); virtual; // override durch Streams
    end;

  TDSoundStreamBuffer = class(TDSoundBuffer)
    protected
      EOFCount: Integer;  // > 1: Datei komplett gespielt
      function GetBufDesc(Flags: Integer): TDSBufferDesc; override;
    public
      constructor Create(AOwner: TComponent); override;
      destructor Destroy; override;
    public
      // Erweiterungen gegenber TDSoundBuffer
      BufferTime: Integer;  // Spielzeit in msec
      procedure Rewind;
      procedure LoadFromFile(FName: TFileName; Flags: Integer); override;
      procedure LoadNextChunk;  // Pufferhlfte nachladen
    end;

// Initialisierung, Deinitialisierung (optional)
function DirectSoundInitialize(Owner: TComponent): Boolean;
function DirectSoundSetPrimaryFormat(Channels, Bits,
  SamplesPerSec: Integer): HResult;
procedure DirectSoundUninitialize;

// Global fr Anwendung und die restlichen Klassen dieser Unit;
// wird ber DirectSoundInitialize gesetzt
var DirectSound: IDirectSound;
// Rckmeldung fr mehr als 63 aktive Notifications (Debug)
var OnMaxEventsExceeded: procedure(Sender: TObject) of object;

implementation

type
  TDSoundThread = class;
  // Verwaltungsklasse um das DirectSound-Objekt herum
  TDSound = class(TComponent)
    public
     SoundBuffers: TList;    // alle TDSoundBuffer-Objekte
     Thread: TDSoundThread;  // ein einziger Thread
    protected
     // Update SoundBuffers beim Create/Destroy von Puffern
     procedure Notification(AComponent: TComponent;
       Operation: TOperation); override;
    public
     constructor Create(AOwner: TComponent); override;
     destructor Destroy; override;
     procedure RebuildEventList;  // siehe TDSoundThread
  end;

// Hintergrund-Thread von TDSound, wartet ber
// MsgWaitForMultipleObjects auf Positions-Rckmeldungen der
// Soundpuffer (d.h. liegt die meiste Zeit still). Die Reaktion
// auf Rckmeldungen luft im Kontext des Vordergrund-Threads!
  TDSoundThread = class(TThread)
    private
      EventIndex: Integer;  // signalisiertes Objekt (Index)
      procedure SynchronizedEvent;  // Rckmeldung
      procedure RebuildEventHandles; // Liste neu aufbauen
    protected
      RebuildListEvent: THandle;  // Rebuild-Ereignis-Objekt
      EventHandles: TList;  // Items = Ereignis-Objekthandles
      Originators: TList;  // Items = TDSoundBuffer
    public
      constructor Create(Owner: TDSound);
      destructor Destroy; override;
      procedure Execute; override;
      procedure Terminate;
  end;

// --- Das (einzige) Objekt der Klasse TDSound --------
var DelphiSoundObject: TDSound;

function DirectSoundInitialize(Owner: TComponent): Boolean;
begin
  DelphiSoundObject := TDSound.Create(Owner);
  Result := DirectSound <> nil;
end;

function DirectSoundSetPrimaryFormat(Channels, Bits,
  SamplesPerSec: Integer): HResult;
var BufDesc: TDSBufferDesc;  // Soundpuffer-Beschreibung
    WaveFmt: TWaveFormatEx;  // gewnschtes Audioformat
    PrimBuffer: IDirectSoundBuffer;
begin
  if DelphiSoundObject = nil then
    raise Exception.Create('Use DirectSoundInitialize first');
  // Primren Soundpuffer anlegen
  FillChar(BufDesc,SizeOf(BufDesc),0);
  with BufDesc do
  begin
    // das erwartet DirectX bei den meisten Strukturen
    dwSize := SizeOf(BufDesc);
    dwFlags := DSBCAPS_PRIMARYBUFFER;  // primrer Soundpuffer
    // dwBufferBytes und lpwfxFormat bleiben 0 bzw. nil
  end;
  Result := DirectSound.CreateSoundBuffer(BufDesc, PrimBuffer, nil);
  if FAILED(Result) then Exit;
  // Gewnschtes Format einsetzen
  FillChar(WaveFmt,SizeOf(WaveFmt),0);
  with WaveFmt do
  begin
    wFormatTag := WAVE_FORMAT_PCM; // was anderes geht eh nicht
    nChannels := Channels;
    wBitsPerSample := Bits; nSamplesPerSec := SamplesPerSec;
    nBlockAlign := wBitsPerSample div 8 * nChannels;
    nAvgBytesPerSec := nSamplesPerSec * nBlockAlign;
  end;
  Result := PrimBuffer.SetFormat(WaveFmt);
end;

procedure DirectSoundUninitialize;
begin
  DelphiSoundObject.Free; DelphiSoundObject := nil;
end;

constructor TDSound.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  SoundBuffers := TList.Create;
  Thread := TDSoundThread.Create(Self);
  DirectSoundCreate(nil, DirectSound, nil);
end;

// rumt alle Soundbuffer und DirectSound ab
destructor TDSound.Destroy;
var x: Integer;
begin
  Thread.Terminate;  // Hintergrund-Thread beenden
  // Abwrts: Puffer werden per opRemove aus der Liste geworfen
  for x := SoundBuffers.Count-1 downto 0 do
    TDSoundBuffer(SoundBuffers[x]).Free;
  SoundBuffers.Free;
  DirectSound := nil;  // IDirectSound-Schnittstelle raus
  inherited Destroy;
end;

// ber diese Methode kriegt das DSound-Objekt mit, wenn jemand
// einen TDSoundBuffer anlegt oder entfernt.
procedure TDSound.Notification(AComponent: TComponent;
   Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (AComponent is TDSound) and (AComponent <> Self) then
    raise Exception.Create('Nur eine DSound-Komponente mglich!');
  if AComponent is TDSoundBuffer then
    if Operation = opInsert then SoundBuffers.Add(AComponent)
      else SoundBuffers.Remove(AComponent);
end;

// Eventliste des Hintergrund-Threads neu aufbauen
// XRefs: TDSoundBuffer.AddNotification, Destroy
procedure TDSound.RebuildEventList;
begin
  SetEvent(Thread.RebuildListEvent);
end;

// ------------- DSoundThread -----------------
constructor TDSoundThread.Create(Owner: TDSound);
begin
  inherited Create(False);
  RebuildListEvent := CreateEvent(nil, False, False, nil);
  EventHandles := TList.Create; Originators := TList.Create;
  FreeOnTerminate := True;
  RebuildEventHandles;
end;

procedure TDSoundThread.Terminate;
begin
  inherited Terminate;
  SetEvent(RebuildListEvent);  // sonst wartet der ewig...
end;

destructor TDSoundThread.Destroy;
begin
  CloseHandle(RebuildListEvent);
  EventHandles.Free; Originators.Free;
  inherited Destroy;
end;

// Neuaufbau der Liste (exakter: des Arrays) mit den Handles der
// Ereignis-Objekte. Luft grundstzlich im Kontext des Vorder-
// grund-Threads; Element 0 ist RebuildListEvent, alle anderen
// Handles stammen aus den SoundBuffer-Objekten. SoundBuffer-
// Objekte mit OnEvent = nil oder FPlaying = False bleiben
// unbercksichtigt.
procedure TDSoundThread.RebuildEventHandles;
var x,y: Integer;
begin
  EventHandles.Clear; Originators.Clear;
  // Items[0] in beiden Listen ist Self
  EventHandles.Add(Pointer(RebuildListEvent));
  Originators.Add(Self);
  if DelphiSoundObject <> nil then  // nicht bei Create...
    with DelphiSoundObject do
      for x := 0 to SoundBuffers.Count-1 do
        with TDSoundBuffer(SoundBuffers[x]) do
          if Assigned(FOnEvent) and Playing then
            for y := 0 to NotifyPositions.Count-1 do
              with PDSBPositionNotify(NotifyPositions[y])^ do
              begin
                // Bei mehr als 64 Events liefert MsgWaitFor...
                // grundstzlich "Fehler" (-1)
                if EventHandles.Count >= MAXIMUM_WAIT_OBJECTS-1
                then begin
                  if Assigned(OnMaxEventsExceeded) then
                     OnMaxEventsExceeded(Self);
                  Exit;
                end;
                EventHandles.Add(Pointer(hEventNotify));
                Originators.Add(SoundBuffers[x]);
              end;
end;

// Wartet auf eine Rckmeldung fr eines der Ereignis-Objekte.
// Ereignis-Objekt 0 ist RebuildListEvent
procedure TDSoundThread.Execute;
var EventCount: Integer; Msg: TMsg;
begin
  while not Terminated do
  begin
    EventCount := EventHandles.Count;
    EventIndex := MsgWaitForMultipleObjects(EventCount,
      EventHandles.List^[0], False, INFINITE, QS_ALLINPUT);
    Dec(EventIndex,WAIT_OBJECT_0);  // Normalisierung
    if EventIndex >= EventCount then
    begin // Botschaft von Windows -> Regulre Verarbeitung
      while (PeekMessage(Msg, 0, 0,0, PM_REMOVE)) do
        if Msg.Message = WM_QUIT then Terminate
          else
          begin
            TranslateMessage(Msg); DispatchMessage(Msg);
          end;
    end else // wurde eines der Ereignis-Objekte signalisiert
      if not Terminated then
         // 0 = explizit, -1 = Ereignis-Objekt gelscht
       if EventIndex <= 0 then Synchronize(RebuildEventHandles)
         else Synchronize(SynchronizedEvent);
  end;
end;

// Wenn sich SoundBuffer x mit Notification y zurckmeldet:
// TDSoundBuffer[x].FEventNo := y; TDSoundBuffer[x].OnEvent.
// Luft im Kontext des Vordergrund-Threads (Synchronize).
procedure TDSoundThread.SynchronizedEvent;
var SoundBuffer: TDSoundBuffer; x: Integer;
begin
  SoundBuffer := Originators[EventIndex];
  if not Assigned(SoundBuffer.FOnEvent) then Exit;
  // das wievielte Ereignis fr diesen Soundbuffer ist das?
  x := EventIndex;
  while (x >= 0) do
    if Originators[x] <> SoundBuffer then Break
     else Dec(x);
  SoundBuffer.FEventNo := EventIndex-x-1;
  if not SoundBuffer.Playing then  // war's ein Stop
     RebuildEventHandles;
  SoundBuffer.OnEvent(SoundBuffer);
end;

// ------------ TDSoundBuffer -------------------
constructor TDSoundBuffer.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  NotifyPositions := TList.Create;  // Positionen & Handles
  FillChar(IOCB,SizeOf(IOCB),0);
end;

destructor TDSoundBuffer.Destroy;
var x: Integer; P: PDSBPositionNotify;
begin
  FPlaying := False; // nicht mehr bei den Events bercksichtigen
  for x := 0 to NotifyPositions.Count-1 do
  begin
    P := NotifyPositions[x];
    // auf diese Freigabe reagiert MsgWaitForMultipleObjects
    // mit dem Ergebnis -1 ("Failure") -> RebuildList,
    // aber ohne diesen SoundBuffer, weil OnEvent = nil
    CloseHandle(P^.hEventNotify);
    Dispose(P);
  end;
  NotifyPositions.Free;
  if BufferObject <> nil then BufferObject.Stop;
  BufferObject := nil;  // IDirectSoundBuffer-Schnittstelle raus
  inherited Destroy;
end;

// Wenn OnEvent vorher nicht gesetzt war, jetzt gesetzt wird,
// und Positionen eingetragen sind, dann Rebuild der Liste
procedure TDSoundBuffer.SetOnEvent(Value: TNotifyEvent);
var NeedRebuild: Boolean;
begin
  NeedRebuild := Playing and Assigned(Value) and
    not Assigned(FOnEvent) and (NotifyPositions.Count > 0);
  FOnEvent := Value;
  if NeedRebuild then DelphiSoundObject.RebuildEventList;
end;

// SoundNotify will Positionen und Handles als Array sehen
function TDSoundBuffer.SetNotifications: HResult;
type TPosArray = Array[0..99] of TDSBPositionNotify;
var x, PNSize: Integer; PN: ^TPosArray;
begin
  with NotifyPositions do
  begin
    PNSize := Count*SizeOf(TDSBPositionNotify);
    GetMem(PN,PNSize);
    for x := 0 to Count-1 do
      PN^[x] := PDSBPositionNotify(Items[x])^;
    Result := SoundNotify.SetNotificationPositions(Count, PN[0]);
    FreeMem(PN,PNSize);
  end;
end;

// Kleines technisches Hindernis: Wenn der Puffer abgespielt
// wird, mu er zum Neusetzen von Benachrichtigungspositionen
// kurz angehalten werden. Dabei sollte sinnvollerweise aber
// keine Notification herauskommen.
function TDSoundBuffer.AddNotification(Offset: Integer): Integer;
var NewNotify: PDSBPositionNotify; Res: HResult;
    CurPlaying: Boolean; CurPlayState: Integer;  // Resume
begin
  Result := -1;
  if BufferObject = nil then raise Exception.Create(
           'TDSoundBuffer.AddNotification: BufferObject = nil');
  if SoundNotify = nil then
  begin
    Res := BufferObject.QueryInterface(IID_IDirectSoundNotify,
                                   SoundNotify);
    if FAILED(Res) then
    begin
      ShowMessage(ErrorString(Res)); Exit;
    end;
  end;
  // Neuen Eintrag in die Liste. Result := Listenindex
  New(NewNotify);
  Result := NotifyPositions.Add(NewNotify);
  NewNotify^.dwOffset := Offset;
  NewNotify^.hEventNotify := CreateEvent(nil,False,False,nil);
  // Abspielen ggf. anhalten, aber ohne Notification
  CurPlaying := FPlaying; SetPlaying(False);  // Rebuild
  if CurPlaying then
  begin
    BufferObject.GetStatus(CurPlayState);
    BufferObject.Stop;
  end
    else CurPlayState := 0; // damit der Compiler nicht mault

  Res := SetNotifications;  // Umstapeln: TList -> Array
  if FAILED(Res) then
    ShowMessage('SetNotificationPositions: '+ErrorString(Res));

  // Abspielen wieder anwerfen, falls erforderlich
  if CurPlaying then
  begin
    SetPlaying(True);  // RebuildEventList, wenn OnEvent <> nil
    if CurPlayState and DSBSTATUS_LOOPING <> 0
     then CurPlayState := DSBPLAY_LOOPING else CurPlayState := 0;
    BufferObject.Play(0,0,CurPlayState);  // wieder anwerfen
  end;
end;

// Dieselbe technische Problemstellung wie bei AddNotification
procedure TDSoundBuffer.DeleteNotification(Index: Integer);
var OldNotify: PDSBPositionNotify; Res: HResult;
    CurPlaying: Boolean; CurPlayState: Integer;  // Resume
begin
  if BufferObject = nil then raise Exception.Create(
       'TDSoundBuffer.DeleteNotification: BufferObject = nil');
  if (Index < 0) or (Index > NotifyPositions.Count-1) then Exit;
  // Spielt der Puffer?
  CurPlaying := FPlaying; FPlaying := False;
  // Event-Handle erst aus der Liste raus, dann lschen
  OldNotify := NotifyPositions[Index];
  NotifyPositions.Delete(Index);
  // Freigabe des Ereignisobjekts. Wenn der Puffer im Moment
  // abgespielt wird -> RebuildList ohne den Puffer (vgl. Destroy),
  // ansonsten steht dieser Handle eh nicht in der Liste
  CloseHandle(OldNotify^.hEventNotify); Dispose(OldNotify);
  // Wenn der Puffer spielt: Anhalten ohne Notify
  if CurPlaying then
  begin
    BufferObject.GetStatus(CurPlayState);
    BufferObject.Stop;  // FPlaying ist im Moment False
  end
    else CurPlayState := 0; // damit der Compiler nicht mault
  Res := SetNotifications;  // Umstapeln TList -> Array
  if FAILED(Res) then
    ShowMessage('SetNotificationPositions: '+ErrorString(Res));

  // Abspielen wieder anwerfen, falls erforderlich
  if CurPlaying then
  begin
    SetPlaying(True);  // -> RebuildList, wenn OnEvent <> nil
    if CurPlayState and DSBSTATUS_LOOPING <> 0
     then CurPlayState := DSBPLAY_LOOPING else CurPlayState := 0;
    BufferObject.Play(0,0,CurPlayState);  // wieder anwerfen
  end;
end;

function TDSoundBuffer.Play(Flags: Integer): HResult;
begin
  if BufferObject = nil then raise Exception.Create(
    'TDSoundBuffer.Play: BufferObject = nil');
  Result := BufferObject.Play(0,0,Flags);
  if Result = DSERR_BUFFERLOST then
  begin
    if SUCCEEDED(BufferObject.Restore)
      then Result := BufferObject.Play(0,0,Flags)
     else if Assigned(FOnBufferLost) then
     begin
       OnBufferLost(Self);  // -> Recreate/Reload
       Result := BufferObject.Play(0,0,Flags);
     end;
  end;
  SetPlaying(Result = DS_OK);
end;

procedure TDSoundBuffer.SetPlaying(Value: Boolean);
begin
  FPlaying := Value;
  if Assigned(FOnEvent) then DelphiSoundObject.RebuildEventList;
end;

procedure TDSoundBuffer.Stop;
begin
  if not Playing then Exit;
  if BufferObject = nil then raise Exception.Create(
    'TDSoundBuffer.Stop: BufferObject = nil');

  // Unerfreulich kompliziert: Wenn ein STOP-Event gesetzt ist,
  // mu der noch durchkommen. Da die Reaktion genauso wie
  // die Methode Stop im Kontext des Vordergrund-Threads
  // luft, rumt SetPlaying die Events ab, bevor der Hinter-
  // grund-Thread sie auswerten kann. Deshalb:
  with NotifyPositions do
    if not Assigned(FOnEvent) or (Count = 0) or
     (PDSBPositionNotify(Items[Count-1])^.dwOffset <>
       DSBPN_OFFSETSTOP) then SetPlaying(False)  // kein STOP
    else FPlaying := False;  // RebuildList nach dem Stop-Event
  BufferObject.Stop;
end;

// XRef: LoadStatic, LoadStreaming
procedure TDSoundBuffer.ClearNotifications;
var CurOnEvent: TNotifyEvent;
begin
  if BufferObject = nil then raise Exception.Create(
         'TDSoundBuffer.ClearNotification: BufferObject = nil');
  CurOnEvent := FOnEvent; OnEvent := nil;
  while NotifyPositions.Count > 0 do DeleteNotification(0);
  OnEvent := CurOnEvent;
end;

// XRef: LoadStatic, LoadStreaming
procedure CheckResult(Action: String; Res: HResult);
begin
  if FAILED(Res) then raise Exception.Create(
    Format('%s: %s',[Action,ErrorString(Res)]));
end;

// Fr statische Puffer: dwBufferBytes = Gre der WAV-Datei
// Stream-Puffer ersetzen diese Methode
function TDSoundBuffer.GetBufDesc(Flags: Integer): TDSBufferDesc;
begin
  FillChar(Result,SizeOf(Result),0);
  with Result do
  begin
    dwSize := SizeOf(Result); // DirectX-Doublecheck
    dwBufferBytes := IOCB.dwDataBytes;  // Gre = Dateigre
    dwFlags := DSBCAPS_STATIC or Flags;  // statischer Puffer
    lpwfxFormat := @IOCB.WaveFormat;  // Format = Dateiformat
  end;
end;

// XRef: LoadFromStream, TDSoundStreamBuffer.Rewind
procedure TDSoundBuffer.LoadEntireBuffer;
var Res: HResult;
    AudioSize, ASize2: Integer; AudioData, AData2: Pointer;
{$IFDEF DX3} BufCaps: TDSBCaps; {$ENDIF}
begin
  // Gesamten Puffer sperren - Schreibaktion folgt
{$IFDEF DX3}
  BufCaps.dwSize := SizeOf(BufCaps);
  Res := BufferObject.GetCaps(BufCaps);
  CheckResult('TDSoundBuffer.LoadFromStream: GetCaps',Res);
  Res := BufferObject.Lock(0,BufCaps.dwBufferBytes,
    AudioData, AudioSize, AData2, ASize2, 0);
{$ELSE} // ab DX5 geht's mit einem Flag (ohne Grenabfrage)
  Res := BufferObject.Lock(0,0,AudioData, AudioSize,
    AData2, ASize2, DSBLOCK_ENTIREBUFFER);
{$ENDIF}
  CheckResult('TDSoundBuffer.LoadFromStream: Lock', Res);
  // WAV-Daten komplett einlesen
  if WaveReadFile(IOCB,AudioSize,AudioData) < AudioSize then
      raise Exception.Create('EOF beim Lesen der Audiodaten');
  // Puffer entsperren
  BufferObject.Unlock(AudioData, AudioSize, nil, 0);
end;

// Flags: DSBCAPS_... (STATIC implizit, POSITIONNOTIFY fr
// Rckmeldungen, LOCHARDWARE/LOCSOFTWARE etc.
// Stream kann nach dem Aufruf freigegeben werden
procedure TDSoundBuffer.LoadFromStream(S: TStream; Flags: Integer);
var BufDesc: TDSBufferDesc; Res: HResult;
begin
  WaveCloseReadFile(IOCB);  // just in case...
  if not WaveOpenStream(S,IOCB) then Exit;
  // Pufferformat & Gre festlegen, Puffer anlegen
  BufDesc := GetBufDesc(Flags);
  // Setzt im Erfolgsfall BufferObject (=IDirectSoundBuffer) neu
  Res := DirectSound.CreateSoundBuffer(BufDesc,BufferObject, nil);
  CheckResult('Pufferobjekt anlegen', Res);
  ClearNotifications;  // eventuell gesetzte Notifications raus
  LoadEntireBuffer;  // gesamten Puffer mit Daten laden
end;

procedure TDSoundBuffer.LoadFromFile(FName: TFileName;
  Flags: Integer);
var S: TFileStream;
begin
  S := TFileStream.Create(FName,fmOpenRead or fmShareDenyNone);
  LoadFromStream(S,Flags);
  S.Free;
end;

// --------- TDSoundStreamBuffer -----------
constructor TDSoundStreamBuffer.Create(AOwner: TComponent);
begin  // Standardvorgabe: 2 Sekunden Spielzeit
  inherited Create(AOwner); BufferTime := 2000;
end;

destructor TDSoundStreamBuffer.Destroy;
begin  // Stream ist hier noch offen
  WaveCloseReadFile(IOCB); inherited Destroy;
end;

procedure TDSoundStreamBuffer.Rewind;
begin  //  la MediaPlayer: zurck zum Anfang
  if BufferObject = nil then Exit;
  Stop;
  WaveStartDataRead(IOCB);  // zurck zum Start der Audiodaten
  LoadEntireBuffer;
  EOFCount := 0;  // siehe LoadNextChunk
end;

// Fr Stream-Puffer: dwBufferBytes = Spielzeit*WAV-Format
function TDSoundStreamBuffer.GetBufDesc(Flags: Integer): TDSBufferDesc;
begin
  EOFCount := 0;  // siehe LoadNextChunk
  FillChar(Result,SizeOf(Result),0);
  // Minimalgre - weniger macht wirklich keinen Sinn
  if BufferTime < 500 then BufferTime := 500;
  with Result do
  begin
    dwSize := SizeOf(Result);
    // Puffergre ber Dateiformat und Spielzeit
    dwBufferBytes := IOCB.WaveFormat.nAvgBytesPerSec *
                       BufferTime div 1000;
    if dwBufferBytes > IOCB.dwDataBytes then
    begin
      Inc(EOFCount);
      dwBufferBytes := IOCB.dwDataBytes;
    end;
    dwFlags := Flags;  // kein DSBCAPS_STATIC
    lpwfxFormat := @IOCB.WaveFormat;  // Format = Dateiformat
  end;
end;

procedure TDSoundStreamBuffer.LoadFromFile(FName: TFileName;
  Flags: Integer);
var S: TFileStream;
begin
  S := TFileStream.Create(FName,fmOpenRead or fmShareDenyNone);
  LoadFromStream(S,Flags);
  // Stream erst beim nchsten LoadFrom... bzw. Destroy abgerumt
end;

// Nachladen des Puffers, sollte als Reaktion auf Positions-
// rckmeldungen "Pufferanfang" und "Puffermitte" aufgerufen
// werden (typischerweise wohl Event[0] und Event[1])
procedure TDSoundStreamBuffer.LoadNextChunk;
var Caps: TDSBCaps; WriteBytes, BytesWritten: Integer;
var PlayPos, WritePos: Integer; Res: HResult;
    AudioSize, ASize2: Integer; AudioData, AData2: Pointer;
    EOF: Boolean; SilenceVal: Byte;
begin
  with Caps do
  begin
    dwSize := SizeOf(Caps); dwBufferBytes := -1;
    if BufferObject <> nil then BufferObject.GetCaps(Caps);
    if dwBufferBytes = -1 then raise Exception.Create(
      'LoadNextChunk: BufferObject = nil or not playing');
    WriteBytes := dwBufferBytes div 2;
  end;
  BufferObject.GetCurrentPosition(PlayPos, WritePos);
  if PlayPos > Caps.dwBufferBytes div 2
    then WritePos := 0  // 2. Hlfte wird gespielt, 1. schreiben
    else WritePos := Caps.dwBufferBytes div 2; // 1. Hlfte ...
  EOF := False;
  Res := BufferObject.Lock(WritePos,WriteBytes,AudioData,AudioSize,
    AData2, ASize2, 0);
  CheckResult('TDSoundStreamBuffer.LoadNextChunk',Res);
  if IOCB.WaveFormat.wBitsPerSample = 16 then SilenceVal := 0
    else SilenceVal := 128;  // 8 Bit: unsigned Bytes!
  BytesWritten := WaveReadFile(IOCB,AudioSize,AudioData);
  if BytesWritten < AudioSize then
  begin  // Rest mit Stille auffllen
    FillChar(PChar(AudioData)[BytesWritten],
                   AudioSize-BytesWritten,SilenceVal);
    EOF := True;
  end;
  if (ASize2 <> 0) and not EOF then
  begin
    BytesWritten := WaveReadFile(IOCB,ASize2,AData2);
    if BytesWritten < ASize2 then
    begin
      FillChar(PChar(AData2)[BytesWritten],
                     ASize2-BytesWritten,SilenceVal);
      EOF := True;
    end;
  end;
  Res := BufferObject.Unlock(AudioData, AudioSize, AData2, ASize2);
  CheckResult('TDSoundStreamBuffer.LoadNextChunk',Res);
  if EOF then Inc(EOFCount);
  if EOFCount > 2 then  // beide Pufferhlften mit 0 gefllt,
     Stop;  // mindestens eine Hlfte bereits mit 0 abgespielt
end;

end.
