unit DDUtil;

{******************************************************************************
    DDUtil Unit

    This is a near direct translation of the ddutil2.cpp unit that ships with
    the Microsoft DirectX SDK, located in the sdk/samples/misc directory. This
    unit provides several functions for loading bitmaps and palettes.

    The major differences between this file and the original Microsoft file
    is that the original provided the means to load bitmaps from either
    files or resources.  This unit only loads bitmaps and palettes from files.
    However, by using the TBitmap object, the code has been dramatically
    simplified, and providing a means to load bitmaps and palettes from
    resources would require simply replacing the LoadFromFile method calls
    with LoadFromStream.

 ******************************************************************************}

interface

uses
  Windows, Graphics, Classes, DDraw;

  function DDLoadPalette(DDraw: IDirectDraw2; PalBitmap: string): IDirectDrawPalette;
  function DDLoadBitmap(DDraw: IDirectDraw2; BitmapName: string): IDirectDrawSurface3;
  function CopyBitmap(DSurface: IDirectDrawSurface3; Bitmap: TBitmap; OrgX, OrgY, Width, Height: Integer): HResult;
  function DDReLoadBitmap(DSurface: IDirectDrawSurface3; BitmapName: string): HResult;
  function DDColorMatch(DSurface: IDirectDrawSurface3; Color: TColor): DWORD;
  function DDSetColorKey(DSurface: IDirectDrawSurface3; Color: TColor): HResult;

implementation

uses
  DXTools, SysUtils;

{ -- loads a palette from the specified bitmap file, returning an IDirectDrawPalette interface -- }
function DDLoadPalette(DDraw: IDirectDraw2; PalBitmap: string): IDirectDrawPalette;
var
  Palette: TColorTable;
  TempBitmap: TBitmap;
begin
  {create the temporary bitmap}
  TempBitmap := TBitmap.Create;

  {initialize the result}
  Result := nil;

  {indicate an error if no direct draw interface was supplied}
  if DDraw = nil then
    raise Exception.Create('No direct draw object specified for creating the palette');

  try
    {load the bitmap, and extract its palette}
    TempBitmap.LoadFromFile(PalBitmap);
    GetPaletteEntries(TempBitmap.Palette, 0, 256, Palette);

    {create the palette object}
    DXCheck( DDraw.CreatePalette(DDPCAPS_8BIT or DDPCAPS_ALLOW256, @Palette, Result, nil) );
  finally
    TempBitmap.Free;
  end;
end;

{ -- create a directdraw surface object and initialize it from a bitmap -- }
function DDLoadBitmap(DDraw: IDirectDraw2; BitmapName: string): IDirectDrawSurface3;
var
  TempBitmap: TBitmap;
  TempSurface: IDirectDrawSurface;
  SurfaceDesc: TDDSurfaceDesc;
begin
  {create the temporary bitmap}
  TempBitmap := TBitmap.Create;

  {initialize the result}
  Result := nil;

  try
    {load the bitmap}
    TempBitmap.LoadFromFile(BitmapName);

    {initialize the attributes for the surface. note: add the DDSCAPS_VIDEOMEMORY flag
     to the SurfaceDesc.ddsCaps.dwCaps property to force the surface to be created in
     video memory. this is dependent upon the amount of free video memory available}
    FillChar(SurfaceDesc, SizeOf(TDDSurfaceDesc), 0);
    SurfaceDesc.dwSize   := SizeOf(TDDSurfaceDesc);
    SurfaceDesc.dwFlags  := DDSD_CAPS or DDSD_HEIGHT or DDSD_WIDTH;
    SurfaceDesc.dwWidth  := TempBitmap.Width;
    SurfaceDesc.dwHeight := TempBitmap.Height;
    SurfaceDesc.ddsCaps.dwCaps := DDSCAPS_OFFSCREENPLAIN;

    {create a temporary direct draw surface object. this will be used to create
     the desired direct draw surface 3 object}
    DXCheck( DDraw.CreateSurface(SurfaceDesc, TempSurface, nil) );

    try
      {we can only get a direct draw surface 3 interface through the QueryInterface
       method of the direct draw surface object}
      DXCheck( TempSurface.QueryInterface(IID_IDirectDrawSurface3, Result) );
    finally
      {now that we have the direct draw surface 3 object, the temporary direct draw
       surface object is no longer needed}
      TempSurface := nil;
    end;

    {draw the bitmap onto the surface}
    CopyBitmap(Result, TempBitmap, 0, 0, TempBitmap.Width, TempBitmap.Height);

  finally
    TempBitmap.Free;
  end;
end;

{ -- draws the specified bitmap onto the specified surface at the specified coordinates -- }
function CopyBitmap(DSurface: IDirectDrawSurface3; Bitmap: TBitmap; OrgX, OrgY, Width, Height: Integer): HResult;
var
  SurfaceDesc: TDDSurfaceDesc;
  SurfaceDC: HDC;
begin
  {indicate an error if the surface or bitmap parameters contain illegal values}
  if DSurface = nil then
    raise Exception.Create('Cannot copy a bitmap to a nil surface');
  if Bitmap = nil then
    raise Exception.Create('Cannot copy a nil bitmap');

  {make sure the surface is restored}
  DXCheck( DSurface.Restore );

  {initialize the surface description record}
  FillChar(SurfaceDesc, SizeOf(TDDSurfaceDesc), 0);
  SurfaceDesc.dwSize   := SizeOf(TDDSurfaceDesc);
  SurfaceDesc.dwFlags  := DDSD_HEIGHT or DDSD_WIDTH;

  {retrieve the width and height of the surface}
  DXCheck( DSurface.GetSurfaceDesc(SurfaceDesc) );

  {retrieve a surface device context}
  Result := DSurface.GetDC(SurfaceDC);

  try
    {use GDI to stretch the bitmap onto the surface}
    StretchBlt(SurfaceDC, 0, 0, SurfaceDesc.dwWidth, SurfaceDesc.dwHeight, Bitmap.Canvas.Handle, OrgX, OrgY, Width, Height, SRCCOPY);
  finally
    {release the device context}
    DSurface.ReleaseDC(SurfaceDC);
  end;
end;

{ -- reloads a bitmap onto the specified surface. useful when restoring surface contents -- }
function DDReLoadBitmap(DSurface: IDirectDrawSurface3; BitmapName: string): HResult;
var
  TempBitmap: TBitmap;
begin
  {create the temporary bitmap object}
  TempBitmap := TBitmap.Create;
  try
    {load the bitmap}
    TempBitmap.LoadFromFile(BitmapName);

    {draw the bitmap onto the surface}
    Result := CopyBitmap(DSurface, TempBitmap, 0, 0, TempBitmap.Width, TempBitmap.Height);
  finally
    {free the bitmap}
    TempBitmap.Free;
  end;
end;

{ -- converts an RGB color into a physical color for the specified surface's color bit depth -- }
function DDColorMatch(DSurface: IDirectDrawSurface3; Color: TColor): DWORD;
var
  SurfaceDC: HDC;
  SurfaceDesc: TDDSurfaceDesc;
  OrgColor: TColor;
  PixelClr: DWORD;
begin
  OrgColor := CLR_INVALID;

  {use GDI SetPixel to color match for us}
  if (Color <> CLR_INVALID) and (DSurface.GetDC(SurfaceDC) = DD_OK) then
  begin
    OrgColor := GetPixel(SurfaceDC, 0, 0);
    SetPixel(SurfaceDC, 0, 0, Color);
    DSurface.ReleaseDC(SurfaceDC);
  end;

  {now lock the surface so we can read back the converted color}
  SurfaceDesc.dwSize := SizeOf(TDDSurfaceDesc);
  DXCheck( DSurface.Lock(nil, SurfaceDesc, DDLOCK_WAIT, 0) );

  {at this point, DXCheck would have fired an exception if there was a problem,
   so we can assume the surface is locked. retrieve the actual color of the
   pixel based on the surface's color depth}
  PixelClr := DWord(SurfaceDesc.lpSurface^);
  PixelClr := PixelClr and (1 shl SurfaceDesc.ddpfPixelFormat.dwRGBBitCount)-1;
  DSurface.Unlock(nil);

  {now put the color that was there back.}
  if (Color <> CLR_INVALID) and (DSurface.GetDC(SurfaceDC) = DD_OK) then
  begin
    SetPixel(SurfaceDC, 0, 0, OrgColor);
    DSurface.ReleaseDC(SurfaceDC);
  end;

  {return the physical color}
  Result := PixelClr;
end;

{ -- sets the source color key for the specified surface from and RGB color value -- }
function DDSetColorKey(DSurface: IDirectDrawSurface3; Color: TColor): HResult;
var
  ColorKey: TDDColorKey;
begin
  {retrieve a physical color value based on the surface's color depth, and
   set the color key appropriately}
  ColorKey.dwColorSpaceLowValue  := DDColorMatch(DSurface, Color);
  ColorKey.dwColorSpaceHighValue := ColorKey.dwColorSpaceLowValue;
  Result := DSurface.SetColorKey(DDCKEY_SRCBLT, ColorKey);
  DXCheck(Result);
end;

end.
